//! Locker math.
#![deny(clippy::integer_arithmetic)]

use crate::*;
use num_traits::ToPrimitive;

impl LockerParams {
    /// Calculates the amount of voting power an [Escrow] has.
    pub fn calculate_voter_power(&self, escrow: &Escrow, now: i64) -> Option<u64> {
        // invalid `now` argument, should never happen.
        if now == -1 {
            return None;
        }
        if escrow.escrow_started_at == -1 {
            return Some(0);
        }
        // Lockup had zero power before the start time.
        // at the end time, lockup also has zero power.
        if now < escrow.escrow_started_at || now >= escrow.escrow_ends_at {
            return Some(0);
        }

        let seconds_until_lockup_expiry = escrow.escrow_ends_at.checked_sub(now)?;
        // elapsed seconds, clamped to the maximum duration
        let relevant_seconds_until_lockup_expiry = seconds_until_lockup_expiry
            .to_u64()?
            .min(self.max_stake_duration);

        // voting power at max lockup
        let power_if_max_lockup = escrow.amount.checked_mul(self.max_stake_vote_multiplier)?;

        // multiply the max lockup power by the fraction of the max stake duration
        let power = (power_if_max_lockup as u128)
            .checked_mul(relevant_seconds_until_lockup_expiry.into())?
            .checked_div(self.max_stake_duration.into())?
            .to_u64()?;

        Some(power)
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_max_lockup() {
        let locker_params = &LockerParams {
            max_stake_duration: 4 * 365 * 24 * 60 * 60,
            max_stake_vote_multiplier: 10,
            ..LockerParams::default()
        };
        let escrow = Escrow {
            escrow_started_at: 100,
            escrow_ends_at: (100 + locker_params.max_stake_duration).to_i64().unwrap(),
            amount: 100_000,
            ..Escrow::default()
        };
        let power = locker_params.calculate_voter_power(&escrow, 100).unwrap();
        assert_eq!(
            power,
            escrow.amount * locker_params.max_stake_vote_multiplier,
            "full power"
        );

        assert_eq!(
            locker_params
                .calculate_voter_power(&escrow, 100 + 2 * 365 * 24 * 60 * 60)
                .unwrap(),
            escrow.amount * locker_params.max_stake_vote_multiplier / 2,
            "half lockup"
        );

        assert_eq!(
            locker_params
                .calculate_voter_power(&escrow, 100 + 5 * 365 * 24 * 60 * 60)
                .unwrap(),
            0,
            "expired lockup"
        );
    }

    #[test]
    fn test_decreased_max_lockup() {
        // if the max lockup is decreased later, the user should keep their lockup parameters
        // but only be subject to the max.
        let locker_params = &LockerParams {
            max_stake_duration: 4 * 365 * 24 * 60 * 60,
            max_stake_vote_multiplier: 10,
            ..LockerParams::default()
        };
        let escrow = Escrow {
            escrow_started_at: 100,
            escrow_ends_at: 1_000_000 + (locker_params.max_stake_duration).to_i64().unwrap(),
            amount: 100_000,
            ..Escrow::default()
        };
        let power = locker_params.calculate_voter_power(&escrow, 600).unwrap();
        assert_eq!(
            power,
            escrow.amount * locker_params.max_stake_vote_multiplier
        );
    }
}
