//! Accounts structs for the [locked_voter] program.

use crate::*;
use govern::{Proposal, Vote};

/// Accounts for [locked_voter::new_locker].
#[derive(Accounts)]
#[instruction(bump: u8)]
pub struct NewLocker<'info> {
    /// Base.
    pub base: Signer<'info>,

    /// [Electorate].
    #[account(
        init,
        seeds = [
            b"Locker".as_ref(),
            base.key().to_bytes().as_ref()
        ],
        bump = bump,
        payer = payer
    )]
    pub locker: Account<'info, Locker>,

    /// Mint of the token that can be used to join the [Locker].
    pub token_mint: Account<'info, Mint>,

    /// Governor associated with the [Electorate].
    pub governor: Account<'info, Governor>,

    /// Payer of the initialization.
    #[account(mut)]
    pub payer: Signer<'info>,

    /// System program.
    pub system_program: Program<'info, System>,
}

/// Accounts for [locked_voter::new_voter].
#[derive(Accounts)]
#[instruction(bump: u8)]
pub struct NewEscrow<'info> {
    /// [Locker].
    pub locker: Account<'info, Locker>,

    /// [Escrow].
    #[account(
        init,
        seeds = [
            b"Escrow".as_ref(),
            locker.key().to_bytes().as_ref(),
            escrow_owner.key().to_bytes().as_ref()
        ],
        bump = bump,
        payer = payer
    )]
    pub escrow: Account<'info, Escrow>,

    /// Authority of the [Voter] to be created.
    pub escrow_owner: UncheckedAccount<'info>,

    /// Payer of the initialization.
    #[account(mut)]
    pub payer: Signer<'info>,

    /// System program.
    pub system_program: Program<'info, System>,
}

/// Accounts for [locked_voter::lock].
#[derive(Accounts)]
pub struct Lock<'info> {
    /// [Locker].
    #[account(mut)]
    pub locker: Account<'info, Locker>,

    #[account(mut)]
    pub escrow: Account<'info, Escrow>,

    #[account(mut)]
    pub escrow_tokens: Account<'info, TokenAccount>,

    /// Authority of the [Escrow] and of the [source_tokens].
    pub escrow_owner: Signer<'info>,
    #[account(mut)]
    pub source_tokens: Account<'info, TokenAccount>,

    /// Token program.
    pub token_program: Program<'info, Token>,
}

/// Accounts for [locked_voter::exit].
#[derive(Accounts)]
pub struct Exit<'info> {
    #[account(mut)]
    pub locker: Account<'info, Locker>,

    #[account(mut)]
    pub escrow: Account<'info, Escrow>,

    /// Authority of the [Escrow].
    pub escrow_owner: Signer<'info>,
    /// Tokens locked up in the [Escrow].
    #[account(mut)]
    pub escrow_tokens: Account<'info, TokenAccount>,
    /// Destination for the tokens to unlock.
    #[account(mut)]
    pub destination_tokens: Account<'info, TokenAccount>,

    /// Token program.
    pub token_program: Program<'info, Token>,
}

/// Accounts for [locked_voter::activate_proposal].
#[derive(Accounts)]
pub struct ActivateProposal<'info> {
    /// The [Locker].
    pub locker: Account<'info, Locker>,
    /// The [Governor].
    pub governor: Account<'info, Governor>,
    /// The [Proposal].
    #[account(mut)]
    pub proposal: Account<'info, Proposal>,
    /// The user's [Escrow].
    pub escrow: Account<'info, Escrow>,
    /// The [Escrow]'s owner.
    pub escrow_owner: Signer<'info>,
    /// The [govern] program.
    pub govern_program: Program<'info, govern::program::Govern>,
}

/// Accounts for [locked_voter::cast_vote].
#[derive(Accounts)]
pub struct CastVote<'info> {
    /// The [Locker].
    pub locker: Account<'info, Locker>,
    /// The [Escrow] that is voting.
    pub escrow: Account<'info, Escrow>,
    /// Vote delegate of the [Escrow].
    pub vote_delegate: Signer<'info>,

    /// The [Proposal] being voted on.
    #[account(mut)]
    pub proposal: Account<'info, Proposal>,
    /// The [Vote].
    #[account(mut)]
    pub vote: Account<'info, Vote>,

    /// The [Governor].
    pub governor: Account<'info, Governor>,
    /// The [govern] program.
    pub govern_program: Program<'info, govern::program::Govern>,
}
