use std::{collections::HashMap, borrow::Cow, fs};
use crate::{Locale, Sources, LocaleError};

#[derive(Debug, Default)]
pub(crate) struct LocaleCache<'a>(
    HashMap<
        &'a str, Locale
    >
);

impl<'a> LocaleCache<'a> {
    pub fn new() -> Self {
        Default::default()
    }

    #[allow(unused)]
    pub fn look_up(&self, tag: &str) -> crate::Result<&Locale> {
        self.0.get(tag).ok_or(crate::LocaleError::CacheExhaustion)
    }
    
    #[allow(unused)]
    pub fn look_up_or_fetch(&self, tag: &str, sources: &Sources) -> crate::Result<Cow<Locale>> {
        if let Some(locale) = self.0.get(tag) {
            log::trace!("Locale `{}` was found in cache.", tag);

            Ok(Cow::Borrowed(locale))
        } else {
            log::trace!("Locale `{}` is missing in cache, fetching sources.", tag);

            if let Some(path) = sources.get(tag) {
                let v = Cow::Owned(json::from_reader(fs::File::open(path)?)?);
                log::trace!("Loaded locale `{}` from disk.", tag);
                Ok(v)
            } else {
                log::error!("Locale `{}` wasn't found in sources. Aborted.", tag);

                Err(LocaleError::LocaleNotFound)
            }
        }
    }

    #[allow(unused, clippy::cast_ref_to_mut)]
    pub fn look_up_or_cache<'w>(&mut self, tag: &'w str, sources: &Sources) -> crate::Result<&Locale> 
    where
        'w: 'a
    {
        if let Some(locale) = self.0.get(tag) {
            log::trace!("Locale `{}` was found in cache.", tag);

            Ok(locale)
        } else if let Some(path) = sources.get(tag) {
            log::trace!("Locale `{}` is missing in cache, fetching sources.", tag);

            let locale = json::from_reader(fs::File::open(path)?)?;
            log::trace!("Loaded locale `{}` from disk.", tag);
            // let key: &'a str = unsafe { transmute(sources.keys().find(|&key| key == tag).unwrap().as_str()) };

            // Mm should be fine because it's other branch, at least I hope so.
            unsafe { &mut *(self as *const Self as *mut Self) }.0.insert(tag, locale);
            log::trace!("Overwritten locale `{}` in cache.", tag);
            Ok(self.0.get(tag).unwrap())

        } else {
            log::error!("Locale `{}` wasn't found in sources. Aborted.", tag);
                
            Err(LocaleError::LocaleNotFound)
        }
    }

    #[allow(unused)]
    pub fn prefer_recache<'w>(&mut self, tag: &'w str, sources: &Sources) -> crate::Result<&Locale> 
    where
        'w: 'a
    {
        if let Some(path) = sources.get(tag) {
            let locale = json::from_reader(fs::File::open(path)?)?;
            self.0.insert(tag, locale);
            Ok(self.0.get(tag).unwrap())
        } else {
            self.look_up(tag)
        }
    }
}