use rapier3d::dynamics::*;
use rapier3d::geometry::*;
use rapier3d::math::Real;
use rapier3d::na::Vector3;
use rapier3d::pipeline::{ChannelEventCollector, PhysicsPipeline};

pub struct Physics {
    pipeline: PhysicsPipeline,
    gravity: Vector3<Real>,
    integration_parameters: IntegrationParameters,
    broad_phase: BroadPhase,
    narrow_phase: NarrowPhase,
    bodies: RigidBodySet,
    colliders: ColliderSet,
    joints: JointSet,
    ccd_solver: CCDSolver,
    event_handler: Option<ChannelEventCollector>,
}

impl Physics {
    pub fn new() -> Self {
        Self {
            pipeline: PhysicsPipeline::new(),
            gravity: Vector3::new(0.0, -10.0, 0.0),
            integration_parameters: IntegrationParameters::default(),
            broad_phase: BroadPhase::new(),
            narrow_phase: NarrowPhase::new(),
            bodies: RigidBodySet::new(),
            colliders: ColliderSet::new(),
            joints: JointSet::new(),
            ccd_solver: CCDSolver::new(),
            event_handler: None,
        }
    }

    pub fn set_event_handler(&mut self, event_handler: Option<ChannelEventCollector>) {
        self.event_handler = event_handler;
    }

    pub fn step(&mut self) {
        match self.event_handler.as_ref() {
            Some(event_handler) => self.pipeline.step(
                &self.gravity,
                &self.integration_parameters,
                &mut self.broad_phase,
                &mut self.narrow_phase,
                &mut self.bodies,
                &mut self.colliders,
                &mut self.joints,
                &mut self.ccd_solver,
                &(),
                event_handler,
            ),
            None => self.pipeline.step(
                &self.gravity,
                &self.integration_parameters,
                &mut self.broad_phase,
                &mut self.narrow_phase,
                &mut self.bodies,
                &mut self.colliders,
                &mut self.joints,
                &mut self.ccd_solver,
                &(),
                &(),
            ),
        }
    }

    pub fn add(
        &mut self,
        body: RigidBody,
        collider: Collider,
    ) -> (RigidBodyHandle, ColliderHandle) {
        let body_handle = self.bodies.insert(body);
        let collider_handle = self
            .colliders
            .insert(collider, body_handle, &mut self.bodies);
        (body_handle, collider_handle)
    }

    pub fn remove(&mut self, body_handle: RigidBodyHandle) {
        self.bodies
            .remove(body_handle, &mut self.colliders, &mut self.joints);
    }

    pub fn body(&self, body_handle: RigidBodyHandle) -> Option<&RigidBody> {
        self.bodies.get(body_handle)
    }

    pub fn body_mut(&mut self, body_handle: RigidBodyHandle) -> Option<&mut RigidBody> {
        self.bodies.get_mut(body_handle)
    }

    pub fn collider(&self, collider_handle: ColliderHandle) -> Option<&Collider> {
        self.colliders.get(collider_handle)
    }

    pub fn collider_mut(&mut self, collider_handle: ColliderHandle) -> Option<&mut Collider> {
        self.colliders.get_mut(collider_handle)
    }
}
