import {
  Graphics,
  ServerMessage,
  Shape,
  shapeInto,
  Material,
  materialInto,
  Isometry,
} from ".";

import { Mesh } from "three";

export class Game<Input> {
  onmessage?: (output: any) => any;
  onchange?: (entity: number, mesh?: Mesh) => any;
  graphics: Graphics;
  private socket?: WebSocket;
  private meshes: Map<number, Mesh>;

  constructor() {
    this.graphics = new Graphics();
    this.meshes = new Map();
    this.graphics.mount(document.body);
    this.graphics.animate();
  }

  connect(uri: string) {
    this.socket = new WebSocket(uri);
    this.socket.onmessage = ({ data }) => this.onMessage(JSON.parse(data));
    this.socket.onclose = () => (this.socket = undefined);
  }

  send(input: Input) {
    this.socket?.send(JSON.stringify(input));
  }

  private onMessage(msg: ServerMessage) {
    switch (msg.t) {
      case "Components":
        msg.c.forEach(([entity, component]) => {
          switch (component.t) {
            case "Shape":
              this.onShape(entity, component.c);
              break;
            case "Material":
              this.onMaterial(entity, component.c);
              break;
            case "Isometry":
              this.onIsometry(entity, component.c);
              break;
          }
          if (this.onchange) this.onchange(entity, this.meshes.get(entity));
        });
        break;
      case "Custom":
        if (this.onmessage) this.onmessage(msg.c);
        break;
    }
  }

  private onShape(entity: number, shape: Shape | null) {
    const mesh = this.meshes.get(entity);
    if (shape) {
      const geometry = shapeInto(shape);
      if (mesh) mesh.geometry = geometry;
      else {
        const mesh = new Mesh(geometry);
        mesh.receiveShadow = true;
        this.meshes.set(entity, mesh);
        this.graphics.scene.add(mesh);
      }
    } else {
      this.meshes.delete(entity);
      if (mesh) this.graphics.scene.remove(mesh);
    }
  }

  private onMaterial(entity: number, material: Material) {
    const mesh = this.meshes.get(entity);
    if (mesh) mesh.material = materialInto(material);
  }

  private onIsometry(entity: number, isometry: Isometry) {
    const mesh = this.meshes.get(entity);
    if (mesh) {
      mesh.position.set(...isometry.translation);
      mesh.quaternion.set(...isometry.rotation);
    }
  }
}
