use unicode_width::UnicodeWidthChar;

pub(crate) fn soft_breaks(s: &str, width: usize) -> Vec<usize> {
    let mut parser = vte::Parser::new();
    let mut performer = BreakCounter {
        line_starts: vec![],
        current_line: 0,
        cursor: 0,
        max_width: width,
    };
    for b in s.bytes() {
        parser.advance(&mut performer, b);
        performer.cursor += 1;
    }
    performer.line_starts
}

#[derive(Debug)]
struct BreakCounter {
    line_starts: Vec<usize>,
    current_line: usize,
    cursor: usize,
    max_width: usize,
}

impl vte::Perform for BreakCounter {
    fn print(&mut self, c: char) {
        if self.current_line % self.max_width == 0 {
            self.line_starts.push(self.cursor);
        }
        self.current_line += c.width().unwrap_or(0);
    }
    fn execute(&mut self, byte: u8) {
        if byte == b'\n' {
            if self.current_line == 0 {
                // Here we only allow exactly 0, no modulo.  That's because
                // a newline which wraps without any other chars is ignored.
                self.line_starts.push(self.cursor);
            }
            self.current_line = 0;
        }
    }
    fn hook(&mut self, _: &vte::Params, _: &[u8], _: bool, _: char) {}
    fn put(&mut self, _: u8) {}
    fn unhook(&mut self) {}
    fn osc_dispatch(&mut self, _: &[&[u8]], _: bool) {}
    fn csi_dispatch(&mut self, _: &vte::Params, _: &[u8], _: bool, _: char) {}
    fn esc_dispatch(&mut self, _: &[u8], _: bool, _: u8) {}
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_ambiguous() {
        assert_eq!(soft_breaks("foobar", 3), vec![0, 3]);
        assert_eq!(soft_breaks("foobar\n", 3), vec![0, 3]);
        assert_eq!(soft_breaks("foobar\nqux", 3), vec![0, 3, 7]);
        assert_eq!(soft_breaks("foobar\nqux\n", 3), vec![0, 3, 7]);
        assert_eq!(soft_breaks("foo\nbar\n", 3), vec![0, 4]);
    }

    #[test]
    fn test_simple() {
        assert_eq!(soft_breaks("foo", 100), vec![0]);
        assert_eq!(soft_breaks("foobarqux", 5), vec![0, 5]);
        assert_eq!(soft_breaks("foobarqux\n", 5), vec![0, 5]);
        assert_eq!(soft_breaks("foo\nbar\nqux", 100), vec![0, 4, 8]);
        assert_eq!(
            soft_breaks("foo\nfoobarquxzap\nfoo", 5),
            vec![0, 4, 9, 14, 17]
        );
        assert_eq!(&"foo\nfoobarquxzap\nfoo"[0..4], "foo\n");
        assert_eq!(&"foo\nfoobarquxzap\nfoo"[4..9], "fooba");
        assert_eq!(&"foo\nfoobarquxzap\nfoo"[9..14], "rquxz");
        assert_eq!(&"foo\nfoobarquxzap\nfoo"[14..17], "ap\n");
        assert_eq!(&"foo\nfoobarquxzap\nfoo"[17..], "foo");
    }

    #[test]
    fn test_rn() {
        assert_eq!(soft_breaks("foo\r\nbar\r\nqux", 100), vec![0, 5, 10]);
    }

    #[test]
    #[ignore] // FIXME: This is a bug
    fn test_control_chars() {
        //       linebreaks:
        //       V...-----------V--...---
        let s = "\x1B[31;1;4mHello\x1B[0m";
        //       0...00000000011111...111
        //       0...12345678901234...567
        eprintln!("{}", s);
        assert_eq!(soft_breaks(s, 3), vec![0, 12]);
    }

    #[test]
    fn test_empty_lines() {
        assert_eq!(soft_breaks("\n\n\n", 3), vec![0, 1, 2]);
        assert_eq!(soft_breaks("foo\n\n", 3), vec![0, 4]);
    }

    #[test]
    #[ignore] // FIXME: This is a bug
    fn test_cjk() {
        let s = "本日はいかがお過ごしでしょうか？";
        assert_eq!(&s[0..24], "本日はいかがお過");
        assert_eq!(&s[24..], "ごしでしょうか？");
        assert_eq!(soft_breaks(s, 16), vec![0, 24]);
    }
}
