/*! Print stuff to the terminal, remove it, repeat!

Here's how to use it:

```
use liveterm::TermPrinter;
use std::fmt::Write;
# fn main() -> Result<(), Box<dyn std::error::Error>> {
let mut tp = TermPrinter::new(std::io::stdout());
for i in 0..10 {
    tp.clear()?;              // clear what we draw last time
    tp.buf.clear();           // clear the buffer
    write!(tp.buf, "{}", i)?; // fill the buffer
    tp.print()?;              // draw the buffer
}
# Ok(())
# }
```
*/

mod soft_breaks;

use crate::soft_breaks::soft_breaks;
use crossterm::{cursor, terminal, terminal::ClearType, QueueableCommand};
use std::io::Write;

type Result<T> = std::result::Result<T, std::io::Error>;

/// Wraps stdout, allowing you to clear stuff you previously wrote.
pub struct TermPrinter<W> {
    wtr: W,
    /// Write the stuff you want to print/clear into this buffer.
    pub buf: String,
    /// Last time, we started at this byte in `buf` and printed to the end.
    last_print_start: usize,
}

impl<W: Write> TermPrinter<W> {
    pub fn new(wtr: W) -> TermPrinter<W> {
        TermPrinter {
            wtr,
            buf: String::new(),
            last_print_start: 0,
        }
    }

    /// Clear the lines we wrote previously.
    ///
    /// Make sure `buf` still has the same contents as the last time you
    /// called [`TermPrinter::print()`].
    ///
    /// Note that in some cases this is impossible: for instance, if we print N
    /// lines and then the user resizes the terminal to something less than N,
    /// those lines _will_ end up in the scrollback buffer - there's nothing
    /// we can do about that (without switching to the alternate screen).
    pub fn clear(&mut self) -> Result<()> {
        // Looks like MoveToPreviousLine(0) still moves up one line, so we
        // need to guard the 0 case
        if self.last_print_start != self.buf.len() {
            let (width, _) = terminal::size()?;
            let line_starts = soft_breaks(&self.buf[self.last_print_start..], width as usize);
            let n = line_starts.len() as u16;
            self.wtr
                .queue(cursor::MoveToPreviousLine(n))?
                .queue(terminal::Clear(ClearType::FromCursorDown))?;
            self.last_print_start = self.buf.len();
        }
        Ok(())
    }

    /// Print the visible tail of `buf` to `wtr`.
    ///
    /// Stuff that scrolls off the top of the terminal end up in the scrollback
    /// buffer, where we can't clear it.  Therefore this method will only
    /// write as many lines as the terminal currently has room for.
    pub fn print(&mut self) -> Result<()> {
        if self.buf.is_empty() {
            return Ok(());
        }
        let (width, height) = terminal::size()?;
        let line_starts = soft_breaks(&self.buf, width as usize);
        let len = line_starts.len();
        let n = (height as usize - 1).min(len);
        let start = line_starts[len - n];
        self.wtr.write_all(&self.buf.as_bytes()[start..])?;
        self.wtr.flush()?;
        self.last_print_start = start;
        Ok(())
    }

    /// Print all of `buf` to `wtr`.
    ///
    /// After this we can't reliably clear what we've written (since it
    /// may have gone off the top of the screen).  Hence, this method drops
    /// the [`TermPrinter`].
    pub fn print_all(mut self) -> Result<()> {
        self.wtr.write_all(self.buf.as_bytes())?;
        self.wtr.flush()?;
        Ok(())
    }
}
