#![feature(test)]

extern crate test;
extern crate littlewing;

use test::Bencher;

use littlewing::color;
use littlewing::eval::Eval;
use littlewing::fen::FEN;
use littlewing::game::Game;
use littlewing::piece_move_generator::PieceMoveGenerator;
use littlewing::piece_move_notation::PieceMoveNotation;
use littlewing::search::Search;

#[bench]
fn bench_next_move(b: &mut Bencher) {
    let mut game = Game::from_fen("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1").unwrap();

    b.iter(|| {
        let mut n = 0;
        game.moves.clear();
        while let Some(_) = game.next_move() {
            n += 1;
        }
        n
    })
}

#[bench]
fn bench_next_move_without_ordering(b: &mut Bencher) {
    let mut game = Game::from_fen("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1").unwrap();
    game.moves.skip_ordering = true;

    b.iter(|| {
        let mut n = 0;
        game.moves.clear();
        while let Some(_) = game.next_move() {
            n += 1;
        }
        n
    })
}

#[bench]
fn bench_make_undo_move(b: &mut Bencher) {
    let mut game = Game::from_fen("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1").unwrap();
    let m = game.move_from_lan("e2e4");

    b.iter(|| {
        game.make_move(m);
        game.undo_move(m);
    })
}

#[bench]
fn bench_eval(b: &mut Bencher) {
    let game = Game::from_fen("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1").unwrap();

    b.iter(|| {
        game.eval()
    })
}

#[bench]
fn bench_eval_material(b: &mut Bencher) {
    let game = Game::from_fen("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1").unwrap();

    b.iter(|| {
        game.eval_material(color::WHITE)
    })
}

#[bench]
fn bench_see(b: &mut Bencher) {
    let mut game = Game::from_fen("rnbqkb1r/pp2pppp/2p2n2/1B1p4/4P3/2N5/PPPP1PPP/R1BQK1NR w KQkq - 0 4").unwrap();
    let m = game.move_from_lan("c2d5");

    b.iter(|| {
        game.see(m)
    })
}

#[bench]
fn bench_search(b: &mut Bencher) {
    let mut game = Game::from_fen("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1").unwrap();

    b.iter(|| {
        game.search(1..5)
    })
}

#[bench]
fn bench_perft(b: &mut Bencher) {
    let mut game = Game::from_fen("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1").unwrap();
    b.iter(|| {
        game.perft(3)
    });
}

#[bench]
fn bench_move_from_lan(b: &mut Bencher) {
    let mut game = Game::from_fen("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1").unwrap();
    b.iter(|| {
        game.move_from_lan("e2e4");
    });
}

#[bench]
fn bench_move_from_san(b: &mut Bencher) {
    let mut game = Game::from_fen("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1").unwrap();
    b.iter(|| {
        game.move_from_san("e4");
    });
}

#[bench]
fn bench_tt_16mb_get(b: &mut Bencher) {
    let mut game = Game::from_fen("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1").unwrap();
    let m = game.move_from_lan("e2e4");
    game.tt_resize(16 << 20); // 16 MB
    game.search(1..5);
    game.make_move(m);
    let hash = game.positions.top().hash;
    b.iter(|| {
        if let Some(t) = game.tt.get(hash) {
            t.score()
        } else {
            0
        }
    })
}

#[bench]
fn bench_tt_256mb_get(b: &mut Bencher) {
    let mut game = Game::from_fen("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1").unwrap();
    let m = game.move_from_lan("e2e4");
    game.tt_resize(256 << 20); // 256 MB
    game.search(1..5);
    game.make_move(m);
    let hash = game.positions.top().hash;
    b.iter(|| {
        if let Some(t) = game.tt.get(hash) {
            t.score()
        } else {
            0
        }
    })
}
