use clap::{AppSettings, Parser, Subcommand};
use std::path::PathBuf;

#[derive(Debug, Parser)]
#[clap(author, version, about)]
#[clap(global_setting(AppSettings::PropagateVersion))]
#[clap(global_setting(AppSettings::InferLongArgs))]
#[clap(global_setting(AppSettings::InferSubcommands))]
#[clap(global_setting(AppSettings::ArgsNegateSubcommands))]
#[clap(global_setting(AppSettings::UseLongFormatForHelpSubcommand))]
pub struct Arguments {
    #[clap(short, long, global(true), parse(from_occurrences))]
    /// Make the subcommand more talkative.
    pub verbose: usize,
    /// The sub-command to execute.
    #[clap(subcommand)]
    pub command: Option<Commands>,
    /// The input stream to read Markdown from. Defaults to STDIN.
    #[clap(short, long)]
    pub input: Option<PathBuf>,
    /// The output stream to write matching fenced code block contents to. Defaults to STDOUT.
    /// The directory path to the file must already exist.
    #[clap(short, long)]
    pub output: Option<PathBuf>,
    /// Overwrite the existing contents in the output stream.
    #[clap(long, requires("output"))]
    pub overwrite: bool,
    #[clap(short, long)]
    /// The language that the fenced code blocks must match to be included in the output.
    pub language: Option<String>,
    #[clap(short, long, requires("language"))]
    /// Require fenced code blocks have a language to be included in the output.
    pub required: bool,
}

impl Arguments {
    /// Creates a new [`Arguments`] instance from the command-line arguments.
    pub fn from_args() -> Self {
        Arguments::parse()
    }
}

#[derive(Debug, Parser)]
/// Walks a directory tree, extracting each matching file found during the walk and outputting the contents to the output directory with the `.md` extension removed.
pub struct WalkCommand {
    /// The input directory to read Markdown from. Defaults to the current directory.
    #[clap(short, long, default_value = ".")]
    pub input: PathBuf,
    /// The output directory to write matching fenced code block contents to.
    /// The directory path to the file must already exist.
    #[clap(short, long)]
    pub output: PathBuf,
    /// Overwrite any existing files in the output directory.
    #[clap(long)]
    pub overwrite: bool,
    /// The file extension used to filter the files during the walk.
    /// Only files matching `.<EXTENSION>.md` will be extracted to the output directory.
    #[clap(short, long)]
    pub extension: String,
    #[clap(short, long)]
    /// The language that the fenced code blocks must match to be included in the output.
    pub language: Option<String>,
    #[clap(short, long, requires("language"))]
    /// Require fenced code blocks have a language to be included in the output.
    pub required: bool,
}

#[derive(Debug, Subcommand)]
#[clap()]
pub enum Commands {
    Walk(WalkCommand),
}
