////!
////! LiteKV -- A tiny key-value store with a simple REST API backed by SQLite.
////! Copyright (c) 2021 SilentByte <https://silentbyte.com/>
////!

use actix_web::http::StatusCode;
use actix_web::ResponseError;

impl std::fmt::Debug for ApiError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        error_chain_fmt(self, f)
    }
}

fn error_chain_fmt(
    e: &impl std::error::Error,
    f: &mut std::fmt::Formatter<'_>,
) -> std::fmt::Result {
    writeln!(f, "{}\n", e)?;
    let mut current = e.source();
    while let Some(cause) = current {
        writeln!(f, "Caused by:\n\t{}", cause)?;
        current = cause.source();
    }
    Ok(())
}

#[derive(thiserror::Error)]
pub enum ApiError {
    #[error(transparent)]
    UnknownError(#[from] anyhow::Error),

    #[error("Value with key '{0}' could not be found")]
    ValueNotFound(String),

    #[error("Data Store is in read-only mode")]
    ReadonlyDataStore,
}

impl ResponseError for ApiError {
    fn status_code(&self) -> StatusCode {
        match self {
            ApiError::UnknownError(_) => StatusCode::INTERNAL_SERVER_ERROR,
            ApiError::ValueNotFound(_) => StatusCode::NOT_FOUND,
            ApiError::ReadonlyDataStore => StatusCode::FORBIDDEN,
        }
    }
}
