#![allow(unused)]

use path::PathBuf;
use std::borrow::Cow;
use std::fs::{File, read_to_string};
use std::io;
use std::io::{BufRead, BufReader};
use std::path;
use std::thread;
use std::time::{Duration, Instant};

use anyhow::{Context, Result};
use colored::Colorize;
use console::Emoji;
use indicatif::{FormattedDuration, HumanDuration, ProgressFinish, ProgressStyle};
use itertools::Itertools;
use log::{error, info, warn};
use pretty_env_logger::env_logger::TimestampPrecision;
use structopt::StructOpt;
use structopt_flags::LogLevel;

use lipen_grrs::find_matches;

/// Search for a pattern in a file and display the lines that contain it.
#[derive(StructOpt, Debug)]
struct Cli {
    /// The pattern to look for.
    pattern: String,

    /// The path to the file to read.
    #[structopt(parse(from_os_str))]
    path: PathBuf,

    /// The verbosity level.
    #[structopt(flatten)]
    verbose: structopt_flags::Verbose,
}

static SPARKLE: Emoji<'_, '_> = Emoji("✨ ", ":-)");

fn main() -> Result<()> {
    let started = Instant::now();
    let args: Cli = Cli::from_args();

    // Setup logging
    // env_logger::builder().filter_level(args.verbose.get_level_filter()).try_init()?;
    let level_filter = args.verbose.get_level_filter();
    pretty_env_logger::formatted_timed_builder()
        .format_timestamp(Some(TimestampPrecision::Seconds))
        .filter_level(level_filter)
        .try_init()?;

    info!("args = {:#?}", args);

    info!("such information");
    warn!("o_O");
    error!("much error");

    let pb = indicatif::ProgressBar::new_spinner();
    pb.set_style(ProgressStyle::default_spinner()
        .tick_chars("⠁⠂⠄⡀⢀⠠⠐⠈ ")
        .template("{spinner} Found: {pos}"));

    let matches = find_matches(&args.path, &args.pattern)?;
    for (number, m) in matches.enumerate() {
        let s = format!("{}: {}", number + 1, m.red().bold());
        if pb.is_hidden() {
            println!("{}", s);
        } else {
            pb.println(s);
        }
        pb.inc(1);
        thread::sleep(Duration::from_millis(10))
    }
    pb.abandon();
    info!("Found {} matching lines", pb.position());

    println!("{} Done in {:.1}s", SPARKLE, started.elapsed().as_secs_f32());
    Ok(())
}
