// This file is part of linux-support. It is subject to the license terms in the COPYRIGHT file found in the top-level directory of this distribution and at https://raw.githubusercontent.com/lemonrock/linux-support/master/COPYRIGHT. No part of linux-support, including this file, may be copied, modified, propagated, or distributed except according to the terms contained in the COPYRIGHT file.
// Copyright © 2020 The developers of linux-support. See the COPYRIGHT file in the top-level directory of this distribution and at https://raw.githubusercontent.com/lemonrock/linux-support/master/COPYRIGHT.


/// Field ordering based on order as generated by `inet6_fill_ifaddr()` in Linux source `addrconf.c` (for Internet Protocol version 6).
#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub(crate) struct GetAddressProcessingMessageState<IPA: InternetProtocolAddress>
{
	pub(crate) unicast_common: GetAddressProcessingMessageStateUnicastCommon<IPA>,
	
	/// May be set for Internet Protocol version 4 (ie can be `Some` or `None`); always `None` for Internet Protocol version 6.
	pub(crate) interface_name: Option<NetworkInterfaceName>,
	
	/// May be set for Internet Protocol version 4 (ie can be `Some` or `None`); always `None` for Internet Protocol version 6.
	pub(crate) broadcast_address: Option<IPA>,
}

#[allow(missing_docs)]
impl<IPA: InternetProtocolAddress> GetAddressProcessingMessageState<IPA>
{
	#[inline(always)]
	pub(crate) fn new(message_header: &ifaddrmsg) -> Result<Self, String>
	{
		Ok
		(
			Self
			{
				unicast_common: GetAddressProcessingMessageStateUnicastCommon::new(message_header)?,
				interface_name: None,
				broadcast_address: None,
			}
		)
	}
}

#[allow(missing_docs)]
impl GetAddressProcessingMessageState<in_addr>
{
	#[inline(always)]
	pub(crate) fn to_processed_message(self) -> Result<GetInternetProtocolVersion4AddressMessageData, String>
	{
		let unicast_common = self.unicast_common.to_processed_message()?;
		
		Ok
		(
			GetInternetProtocolVersion4AddressMessageData
			{
				unicast_common,
				
				interface_name: self.interface_name,
				
				broadcast_address: self.broadcast_address,
			}
		)
	}
}

#[allow(missing_docs)]
impl GetAddressProcessingMessageState<in6_addr>
{
	#[inline(always)]
	pub(crate) fn to_processed_message(self) -> Result<GetInternetProtocolVersion6AddressMessageData, String>
	{
		let unicast_common = self.unicast_common.to_processed_message()?;
		
		debug_assert!(self.interface_name.is_none());
		
		debug_assert!(self.broadcast_address.is_none());
		
		Ok
		(
			GetInternetProtocolVersion6AddressMessageData
			{
				unicast_common,
			}
		)
	}
}
