// cmdline.rs
//
// Copyright 2019 Alberto Ruiz <aruiz@gnome.org>
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.
//
// SPDX-License-Identifier: MPL-2.0

#![cfg_attr(not(feature = "std"), no_std)]

#![cfg(not(feature = "std"))]
extern crate alloc;

#[cfg(not(feature = "std"))]
use alloc::string::String;
#[cfg(not(feature = "std"))]
use alloc::vec::Vec;

pub type CmdlineParam = Vec<Option<String>>;
pub type Cmdline = Vec<(String, CmdlineParam)>;

pub fn cmdline_parse(args: &str) -> Result<Cmdline, &'static str> {
    match Cmdline::parse(args) {
        Ok(cmdline) => Ok(cmdline),
        Err(_) => Err("could not parse kernel cmdline"),
    }
}

pub trait VecOfTuplesAsDict {
    fn get<'a>(&'a mut self, key: &String) -> Option<&'a mut CmdlineParam>;
    fn take_from_key(&mut self, key: &str) -> Option<CmdlineParam>;
    fn entry_or_insert<'a>(&'a mut self, key: String, insert: CmdlineParam)
        -> &'a mut CmdlineParam;
}

impl VecOfTuplesAsDict for Cmdline {
    fn get<'a>(&'a mut self, key: &String) -> Option<&'a mut CmdlineParam> {
        for item in self {
            if &item.0 == key {
                return Some(&mut item.1);
            }
        }
        None
    }

    fn take_from_key(&mut self, key: &str) -> Option<CmdlineParam> {
        match { self.iter().position(|x| &x.0 == key) } {
            Some(pos) => Some(self.remove(pos).1),
            None => None,
        }
    }

    fn entry_or_insert<'a>(
        &'a mut self,
        key: String,
        insert: CmdlineParam,
    ) -> &'a mut CmdlineParam {
        let pos = { self.iter().position(|(k, _)| k == &key) };

        match pos {
            Some(index) => &mut self[index].1,
            None => {
                self.push((key, insert));
                let len = { self.len() };
                &mut self[len - 1].1
            }
        }
    }
}

pub trait CmdlineStore {
    fn cmdline_store(&mut self, cmdline: &Cmdline) -> Result<(), &'static str>;
    fn cmdline(&self) -> Result<Cmdline, &'static str>;
}

pub trait CmdlineMut {
    fn parse(buffer: &str) -> Result<Cmdline, (usize, &'static str)>;
    fn render(&self) -> Result<String, &'static str>;
    fn add_param(&mut self, key: String, value: Option<String>);
}

impl CmdlineMut for Cmdline {
    fn parse(buffer: &str) -> Result<Cmdline, (usize, &'static str)> {
        #[derive(Debug)]
        enum Scope {
            InValueQuoted,
            InValueUnquoted,
            InKey,
            InEqual,
            InSpace,
        }

        let mut key = String::new();
        let mut value = String::new();

        let mut result = Cmdline::new();
        let mut scope = Scope::InSpace;

        let mut i: usize = 0;
        for c in buffer.chars() {
            match c {
                ' ' => match scope {
                    Scope::InValueQuoted => {
                        value.push(c);
                    }
                    Scope::InValueUnquoted => {
                        result.add_param(key.drain(..).collect(), Some(value.drain(..).collect()));
                        scope = Scope::InSpace;
                    }
                    Scope::InSpace => {}
                    Scope::InEqual => {
                        return Err((i, "empty parameter value"));
                    }
                    Scope::InKey => {
                        result.add_param(key.drain(..).collect(), None);
                    }
                },
                '"' => match scope {
                    Scope::InValueQuoted => {
                        scope = Scope::InValueUnquoted;
                    }
                    Scope::InEqual => {
                        scope = Scope::InValueQuoted;
                    }
                    Scope::InKey => {
                        return Err((i, "quote in parameter name"));
                    }
                    Scope::InValueUnquoted => {
                        scope = Scope::InValueQuoted;
                    }
                    Scope::InSpace => {
                        return Err((i, "quote after unquoted space"));
                    }
                },
                '=' => match scope {
                    Scope::InKey => {
                        scope = Scope::InEqual;
                    }
                    Scope::InValueQuoted | Scope::InValueUnquoted => {
                        value.push(c);
                    }
                    Scope::InEqual => {
                        scope = Scope::InValueUnquoted;
                        value.push(c)
                    }
                    Scope::InSpace => {
                        return Err((i, "equals after space"));
                    }
                },
                _ => match scope {
                    Scope::InKey => {
                        key.push(c);
                    }
                    Scope::InValueQuoted => {
                        value.push(c);
                    }
                    Scope::InValueUnquoted => {
                        value.push(c);
                    }
                    Scope::InSpace => {
                        scope = Scope::InKey;
                        key.push(c);
                    }
                    Scope::InEqual => {
                        scope = Scope::InValueUnquoted;
                        value.push(c);
                    }
                },
            };
            i += 1;
        }

        match scope {
            Scope::InKey => {
                result.add_param(key.drain(..).collect(), None);
            }
            Scope::InValueQuoted => {
                return Err((i, "unclosed quote in parameter value"));
            }
            Scope::InValueUnquoted => {
                result.add_param(key.drain(..).collect(), Some(value.drain(..).collect()))
            }
            Scope::InEqual => {
                return Err((i, "empty parameter value"));
            }
            Scope::InSpace => {}
        }

        Ok(result)
    }

    fn add_param(&mut self, key: String, value: Option<String>) {
        let vec = self.entry_or_insert(key, Vec::new());
        vec.push(value);
    }

    fn render(&self) -> Result<String, &'static str> {
        let mut render = String::new();
        for (param, values) in self {
            for value in values {
                match value {
                    Some(value) => {
                        render.push_str(&param);
                        render.push('=');
                        if value.contains('"') {
                            return Err("cannot escape quote character");
                        }
                        if value.contains(' ') {
                            render.push('"');
                            render.push_str(&value);
                            render.push('"');
                        } else {
                            render.push_str(&value);
                        }
                    }
                    _ => {
                        render.push_str(&param);
                    }
                }
                render.push(' ');
            }
        }
        render.pop();

        Ok(render)
    }
}

pub trait CmdlineHandler {
    fn cmdline_render(&self) -> Result<String, &'static str>;
    fn cmdline_set(&mut self, params: &[String]) -> Result<(), &'static str>;
    fn cmdline_get(&self, param: &str) -> Result<CmdlineParam, &'static str>;
    fn cmdline_add(&mut self, params: &[String]) -> Result<(), &'static str>;
    fn cmdline_remove(&mut self, params: &[String]) -> Result<(), &'static str>;
    fn cmdline_clear(&mut self, param: &[String]) -> Result<(), &'static str>;
}

impl<T> CmdlineHandler for T
where
    T: CmdlineStore,
{
    fn cmdline_render(&self) -> Result<String, &'static str> {
        let cmdline = self.cmdline()?;
        cmdline.render()
    }

    fn cmdline_set(&mut self, params: &[String]) -> Result<(), &'static str> {
        let params = cmdline_parse(&params.join(" ").as_str())?;
        let mut cmdline: Cmdline = match self.cmdline() {
            Ok(cmdline) => Ok(cmdline),
            Err(e) => Err(e)
        }?;
        let mut commit = false;
        for (set_key, set_values) in params {
            let values = cmdline.entry_or_insert(set_key, Vec::new());

            for set_value in set_values {
                if !values.contains(&set_value) {
                    commit = true;
                    values.pop(); // We replace the last instance
                    values.push(set_value);
                }
            }
        }
        match commit {
            true => self.cmdline_store(&cmdline),
            false => Ok(()),
        }
    }

    fn cmdline_get(&self, param: &str) -> Result<CmdlineParam, &'static str> {
        let mut cmdline = self.cmdline()?;
        match { cmdline.take_from_key(param) } {
            Some(values) => Ok(values),
            None => Err("parmeter not present in kernel cmdline")
        }
    }

    fn cmdline_add(&mut self, params: &[String]) -> Result<(), &'static str> {
        let mut cmdline = self.cmdline()?;
        let params = cmdline_parse(&params.join(" ").as_str())?;
        let mut commit = false;
        for (add_key, add_values) in params {
            let mut found = false;
            for (k, params_for_key) in cmdline.iter_mut() {
                if k != &add_key {
                    continue;
                };
                found = true;
                for val in &add_values {
                    if !params_for_key.contains(&val) {
                        commit = true;
                        params_for_key.push(val.clone());
                    }
                }
            }

            if !found {
                commit = true;
                cmdline.push((add_key.clone(), add_values.clone()));
            }
        }

        match commit {
            true => self.cmdline_store(&cmdline),
            false => Ok(()),
        }
    }

    fn cmdline_remove(&mut self, params: &[String]) -> Result<(), &'static str> {
        let mut cmdline = self.cmdline()?;
        let params = cmdline_parse(&params.join(" ").as_str())?;
        let mut commit = false;

        for (rem_key, rem_values) in params {
            let params_for_key = cmdline.entry_or_insert(rem_key, rem_values.clone());
            for val in rem_values {
                while let Some(index) = params_for_key.iter().position(|v| v == &val) {
                    commit = true;
                    params_for_key.remove(index);
                }
            }
        }

        match commit {
            true => self.cmdline_store(&cmdline),
            false => Ok(()),
        }
    }

    fn cmdline_clear(&mut self, params: &[String]) -> Result<(), &'static str> {
        let mut cmdline = self.cmdline()?;
        for param in params {
            if let Some(_) = cmdline.take_from_key(param) {
                self.cmdline_store(&cmdline)?;
            }
        }
        Ok(())
    }
}

#[cfg(test)]
mod cmdline_tests {
    use super::*;
    #[cfg(not(feature = "std"))]
    use alloc::vec;

    struct SimpleCmdlineStore {
        content: String,
    }

    impl SimpleCmdlineStore {
        fn new() -> SimpleCmdlineStore {
            SimpleCmdlineStore {
                content: String::new(),
            }
        }
    }

    impl CmdlineStore for SimpleCmdlineStore {
        fn cmdline_store(&mut self, cmdline: &Cmdline) -> Result<(), &'static str> {
            self.content = cmdline.render()?;
            Ok(())
        }
        fn cmdline(&self) -> Result<Cmdline, &'static str> {
            cmdline_parse(self.content.as_str())
        }
    }

    #[test]
    fn cmdline_handler_trait_for_bls() {
        // Set arguments
        let mut a = SimpleCmdlineStore::new();
        let args = "a=b b c=d"
            .split(" ")
            .map(|arg| String::from(arg))
            .collect::<Vec<String>>();
        a.cmdline_set(&args)
            .expect("Could not set cmdline arguments into BLS entry");

        // Assert with cmdline_get
        assert_eq!(
            a.cmdline_get("a")
                .expect("Could not get 'a' cmdline argument"),
            vec![Some(String::from("b"))]
        );
        assert_eq!(
            a.cmdline_get("b")
                .expect("Could not get 'b' cmdline argument"),
            vec![None]
        );
        assert_eq!(
            a.cmdline_get("c")
                .expect("Could not get 'c' cmdline argument"),
            vec![Some(String::from("d"))]
        );

        // Assert with contents
        assert_eq!(a.content.as_str(), "a=b b c=d");

        // Add arguments
        let args = vec![String::from("a=c"), String::from("a=d"), String::from("a")];
        a.cmdline_add(&args)
            .expect("Could not add 'a=c a=d a' as cmdline args");

        // Assert with cmdline_get
        assert_eq!(
            a.cmdline_get("a")
                .expect("Could not get 'a' cmdline argument"),
            vec![
                Some(String::from("b")),
                Some(String::from("c")),
                Some(String::from("d")),
                None
            ]
        );

        // Assert with cmdline_render
        assert_eq!(
            a.cmdline_render()
                .expect("Could not render cmdline from env file"),
            String::from("a=b a=c a=d a b c=d")
        );

        // Assert with contents
        assert_eq!(a.content.as_str(), "a=b a=c a=d a b c=d");

        // Remove one argument assignment
        let args = "a=c"
            .split(" ")
            .map(|arg| String::from(arg))
            .collect::<Vec<String>>();
        a.cmdline_remove(&args)
            .expect("Could not set cmdline arguments into BLS entry");
        assert_eq!(a.content.as_str(), "a=b a=d a b c=d");

        // Clear a and b arguments
        let args = "a b"
            .split(" ")
            .map(|arg| String::from(arg))
            .collect::<Vec<String>>();
        a.cmdline_clear(&args)
            .expect("Could not set cmdline arguments into BLS entry");
        assert_eq!(a.content.as_str(), "c=d");
    }

    #[test]
    fn cmdline_parse_test() {
        let test = "a=test";
        assert_eq!(
            Cmdline::parse(test),
            Ok(vec![(String::from("a"), vec![Some(String::from("test"))])])
        );

        let test = "a=te\"s\"t";
        assert_eq!(
            Cmdline::parse(test),
            Ok(vec![(String::from("a"), vec![Some(String::from("test"))])])
        );

        let test = "a b c";
        assert_eq!(
            Cmdline::parse(test),
            Ok(vec![
                (String::from("a"), vec![None]),
                (String::from("b"), vec![None]),
                (String::from("c"), vec![None])
            ])
        );

        let test = "a=test a a=test2 c a=test3";
        assert_eq!(
            Cmdline::parse(test),
            Ok(vec![
                (
                    String::from("a"),
                    vec![
                        Some(String::from("test")),
                        None,
                        Some(String::from("test2")),
                        Some(String::from("test3"))
                    ]
                ),
                (String::from("c"), vec![None])
            ])
        );

        let test = "a=3 =asd";
        assert!(Cmdline::parse(test).is_err());

        let test = "a=3 b= ";
        assert!(Cmdline::parse(test).is_err());

        let test = "a=3 b=";
        assert!(Cmdline::parse(test).is_err());

        let test = "\"quoted param\"=should_error";
        assert!(Cmdline::parse(test).is_err());

        let test = "quot\"ed param\"=should_error";
        assert!(Cmdline::parse(test).is_err());

        let test = "arg1 \"quoted param\"=should_error";
        assert!(Cmdline::parse(test).is_err());

        let test = "param=\"unclosed quote";
        assert!(Cmdline::parse(test).is_err());
    }
}
