use freedesktop_entry_parser::ParseError as PE;
use std::{io, num::ParseIntError, path::PathBuf, str::Utf8Error};
use thiserror::Error;
#[cfg(feature = "expand-paths")]
use shellexpand::LookupError;
#[cfg(feature = "expand-paths")]
use std::env::VarError;

pub type Result<T> = std::result::Result<T, LiniconError>;

/// Linicon error
#[derive(Debug, Error)]
pub enum LiniconError {
    /// Theme with `theme name` not found
    #[error("Theme `{0}` not found")]
    ThemeNotFound(String),
    /// Index file for `theme name` not found
    #[error("Index file not found for `{0}`")]
    IndexFileNotFound(String),
    /// Error paring index file at `path`
    #[error("Error parsing index file `{path}`")]
    ParseIndex {
        path: PathBuf,
        #[source]
        source: ParseError,
    },
    #[error("Error opening index file `{path}`")]
    OpenIndex {
        path: PathBuf,
        #[source]
        source: io::Error,
    },
    #[cfg(feature = "expand-paths")]
    #[error("Error expanding path")]
    ExapandPath(#[from] LookupError<VarError>)
}

/// Error from parsing an icon theme index file
#[derive(Debug, Error)]
pub enum ParseError {
    /// Wrapper around [`freedesktop_entry_parser::ParseError`](https://docs.rs/freedesktop_entry_parser/*/freedesktop_entry_parser/errors/enum.ParseError.html)
    #[error(transparent)]
    ParseError(#[from] PE),
    /// Index file conatins invalid UTF-8
    #[error("Index file conatins invalid UTF-8")]
    UTT8Error(#[from] Utf8Error),
    /// Field in the index file contains an invalid numeric value
    #[error("Field in the index file contains an invalid numeric value")]
    ParseIntError(#[from] ParseIntError),
    /// Index file is missing it's header
    #[error("Index file is missing it's header")]
    MissingHeader,
    /// Icon in the index file is missing the size field, which is required
    #[error(
        "Icon {0} in the index file is missing the size field, which is required"
    )]
    MissingSize(String),
    /// Icon in the index file has an invalid type
    #[error("Icon {0} in the index file has an invalid type")]
    InvalidType(String),
    /// Index file doesn't contain any sections
    #[error("Index file doesn't contain any sections")]
    EmptyIndexFile,
    /// The first section of the index file is not titled `Icon Theme`
    #[error("The first section of the index file is not titled `Icon Theme`")]
    NoIconTheme,
}
