// This file is Copyright its original authors, visible in version control
// history.
//
// This file is licensed under the Apache License, Version 2.0 <LICENSE-APACHE
// or http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your option.
// You may not use this file except in accordance with one or both of these
// licenses.

//! The logic to monitor for on-chain transactions and create the relevant claim responses lives
//! here.
//!
//! ChannelMonitor objects are generated by ChannelManager in response to relevant
//! messages/actions, and MUST be persisted to disk (and, preferably, remotely) before progress can
//! be made in responding to certain messages, see [`chain::Watch`] for more.
//!
//! Note that ChannelMonitors are an important part of the lightning trust model and a copy of the
//! latest ChannelMonitor must always be actively monitoring for chain updates (and no out-of-date
//! ChannelMonitors should do so). Thus, if you're building rust-lightning into an HSM or other
//! security-domain-separated system design, you should consider having multiple paths for
//! ChannelMonitors to get out of the HSM and onto monitoring devices.

use bitcoin::blockdata::block::{Block, BlockHeader};
use bitcoin::blockdata::transaction::{TxOut,Transaction};
use bitcoin::blockdata::transaction::OutPoint as BitcoinOutPoint;
use bitcoin::blockdata::script::{Script, Builder};
use bitcoin::blockdata::opcodes;

use bitcoin::hashes::Hash;
use bitcoin::hashes::sha256::Hash as Sha256;
use bitcoin::hash_types::{Txid, BlockHash, WPubkeyHash};

use bitcoin::secp256k1::{Secp256k1,Signature};
use bitcoin::secp256k1::key::{SecretKey,PublicKey};
use bitcoin::secp256k1;

use ln::{PaymentHash, PaymentPreimage};
use ln::msgs::DecodeError;
use ln::chan_utils;
use ln::chan_utils::{CounterpartyCommitmentSecrets, HTLCOutputInCommitment, HTLCType, ChannelTransactionParameters, HolderCommitmentTransaction};
use ln::channelmanager::{BestBlock, HTLCSource};
use ln::onchaintx::{OnchainTxHandler, InputDescriptors};
use chain;
use chain::WatchedOutput;
use chain::chaininterface::{BroadcasterInterface, FeeEstimator};
use chain::transaction::{OutPoint, TransactionData};
use chain::keysinterface::{SpendableOutputDescriptor, StaticPaymentOutputDescriptor, DelayedPaymentOutputDescriptor, Sign, KeysInterface};
use chain::Filter;
use util::logger::Logger;
use util::ser::{Readable, ReadableArgs, MaybeReadable, Writer, Writeable, U48};
use util::byte_utils;
use util::events::Event;

use std::collections::{HashMap, HashSet};
use std::{cmp, mem};
use std::io::Error;
use std::ops::Deref;
use std::sync::Mutex;

/// An update generated by the underlying Channel itself which contains some new information the
/// ChannelMonitor should be made aware of.
#[cfg_attr(any(test, feature = "fuzztarget", feature = "_test_utils"), derive(PartialEq))]
#[derive(Clone)]
#[must_use]
pub struct ChannelMonitorUpdate {
	pub(crate) updates: Vec<ChannelMonitorUpdateStep>,
	/// The sequence number of this update. Updates *must* be replayed in-order according to this
	/// sequence number (and updates may panic if they are not). The update_id values are strictly
	/// increasing and increase by one for each new update, with one exception specified below.
	///
	/// This sequence number is also used to track up to which points updates which returned
	/// ChannelMonitorUpdateErr::TemporaryFailure have been applied to all copies of a given
	/// ChannelMonitor when ChannelManager::channel_monitor_updated is called.
	///
	/// The only instance where update_id values are not strictly increasing is the case where we
	/// allow post-force-close updates with a special update ID of [`CLOSED_CHANNEL_UPDATE_ID`]. See
	/// its docs for more details.
	pub update_id: u64,
}

/// If:
///    (1) a channel has been force closed and
///    (2) we receive a preimage from a forward link that allows us to spend an HTLC output on
///        this channel's (the backward link's) broadcasted commitment transaction
/// then we allow the `ChannelManager` to send a `ChannelMonitorUpdate` with this update ID,
/// with the update providing said payment preimage. No other update types are allowed after
/// force-close.
pub const CLOSED_CHANNEL_UPDATE_ID: u64 = std::u64::MAX;

impl Writeable for ChannelMonitorUpdate {
	fn write<W: Writer>(&self, w: &mut W) -> Result<(), ::std::io::Error> {
		self.update_id.write(w)?;
		(self.updates.len() as u64).write(w)?;
		for update_step in self.updates.iter() {
			update_step.write(w)?;
		}
		Ok(())
	}
}
impl Readable for ChannelMonitorUpdate {
	fn read<R: ::std::io::Read>(r: &mut R) -> Result<Self, DecodeError> {
		let update_id: u64 = Readable::read(r)?;
		let len: u64 = Readable::read(r)?;
		let mut updates = Vec::with_capacity(cmp::min(len as usize, MAX_ALLOC_SIZE / ::std::mem::size_of::<ChannelMonitorUpdateStep>()));
		for _ in 0..len {
			updates.push(Readable::read(r)?);
		}
		Ok(Self { update_id, updates })
	}
}

/// An error enum representing a failure to persist a channel monitor update.
#[derive(Clone, Debug)]
pub enum ChannelMonitorUpdateErr {
	/// Used to indicate a temporary failure (eg connection to a watchtower or remote backup of
	/// our state failed, but is expected to succeed at some point in the future).
	///
	/// Such a failure will "freeze" a channel, preventing us from revoking old states or
	/// submitting new commitment transactions to the counterparty. Once the update(s) which failed
	/// have been successfully applied, ChannelManager::channel_monitor_updated can be used to
	/// restore the channel to an operational state.
	///
	/// Note that a given ChannelManager will *never* re-generate a given ChannelMonitorUpdate. If
	/// you return a TemporaryFailure you must ensure that it is written to disk safely before
	/// writing out the latest ChannelManager state.
	///
	/// Even when a channel has been "frozen" updates to the ChannelMonitor can continue to occur
	/// (eg if an inbound HTLC which we forwarded was claimed upstream resulting in us attempting
	/// to claim it on this channel) and those updates must be applied wherever they can be. At
	/// least one such updated ChannelMonitor must be persisted otherwise PermanentFailure should
	/// be returned to get things on-chain ASAP using only the in-memory copy. Obviously updates to
	/// the channel which would invalidate previous ChannelMonitors are not made when a channel has
	/// been "frozen".
	///
	/// Note that even if updates made after TemporaryFailure succeed you must still call
	/// channel_monitor_updated to ensure you have the latest monitor and re-enable normal channel
	/// operation.
	///
	/// Note that the update being processed here will not be replayed for you when you call
	/// ChannelManager::channel_monitor_updated, so you must store the update itself along
	/// with the persisted ChannelMonitor on your own local disk prior to returning a
	/// TemporaryFailure. You may, of course, employ a journaling approach, storing only the
	/// ChannelMonitorUpdate on disk without updating the monitor itself, replaying the journal at
	/// reload-time.
	///
	/// For deployments where a copy of ChannelMonitors and other local state are backed up in a
	/// remote location (with local copies persisted immediately), it is anticipated that all
	/// updates will return TemporaryFailure until the remote copies could be updated.
	TemporaryFailure,
	/// Used to indicate no further channel monitor updates will be allowed (eg we've moved on to a
	/// different watchtower and cannot update with all watchtowers that were previously informed
	/// of this channel).
	///
	/// At reception of this error, ChannelManager will force-close the channel and return at
	/// least a final ChannelMonitorUpdate::ChannelForceClosed which must be delivered to at
	/// least one ChannelMonitor copy. Revocation secret MUST NOT be released and offchain channel
	/// update must be rejected.
	///
	/// This failure may also signal a failure to update the local persisted copy of one of
	/// the channel monitor instance.
	///
	/// Note that even when you fail a holder commitment transaction update, you must store the
	/// update to ensure you can claim from it in case of a duplicate copy of this ChannelMonitor
	/// broadcasts it (e.g distributed channel-monitor deployment)
	///
	/// In case of distributed watchtowers deployment, the new version must be written to disk, as
	/// state may have been stored but rejected due to a block forcing a commitment broadcast. This
	/// storage is used to claim outputs of rejected state confirmed onchain by another watchtower,
	/// lagging behind on block processing.
	PermanentFailure,
}

/// General Err type for ChannelMonitor actions. Generally, this implies that the data provided is
/// inconsistent with the ChannelMonitor being called. eg for ChannelMonitor::update_monitor this
/// means you tried to update a monitor for a different channel or the ChannelMonitorUpdate was
/// corrupted.
/// Contains a developer-readable error message.
#[derive(Clone, Debug)]
pub struct MonitorUpdateError(pub &'static str);

/// An event to be processed by the ChannelManager.
#[derive(Clone, PartialEq)]
pub enum MonitorEvent {
	/// A monitor event containing an HTLCUpdate.
	HTLCEvent(HTLCUpdate),

	/// A monitor event that the Channel's commitment transaction was broadcasted.
	CommitmentTxBroadcasted(OutPoint),
}

/// Simple structure sent back by `chain::Watch` when an HTLC from a forward channel is detected on
/// chain. Used to update the corresponding HTLC in the backward channel. Failing to pass the
/// preimage claim backward will lead to loss of funds.
#[derive(Clone, PartialEq)]
pub struct HTLCUpdate {
	pub(crate) payment_hash: PaymentHash,
	pub(crate) payment_preimage: Option<PaymentPreimage>,
	pub(crate) source: HTLCSource
}
impl_writeable!(HTLCUpdate, 0, { payment_hash, payment_preimage, source });

/// If an HTLC expires within this many blocks, don't try to claim it in a shared transaction,
/// instead claiming it in its own individual transaction.
pub(crate) const CLTV_SHARED_CLAIM_BUFFER: u32 = 12;
/// If an HTLC expires within this many blocks, force-close the channel to broadcast the
/// HTLC-Success transaction.
/// In other words, this is an upper bound on how many blocks we think it can take us to get a
/// transaction confirmed (and we use it in a few more, equivalent, places).
pub(crate) const CLTV_CLAIM_BUFFER: u32 = 6;
/// Number of blocks by which point we expect our counterparty to have seen new blocks on the
/// network and done a full update_fail_htlc/commitment_signed dance (+ we've updated all our
/// copies of ChannelMonitors, including watchtowers). We could enforce the contract by failing
/// at CLTV expiration height but giving a grace period to our peer may be profitable for us if he
/// can provide an over-late preimage. Nevertheless, grace period has to be accounted in our
/// CLTV_EXPIRY_DELTA to be secure. Following this policy we may decrease the rate of channel failures
/// due to expiration but increase the cost of funds being locked longuer in case of failure.
/// This delay also cover a low-power peer being slow to process blocks and so being behind us on
/// accurate block height.
/// In case of onchain failure to be pass backward we may see the last block of ANTI_REORG_DELAY
/// with at worst this delay, so we are not only using this value as a mercy for them but also
/// us as a safeguard to delay with enough time.
pub(crate) const LATENCY_GRACE_PERIOD_BLOCKS: u32 = 3;
/// Number of blocks we wait on seeing a HTLC output being solved before we fail corresponding inbound
/// HTLCs. This prevents us from failing backwards and then getting a reorg resulting in us losing money.
/// We use also this delay to be sure we can remove our in-flight claim txn from bump candidates buffer.
/// It may cause spurrious generation of bumped claim txn but that's allright given the outpoint is already
/// solved by a previous claim tx. What we want to avoid is reorg evicting our claim tx and us not
/// keeping bumping another claim tx to solve the outpoint.
pub(crate) const ANTI_REORG_DELAY: u32 = 6;
/// Number of blocks before confirmation at which we fail back an un-relayed HTLC or at which we
/// refuse to accept a new HTLC.
///
/// This is used for a few separate purposes:
/// 1) if we've received an MPP HTLC to us and it expires within this many blocks and we are
///    waiting on additional parts (or waiting on the preimage for any HTLC from the user), we will
///    fail this HTLC,
/// 2) if we receive an HTLC within this many blocks of its expiry (plus one to avoid a race
///    condition with the above), we will fail this HTLC without telling the user we received it,
/// 3) if we are waiting on a connection or a channel state update to send an HTLC to a peer, and
///    that HTLC expires within this many blocks, we will simply fail the HTLC instead.
///
/// (1) is all about protecting us - we need enough time to update the channel state before we hit
/// CLTV_CLAIM_BUFFER, at which point we'd go on chain to claim the HTLC with the preimage.
///
/// (2) is the same, but with an additional buffer to avoid accepting an HTLC which is immediately
/// in a race condition between the user connecting a block (which would fail it) and the user
/// providing us the preimage (which would claim it).
///
/// (3) is about our counterparty - we don't want to relay an HTLC to a counterparty when they may
/// end up force-closing the channel on us to claim it.
pub(crate) const HTLC_FAIL_BACK_BUFFER: u32 = CLTV_CLAIM_BUFFER + LATENCY_GRACE_PERIOD_BLOCKS;

// TODO(devrandom) replace this with HolderCommitmentTransaction
#[derive(Clone, PartialEq)]
struct HolderSignedTx {
	/// txid of the transaction in tx, just used to make comparison faster
	txid: Txid,
	revocation_key: PublicKey,
	a_htlc_key: PublicKey,
	b_htlc_key: PublicKey,
	delayed_payment_key: PublicKey,
	per_commitment_point: PublicKey,
	feerate_per_kw: u32,
	htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Signature>, Option<HTLCSource>)>,
}

/// We use this to track counterparty commitment transactions and htlcs outputs and
/// use it to generate any justice or 2nd-stage preimage/timeout transactions.
#[derive(PartialEq)]
struct CounterpartyCommitmentTransaction {
	counterparty_delayed_payment_base_key: PublicKey,
	counterparty_htlc_base_key: PublicKey,
	on_counterparty_tx_csv: u16,
	per_htlc: HashMap<Txid, Vec<HTLCOutputInCommitment>>
}

impl Writeable for CounterpartyCommitmentTransaction {
	fn write<W: Writer>(&self, w: &mut W) -> Result<(), ::std::io::Error> {
		self.counterparty_delayed_payment_base_key.write(w)?;
		self.counterparty_htlc_base_key.write(w)?;
		w.write_all(&byte_utils::be16_to_array(self.on_counterparty_tx_csv))?;
		w.write_all(&byte_utils::be64_to_array(self.per_htlc.len() as u64))?;
		for (ref txid, ref htlcs) in self.per_htlc.iter() {
			w.write_all(&txid[..])?;
			w.write_all(&byte_utils::be64_to_array(htlcs.len() as u64))?;
			for &ref htlc in htlcs.iter() {
				htlc.write(w)?;
			}
		}
		Ok(())
	}
}
impl Readable for CounterpartyCommitmentTransaction {
	fn read<R: ::std::io::Read>(r: &mut R) -> Result<Self, DecodeError> {
		let counterparty_commitment_transaction = {
			let counterparty_delayed_payment_base_key = Readable::read(r)?;
			let counterparty_htlc_base_key = Readable::read(r)?;
			let on_counterparty_tx_csv: u16 = Readable::read(r)?;
			let per_htlc_len: u64 = Readable::read(r)?;
			let mut per_htlc = HashMap::with_capacity(cmp::min(per_htlc_len as usize, MAX_ALLOC_SIZE / 64));
			for _  in 0..per_htlc_len {
				let txid: Txid = Readable::read(r)?;
				let htlcs_count: u64 = Readable::read(r)?;
				let mut htlcs = Vec::with_capacity(cmp::min(htlcs_count as usize, MAX_ALLOC_SIZE / 32));
				for _ in 0..htlcs_count {
					let htlc = Readable::read(r)?;
					htlcs.push(htlc);
				}
				if let Some(_) = per_htlc.insert(txid, htlcs) {
					return Err(DecodeError::InvalidValue);
				}
			}
			CounterpartyCommitmentTransaction {
				counterparty_delayed_payment_base_key,
				counterparty_htlc_base_key,
				on_counterparty_tx_csv,
				per_htlc,
			}
		};
		Ok(counterparty_commitment_transaction)
	}
}

/// When ChannelMonitor discovers an onchain outpoint being a step of a channel and that it needs
/// to generate a tx to push channel state forward, we cache outpoint-solving tx material to build
/// a new bumped one in case of lenghty confirmation delay
#[derive(Clone, PartialEq)]
pub(crate) enum InputMaterial {
	Revoked {
		per_commitment_point: PublicKey,
		counterparty_delayed_payment_base_key: PublicKey,
		counterparty_htlc_base_key: PublicKey,
		per_commitment_key: SecretKey,
		input_descriptor: InputDescriptors,
		amount: u64,
		htlc: Option<HTLCOutputInCommitment>,
		on_counterparty_tx_csv: u16,
	},
	CounterpartyHTLC {
		per_commitment_point: PublicKey,
		counterparty_delayed_payment_base_key: PublicKey,
		counterparty_htlc_base_key: PublicKey,
		preimage: Option<PaymentPreimage>,
		htlc: HTLCOutputInCommitment
	},
	HolderHTLC {
		preimage: Option<PaymentPreimage>,
		amount: u64,
	},
	Funding {
		funding_redeemscript: Script,
	}
}

impl Writeable for InputMaterial  {
	fn write<W: Writer>(&self, writer: &mut W) -> Result<(), ::std::io::Error> {
		match self {
			&InputMaterial::Revoked { ref per_commitment_point, ref counterparty_delayed_payment_base_key, ref counterparty_htlc_base_key, ref per_commitment_key, ref input_descriptor, ref amount, ref htlc, ref on_counterparty_tx_csv} => {
				writer.write_all(&[0; 1])?;
				per_commitment_point.write(writer)?;
				counterparty_delayed_payment_base_key.write(writer)?;
				counterparty_htlc_base_key.write(writer)?;
				writer.write_all(&per_commitment_key[..])?;
				input_descriptor.write(writer)?;
				writer.write_all(&byte_utils::be64_to_array(*amount))?;
				htlc.write(writer)?;
				on_counterparty_tx_csv.write(writer)?;
			},
			&InputMaterial::CounterpartyHTLC { ref per_commitment_point, ref counterparty_delayed_payment_base_key, ref counterparty_htlc_base_key, ref preimage, ref htlc} => {
				writer.write_all(&[1; 1])?;
				per_commitment_point.write(writer)?;
				counterparty_delayed_payment_base_key.write(writer)?;
				counterparty_htlc_base_key.write(writer)?;
				preimage.write(writer)?;
				htlc.write(writer)?;
			},
			&InputMaterial::HolderHTLC { ref preimage, ref amount } => {
				writer.write_all(&[2; 1])?;
				preimage.write(writer)?;
				writer.write_all(&byte_utils::be64_to_array(*amount))?;
			},
			&InputMaterial::Funding { ref funding_redeemscript } => {
				writer.write_all(&[3; 1])?;
				funding_redeemscript.write(writer)?;
			}
		}
		Ok(())
	}
}

impl Readable for InputMaterial {
	fn read<R: ::std::io::Read>(reader: &mut R) -> Result<Self, DecodeError> {
		let input_material = match <u8 as Readable>::read(reader)? {
			0 => {
				let per_commitment_point = Readable::read(reader)?;
				let counterparty_delayed_payment_base_key = Readable::read(reader)?;
				let counterparty_htlc_base_key = Readable::read(reader)?;
				let per_commitment_key = Readable::read(reader)?;
				let input_descriptor = Readable::read(reader)?;
				let amount = Readable::read(reader)?;
				let htlc = Readable::read(reader)?;
				let on_counterparty_tx_csv = Readable::read(reader)?;
				InputMaterial::Revoked {
					per_commitment_point,
					counterparty_delayed_payment_base_key,
					counterparty_htlc_base_key,
					per_commitment_key,
					input_descriptor,
					amount,
					htlc,
					on_counterparty_tx_csv
				}
			},
			1 => {
				let per_commitment_point = Readable::read(reader)?;
				let counterparty_delayed_payment_base_key = Readable::read(reader)?;
				let counterparty_htlc_base_key = Readable::read(reader)?;
				let preimage = Readable::read(reader)?;
				let htlc = Readable::read(reader)?;
				InputMaterial::CounterpartyHTLC {
					per_commitment_point,
					counterparty_delayed_payment_base_key,
					counterparty_htlc_base_key,
					preimage,
					htlc
				}
			},
			2 => {
				let preimage = Readable::read(reader)?;
				let amount = Readable::read(reader)?;
				InputMaterial::HolderHTLC {
					preimage,
					amount,
				}
			},
			3 => {
				InputMaterial::Funding {
					funding_redeemscript: Readable::read(reader)?,
				}
			}
			_ => return Err(DecodeError::InvalidValue),
		};
		Ok(input_material)
	}
}

/// ClaimRequest is a descriptor structure to communicate between detection
/// and reaction module. They are generated by ChannelMonitor while parsing
/// onchain txn leaked from a channel and handed over to OnchainTxHandler which
/// is responsible for opportunistic aggregation, selecting and enforcing
/// bumping logic, building and signing transactions.
pub(crate) struct ClaimRequest {
	// Block height before which claiming is exclusive to one party,
	// after reaching it, claiming may be contentious.
	pub(crate) absolute_timelock: u32,
	// Timeout tx must have nLocktime set which means aggregating multiple
	// ones must take the higher nLocktime among them to satisfy all of them.
	// Sadly it has few pitfalls, a) it takes longuer to get fund back b) CLTV_DELTA
	// of a sooner-HTLC could be swallowed by the highest nLocktime of the HTLC set.
	// Do simplify we mark them as non-aggregable.
	pub(crate) aggregable: bool,
	// Basic bitcoin outpoint (txid, vout)
	pub(crate) outpoint: BitcoinOutPoint,
	// Following outpoint type, set of data needed to generate transaction digest
	// and satisfy witness program.
	pub(crate) witness_data: InputMaterial
}

/// An entry for an [`OnchainEvent`], stating the block height when the event was observed and the
/// transaction causing it.
///
/// Used to determine when the on-chain event can be considered safe from a chain reorganization.
#[derive(PartialEq)]
struct OnchainEventEntry {
	txid: Txid,
	height: u32,
	event: OnchainEvent,
}

impl OnchainEventEntry {
	fn confirmation_threshold(&self) -> u32 {
		self.height + ANTI_REORG_DELAY - 1
	}

	fn has_reached_confirmation_threshold(&self, height: u32) -> bool {
		height >= self.confirmation_threshold()
	}
}

/// Upon discovering of some classes of onchain tx by ChannelMonitor, we may have to take actions on it
/// once they mature to enough confirmations (ANTI_REORG_DELAY)
#[derive(PartialEq)]
enum OnchainEvent {
	/// HTLC output getting solved by a timeout, at maturation we pass upstream payment source information to solve
	/// inbound HTLC in backward channel. Note, in case of preimage, we pass info to upstream without delay as we can
	/// only win from it, so it's never an OnchainEvent
	HTLCUpdate {
		htlc_update: (HTLCSource, PaymentHash),
	},
	MaturingOutput {
		descriptor: SpendableOutputDescriptor,
	},
}

const SERIALIZATION_VERSION: u8 = 1;
const MIN_SERIALIZATION_VERSION: u8 = 1;

#[cfg_attr(any(test, feature = "fuzztarget", feature = "_test_utils"), derive(PartialEq))]
#[derive(Clone)]
pub(crate) enum ChannelMonitorUpdateStep {
	LatestHolderCommitmentTXInfo {
		commitment_tx: HolderCommitmentTransaction,
		htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Signature>, Option<HTLCSource>)>,
	},
	LatestCounterpartyCommitmentTXInfo {
		commitment_txid: Txid,
		htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Box<HTLCSource>>)>,
		commitment_number: u64,
		their_revocation_point: PublicKey,
	},
	PaymentPreimage {
		payment_preimage: PaymentPreimage,
	},
	CommitmentSecret {
		idx: u64,
		secret: [u8; 32],
	},
	/// Used to indicate that the no future updates will occur, and likely that the latest holder
	/// commitment transaction(s) should be broadcast, as the channel has been force-closed.
	ChannelForceClosed {
		/// If set to false, we shouldn't broadcast the latest holder commitment transaction as we
		/// think we've fallen behind!
		should_broadcast: bool,
	},
}

impl Writeable for ChannelMonitorUpdateStep {
	fn write<W: Writer>(&self, w: &mut W) -> Result<(), ::std::io::Error> {
		match self {
			&ChannelMonitorUpdateStep::LatestHolderCommitmentTXInfo { ref commitment_tx, ref htlc_outputs } => {
				0u8.write(w)?;
				commitment_tx.write(w)?;
				(htlc_outputs.len() as u64).write(w)?;
				for &(ref output, ref signature, ref source) in htlc_outputs.iter() {
					output.write(w)?;
					signature.write(w)?;
					source.write(w)?;
				}
			}
			&ChannelMonitorUpdateStep::LatestCounterpartyCommitmentTXInfo { commitment_txid, ref htlc_outputs, ref commitment_number, ref their_revocation_point } => {
				1u8.write(w)?;
				commitment_txid.write(w)?;
				commitment_number.write(w)?;
				their_revocation_point.write(w)?;
				(htlc_outputs.len() as u64).write(w)?;
				for &(ref output, ref source) in htlc_outputs.iter() {
					output.write(w)?;
					source.as_ref().map(|b| b.as_ref()).write(w)?;
				}
			},
			&ChannelMonitorUpdateStep::PaymentPreimage { ref payment_preimage } => {
				2u8.write(w)?;
				payment_preimage.write(w)?;
			},
			&ChannelMonitorUpdateStep::CommitmentSecret { ref idx, ref secret } => {
				3u8.write(w)?;
				idx.write(w)?;
				secret.write(w)?;
			},
			&ChannelMonitorUpdateStep::ChannelForceClosed { ref should_broadcast } => {
				4u8.write(w)?;
				should_broadcast.write(w)?;
			},
		}
		Ok(())
	}
}
impl Readable for ChannelMonitorUpdateStep {
	fn read<R: ::std::io::Read>(r: &mut R) -> Result<Self, DecodeError> {
		match Readable::read(r)? {
			0u8 => {
				Ok(ChannelMonitorUpdateStep::LatestHolderCommitmentTXInfo {
					commitment_tx: Readable::read(r)?,
					htlc_outputs: {
						let len: u64 = Readable::read(r)?;
						let mut res = Vec::new();
						for _ in 0..len {
							res.push((Readable::read(r)?, Readable::read(r)?, Readable::read(r)?));
						}
						res
					},
				})
			},
			1u8 => {
				Ok(ChannelMonitorUpdateStep::LatestCounterpartyCommitmentTXInfo {
					commitment_txid: Readable::read(r)?,
					commitment_number: Readable::read(r)?,
					their_revocation_point: Readable::read(r)?,
					htlc_outputs: {
						let len: u64 = Readable::read(r)?;
						let mut res = Vec::new();
						for _ in 0..len {
							res.push((Readable::read(r)?, <Option<HTLCSource> as Readable>::read(r)?.map(|o| Box::new(o))));
						}
						res
					},
				})
			},
			2u8 => {
				Ok(ChannelMonitorUpdateStep::PaymentPreimage {
					payment_preimage: Readable::read(r)?,
				})
			},
			3u8 => {
				Ok(ChannelMonitorUpdateStep::CommitmentSecret {
					idx: Readable::read(r)?,
					secret: Readable::read(r)?,
				})
			},
			4u8 => {
				Ok(ChannelMonitorUpdateStep::ChannelForceClosed {
					should_broadcast: Readable::read(r)?
				})
			},
			_ => Err(DecodeError::InvalidValue),
		}
	}
}

/// A ChannelMonitor handles chain events (blocks connected and disconnected) and generates
/// on-chain transactions to ensure no loss of funds occurs.
///
/// You MUST ensure that no ChannelMonitors for a given channel anywhere contain out-of-date
/// information and are actively monitoring the chain.
///
/// Pending Events or updated HTLCs which have not yet been read out by
/// get_and_clear_pending_monitor_events or get_and_clear_pending_events are serialized to disk and
/// reloaded at deserialize-time. Thus, you must ensure that, when handling events, all events
/// gotten are fully handled before re-serializing the new state.
///
/// Note that the deserializer is only implemented for (BlockHash, ChannelMonitor), which
/// tells you the last block hash which was block_connect()ed. You MUST rescan any blocks along
/// the "reorg path" (ie disconnecting blocks until you find a common ancestor from both the
/// returned block hash and the the current chain and then reconnecting blocks to get to the
/// best chain) upon deserializing the object!
pub struct ChannelMonitor<Signer: Sign> {
	#[cfg(test)]
	pub(crate) inner: Mutex<ChannelMonitorImpl<Signer>>,
	#[cfg(not(test))]
	inner: Mutex<ChannelMonitorImpl<Signer>>,
}

pub(crate) struct ChannelMonitorImpl<Signer: Sign> {
	latest_update_id: u64,
	commitment_transaction_number_obscure_factor: u64,

	destination_script: Script,
	broadcasted_holder_revokable_script: Option<(Script, PublicKey, PublicKey)>,
	counterparty_payment_script: Script,
	shutdown_script: Script,

	channel_keys_id: [u8; 32],
	holder_revocation_basepoint: PublicKey,
	funding_info: (OutPoint, Script),
	current_counterparty_commitment_txid: Option<Txid>,
	prev_counterparty_commitment_txid: Option<Txid>,

	counterparty_tx_cache: CounterpartyCommitmentTransaction,
	funding_redeemscript: Script,
	channel_value_satoshis: u64,
	// first is the idx of the first of the two revocation points
	their_cur_revocation_points: Option<(u64, PublicKey, Option<PublicKey>)>,

	on_holder_tx_csv: u16,

	commitment_secrets: CounterpartyCommitmentSecrets,
	counterparty_claimable_outpoints: HashMap<Txid, Vec<(HTLCOutputInCommitment, Option<Box<HTLCSource>>)>>,
	/// We cannot identify HTLC-Success or HTLC-Timeout transactions by themselves on the chain.
	/// Nor can we figure out their commitment numbers without the commitment transaction they are
	/// spending. Thus, in order to claim them via revocation key, we track all the counterparty
	/// commitment transactions which we find on-chain, mapping them to the commitment number which
	/// can be used to derive the revocation key and claim the transactions.
	counterparty_commitment_txn_on_chain: HashMap<Txid, u64>,
	/// Cache used to make pruning of payment_preimages faster.
	/// Maps payment_hash values to commitment numbers for counterparty transactions for non-revoked
	/// counterparty transactions (ie should remain pretty small).
	/// Serialized to disk but should generally not be sent to Watchtowers.
	counterparty_hash_commitment_number: HashMap<PaymentHash, u64>,

	// We store two holder commitment transactions to avoid any race conditions where we may update
	// some monitors (potentially on watchtowers) but then fail to update others, resulting in the
	// various monitors for one channel being out of sync, and us broadcasting a holder
	// transaction for which we have deleted claim information on some watchtowers.
	prev_holder_signed_commitment_tx: Option<HolderSignedTx>,
	current_holder_commitment_tx: HolderSignedTx,

	// Used just for ChannelManager to make sure it has the latest channel data during
	// deserialization
	current_counterparty_commitment_number: u64,
	// Used just for ChannelManager to make sure it has the latest channel data during
	// deserialization
	current_holder_commitment_number: u64,

	payment_preimages: HashMap<PaymentHash, PaymentPreimage>,

	pending_monitor_events: Vec<MonitorEvent>,
	pending_events: Vec<Event>,

	// Used to track on-chain events (i.e., transactions part of channels confirmed on chain) on
	// which to take actions once they reach enough confirmations. Each entry includes the
	// transaction's id and the height when the transaction was confirmed on chain.
	onchain_events_awaiting_threshold_conf: Vec<OnchainEventEntry>,

	// If we get serialized out and re-read, we need to make sure that the chain monitoring
	// interface knows about the TXOs that we want to be notified of spends of. We could probably
	// be smart and derive them from the above storage fields, but its much simpler and more
	// Obviously Correct (tm) if we just keep track of them explicitly.
	outputs_to_watch: HashMap<Txid, Vec<(u32, Script)>>,

	#[cfg(test)]
	pub onchain_tx_handler: OnchainTxHandler<Signer>,
	#[cfg(not(test))]
	onchain_tx_handler: OnchainTxHandler<Signer>,

	// This is set when the Channel[Manager] generated a ChannelMonitorUpdate which indicated the
	// channel has been force-closed. After this is set, no further holder commitment transaction
	// updates may occur, and we panic!() if one is provided.
	lockdown_from_offchain: bool,

	// Set once we've signed a holder commitment transaction and handed it over to our
	// OnchainTxHandler. After this is set, no future updates to our holder commitment transactions
	// may occur, and we fail any such monitor updates.
	//
	// In case of update rejection due to a locally already signed commitment transaction, we
	// nevertheless store update content to track in case of concurrent broadcast by another
	// remote monitor out-of-order with regards to the block view.
	holder_tx_signed: bool,

	// We simply modify best_block in Channel's block_connected so that serialization is
	// consistent but hopefully the users' copy handles block_connected in a consistent way.
	// (we do *not*, however, update them in update_monitor to ensure any local user copies keep
	// their best_block from its state and not based on updated copies that didn't run through
	// the full block_connected).
	best_block: BestBlock,

	secp_ctx: Secp256k1<secp256k1::All>, //TODO: dedup this a bit...
}

/// Transaction outputs to watch for on-chain spends.
pub type TransactionOutputs = (Txid, Vec<(u32, TxOut)>);

#[cfg(any(test, feature = "fuzztarget", feature = "_test_utils"))]
/// Used only in testing and fuzztarget to check serialization roundtrips don't change the
/// underlying object
impl<Signer: Sign> PartialEq for ChannelMonitor<Signer> {
	fn eq(&self, other: &Self) -> bool {
		let inner = self.inner.lock().unwrap();
		let other = other.inner.lock().unwrap();
		inner.eq(&other)
	}
}

#[cfg(any(test, feature = "fuzztarget", feature = "_test_utils"))]
/// Used only in testing and fuzztarget to check serialization roundtrips don't change the
/// underlying object
impl<Signer: Sign> PartialEq for ChannelMonitorImpl<Signer> {
	fn eq(&self, other: &Self) -> bool {
		if self.latest_update_id != other.latest_update_id ||
			self.commitment_transaction_number_obscure_factor != other.commitment_transaction_number_obscure_factor ||
			self.destination_script != other.destination_script ||
			self.broadcasted_holder_revokable_script != other.broadcasted_holder_revokable_script ||
			self.counterparty_payment_script != other.counterparty_payment_script ||
			self.channel_keys_id != other.channel_keys_id ||
			self.holder_revocation_basepoint != other.holder_revocation_basepoint ||
			self.funding_info != other.funding_info ||
			self.current_counterparty_commitment_txid != other.current_counterparty_commitment_txid ||
			self.prev_counterparty_commitment_txid != other.prev_counterparty_commitment_txid ||
			self.counterparty_tx_cache != other.counterparty_tx_cache ||
			self.funding_redeemscript != other.funding_redeemscript ||
			self.channel_value_satoshis != other.channel_value_satoshis ||
			self.their_cur_revocation_points != other.their_cur_revocation_points ||
			self.on_holder_tx_csv != other.on_holder_tx_csv ||
			self.commitment_secrets != other.commitment_secrets ||
			self.counterparty_claimable_outpoints != other.counterparty_claimable_outpoints ||
			self.counterparty_commitment_txn_on_chain != other.counterparty_commitment_txn_on_chain ||
			self.counterparty_hash_commitment_number != other.counterparty_hash_commitment_number ||
			self.prev_holder_signed_commitment_tx != other.prev_holder_signed_commitment_tx ||
			self.current_counterparty_commitment_number != other.current_counterparty_commitment_number ||
			self.current_holder_commitment_number != other.current_holder_commitment_number ||
			self.current_holder_commitment_tx != other.current_holder_commitment_tx ||
			self.payment_preimages != other.payment_preimages ||
			self.pending_monitor_events != other.pending_monitor_events ||
			self.pending_events.len() != other.pending_events.len() || // We trust events to round-trip properly
			self.onchain_events_awaiting_threshold_conf != other.onchain_events_awaiting_threshold_conf ||
			self.outputs_to_watch != other.outputs_to_watch ||
			self.lockdown_from_offchain != other.lockdown_from_offchain ||
			self.holder_tx_signed != other.holder_tx_signed
		{
			false
		} else {
			true
		}
	}
}

impl<Signer: Sign> Writeable for ChannelMonitor<Signer> {
	fn write<W: Writer>(&self, writer: &mut W) -> Result<(), Error> {
		//TODO: We still write out all the serialization here manually instead of using the fancy
		//serialization framework we have, we should migrate things over to it.
		writer.write_all(&[SERIALIZATION_VERSION; 1])?;
		writer.write_all(&[MIN_SERIALIZATION_VERSION; 1])?;

		self.inner.lock().unwrap().write(writer)
	}
}

impl<Signer: Sign> Writeable for ChannelMonitorImpl<Signer> {
	fn write<W: Writer>(&self, writer: &mut W) -> Result<(), Error> {
		self.latest_update_id.write(writer)?;

		// Set in initial Channel-object creation, so should always be set by now:
		U48(self.commitment_transaction_number_obscure_factor).write(writer)?;

		self.destination_script.write(writer)?;
		if let Some(ref broadcasted_holder_revokable_script) = self.broadcasted_holder_revokable_script {
			writer.write_all(&[0; 1])?;
			broadcasted_holder_revokable_script.0.write(writer)?;
			broadcasted_holder_revokable_script.1.write(writer)?;
			broadcasted_holder_revokable_script.2.write(writer)?;
		} else {
			writer.write_all(&[1; 1])?;
		}

		self.counterparty_payment_script.write(writer)?;
		self.shutdown_script.write(writer)?;

		self.channel_keys_id.write(writer)?;
		self.holder_revocation_basepoint.write(writer)?;
		writer.write_all(&self.funding_info.0.txid[..])?;
		writer.write_all(&byte_utils::be16_to_array(self.funding_info.0.index))?;
		self.funding_info.1.write(writer)?;
		self.current_counterparty_commitment_txid.write(writer)?;
		self.prev_counterparty_commitment_txid.write(writer)?;

		self.counterparty_tx_cache.write(writer)?;
		self.funding_redeemscript.write(writer)?;
		self.channel_value_satoshis.write(writer)?;

		match self.their_cur_revocation_points {
			Some((idx, pubkey, second_option)) => {
				writer.write_all(&byte_utils::be48_to_array(idx))?;
				writer.write_all(&pubkey.serialize())?;
				match second_option {
					Some(second_pubkey) => {
						writer.write_all(&second_pubkey.serialize())?;
					},
					None => {
						writer.write_all(&[0; 33])?;
					},
				}
			},
			None => {
				writer.write_all(&byte_utils::be48_to_array(0))?;
			},
		}

		writer.write_all(&byte_utils::be16_to_array(self.on_holder_tx_csv))?;

		self.commitment_secrets.write(writer)?;

		macro_rules! serialize_htlc_in_commitment {
			($htlc_output: expr) => {
				writer.write_all(&[$htlc_output.offered as u8; 1])?;
				writer.write_all(&byte_utils::be64_to_array($htlc_output.amount_msat))?;
				writer.write_all(&byte_utils::be32_to_array($htlc_output.cltv_expiry))?;
				writer.write_all(&$htlc_output.payment_hash.0[..])?;
				$htlc_output.transaction_output_index.write(writer)?;
			}
		}

		writer.write_all(&byte_utils::be64_to_array(self.counterparty_claimable_outpoints.len() as u64))?;
		for (ref txid, ref htlc_infos) in self.counterparty_claimable_outpoints.iter() {
			writer.write_all(&txid[..])?;
			writer.write_all(&byte_utils::be64_to_array(htlc_infos.len() as u64))?;
			for &(ref htlc_output, ref htlc_source) in htlc_infos.iter() {
				serialize_htlc_in_commitment!(htlc_output);
				htlc_source.as_ref().map(|b| b.as_ref()).write(writer)?;
			}
		}

		writer.write_all(&byte_utils::be64_to_array(self.counterparty_commitment_txn_on_chain.len() as u64))?;
		for (ref txid, commitment_number) in self.counterparty_commitment_txn_on_chain.iter() {
			writer.write_all(&txid[..])?;
			writer.write_all(&byte_utils::be48_to_array(*commitment_number))?;
		}

		writer.write_all(&byte_utils::be64_to_array(self.counterparty_hash_commitment_number.len() as u64))?;
		for (ref payment_hash, commitment_number) in self.counterparty_hash_commitment_number.iter() {
			writer.write_all(&payment_hash.0[..])?;
			writer.write_all(&byte_utils::be48_to_array(*commitment_number))?;
		}

		macro_rules! serialize_holder_tx {
			($holder_tx: expr) => {
				$holder_tx.txid.write(writer)?;
				writer.write_all(&$holder_tx.revocation_key.serialize())?;
				writer.write_all(&$holder_tx.a_htlc_key.serialize())?;
				writer.write_all(&$holder_tx.b_htlc_key.serialize())?;
				writer.write_all(&$holder_tx.delayed_payment_key.serialize())?;
				writer.write_all(&$holder_tx.per_commitment_point.serialize())?;

				writer.write_all(&byte_utils::be32_to_array($holder_tx.feerate_per_kw))?;
				writer.write_all(&byte_utils::be64_to_array($holder_tx.htlc_outputs.len() as u64))?;
				for &(ref htlc_output, ref sig, ref htlc_source) in $holder_tx.htlc_outputs.iter() {
					serialize_htlc_in_commitment!(htlc_output);
					if let &Some(ref their_sig) = sig {
						1u8.write(writer)?;
						writer.write_all(&their_sig.serialize_compact())?;
					} else {
						0u8.write(writer)?;
					}
					htlc_source.write(writer)?;
				}
			}
		}

		if let Some(ref prev_holder_tx) = self.prev_holder_signed_commitment_tx {
			writer.write_all(&[1; 1])?;
			serialize_holder_tx!(prev_holder_tx);
		} else {
			writer.write_all(&[0; 1])?;
		}

		serialize_holder_tx!(self.current_holder_commitment_tx);

		writer.write_all(&byte_utils::be48_to_array(self.current_counterparty_commitment_number))?;
		writer.write_all(&byte_utils::be48_to_array(self.current_holder_commitment_number))?;

		writer.write_all(&byte_utils::be64_to_array(self.payment_preimages.len() as u64))?;
		for payment_preimage in self.payment_preimages.values() {
			writer.write_all(&payment_preimage.0[..])?;
		}

		writer.write_all(&byte_utils::be64_to_array(self.pending_monitor_events.len() as u64))?;
		for event in self.pending_monitor_events.iter() {
			match event {
				MonitorEvent::HTLCEvent(upd) => {
					0u8.write(writer)?;
					upd.write(writer)?;
				},
				MonitorEvent::CommitmentTxBroadcasted(_) => 1u8.write(writer)?
			}
		}

		writer.write_all(&byte_utils::be64_to_array(self.pending_events.len() as u64))?;
		for event in self.pending_events.iter() {
			event.write(writer)?;
		}

		self.best_block.block_hash().write(writer)?;
		writer.write_all(&byte_utils::be32_to_array(self.best_block.height()))?;

		writer.write_all(&byte_utils::be64_to_array(self.onchain_events_awaiting_threshold_conf.len() as u64))?;
		for ref entry in self.onchain_events_awaiting_threshold_conf.iter() {
			entry.txid.write(writer)?;
			writer.write_all(&byte_utils::be32_to_array(entry.height))?;
			match entry.event {
				OnchainEvent::HTLCUpdate { ref htlc_update } => {
					0u8.write(writer)?;
					htlc_update.0.write(writer)?;
					htlc_update.1.write(writer)?;
				},
				OnchainEvent::MaturingOutput { ref descriptor } => {
					1u8.write(writer)?;
					descriptor.write(writer)?;
				},
			}
		}

		(self.outputs_to_watch.len() as u64).write(writer)?;
		for (txid, idx_scripts) in self.outputs_to_watch.iter() {
			txid.write(writer)?;
			(idx_scripts.len() as u64).write(writer)?;
			for (idx, script) in idx_scripts.iter() {
				idx.write(writer)?;
				script.write(writer)?;
			}
		}
		self.onchain_tx_handler.write(writer)?;

		self.lockdown_from_offchain.write(writer)?;
		self.holder_tx_signed.write(writer)?;

		Ok(())
	}
}

impl<Signer: Sign> ChannelMonitor<Signer> {
	pub(crate) fn new(secp_ctx: Secp256k1<secp256k1::All>, keys: Signer, shutdown_pubkey: &PublicKey,
	                  on_counterparty_tx_csv: u16, destination_script: &Script, funding_info: (OutPoint, Script),
	                  channel_parameters: &ChannelTransactionParameters,
	                  funding_redeemscript: Script, channel_value_satoshis: u64,
	                  commitment_transaction_number_obscure_factor: u64,
	                  initial_holder_commitment_tx: HolderCommitmentTransaction,
	                  best_block: BestBlock) -> ChannelMonitor<Signer> {

		assert!(commitment_transaction_number_obscure_factor <= (1 << 48));
		let our_channel_close_key_hash = WPubkeyHash::hash(&shutdown_pubkey.serialize());
		let shutdown_script = Builder::new().push_opcode(opcodes::all::OP_PUSHBYTES_0).push_slice(&our_channel_close_key_hash[..]).into_script();
		let payment_key_hash = WPubkeyHash::hash(&keys.pubkeys().payment_point.serialize());
		let counterparty_payment_script = Builder::new().push_opcode(opcodes::all::OP_PUSHBYTES_0).push_slice(&payment_key_hash[..]).into_script();

		let counterparty_channel_parameters = channel_parameters.counterparty_parameters.as_ref().unwrap();
		let counterparty_delayed_payment_base_key = counterparty_channel_parameters.pubkeys.delayed_payment_basepoint;
		let counterparty_htlc_base_key = counterparty_channel_parameters.pubkeys.htlc_basepoint;
		let counterparty_tx_cache = CounterpartyCommitmentTransaction { counterparty_delayed_payment_base_key, counterparty_htlc_base_key, on_counterparty_tx_csv, per_htlc: HashMap::new() };

		let channel_keys_id = keys.channel_keys_id();
		let holder_revocation_basepoint = keys.pubkeys().revocation_basepoint;

		// block for Rust 1.34 compat
		let (holder_commitment_tx, current_holder_commitment_number) = {
			let trusted_tx = initial_holder_commitment_tx.trust();
			let txid = trusted_tx.txid();

			let tx_keys = trusted_tx.keys();
			let holder_commitment_tx = HolderSignedTx {
				txid,
				revocation_key: tx_keys.revocation_key,
				a_htlc_key: tx_keys.broadcaster_htlc_key,
				b_htlc_key: tx_keys.countersignatory_htlc_key,
				delayed_payment_key: tx_keys.broadcaster_delayed_payment_key,
				per_commitment_point: tx_keys.per_commitment_point,
				feerate_per_kw: trusted_tx.feerate_per_kw(),
				htlc_outputs: Vec::new(), // There are never any HTLCs in the initial commitment transactions
			};
			(holder_commitment_tx, trusted_tx.commitment_number())
		};

		let onchain_tx_handler =
			OnchainTxHandler::new(destination_script.clone(), keys,
			channel_parameters.clone(), initial_holder_commitment_tx, secp_ctx.clone());

		let mut outputs_to_watch = HashMap::new();
		outputs_to_watch.insert(funding_info.0.txid, vec![(funding_info.0.index as u32, funding_info.1.clone())]);

		ChannelMonitor {
			inner: Mutex::new(ChannelMonitorImpl {
				latest_update_id: 0,
				commitment_transaction_number_obscure_factor,

				destination_script: destination_script.clone(),
				broadcasted_holder_revokable_script: None,
				counterparty_payment_script,
				shutdown_script,

				channel_keys_id,
				holder_revocation_basepoint,
				funding_info,
				current_counterparty_commitment_txid: None,
				prev_counterparty_commitment_txid: None,

				counterparty_tx_cache,
				funding_redeemscript,
				channel_value_satoshis,
				their_cur_revocation_points: None,

				on_holder_tx_csv: counterparty_channel_parameters.selected_contest_delay,

				commitment_secrets: CounterpartyCommitmentSecrets::new(),
				counterparty_claimable_outpoints: HashMap::new(),
				counterparty_commitment_txn_on_chain: HashMap::new(),
				counterparty_hash_commitment_number: HashMap::new(),

				prev_holder_signed_commitment_tx: None,
				current_holder_commitment_tx: holder_commitment_tx,
				current_counterparty_commitment_number: 1 << 48,
				current_holder_commitment_number,

				payment_preimages: HashMap::new(),
				pending_monitor_events: Vec::new(),
				pending_events: Vec::new(),

				onchain_events_awaiting_threshold_conf: Vec::new(),
				outputs_to_watch,

				onchain_tx_handler,

				lockdown_from_offchain: false,
				holder_tx_signed: false,

				best_block,

				secp_ctx,
			}),
		}
	}

	#[cfg(test)]
	fn provide_secret(&self, idx: u64, secret: [u8; 32]) -> Result<(), MonitorUpdateError> {
		self.inner.lock().unwrap().provide_secret(idx, secret)
	}

	/// Informs this monitor of the latest counterparty (ie non-broadcastable) commitment transaction.
	/// The monitor watches for it to be broadcasted and then uses the HTLC information (and
	/// possibly future revocation/preimage information) to claim outputs where possible.
	/// We cache also the mapping hash:commitment number to lighten pruning of old preimages by watchtowers.
	pub(crate) fn provide_latest_counterparty_commitment_tx<L: Deref>(
		&self,
		txid: Txid,
		htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Box<HTLCSource>>)>,
		commitment_number: u64,
		their_revocation_point: PublicKey,
		logger: &L,
	) where L::Target: Logger {
		self.inner.lock().unwrap().provide_latest_counterparty_commitment_tx(
			txid, htlc_outputs, commitment_number, their_revocation_point, logger)
	}

	#[cfg(test)]
	fn provide_latest_holder_commitment_tx(
		&self,
		holder_commitment_tx: HolderCommitmentTransaction,
		htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Signature>, Option<HTLCSource>)>,
	) -> Result<(), MonitorUpdateError> {
		self.inner.lock().unwrap().provide_latest_holder_commitment_tx(
			holder_commitment_tx, htlc_outputs)
	}

	#[cfg(test)]
	pub(crate) fn provide_payment_preimage<B: Deref, F: Deref, L: Deref>(
		&self,
		payment_hash: &PaymentHash,
		payment_preimage: &PaymentPreimage,
		broadcaster: &B,
		fee_estimator: &F,
		logger: &L,
	) where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().provide_payment_preimage(
			payment_hash, payment_preimage, broadcaster, fee_estimator, logger)
	}

	pub(crate) fn broadcast_latest_holder_commitment_txn<B: Deref, L: Deref>(
		&self,
		broadcaster: &B,
		logger: &L,
	) where
		B::Target: BroadcasterInterface,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().broadcast_latest_holder_commitment_txn(broadcaster, logger)
	}

	/// Updates a ChannelMonitor on the basis of some new information provided by the Channel
	/// itself.
	///
	/// panics if the given update is not the next update by update_id.
	pub fn update_monitor<B: Deref, F: Deref, L: Deref>(
		&self,
		updates: &ChannelMonitorUpdate,
		broadcaster: &B,
		fee_estimator: &F,
		logger: &L,
	) -> Result<(), MonitorUpdateError>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().update_monitor(updates, broadcaster, fee_estimator, logger)
	}

	/// Gets the update_id from the latest ChannelMonitorUpdate which was applied to this
	/// ChannelMonitor.
	pub fn get_latest_update_id(&self) -> u64 {
		self.inner.lock().unwrap().get_latest_update_id()
	}

	/// Gets the funding transaction outpoint of the channel this ChannelMonitor is monitoring for.
	pub fn get_funding_txo(&self) -> (OutPoint, Script) {
		self.inner.lock().unwrap().get_funding_txo().clone()
	}

	/// Gets a list of txids, with their output scripts (in the order they appear in the
	/// transaction), which we must learn about spends of via block_connected().
	pub fn get_outputs_to_watch(&self) -> Vec<(Txid, Vec<(u32, Script)>)> {
		self.inner.lock().unwrap().get_outputs_to_watch()
			.iter().map(|(txid, outputs)| (*txid, outputs.clone())).collect()
	}

	/// Loads the funding txo and outputs to watch into the given `chain::Filter` by repeatedly
	/// calling `chain::Filter::register_output` and `chain::Filter::register_tx` until all outputs
	/// have been registered.
	pub fn load_outputs_to_watch<F: Deref>(&self, filter: &F) where F::Target: chain::Filter {
		let lock = self.inner.lock().unwrap();
		filter.register_tx(&lock.get_funding_txo().0.txid, &lock.get_funding_txo().1);
		for (txid, outputs) in lock.get_outputs_to_watch().iter() {
			for (index, script_pubkey) in outputs.iter() {
				assert!(*index <= u16::max_value() as u32);
				filter.register_output(WatchedOutput {
					block_hash: None,
					outpoint: OutPoint { txid: *txid, index: *index as u16 },
					script_pubkey: script_pubkey.clone(),
				});
			}
		}
	}

	/// Get the list of HTLCs who's status has been updated on chain. This should be called by
	/// ChannelManager via [`chain::Watch::release_pending_monitor_events`].
	pub fn get_and_clear_pending_monitor_events(&self) -> Vec<MonitorEvent> {
		self.inner.lock().unwrap().get_and_clear_pending_monitor_events()
	}

	/// Gets the list of pending events which were generated by previous actions, clearing the list
	/// in the process.
	///
	/// This is called by ChainMonitor::get_and_clear_pending_events() and is equivalent to
	/// EventsProvider::get_and_clear_pending_events() except that it requires &mut self as we do
	/// no internal locking in ChannelMonitors.
	pub fn get_and_clear_pending_events(&self) -> Vec<Event> {
		self.inner.lock().unwrap().get_and_clear_pending_events()
	}

	pub(crate) fn get_min_seen_secret(&self) -> u64 {
		self.inner.lock().unwrap().get_min_seen_secret()
	}

	pub(crate) fn get_cur_counterparty_commitment_number(&self) -> u64 {
		self.inner.lock().unwrap().get_cur_counterparty_commitment_number()
	}

	pub(crate) fn get_cur_holder_commitment_number(&self) -> u64 {
		self.inner.lock().unwrap().get_cur_holder_commitment_number()
	}

	/// Used by ChannelManager deserialization to broadcast the latest holder state if its copy of
	/// the Channel was out-of-date. You may use it to get a broadcastable holder toxic tx in case of
	/// fallen-behind, i.e when receiving a channel_reestablish with a proof that our counterparty side knows
	/// a higher revocation secret than the holder commitment number we are aware of. Broadcasting these
	/// transactions are UNSAFE, as they allow counterparty side to punish you. Nevertheless you may want to
	/// broadcast them if counterparty don't close channel with his higher commitment transaction after a
	/// substantial amount of time (a month or even a year) to get back funds. Best may be to contact
	/// out-of-band the other node operator to coordinate with him if option is available to you.
	/// In any-case, choice is up to the user.
	pub fn get_latest_holder_commitment_txn<L: Deref>(&self, logger: &L) -> Vec<Transaction>
	where L::Target: Logger {
		self.inner.lock().unwrap().get_latest_holder_commitment_txn(logger)
	}

	/// Unsafe test-only version of get_latest_holder_commitment_txn used by our test framework
	/// to bypass HolderCommitmentTransaction state update lockdown after signature and generate
	/// revoked commitment transaction.
	#[cfg(any(test, feature = "unsafe_revoked_tx_signing"))]
	pub fn unsafe_get_latest_holder_commitment_txn<L: Deref>(&self, logger: &L) -> Vec<Transaction>
	where L::Target: Logger {
		self.inner.lock().unwrap().unsafe_get_latest_holder_commitment_txn(logger)
	}

	/// Processes transactions in a newly connected block, which may result in any of the following:
	/// - update the monitor's state against resolved HTLCs
	/// - punish the counterparty in the case of seeing a revoked commitment transaction
	/// - force close the channel and claim/timeout incoming/outgoing HTLCs if near expiration
	/// - detect settled outputs for later spending
	/// - schedule and bump any in-flight claims
	///
	/// Returns any new outputs to watch from `txdata`; after called, these are also included in
	/// [`get_outputs_to_watch`].
	///
	/// [`get_outputs_to_watch`]: #method.get_outputs_to_watch
	pub fn block_connected<B: Deref, F: Deref, L: Deref>(
		&self,
		header: &BlockHeader,
		txdata: &TransactionData,
		height: u32,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) -> Vec<TransactionOutputs>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().block_connected(
			header, txdata, height, broadcaster, fee_estimator, logger)
	}

	/// Determines if the disconnected block contained any transactions of interest and updates
	/// appropriately.
	pub fn block_disconnected<B: Deref, F: Deref, L: Deref>(
		&self,
		header: &BlockHeader,
		height: u32,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().block_disconnected(
			header, height, broadcaster, fee_estimator, logger)
	}

	/// Processes transactions confirmed in a block with the given header and height, returning new
	/// outputs to watch. See [`block_connected`] for details.
	///
	/// Used instead of [`block_connected`] by clients that are notified of transactions rather than
	/// blocks. See [`chain::Confirm`] for calling expectations.
	///
	/// [`block_connected`]: Self::block_connected
	pub fn transactions_confirmed<B: Deref, F: Deref, L: Deref>(
		&self,
		header: &BlockHeader,
		txdata: &TransactionData,
		height: u32,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) -> Vec<TransactionOutputs>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().transactions_confirmed(
			header, txdata, height, broadcaster, fee_estimator, logger)
	}

	/// Processes a transaction that was reorganized out of the chain.
	///
	/// Used instead of [`block_disconnected`] by clients that are notified of transactions rather
	/// than blocks. See [`chain::Confirm`] for calling expectations.
	///
	/// [`block_disconnected`]: Self::block_disconnected
	pub fn transaction_unconfirmed<B: Deref, F: Deref, L: Deref>(
		&self,
		txid: &Txid,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().transaction_unconfirmed(
			txid, broadcaster, fee_estimator, logger);
	}

	/// Updates the monitor with the current best chain tip, returning new outputs to watch. See
	/// [`block_connected`] for details.
	///
	/// Used instead of [`block_connected`] by clients that are notified of transactions rather than
	/// blocks. See [`chain::Confirm`] for calling expectations.
	///
	/// [`block_connected`]: Self::block_connected
	pub fn best_block_updated<B: Deref, F: Deref, L: Deref>(
		&self,
		header: &BlockHeader,
		height: u32,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) -> Vec<TransactionOutputs>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().best_block_updated(
			header, height, broadcaster, fee_estimator, logger)
	}

	/// Returns the set of txids that should be monitored for re-organization out of the chain.
	pub fn get_relevant_txids(&self) -> Vec<Txid> {
		let inner = self.inner.lock().unwrap();
		let mut txids: Vec<Txid> = inner.onchain_events_awaiting_threshold_conf
			.iter()
			.map(|entry| entry.txid)
			.chain(inner.onchain_tx_handler.get_relevant_txids().into_iter())
			.collect();
		txids.sort_unstable();
		txids.dedup();
		txids
	}
}

impl<Signer: Sign> ChannelMonitorImpl<Signer> {
	/// Inserts a revocation secret into this channel monitor. Prunes old preimages if neither
	/// needed by holder commitment transactions HTCLs nor by counterparty ones. Unless we haven't already seen
	/// counterparty commitment transaction's secret, they are de facto pruned (we can use revocation key).
	fn provide_secret(&mut self, idx: u64, secret: [u8; 32]) -> Result<(), MonitorUpdateError> {
		if let Err(()) = self.commitment_secrets.provide_secret(idx, secret) {
			return Err(MonitorUpdateError("Previous secret did not match new one"));
		}

		// Prune HTLCs from the previous counterparty commitment tx so we don't generate failure/fulfill
		// events for now-revoked/fulfilled HTLCs.
		if let Some(txid) = self.prev_counterparty_commitment_txid.take() {
			for &mut (_, ref mut source) in self.counterparty_claimable_outpoints.get_mut(&txid).unwrap() {
				*source = None;
			}
		}

		if !self.payment_preimages.is_empty() {
			let cur_holder_signed_commitment_tx = &self.current_holder_commitment_tx;
			let prev_holder_signed_commitment_tx = self.prev_holder_signed_commitment_tx.as_ref();
			let min_idx = self.get_min_seen_secret();
			let counterparty_hash_commitment_number = &mut self.counterparty_hash_commitment_number;

			self.payment_preimages.retain(|&k, _| {
				for &(ref htlc, _, _) in cur_holder_signed_commitment_tx.htlc_outputs.iter() {
					if k == htlc.payment_hash {
						return true
					}
				}
				if let Some(prev_holder_commitment_tx) = prev_holder_signed_commitment_tx {
					for &(ref htlc, _, _) in prev_holder_commitment_tx.htlc_outputs.iter() {
						if k == htlc.payment_hash {
							return true
						}
					}
				}
				let contains = if let Some(cn) = counterparty_hash_commitment_number.get(&k) {
					if *cn < min_idx {
						return true
					}
					true
				} else { false };
				if contains {
					counterparty_hash_commitment_number.remove(&k);
				}
				false
			});
		}

		Ok(())
	}

	pub(crate) fn provide_latest_counterparty_commitment_tx<L: Deref>(&mut self, txid: Txid, htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Box<HTLCSource>>)>, commitment_number: u64, their_revocation_point: PublicKey, logger: &L) where L::Target: Logger {
		// TODO: Encrypt the htlc_outputs data with the single-hash of the commitment transaction
		// so that a remote monitor doesn't learn anything unless there is a malicious close.
		// (only maybe, sadly we cant do the same for local info, as we need to be aware of
		// timeouts)
		for &(ref htlc, _) in &htlc_outputs {
			self.counterparty_hash_commitment_number.insert(htlc.payment_hash, commitment_number);
		}

		log_trace!(logger, "Tracking new counterparty commitment transaction with txid {} at commitment number {} with {} HTLC outputs", txid, commitment_number, htlc_outputs.len());
		self.prev_counterparty_commitment_txid = self.current_counterparty_commitment_txid.take();
		self.current_counterparty_commitment_txid = Some(txid);
		self.counterparty_claimable_outpoints.insert(txid, htlc_outputs.clone());
		self.current_counterparty_commitment_number = commitment_number;
		//TODO: Merge this into the other per-counterparty-transaction output storage stuff
		match self.their_cur_revocation_points {
			Some(old_points) => {
				if old_points.0 == commitment_number + 1 {
					self.their_cur_revocation_points = Some((old_points.0, old_points.1, Some(their_revocation_point)));
				} else if old_points.0 == commitment_number + 2 {
					if let Some(old_second_point) = old_points.2 {
						self.their_cur_revocation_points = Some((old_points.0 - 1, old_second_point, Some(their_revocation_point)));
					} else {
						self.their_cur_revocation_points = Some((commitment_number, their_revocation_point, None));
					}
				} else {
					self.their_cur_revocation_points = Some((commitment_number, their_revocation_point, None));
				}
			},
			None => {
				self.their_cur_revocation_points = Some((commitment_number, their_revocation_point, None));
			}
		}
		let mut htlcs = Vec::with_capacity(htlc_outputs.len());
		for htlc in htlc_outputs {
			if htlc.0.transaction_output_index.is_some() {
				htlcs.push(htlc.0);
			}
		}
		self.counterparty_tx_cache.per_htlc.insert(txid, htlcs);
	}

	/// Informs this monitor of the latest holder (ie broadcastable) commitment transaction. The
	/// monitor watches for timeouts and may broadcast it if we approach such a timeout. Thus, it
	/// is important that any clones of this channel monitor (including remote clones) by kept
	/// up-to-date as our holder commitment transaction is updated.
	/// Panics if set_on_holder_tx_csv has never been called.
	fn provide_latest_holder_commitment_tx(&mut self, holder_commitment_tx: HolderCommitmentTransaction, htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Signature>, Option<HTLCSource>)>) -> Result<(), MonitorUpdateError> {
		// block for Rust 1.34 compat
		let mut new_holder_commitment_tx = {
			let trusted_tx = holder_commitment_tx.trust();
			let txid = trusted_tx.txid();
			let tx_keys = trusted_tx.keys();
			self.current_holder_commitment_number = trusted_tx.commitment_number();
			HolderSignedTx {
				txid,
				revocation_key: tx_keys.revocation_key,
				a_htlc_key: tx_keys.broadcaster_htlc_key,
				b_htlc_key: tx_keys.countersignatory_htlc_key,
				delayed_payment_key: tx_keys.broadcaster_delayed_payment_key,
				per_commitment_point: tx_keys.per_commitment_point,
				feerate_per_kw: trusted_tx.feerate_per_kw(),
				htlc_outputs,
			}
		};
		self.onchain_tx_handler.provide_latest_holder_tx(holder_commitment_tx);
		mem::swap(&mut new_holder_commitment_tx, &mut self.current_holder_commitment_tx);
		self.prev_holder_signed_commitment_tx = Some(new_holder_commitment_tx);
		if self.holder_tx_signed {
			return Err(MonitorUpdateError("Latest holder commitment signed has already been signed, update is rejected"));
		}
		Ok(())
	}

	/// Provides a payment_hash->payment_preimage mapping. Will be automatically pruned when all
	/// commitment_tx_infos which contain the payment hash have been revoked.
	fn provide_payment_preimage<B: Deref, F: Deref, L: Deref>(&mut self, payment_hash: &PaymentHash, payment_preimage: &PaymentPreimage, broadcaster: &B, fee_estimator: &F, logger: &L)
	where B::Target: BroadcasterInterface,
		    F::Target: FeeEstimator,
		    L::Target: Logger,
	{
		self.payment_preimages.insert(payment_hash.clone(), payment_preimage.clone());

		// If the channel is force closed, try to claim the output from this preimage.
		// First check if a counterparty commitment transaction has been broadcasted:
		macro_rules! claim_htlcs {
			($commitment_number: expr, $txid: expr) => {
				let htlc_claim_reqs = self.get_counterparty_htlc_output_claim_reqs($commitment_number, $txid, None);
				self.onchain_tx_handler.update_claims_view(&Vec::new(), htlc_claim_reqs, None, broadcaster, fee_estimator, logger);
			}
		}
		if let Some(txid) = self.current_counterparty_commitment_txid {
			if let Some(commitment_number) = self.counterparty_commitment_txn_on_chain.get(&txid) {
				claim_htlcs!(*commitment_number, txid);
				return;
			}
		}
		if let Some(txid) = self.prev_counterparty_commitment_txid {
			if let Some(commitment_number) = self.counterparty_commitment_txn_on_chain.get(&txid) {
				claim_htlcs!(*commitment_number, txid);
				return;
			}
		}

		// Then if a holder commitment transaction has been seen on-chain, broadcast transactions
		// claiming the HTLC output from each of the holder commitment transactions.
		// Note that we can't just use `self.holder_tx_signed`, because that only covers the case where
		// *we* sign a holder commitment transaction, not when e.g. a watchtower broadcasts one of our
		// holder commitment transactions.
		if self.broadcasted_holder_revokable_script.is_some() {
			let (claim_reqs, _) = self.get_broadcasted_holder_claims(&self.current_holder_commitment_tx);
			self.onchain_tx_handler.update_claims_view(&Vec::new(), claim_reqs, None, broadcaster, fee_estimator, logger);
			if let Some(ref tx) = self.prev_holder_signed_commitment_tx {
				let (claim_reqs, _) = self.get_broadcasted_holder_claims(&tx);
				self.onchain_tx_handler.update_claims_view(&Vec::new(), claim_reqs, None, broadcaster, fee_estimator, logger);
			}
		}
	}

	pub(crate) fn broadcast_latest_holder_commitment_txn<B: Deref, L: Deref>(&mut self, broadcaster: &B, logger: &L)
		where B::Target: BroadcasterInterface,
					L::Target: Logger,
	{
		for tx in self.get_latest_holder_commitment_txn(logger).iter() {
			broadcaster.broadcast_transaction(tx);
		}
		self.pending_monitor_events.push(MonitorEvent::CommitmentTxBroadcasted(self.funding_info.0));
	}

	pub fn update_monitor<B: Deref, F: Deref, L: Deref>(&mut self, updates: &ChannelMonitorUpdate, broadcaster: &B, fee_estimator: &F, logger: &L) -> Result<(), MonitorUpdateError>
	where B::Target: BroadcasterInterface,
		    F::Target: FeeEstimator,
		    L::Target: Logger,
	{
		// ChannelMonitor updates may be applied after force close if we receive a
		// preimage for a broadcasted commitment transaction HTLC output that we'd
		// like to claim on-chain. If this is the case, we no longer have guaranteed
		// access to the monitor's update ID, so we use a sentinel value instead.
		if updates.update_id == CLOSED_CHANNEL_UPDATE_ID {
			match updates.updates[0] {
				ChannelMonitorUpdateStep::PaymentPreimage { .. } => {},
				_ => panic!("Attempted to apply post-force-close ChannelMonitorUpdate that wasn't providing a payment preimage"),
			}
			assert_eq!(updates.updates.len(), 1);
		} else if self.latest_update_id + 1 != updates.update_id {
			panic!("Attempted to apply ChannelMonitorUpdates out of order, check the update_id before passing an update to update_monitor!");
		}
		for update in updates.updates.iter() {
			match update {
				ChannelMonitorUpdateStep::LatestHolderCommitmentTXInfo { commitment_tx, htlc_outputs } => {
					log_trace!(logger, "Updating ChannelMonitor with latest holder commitment transaction info");
					if self.lockdown_from_offchain { panic!(); }
					self.provide_latest_holder_commitment_tx(commitment_tx.clone(), htlc_outputs.clone())?
				}
				ChannelMonitorUpdateStep::LatestCounterpartyCommitmentTXInfo { commitment_txid, htlc_outputs, commitment_number, their_revocation_point } => {
					log_trace!(logger, "Updating ChannelMonitor with latest counterparty commitment transaction info");
					self.provide_latest_counterparty_commitment_tx(*commitment_txid, htlc_outputs.clone(), *commitment_number, *their_revocation_point, logger)
				},
				ChannelMonitorUpdateStep::PaymentPreimage { payment_preimage } => {
					log_trace!(logger, "Updating ChannelMonitor with payment preimage");
					self.provide_payment_preimage(&PaymentHash(Sha256::hash(&payment_preimage.0[..]).into_inner()), &payment_preimage, broadcaster, fee_estimator, logger)
				},
				ChannelMonitorUpdateStep::CommitmentSecret { idx, secret } => {
					log_trace!(logger, "Updating ChannelMonitor with commitment secret");
					self.provide_secret(*idx, *secret)?
				},
				ChannelMonitorUpdateStep::ChannelForceClosed { should_broadcast } => {
					log_trace!(logger, "Updating ChannelMonitor: channel force closed, should broadcast: {}", should_broadcast);
					self.lockdown_from_offchain = true;
					if *should_broadcast {
						self.broadcast_latest_holder_commitment_txn(broadcaster, logger);
					} else if !self.holder_tx_signed {
						log_error!(logger, "You have a toxic holder commitment transaction avaible in channel monitor, read comment in ChannelMonitor::get_latest_holder_commitment_txn to be informed of manual action to take");
					} else {
						// If we generated a MonitorEvent::CommitmentTxBroadcasted, the ChannelManager
						// will still give us a ChannelForceClosed event with !should_broadcast, but we
						// shouldn't print the scary warning above.
						log_info!(logger, "Channel off-chain state closed after we broadcasted our latest commitment transaction.");
					}
				}
			}
		}
		self.latest_update_id = updates.update_id;
		Ok(())
	}

	pub fn get_latest_update_id(&self) -> u64 {
		self.latest_update_id
	}

	pub fn get_funding_txo(&self) -> &(OutPoint, Script) {
		&self.funding_info
	}

	pub fn get_outputs_to_watch(&self) -> &HashMap<Txid, Vec<(u32, Script)>> {
		// If we've detected a counterparty commitment tx on chain, we must include it in the set
		// of outputs to watch for spends of, otherwise we're likely to lose user funds. Because
		// its trivial to do, double-check that here.
		for (txid, _) in self.counterparty_commitment_txn_on_chain.iter() {
			self.outputs_to_watch.get(txid).expect("Counterparty commitment txn which have been broadcast should have outputs registered");
		}
		&self.outputs_to_watch
	}

	pub fn get_and_clear_pending_monitor_events(&mut self) -> Vec<MonitorEvent> {
		let mut ret = Vec::new();
		mem::swap(&mut ret, &mut self.pending_monitor_events);
		ret
	}

	pub fn get_and_clear_pending_events(&mut self) -> Vec<Event> {
		let mut ret = Vec::new();
		mem::swap(&mut ret, &mut self.pending_events);
		ret
	}

	/// Can only fail if idx is < get_min_seen_secret
	fn get_secret(&self, idx: u64) -> Option<[u8; 32]> {
		self.commitment_secrets.get_secret(idx)
	}

	pub(crate) fn get_min_seen_secret(&self) -> u64 {
		self.commitment_secrets.get_min_seen_secret()
	}

	pub(crate) fn get_cur_counterparty_commitment_number(&self) -> u64 {
		self.current_counterparty_commitment_number
	}

	pub(crate) fn get_cur_holder_commitment_number(&self) -> u64 {
		self.current_holder_commitment_number
	}

	/// Attempts to claim a counterparty commitment transaction's outputs using the revocation key and
	/// data in counterparty_claimable_outpoints. Will directly claim any HTLC outputs which expire at a
	/// height > height + CLTV_SHARED_CLAIM_BUFFER. In any case, will install monitoring for
	/// HTLC-Success/HTLC-Timeout transactions.
	/// Return updates for HTLC pending in the channel and failed automatically by the broadcast of
	/// revoked counterparty commitment tx
	fn check_spend_counterparty_transaction<L: Deref>(&mut self, tx: &Transaction, height: u32, logger: &L) -> (Vec<ClaimRequest>, TransactionOutputs) where L::Target: Logger {
		// Most secp and related errors trying to create keys means we have no hope of constructing
		// a spend transaction...so we return no transactions to broadcast
		let mut claimable_outpoints = Vec::new();
		let mut watch_outputs = Vec::new();

		let commitment_txid = tx.txid(); //TODO: This is gonna be a performance bottleneck for watchtowers!
		let per_commitment_option = self.counterparty_claimable_outpoints.get(&commitment_txid);

		macro_rules! ignore_error {
			( $thing : expr ) => {
				match $thing {
					Ok(a) => a,
					Err(_) => return (claimable_outpoints, (commitment_txid, watch_outputs))
				}
			};
		}

		let commitment_number = 0xffffffffffff - ((((tx.input[0].sequence as u64 & 0xffffff) << 3*8) | (tx.lock_time as u64 & 0xffffff)) ^ self.commitment_transaction_number_obscure_factor);
		if commitment_number >= self.get_min_seen_secret() {
			let secret = self.get_secret(commitment_number).unwrap();
			let per_commitment_key = ignore_error!(SecretKey::from_slice(&secret));
			let per_commitment_point = PublicKey::from_secret_key(&self.secp_ctx, &per_commitment_key);
			let revocation_pubkey = ignore_error!(chan_utils::derive_public_revocation_key(&self.secp_ctx, &per_commitment_point, &self.holder_revocation_basepoint));
			let delayed_key = ignore_error!(chan_utils::derive_public_key(&self.secp_ctx, &PublicKey::from_secret_key(&self.secp_ctx, &per_commitment_key), &self.counterparty_tx_cache.counterparty_delayed_payment_base_key));

			let revokeable_redeemscript = chan_utils::get_revokeable_redeemscript(&revocation_pubkey, self.counterparty_tx_cache.on_counterparty_tx_csv, &delayed_key);
			let revokeable_p2wsh = revokeable_redeemscript.to_v0_p2wsh();

			// First, process non-htlc outputs (to_holder & to_counterparty)
			for (idx, outp) in tx.output.iter().enumerate() {
				if outp.script_pubkey == revokeable_p2wsh {
					let witness_data = InputMaterial::Revoked { per_commitment_point, counterparty_delayed_payment_base_key: self.counterparty_tx_cache.counterparty_delayed_payment_base_key, counterparty_htlc_base_key: self.counterparty_tx_cache.counterparty_htlc_base_key, per_commitment_key, input_descriptor: InputDescriptors::RevokedOutput, amount: outp.value, htlc: None, on_counterparty_tx_csv: self.counterparty_tx_cache.on_counterparty_tx_csv};
					claimable_outpoints.push(ClaimRequest { absolute_timelock: height + self.counterparty_tx_cache.on_counterparty_tx_csv as u32, aggregable: true, outpoint: BitcoinOutPoint { txid: commitment_txid, vout: idx as u32 }, witness_data});
				}
			}

			// Then, try to find revoked htlc outputs
			if let Some(ref per_commitment_data) = per_commitment_option {
				for (_, &(ref htlc, _)) in per_commitment_data.iter().enumerate() {
					if let Some(transaction_output_index) = htlc.transaction_output_index {
						if transaction_output_index as usize >= tx.output.len() ||
								tx.output[transaction_output_index as usize].value != htlc.amount_msat / 1000 {
							return (claimable_outpoints, (commitment_txid, watch_outputs)); // Corrupted per_commitment_data, fuck this user
						}
						let witness_data = InputMaterial::Revoked { per_commitment_point, counterparty_delayed_payment_base_key: self.counterparty_tx_cache.counterparty_delayed_payment_base_key, counterparty_htlc_base_key: self.counterparty_tx_cache.counterparty_htlc_base_key, per_commitment_key, input_descriptor: if htlc.offered { InputDescriptors::RevokedOfferedHTLC } else { InputDescriptors::RevokedReceivedHTLC }, amount: tx.output[transaction_output_index as usize].value, htlc: Some(htlc.clone()), on_counterparty_tx_csv: self.counterparty_tx_cache.on_counterparty_tx_csv};
						claimable_outpoints.push(ClaimRequest { absolute_timelock: htlc.cltv_expiry, aggregable: true, outpoint: BitcoinOutPoint { txid: commitment_txid, vout: transaction_output_index }, witness_data });
					}
				}
			}

			// Last, track onchain revoked commitment transaction and fail backward outgoing HTLCs as payment path is broken
			if !claimable_outpoints.is_empty() || per_commitment_option.is_some() { // ie we're confident this is actually ours
				// We're definitely a counterparty commitment transaction!
				log_trace!(logger, "Got broadcast of revoked counterparty commitment transaction, going to generate general spend tx with {} inputs", claimable_outpoints.len());
				for (idx, outp) in tx.output.iter().enumerate() {
					watch_outputs.push((idx as u32, outp.clone()));
				}
				self.counterparty_commitment_txn_on_chain.insert(commitment_txid, commitment_number);

				macro_rules! check_htlc_fails {
					($txid: expr, $commitment_tx: expr) => {
						if let Some(ref outpoints) = self.counterparty_claimable_outpoints.get($txid) {
							for &(ref htlc, ref source_option) in outpoints.iter() {
								if let &Some(ref source) = source_option {
									self.onchain_events_awaiting_threshold_conf.retain(|ref entry| {
										if entry.height != height { return true; }
										match entry.event {
											 OnchainEvent::HTLCUpdate { ref htlc_update } => {
												 htlc_update.0 != **source
											 },
											 _ => true,
										}
									});
									let entry = OnchainEventEntry {
										txid: *$txid,
										height,
										event: OnchainEvent::HTLCUpdate {
											htlc_update: ((**source).clone(), htlc.payment_hash.clone())
										},
									};
									log_info!(logger, "Failing HTLC with payment_hash {} from {} counterparty commitment tx due to broadcast of revoked counterparty commitment transaction, waiting for confirmation (at height {})", log_bytes!(htlc.payment_hash.0), $commitment_tx, entry.confirmation_threshold());
									self.onchain_events_awaiting_threshold_conf.push(entry);
								}
							}
						}
					}
				}
				if let Some(ref txid) = self.current_counterparty_commitment_txid {
					check_htlc_fails!(txid, "current");
				}
				if let Some(ref txid) = self.prev_counterparty_commitment_txid {
					check_htlc_fails!(txid, "counterparty");
				}
				// No need to check holder commitment txn, symmetric HTLCSource must be present as per-htlc data on counterparty commitment tx
			}
		} else if let Some(per_commitment_data) = per_commitment_option {
			// While this isn't useful yet, there is a potential race where if a counterparty
			// revokes a state at the same time as the commitment transaction for that state is
			// confirmed, and the watchtower receives the block before the user, the user could
			// upload a new ChannelMonitor with the revocation secret but the watchtower has
			// already processed the block, resulting in the counterparty_commitment_txn_on_chain entry
			// not being generated by the above conditional. Thus, to be safe, we go ahead and
			// insert it here.
			for (idx, outp) in tx.output.iter().enumerate() {
				watch_outputs.push((idx as u32, outp.clone()));
			}
			self.counterparty_commitment_txn_on_chain.insert(commitment_txid, commitment_number);

			log_trace!(logger, "Got broadcast of non-revoked counterparty commitment transaction {}", commitment_txid);

			macro_rules! check_htlc_fails {
				($txid: expr, $commitment_tx: expr, $id: tt) => {
					if let Some(ref latest_outpoints) = self.counterparty_claimable_outpoints.get($txid) {
						$id: for &(ref htlc, ref source_option) in latest_outpoints.iter() {
							if let &Some(ref source) = source_option {
								// Check if the HTLC is present in the commitment transaction that was
								// broadcast, but not if it was below the dust limit, which we should
								// fail backwards immediately as there is no way for us to learn the
								// payment_preimage.
								// Note that if the dust limit were allowed to change between
								// commitment transactions we'd want to be check whether *any*
								// broadcastable commitment transaction has the HTLC in it, but it
								// cannot currently change after channel initialization, so we don't
								// need to here.
								for &(ref broadcast_htlc, ref broadcast_source) in per_commitment_data.iter() {
									if broadcast_htlc.transaction_output_index.is_some() && Some(source) == broadcast_source.as_ref() {
										continue $id;
									}
								}
								log_trace!(logger, "Failing HTLC with payment_hash {} from {} counterparty commitment tx due to broadcast of counterparty commitment transaction", log_bytes!(htlc.payment_hash.0), $commitment_tx);
								self.onchain_events_awaiting_threshold_conf.retain(|ref entry| {
									if entry.height != height { return true; }
									match entry.event {
										 OnchainEvent::HTLCUpdate { ref htlc_update } => {
											 htlc_update.0 != **source
										 },
										 _ => true,
									}
								});
								self.onchain_events_awaiting_threshold_conf.push(OnchainEventEntry {
									txid: *$txid,
									height,
									event: OnchainEvent::HTLCUpdate {
										htlc_update: ((**source).clone(), htlc.payment_hash.clone())
									},
								});
							}
						}
					}
				}
			}
			if let Some(ref txid) = self.current_counterparty_commitment_txid {
				check_htlc_fails!(txid, "current", 'current_loop);
			}
			if let Some(ref txid) = self.prev_counterparty_commitment_txid {
				check_htlc_fails!(txid, "previous", 'prev_loop);
			}

			let htlc_claim_reqs = self.get_counterparty_htlc_output_claim_reqs(commitment_number, commitment_txid, Some(tx));
			for req in htlc_claim_reqs {
				claimable_outpoints.push(req);
			}

		}
		(claimable_outpoints, (commitment_txid, watch_outputs))
	}

	fn get_counterparty_htlc_output_claim_reqs(&self, commitment_number: u64, commitment_txid: Txid, tx: Option<&Transaction>) -> Vec<ClaimRequest> {
		let mut claims = Vec::new();
		if let Some(htlc_outputs) = self.counterparty_claimable_outpoints.get(&commitment_txid) {
			if let Some(revocation_points) = self.their_cur_revocation_points {
				let revocation_point_option =
					// If the counterparty commitment tx is the latest valid state, use their latest
					// per-commitment point
					if revocation_points.0 == commitment_number { Some(&revocation_points.1) }
					else if let Some(point) = revocation_points.2.as_ref() {
						// If counterparty commitment tx is the state previous to the latest valid state, use
						// their previous per-commitment point (non-atomicity of revocation means it's valid for
						// them to temporarily have two valid commitment txns from our viewpoint)
						if revocation_points.0 == commitment_number + 1 { Some(point) } else { None }
					} else { None };
				if let Some(revocation_point) = revocation_point_option {
					for (_, &(ref htlc, _)) in htlc_outputs.iter().enumerate() {
						if let Some(transaction_output_index) = htlc.transaction_output_index {
							if let Some(transaction) = tx {
								if transaction_output_index as usize >= transaction.output.len() ||
									transaction.output[transaction_output_index as usize].value != htlc.amount_msat / 1000 {
										return claims; // Corrupted per_commitment_data, fuck this user
									}
							}
							let preimage =
								if htlc.offered {
									if let Some(p) = self.payment_preimages.get(&htlc.payment_hash) {
										Some(*p)
									} else { None }
								} else { None };
							let aggregable = if !htlc.offered { false } else { true };
							if preimage.is_some() || !htlc.offered {
								let witness_data = InputMaterial::CounterpartyHTLC { per_commitment_point: *revocation_point, counterparty_delayed_payment_base_key: self.counterparty_tx_cache.counterparty_delayed_payment_base_key, counterparty_htlc_base_key: self.counterparty_tx_cache.counterparty_htlc_base_key, preimage, htlc: htlc.clone() };
								claims.push(ClaimRequest { absolute_timelock: htlc.cltv_expiry, aggregable, outpoint: BitcoinOutPoint { txid: commitment_txid, vout: transaction_output_index }, witness_data });
							}
						}
					}
				}
			}
		}
		claims
	}

	/// Attempts to claim a counterparty HTLC-Success/HTLC-Timeout's outputs using the revocation key
	fn check_spend_counterparty_htlc<L: Deref>(&mut self, tx: &Transaction, commitment_number: u64, height: u32, logger: &L) -> (Vec<ClaimRequest>, Option<TransactionOutputs>) where L::Target: Logger {
		let htlc_txid = tx.txid();
		if tx.input.len() != 1 || tx.output.len() != 1 || tx.input[0].witness.len() != 5 {
			return (Vec::new(), None)
		}

		macro_rules! ignore_error {
			( $thing : expr ) => {
				match $thing {
					Ok(a) => a,
					Err(_) => return (Vec::new(), None)
				}
			};
		}

		let secret = if let Some(secret) = self.get_secret(commitment_number) { secret } else { return (Vec::new(), None); };
		let per_commitment_key = ignore_error!(SecretKey::from_slice(&secret));
		let per_commitment_point = PublicKey::from_secret_key(&self.secp_ctx, &per_commitment_key);

		log_trace!(logger, "Counterparty HTLC broadcast {}:{}", htlc_txid, 0);
		let witness_data = InputMaterial::Revoked { per_commitment_point, counterparty_delayed_payment_base_key: self.counterparty_tx_cache.counterparty_delayed_payment_base_key, counterparty_htlc_base_key: self.counterparty_tx_cache.counterparty_htlc_base_key,  per_commitment_key, input_descriptor: InputDescriptors::RevokedOutput, amount: tx.output[0].value, htlc: None, on_counterparty_tx_csv: self.counterparty_tx_cache.on_counterparty_tx_csv };
		let claimable_outpoints = vec!(ClaimRequest { absolute_timelock: height + self.counterparty_tx_cache.on_counterparty_tx_csv as u32, aggregable: true, outpoint: BitcoinOutPoint { txid: htlc_txid, vout: 0}, witness_data });
		let outputs = vec![(0, tx.output[0].clone())];
		(claimable_outpoints, Some((htlc_txid, outputs)))
	}

	// Returns (1) `ClaimRequest`s that can be given to the OnChainTxHandler, so that the handler can
	// broadcast transactions claiming holder HTLC commitment outputs and (2) a holder revokable
	// script so we can detect whether a holder transaction has been seen on-chain.
	fn get_broadcasted_holder_claims(&self, holder_tx: &HolderSignedTx) -> (Vec<ClaimRequest>, Option<(Script, PublicKey, PublicKey)>) {
		let mut claim_requests = Vec::with_capacity(holder_tx.htlc_outputs.len());

		let redeemscript = chan_utils::get_revokeable_redeemscript(&holder_tx.revocation_key, self.on_holder_tx_csv, &holder_tx.delayed_payment_key);
		let broadcasted_holder_revokable_script = Some((redeemscript.to_v0_p2wsh(), holder_tx.per_commitment_point.clone(), holder_tx.revocation_key.clone()));

		for &(ref htlc, _, _) in holder_tx.htlc_outputs.iter() {
			if let Some(transaction_output_index) = htlc.transaction_output_index {
				claim_requests.push(ClaimRequest { absolute_timelock: ::std::u32::MAX, aggregable: false, outpoint: BitcoinOutPoint { txid: holder_tx.txid, vout: transaction_output_index as u32 },
					witness_data: InputMaterial::HolderHTLC {
						preimage: if !htlc.offered {
								if let Some(preimage) = self.payment_preimages.get(&htlc.payment_hash) {
									Some(preimage.clone())
								} else {
									// We can't build an HTLC-Success transaction without the preimage
									continue;
								}
							} else { None },
						amount: htlc.amount_msat,
				}});
			}
		}

		(claim_requests, broadcasted_holder_revokable_script)
	}

	// Returns holder HTLC outputs to watch and react to in case of spending.
	fn get_broadcasted_holder_watch_outputs(&self, holder_tx: &HolderSignedTx, commitment_tx: &Transaction) -> Vec<(u32, TxOut)> {
		let mut watch_outputs = Vec::with_capacity(holder_tx.htlc_outputs.len());
		for &(ref htlc, _, _) in holder_tx.htlc_outputs.iter() {
			if let Some(transaction_output_index) = htlc.transaction_output_index {
				watch_outputs.push((transaction_output_index, commitment_tx.output[transaction_output_index as usize].clone()));
			}
		}
		watch_outputs
	}

	/// Attempts to claim any claimable HTLCs in a commitment transaction which was not (yet)
	/// revoked using data in holder_claimable_outpoints.
	/// Should not be used if check_spend_revoked_transaction succeeds.
	fn check_spend_holder_transaction<L: Deref>(&mut self, tx: &Transaction, height: u32, logger: &L) -> (Vec<ClaimRequest>, TransactionOutputs) where L::Target: Logger {
		let commitment_txid = tx.txid();
		let mut claim_requests = Vec::new();
		let mut watch_outputs = Vec::new();

		macro_rules! wait_threshold_conf {
			($source: expr, $commitment_tx: expr, $payment_hash: expr) => {
				self.onchain_events_awaiting_threshold_conf.retain(|ref entry| {
					if entry.height != height { return true; }
					match entry.event {
						 OnchainEvent::HTLCUpdate { ref htlc_update } => {
							 htlc_update.0 != $source
						 },
						 _ => true,
					}
				});
				let entry = OnchainEventEntry {
					txid: commitment_txid,
					height,
					event: OnchainEvent::HTLCUpdate { htlc_update: ($source, $payment_hash) },
				};
				log_trace!(logger, "Failing HTLC with payment_hash {} from {} holder commitment tx due to broadcast of transaction, waiting confirmation (at height{})", log_bytes!($payment_hash.0), $commitment_tx, entry.confirmation_threshold());
				self.onchain_events_awaiting_threshold_conf.push(entry);
			}
		}

		macro_rules! append_onchain_update {
			($updates: expr, $to_watch: expr) => {
				claim_requests = $updates.0;
				self.broadcasted_holder_revokable_script = $updates.1;
				watch_outputs.append(&mut $to_watch);
			}
		}

		// HTLCs set may differ between last and previous holder commitment txn, in case of one them hitting chain, ensure we cancel all HTLCs backward
		let mut is_holder_tx = false;

		if self.current_holder_commitment_tx.txid == commitment_txid {
			is_holder_tx = true;
			log_trace!(logger, "Got latest holder commitment tx broadcast, searching for available HTLCs to claim");
			let res = self.get_broadcasted_holder_claims(&self.current_holder_commitment_tx);
			let mut to_watch = self.get_broadcasted_holder_watch_outputs(&self.current_holder_commitment_tx, tx);
			append_onchain_update!(res, to_watch);
		} else if let &Some(ref holder_tx) = &self.prev_holder_signed_commitment_tx {
			if holder_tx.txid == commitment_txid {
				is_holder_tx = true;
				log_trace!(logger, "Got previous holder commitment tx broadcast, searching for available HTLCs to claim");
				let res = self.get_broadcasted_holder_claims(holder_tx);
				let mut to_watch = self.get_broadcasted_holder_watch_outputs(holder_tx, tx);
				append_onchain_update!(res, to_watch);
			}
		}

		macro_rules! fail_dust_htlcs_after_threshold_conf {
			($holder_tx: expr) => {
				for &(ref htlc, _, ref source) in &$holder_tx.htlc_outputs {
					if htlc.transaction_output_index.is_none() {
						if let &Some(ref source) = source {
							wait_threshold_conf!(source.clone(), "lastest", htlc.payment_hash.clone());
						}
					}
				}
			}
		}

		if is_holder_tx {
			fail_dust_htlcs_after_threshold_conf!(self.current_holder_commitment_tx);
			if let &Some(ref holder_tx) = &self.prev_holder_signed_commitment_tx {
				fail_dust_htlcs_after_threshold_conf!(holder_tx);
			}
		}

		(claim_requests, (commitment_txid, watch_outputs))
	}

	pub fn get_latest_holder_commitment_txn<L: Deref>(&mut self, logger: &L) -> Vec<Transaction> where L::Target: Logger {
		log_trace!(logger, "Getting signed latest holder commitment transaction!");
		self.holder_tx_signed = true;
		let commitment_tx = self.onchain_tx_handler.get_fully_signed_holder_tx(&self.funding_redeemscript);
		let txid = commitment_tx.txid();
		let mut res = vec![commitment_tx];
		for htlc in self.current_holder_commitment_tx.htlc_outputs.iter() {
			if let Some(vout) = htlc.0.transaction_output_index {
				let preimage = if !htlc.0.offered {
					if let Some(preimage) = self.payment_preimages.get(&htlc.0.payment_hash) { Some(preimage.clone()) } else {
						// We can't build an HTLC-Success transaction without the preimage
						continue;
					}
				} else { None };
				if let Some(htlc_tx) = self.onchain_tx_handler.get_fully_signed_htlc_tx(
					&::bitcoin::OutPoint { txid, vout }, &preimage) {
					res.push(htlc_tx);
				}
			}
		}
		// We throw away the generated waiting_first_conf data as we aren't (yet) confirmed and we don't actually know what the caller wants to do.
		// The data will be re-generated and tracked in check_spend_holder_transaction if we get a confirmation.
		return res;
	}

	#[cfg(any(test,feature = "unsafe_revoked_tx_signing"))]
	fn unsafe_get_latest_holder_commitment_txn<L: Deref>(&mut self, logger: &L) -> Vec<Transaction> where L::Target: Logger {
		log_trace!(logger, "Getting signed copy of latest holder commitment transaction!");
		let commitment_tx = self.onchain_tx_handler.get_fully_signed_copy_holder_tx(&self.funding_redeemscript);
		let txid = commitment_tx.txid();
		let mut res = vec![commitment_tx];
		for htlc in self.current_holder_commitment_tx.htlc_outputs.iter() {
			if let Some(vout) = htlc.0.transaction_output_index {
				let preimage = if !htlc.0.offered {
					if let Some(preimage) = self.payment_preimages.get(&htlc.0.payment_hash) { Some(preimage.clone()) } else {
						// We can't build an HTLC-Success transaction without the preimage
						continue;
					}
				} else { None };
				if let Some(htlc_tx) = self.onchain_tx_handler.unsafe_get_fully_signed_htlc_tx(
					&::bitcoin::OutPoint { txid, vout }, &preimage) {
					res.push(htlc_tx);
				}
			}
		}
		return res
	}

	pub fn block_connected<B: Deref, F: Deref, L: Deref>(&mut self, header: &BlockHeader, txdata: &TransactionData, height: u32, broadcaster: B, fee_estimator: F, logger: L) -> Vec<TransactionOutputs>
		where B::Target: BroadcasterInterface,
		      F::Target: FeeEstimator,
					L::Target: Logger,
	{
		let block_hash = header.block_hash();
		log_trace!(logger, "New best block {} at height {}", block_hash, height);
		self.best_block = BestBlock::new(block_hash, height);

		self.transactions_confirmed(header, txdata, height, broadcaster, fee_estimator, logger)
	}

	fn best_block_updated<B: Deref, F: Deref, L: Deref>(
		&mut self,
		header: &BlockHeader,
		height: u32,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) -> Vec<TransactionOutputs>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		let block_hash = header.block_hash();
		log_trace!(logger, "New best block {} at height {}", block_hash, height);

		if height > self.best_block.height() {
			self.best_block = BestBlock::new(block_hash, height);
			self.block_confirmed(height, vec![], vec![], vec![], broadcaster, fee_estimator, logger)
		} else {
			self.best_block = BestBlock::new(block_hash, height);
			self.onchain_events_awaiting_threshold_conf.retain(|ref entry| entry.height <= height);
			self.onchain_tx_handler.block_disconnected(height + 1, broadcaster, fee_estimator, logger);
			Vec::new()
		}
	}

	fn transactions_confirmed<B: Deref, F: Deref, L: Deref>(
		&mut self,
		header: &BlockHeader,
		txdata: &TransactionData,
		height: u32,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) -> Vec<TransactionOutputs>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		let txn_matched = self.filter_block(txdata);
		for tx in &txn_matched {
			let mut output_val = 0;
			for out in tx.output.iter() {
				if out.value > 21_000_000_0000_0000 { panic!("Value-overflowing transaction provided to block connected"); }
				output_val += out.value;
				if output_val > 21_000_000_0000_0000 { panic!("Value-overflowing transaction provided to block connected"); }
			}
		}

		let block_hash = header.block_hash();
		log_trace!(logger, "Block {} at height {} connected with {} txn matched", block_hash, height, txn_matched.len());

		let mut watch_outputs = Vec::new();
		let mut claimable_outpoints = Vec::new();
		for tx in &txn_matched {
			if tx.input.len() == 1 {
				// Assuming our keys were not leaked (in which case we're screwed no matter what),
				// commitment transactions and HTLC transactions will all only ever have one input,
				// which is an easy way to filter out any potential non-matching txn for lazy
				// filters.
				let prevout = &tx.input[0].previous_output;
				if prevout.txid == self.funding_info.0.txid && prevout.vout == self.funding_info.0.index as u32 {
					if (tx.input[0].sequence >> 8*3) as u8 == 0x80 && (tx.lock_time >> 8*3) as u8 == 0x20 {
						let (mut new_outpoints, new_outputs) = self.check_spend_counterparty_transaction(&tx, height, &logger);
						if !new_outputs.1.is_empty() {
							watch_outputs.push(new_outputs);
						}
						if new_outpoints.is_empty() {
							let (mut new_outpoints, new_outputs) = self.check_spend_holder_transaction(&tx, height, &logger);
							if !new_outputs.1.is_empty() {
								watch_outputs.push(new_outputs);
							}
							claimable_outpoints.append(&mut new_outpoints);
						}
						claimable_outpoints.append(&mut new_outpoints);
					}
				} else {
					if let Some(&commitment_number) = self.counterparty_commitment_txn_on_chain.get(&prevout.txid) {
						let (mut new_outpoints, new_outputs_option) = self.check_spend_counterparty_htlc(&tx, commitment_number, height, &logger);
						claimable_outpoints.append(&mut new_outpoints);
						if let Some(new_outputs) = new_outputs_option {
							watch_outputs.push(new_outputs);
						}
					}
				}
			}
			// While all commitment/HTLC-Success/HTLC-Timeout transactions have one input, HTLCs
			// can also be resolved in a few other ways which can have more than one output. Thus,
			// we call is_resolving_htlc_output here outside of the tx.input.len() == 1 check.
			self.is_resolving_htlc_output(&tx, height, &logger);

			self.is_paying_spendable_output(&tx, height, &logger);
		}

		self.block_confirmed(height, txn_matched, watch_outputs, claimable_outpoints, broadcaster, fee_estimator, logger)
	}

	fn block_confirmed<B: Deref, F: Deref, L: Deref>(
		&mut self,
		height: u32,
		txn_matched: Vec<&Transaction>,
		mut watch_outputs: Vec<TransactionOutputs>,
		mut claimable_outpoints: Vec<ClaimRequest>,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) -> Vec<TransactionOutputs>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		let should_broadcast = self.would_broadcast_at_height(height, &logger);
		if should_broadcast {
			claimable_outpoints.push(ClaimRequest { absolute_timelock: height, aggregable: false, outpoint: BitcoinOutPoint { txid: self.funding_info.0.txid.clone(), vout: self.funding_info.0.index as u32 }, witness_data: InputMaterial::Funding { funding_redeemscript: self.funding_redeemscript.clone() }});
			self.pending_monitor_events.push(MonitorEvent::CommitmentTxBroadcasted(self.funding_info.0));
			let commitment_tx = self.onchain_tx_handler.get_fully_signed_holder_tx(&self.funding_redeemscript);
			self.holder_tx_signed = true;
			let (mut new_outpoints, _) = self.get_broadcasted_holder_claims(&self.current_holder_commitment_tx);
			let new_outputs = self.get_broadcasted_holder_watch_outputs(&self.current_holder_commitment_tx, &commitment_tx);
			if !new_outputs.is_empty() {
				watch_outputs.push((self.current_holder_commitment_tx.txid.clone(), new_outputs));
			}
			claimable_outpoints.append(&mut new_outpoints);
		}

		// Find which on-chain events have reached their confirmation threshold.
		let onchain_events_awaiting_threshold_conf =
			self.onchain_events_awaiting_threshold_conf.drain(..).collect::<Vec<_>>();
		let mut onchain_events_reaching_threshold_conf = Vec::new();
		for entry in onchain_events_awaiting_threshold_conf {
			if entry.has_reached_confirmation_threshold(height) {
				onchain_events_reaching_threshold_conf.push(entry);
			} else {
				self.onchain_events_awaiting_threshold_conf.push(entry);
			}
		}

		// Used to check for duplicate HTLC resolutions.
		#[cfg(debug_assertions)]
		let unmatured_htlcs: Vec<_> = self.onchain_events_awaiting_threshold_conf
			.iter()
			.filter_map(|entry| match &entry.event {
				OnchainEvent::HTLCUpdate { htlc_update } => Some(htlc_update.0.clone()),
				OnchainEvent::MaturingOutput { .. } => None,
			})
			.collect();
		#[cfg(debug_assertions)]
		let mut matured_htlcs = Vec::new();

		// Produce actionable events from on-chain events having reached their threshold.
		for entry in onchain_events_reaching_threshold_conf.drain(..) {
			match entry.event {
				OnchainEvent::HTLCUpdate { htlc_update } => {
					// Check for duplicate HTLC resolutions.
					#[cfg(debug_assertions)]
					{
						debug_assert!(
							unmatured_htlcs.iter().find(|&htlc| htlc == &htlc_update.0).is_none(),
							"An unmature HTLC transaction conflicts with a maturing one; failed to \
							 call either transaction_unconfirmed for the conflicting transaction \
							 or block_disconnected for a block containing it.");
						debug_assert!(
							matured_htlcs.iter().find(|&htlc| htlc == &htlc_update.0).is_none(),
							"A matured HTLC transaction conflicts with a maturing one; failed to \
							 call either transaction_unconfirmed for the conflicting transaction \
							 or block_disconnected for a block containing it.");
						matured_htlcs.push(htlc_update.0.clone());
					}

					log_trace!(logger, "HTLC {} failure update has got enough confirmations to be passed upstream", log_bytes!((htlc_update.1).0));
					self.pending_monitor_events.push(MonitorEvent::HTLCEvent(HTLCUpdate {
						payment_hash: htlc_update.1,
						payment_preimage: None,
						source: htlc_update.0,
					}));
				},
				OnchainEvent::MaturingOutput { descriptor } => {
					log_trace!(logger, "Descriptor {} has got enough confirmations to be passed upstream", log_spendable!(descriptor));
					self.pending_events.push(Event::SpendableOutputs {
						outputs: vec![descriptor]
					});
				}
			}
		}

		self.onchain_tx_handler.update_claims_view(&txn_matched, claimable_outpoints, Some(height), &&*broadcaster, &&*fee_estimator, &&*logger);

		// Determine new outputs to watch by comparing against previously known outputs to watch,
		// updating the latter in the process.
		watch_outputs.retain(|&(ref txid, ref txouts)| {
			let idx_and_scripts = txouts.iter().map(|o| (o.0, o.1.script_pubkey.clone())).collect();
			self.outputs_to_watch.insert(txid.clone(), idx_and_scripts).is_none()
		});
		#[cfg(test)]
		{
		        // If we see a transaction for which we registered outputs previously,
			// make sure the registered scriptpubkey at the expected index match
			// the actual transaction output one. We failed this case before #653.
			for tx in &txn_matched {
				if let Some(outputs) = self.get_outputs_to_watch().get(&tx.txid()) {
					for idx_and_script in outputs.iter() {
						assert!((idx_and_script.0 as usize) < tx.output.len());
						assert_eq!(tx.output[idx_and_script.0 as usize].script_pubkey, idx_and_script.1);
					}
				}
			}
		}
		watch_outputs
	}

	pub fn block_disconnected<B: Deref, F: Deref, L: Deref>(&mut self, header: &BlockHeader, height: u32, broadcaster: B, fee_estimator: F, logger: L)
		where B::Target: BroadcasterInterface,
		      F::Target: FeeEstimator,
		      L::Target: Logger,
	{
		log_trace!(logger, "Block {} at height {} disconnected", header.block_hash(), height);

		//We may discard:
		//- htlc update there as failure-trigger tx (revoked commitment tx, non-revoked commitment tx, HTLC-timeout tx) has been disconnected
		//- maturing spendable output has transaction paying us has been disconnected
		self.onchain_events_awaiting_threshold_conf.retain(|ref entry| entry.height < height);

		self.onchain_tx_handler.block_disconnected(height, broadcaster, fee_estimator, logger);

		self.best_block = BestBlock::new(header.prev_blockhash, height - 1);
	}

	fn transaction_unconfirmed<B: Deref, F: Deref, L: Deref>(
		&mut self,
		txid: &Txid,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.onchain_events_awaiting_threshold_conf.retain(|ref entry| entry.txid != *txid);
		self.onchain_tx_handler.transaction_unconfirmed(txid, broadcaster, fee_estimator, logger);
	}

	/// Filters a block's `txdata` for transactions spending watched outputs or for any child
	/// transactions thereof.
	fn filter_block<'a>(&self, txdata: &TransactionData<'a>) -> Vec<&'a Transaction> {
		let mut matched_txn = HashSet::new();
		txdata.iter().filter(|&&(_, tx)| {
			let mut matches = self.spends_watched_output(tx);
			for input in tx.input.iter() {
				if matches { break; }
				if matched_txn.contains(&input.previous_output.txid) {
					matches = true;
				}
			}
			if matches {
				matched_txn.insert(tx.txid());
			}
			matches
		}).map(|(_, tx)| *tx).collect()
	}

	/// Checks if a given transaction spends any watched outputs.
	fn spends_watched_output(&self, tx: &Transaction) -> bool {
		for input in tx.input.iter() {
			if let Some(outputs) = self.get_outputs_to_watch().get(&input.previous_output.txid) {
				for (idx, _script_pubkey) in outputs.iter() {
					if *idx == input.previous_output.vout {
						#[cfg(test)]
						{
						        // If the expected script is a known type, check that the witness
						        // appears to be spending the correct type (ie that the match would
						        // actually succeed in BIP 158/159-style filters).
						        if _script_pubkey.is_v0_p2wsh() {
						                assert_eq!(&bitcoin::Address::p2wsh(&Script::from(input.witness.last().unwrap().clone()), bitcoin::Network::Bitcoin).script_pubkey(), _script_pubkey);
						        } else if _script_pubkey.is_v0_p2wpkh() {
						                assert_eq!(&bitcoin::Address::p2wpkh(&bitcoin::PublicKey::from_slice(&input.witness.last().unwrap()).unwrap(), bitcoin::Network::Bitcoin).unwrap().script_pubkey(), _script_pubkey);
						        } else { panic!(); }
						}
						return true;
					}
				}
			}
		}

		false
	}

	fn would_broadcast_at_height<L: Deref>(&self, height: u32, logger: &L) -> bool where L::Target: Logger {
		// We need to consider all HTLCs which are:
		//  * in any unrevoked counterparty commitment transaction, as they could broadcast said
		//    transactions and we'd end up in a race, or
		//  * are in our latest holder commitment transaction, as this is the thing we will
		//    broadcast if we go on-chain.
		// Note that we consider HTLCs which were below dust threshold here - while they don't
		// strictly imply that we need to fail the channel, we need to go ahead and fail them back
		// to the source, and if we don't fail the channel we will have to ensure that the next
		// updates that peer sends us are update_fails, failing the channel if not. It's probably
		// easier to just fail the channel as this case should be rare enough anyway.
		macro_rules! scan_commitment {
			($htlcs: expr, $holder_tx: expr) => {
				for ref htlc in $htlcs {
					// For inbound HTLCs which we know the preimage for, we have to ensure we hit the
					// chain with enough room to claim the HTLC without our counterparty being able to
					// time out the HTLC first.
					// For outbound HTLCs which our counterparty hasn't failed/claimed, our primary
					// concern is being able to claim the corresponding inbound HTLC (on another
					// channel) before it expires. In fact, we don't even really care if our
					// counterparty here claims such an outbound HTLC after it expired as long as we
					// can still claim the corresponding HTLC. Thus, to avoid needlessly hitting the
					// chain when our counterparty is waiting for expiration to off-chain fail an HTLC
					// we give ourselves a few blocks of headroom after expiration before going
					// on-chain for an expired HTLC.
					// Note that, to avoid a potential attack whereby a node delays claiming an HTLC
					// from us until we've reached the point where we go on-chain with the
					// corresponding inbound HTLC, we must ensure that outbound HTLCs go on chain at
					// least CLTV_CLAIM_BUFFER blocks prior to the inbound HTLC.
					//  aka outbound_cltv + LATENCY_GRACE_PERIOD_BLOCKS == height - CLTV_CLAIM_BUFFER
					//      inbound_cltv == height + CLTV_CLAIM_BUFFER
					//      outbound_cltv + LATENCY_GRACE_PERIOD_BLOCKS + CLTV_CLAIM_BUFFER <= inbound_cltv - CLTV_CLAIM_BUFFER
					//      LATENCY_GRACE_PERIOD_BLOCKS + 2*CLTV_CLAIM_BUFFER <= inbound_cltv - outbound_cltv
					//      CLTV_EXPIRY_DELTA <= inbound_cltv - outbound_cltv (by check in ChannelManager::decode_update_add_htlc_onion)
					//      LATENCY_GRACE_PERIOD_BLOCKS + 2*CLTV_CLAIM_BUFFER <= CLTV_EXPIRY_DELTA
					//  The final, above, condition is checked for statically in channelmanager
					//  with CHECK_CLTV_EXPIRY_SANITY_2.
					let htlc_outbound = $holder_tx == htlc.offered;
					if ( htlc_outbound && htlc.cltv_expiry + LATENCY_GRACE_PERIOD_BLOCKS <= height) ||
					   (!htlc_outbound && htlc.cltv_expiry <= height + CLTV_CLAIM_BUFFER && self.payment_preimages.contains_key(&htlc.payment_hash)) {
						log_info!(logger, "Force-closing channel due to {} HTLC timeout, HTLC expiry is {}", if htlc_outbound { "outbound" } else { "inbound "}, htlc.cltv_expiry);
						return true;
					}
				}
			}
		}

		scan_commitment!(self.current_holder_commitment_tx.htlc_outputs.iter().map(|&(ref a, _, _)| a), true);

		if let Some(ref txid) = self.current_counterparty_commitment_txid {
			if let Some(ref htlc_outputs) = self.counterparty_claimable_outpoints.get(txid) {
				scan_commitment!(htlc_outputs.iter().map(|&(ref a, _)| a), false);
			}
		}
		if let Some(ref txid) = self.prev_counterparty_commitment_txid {
			if let Some(ref htlc_outputs) = self.counterparty_claimable_outpoints.get(txid) {
				scan_commitment!(htlc_outputs.iter().map(|&(ref a, _)| a), false);
			}
		}

		false
	}

	/// Check if any transaction broadcasted is resolving HTLC output by a success or timeout on a holder
	/// or counterparty commitment tx, if so send back the source, preimage if found and payment_hash of resolved HTLC
	fn is_resolving_htlc_output<L: Deref>(&mut self, tx: &Transaction, height: u32, logger: &L) where L::Target: Logger {
		'outer_loop: for input in &tx.input {
			let mut payment_data = None;
			let revocation_sig_claim = (input.witness.len() == 3 && HTLCType::scriptlen_to_htlctype(input.witness[2].len()) == Some(HTLCType::OfferedHTLC) && input.witness[1].len() == 33)
				|| (input.witness.len() == 3 && HTLCType::scriptlen_to_htlctype(input.witness[2].len()) == Some(HTLCType::AcceptedHTLC) && input.witness[1].len() == 33);
			let accepted_preimage_claim = input.witness.len() == 5 && HTLCType::scriptlen_to_htlctype(input.witness[4].len()) == Some(HTLCType::AcceptedHTLC);
			let offered_preimage_claim = input.witness.len() == 3 && HTLCType::scriptlen_to_htlctype(input.witness[2].len()) == Some(HTLCType::OfferedHTLC);

			macro_rules! log_claim {
				($tx_info: expr, $holder_tx: expr, $htlc: expr, $source_avail: expr) => {
					// We found the output in question, but aren't failing it backwards
					// as we have no corresponding source and no valid counterparty commitment txid
					// to try a weak source binding with same-hash, same-value still-valid offered HTLC.
					// This implies either it is an inbound HTLC or an outbound HTLC on a revoked transaction.
					let outbound_htlc = $holder_tx == $htlc.offered;
					if ($holder_tx && revocation_sig_claim) ||
							(outbound_htlc && !$source_avail && (accepted_preimage_claim || offered_preimage_claim)) {
						log_error!(logger, "Input spending {} ({}:{}) in {} resolves {} HTLC with payment hash {} with {}!",
							$tx_info, input.previous_output.txid, input.previous_output.vout, tx.txid(),
							if outbound_htlc { "outbound" } else { "inbound" }, log_bytes!($htlc.payment_hash.0),
							if revocation_sig_claim { "revocation sig" } else { "preimage claim after we'd passed the HTLC resolution back" });
					} else {
						log_info!(logger, "Input spending {} ({}:{}) in {} resolves {} HTLC with payment hash {} with {}",
							$tx_info, input.previous_output.txid, input.previous_output.vout, tx.txid(),
							if outbound_htlc { "outbound" } else { "inbound" }, log_bytes!($htlc.payment_hash.0),
							if revocation_sig_claim { "revocation sig" } else if accepted_preimage_claim || offered_preimage_claim { "preimage" } else { "timeout" });
					}
				}
			}

			macro_rules! check_htlc_valid_counterparty {
				($counterparty_txid: expr, $htlc_output: expr) => {
					if let Some(txid) = $counterparty_txid {
						for &(ref pending_htlc, ref pending_source) in self.counterparty_claimable_outpoints.get(&txid).unwrap() {
							if pending_htlc.payment_hash == $htlc_output.payment_hash && pending_htlc.amount_msat == $htlc_output.amount_msat {
								if let &Some(ref source) = pending_source {
									log_claim!("revoked counterparty commitment tx", false, pending_htlc, true);
									payment_data = Some(((**source).clone(), $htlc_output.payment_hash));
									break;
								}
							}
						}
					}
				}
			}

			macro_rules! scan_commitment {
				($htlcs: expr, $tx_info: expr, $holder_tx: expr) => {
					for (ref htlc_output, source_option) in $htlcs {
						if Some(input.previous_output.vout) == htlc_output.transaction_output_index {
							if let Some(ref source) = source_option {
								log_claim!($tx_info, $holder_tx, htlc_output, true);
								// We have a resolution of an HTLC either from one of our latest
								// holder commitment transactions or an unrevoked counterparty commitment
								// transaction. This implies we either learned a preimage, the HTLC
								// has timed out, or we screwed up. In any case, we should now
								// resolve the source HTLC with the original sender.
								payment_data = Some(((*source).clone(), htlc_output.payment_hash));
							} else if !$holder_tx {
									check_htlc_valid_counterparty!(self.current_counterparty_commitment_txid, htlc_output);
								if payment_data.is_none() {
									check_htlc_valid_counterparty!(self.prev_counterparty_commitment_txid, htlc_output);
								}
							}
							if payment_data.is_none() {
								log_claim!($tx_info, $holder_tx, htlc_output, false);
								continue 'outer_loop;
							}
						}
					}
				}
			}

			if input.previous_output.txid == self.current_holder_commitment_tx.txid {
				scan_commitment!(self.current_holder_commitment_tx.htlc_outputs.iter().map(|&(ref a, _, ref b)| (a, b.as_ref())),
					"our latest holder commitment tx", true);
			}
			if let Some(ref prev_holder_signed_commitment_tx) = self.prev_holder_signed_commitment_tx {
				if input.previous_output.txid == prev_holder_signed_commitment_tx.txid {
					scan_commitment!(prev_holder_signed_commitment_tx.htlc_outputs.iter().map(|&(ref a, _, ref b)| (a, b.as_ref())),
						"our previous holder commitment tx", true);
				}
			}
			if let Some(ref htlc_outputs) = self.counterparty_claimable_outpoints.get(&input.previous_output.txid) {
				scan_commitment!(htlc_outputs.iter().map(|&(ref a, ref b)| (a, (b.as_ref().clone()).map(|boxed| &**boxed))),
					"counterparty commitment tx", false);
			}

			// Check that scan_commitment, above, decided there is some source worth relaying an
			// HTLC resolution backwards to and figure out whether we learned a preimage from it.
			if let Some((source, payment_hash)) = payment_data {
				let mut payment_preimage = PaymentPreimage([0; 32]);
				if accepted_preimage_claim {
					if !self.pending_monitor_events.iter().any(
						|update| if let &MonitorEvent::HTLCEvent(ref upd) = update { upd.source == source } else { false }) {
						payment_preimage.0.copy_from_slice(&input.witness[3]);
						self.pending_monitor_events.push(MonitorEvent::HTLCEvent(HTLCUpdate {
							source,
							payment_preimage: Some(payment_preimage),
							payment_hash
						}));
					}
				} else if offered_preimage_claim {
					if !self.pending_monitor_events.iter().any(
						|update| if let &MonitorEvent::HTLCEvent(ref upd) = update {
							upd.source == source
						} else { false }) {
						payment_preimage.0.copy_from_slice(&input.witness[1]);
						self.pending_monitor_events.push(MonitorEvent::HTLCEvent(HTLCUpdate {
							source,
							payment_preimage: Some(payment_preimage),
							payment_hash
						}));
					}
				} else {
					self.onchain_events_awaiting_threshold_conf.retain(|ref entry| {
						if entry.height != height { return true; }
						match entry.event {
							 OnchainEvent::HTLCUpdate { ref htlc_update } => {
								 htlc_update.0 != source
							 },
							 _ => true,
						}
					});
					let entry = OnchainEventEntry {
						txid: tx.txid(),
						height,
						event: OnchainEvent::HTLCUpdate { htlc_update: (source, payment_hash) },
					};
					log_info!(logger, "Failing HTLC with payment_hash {} timeout by a spend tx, waiting for confirmation (at height{})", log_bytes!(payment_hash.0), entry.confirmation_threshold());
					self.onchain_events_awaiting_threshold_conf.push(entry);
				}
			}
		}
	}

	/// Check if any transaction broadcasted is paying fund back to some address we can assume to own
	fn is_paying_spendable_output<L: Deref>(&mut self, tx: &Transaction, height: u32, logger: &L) where L::Target: Logger {
		let mut spendable_output = None;
		for (i, outp) in tx.output.iter().enumerate() { // There is max one spendable output for any channel tx, including ones generated by us
			if i > ::std::u16::MAX as usize {
				// While it is possible that an output exists on chain which is greater than the
				// 2^16th output in a given transaction, this is only possible if the output is not
				// in a lightning transaction and was instead placed there by some third party who
				// wishes to give us money for no reason.
				// Namely, any lightning transactions which we pre-sign will never have anywhere
				// near 2^16 outputs both because such transactions must have ~2^16 outputs who's
				// scripts are not longer than one byte in length and because they are inherently
				// non-standard due to their size.
				// Thus, it is completely safe to ignore such outputs, and while it may result in
				// us ignoring non-lightning fund to us, that is only possible if someone fills
				// nearly a full block with garbage just to hit this case.
				continue;
			}
			if outp.script_pubkey == self.destination_script {
				spendable_output =  Some(SpendableOutputDescriptor::StaticOutput {
					outpoint: OutPoint { txid: tx.txid(), index: i as u16 },
					output: outp.clone(),
				});
				break;
			} else if let Some(ref broadcasted_holder_revokable_script) = self.broadcasted_holder_revokable_script {
				if broadcasted_holder_revokable_script.0 == outp.script_pubkey {
					spendable_output =  Some(SpendableOutputDescriptor::DelayedPaymentOutput(DelayedPaymentOutputDescriptor {
						outpoint: OutPoint { txid: tx.txid(), index: i as u16 },
						per_commitment_point: broadcasted_holder_revokable_script.1,
						to_self_delay: self.on_holder_tx_csv,
						output: outp.clone(),
						revocation_pubkey: broadcasted_holder_revokable_script.2.clone(),
						channel_keys_id: self.channel_keys_id,
						channel_value_satoshis: self.channel_value_satoshis,
					}));
					break;
				}
			} else if self.counterparty_payment_script == outp.script_pubkey {
				spendable_output = Some(SpendableOutputDescriptor::StaticPaymentOutput(StaticPaymentOutputDescriptor {
					outpoint: OutPoint { txid: tx.txid(), index: i as u16 },
					output: outp.clone(),
					channel_keys_id: self.channel_keys_id,
					channel_value_satoshis: self.channel_value_satoshis,
				}));
				break;
			} else if outp.script_pubkey == self.shutdown_script {
				spendable_output = Some(SpendableOutputDescriptor::StaticOutput {
					outpoint: OutPoint { txid: tx.txid(), index: i as u16 },
					output: outp.clone(),
				});
			}
		}
		if let Some(spendable_output) = spendable_output {
			let entry = OnchainEventEntry {
				txid: tx.txid(),
				height: height,
				event: OnchainEvent::MaturingOutput { descriptor: spendable_output.clone() },
			};
			log_trace!(logger, "Maturing {} until {}", log_spendable!(spendable_output), entry.confirmation_threshold());
			self.onchain_events_awaiting_threshold_conf.push(entry);
		}
	}
}

/// `Persist` defines behavior for persisting channel monitors: this could mean
/// writing once to disk, and/or uploading to one or more backup services.
///
/// Note that for every new monitor, you **must** persist the new `ChannelMonitor`
/// to disk/backups. And, on every update, you **must** persist either the
/// `ChannelMonitorUpdate` or the updated monitor itself. Otherwise, there is risk
/// of situations such as revoking a transaction, then crashing before this
/// revocation can be persisted, then unintentionally broadcasting a revoked
/// transaction and losing money. This is a risk because previous channel states
/// are toxic, so it's important that whatever channel state is persisted is
/// kept up-to-date.
pub trait Persist<ChannelSigner: Sign> {
	/// Persist a new channel's data. The data can be stored any way you want, but
	/// the identifier provided by Rust-Lightning is the channel's outpoint (and
	/// it is up to you to maintain a correct mapping between the outpoint and the
	/// stored channel data). Note that you **must** persist every new monitor to
	/// disk. See the `Persist` trait documentation for more details.
	///
	/// See [`ChannelMonitor::write`] for writing out a `ChannelMonitor`,
	/// and [`ChannelMonitorUpdateErr`] for requirements when returning errors.
	fn persist_new_channel(&self, id: OutPoint, data: &ChannelMonitor<ChannelSigner>) -> Result<(), ChannelMonitorUpdateErr>;

	/// Update one channel's data. The provided `ChannelMonitor` has already
	/// applied the given update.
	///
	/// Note that on every update, you **must** persist either the
	/// `ChannelMonitorUpdate` or the updated monitor itself to disk/backups. See
	/// the `Persist` trait documentation for more details.
	///
	/// If an implementer chooses to persist the updates only, they need to make
	/// sure that all the updates are applied to the `ChannelMonitors` *before*
	/// the set of channel monitors is given to the `ChannelManager`
	/// deserialization routine. See [`ChannelMonitor::update_monitor`] for
	/// applying a monitor update to a monitor. If full `ChannelMonitors` are
	/// persisted, then there is no need to persist individual updates.
	///
	/// Note that there could be a performance tradeoff between persisting complete
	/// channel monitors on every update vs. persisting only updates and applying
	/// them in batches. The size of each monitor grows `O(number of state updates)`
	/// whereas updates are small and `O(1)`.
	///
	/// See [`ChannelMonitor::write`] for writing out a `ChannelMonitor`,
	/// [`ChannelMonitorUpdate::write`] for writing out an update, and
	/// [`ChannelMonitorUpdateErr`] for requirements when returning errors.
	fn update_persisted_channel(&self, id: OutPoint, update: &ChannelMonitorUpdate, data: &ChannelMonitor<ChannelSigner>) -> Result<(), ChannelMonitorUpdateErr>;
}

impl<Signer: Sign, T: Deref, F: Deref, L: Deref> chain::Listen for (ChannelMonitor<Signer>, T, F, L)
where
	T::Target: BroadcasterInterface,
	F::Target: FeeEstimator,
	L::Target: Logger,
{
	fn block_connected(&self, block: &Block, height: u32) {
		let txdata: Vec<_> = block.txdata.iter().enumerate().collect();
		self.0.block_connected(&block.header, &txdata, height, &*self.1, &*self.2, &*self.3);
	}

	fn block_disconnected(&self, header: &BlockHeader, height: u32) {
		self.0.block_disconnected(header, height, &*self.1, &*self.2, &*self.3);
	}
}

impl<Signer: Sign, T: Deref, F: Deref, L: Deref> chain::Confirm for (ChannelMonitor<Signer>, T, F, L)
where
	T::Target: BroadcasterInterface,
	F::Target: FeeEstimator,
	L::Target: Logger,
{
	fn transactions_confirmed(&self, header: &BlockHeader, txdata: &TransactionData, height: u32) {
		self.0.transactions_confirmed(header, txdata, height, &*self.1, &*self.2, &*self.3);
	}

	fn transaction_unconfirmed(&self, txid: &Txid) {
		self.0.transaction_unconfirmed(txid, &*self.1, &*self.2, &*self.3);
	}

	fn best_block_updated(&self, header: &BlockHeader, height: u32) {
		self.0.best_block_updated(header, height, &*self.1, &*self.2, &*self.3);
	}

	fn get_relevant_txids(&self) -> Vec<Txid> {
		self.0.get_relevant_txids()
	}
}

const MAX_ALLOC_SIZE: usize = 64*1024;

impl<'a, Signer: Sign, K: KeysInterface<Signer = Signer>> ReadableArgs<&'a K>
		for (BlockHash, ChannelMonitor<Signer>) {
	fn read<R: ::std::io::Read>(reader: &mut R, keys_manager: &'a K) -> Result<Self, DecodeError> {
		macro_rules! unwrap_obj {
			($key: expr) => {
				match $key {
					Ok(res) => res,
					Err(_) => return Err(DecodeError::InvalidValue),
				}
			}
		}

		let _ver: u8 = Readable::read(reader)?;
		let min_ver: u8 = Readable::read(reader)?;
		if min_ver > SERIALIZATION_VERSION {
			return Err(DecodeError::UnknownVersion);
		}

		let latest_update_id: u64 = Readable::read(reader)?;
		let commitment_transaction_number_obscure_factor = <U48 as Readable>::read(reader)?.0;

		let destination_script = Readable::read(reader)?;
		let broadcasted_holder_revokable_script = match <u8 as Readable>::read(reader)? {
			0 => {
				let revokable_address = Readable::read(reader)?;
				let per_commitment_point = Readable::read(reader)?;
				let revokable_script = Readable::read(reader)?;
				Some((revokable_address, per_commitment_point, revokable_script))
			},
			1 => { None },
			_ => return Err(DecodeError::InvalidValue),
		};
		let counterparty_payment_script = Readable::read(reader)?;
		let shutdown_script = Readable::read(reader)?;

		let channel_keys_id = Readable::read(reader)?;
		let holder_revocation_basepoint = Readable::read(reader)?;
		// Technically this can fail and serialize fail a round-trip, but only for serialization of
		// barely-init'd ChannelMonitors that we can't do anything with.
		let outpoint = OutPoint {
			txid: Readable::read(reader)?,
			index: Readable::read(reader)?,
		};
		let funding_info = (outpoint, Readable::read(reader)?);
		let current_counterparty_commitment_txid = Readable::read(reader)?;
		let prev_counterparty_commitment_txid = Readable::read(reader)?;

		let counterparty_tx_cache = Readable::read(reader)?;
		let funding_redeemscript = Readable::read(reader)?;
		let channel_value_satoshis = Readable::read(reader)?;

		let their_cur_revocation_points = {
			let first_idx = <U48 as Readable>::read(reader)?.0;
			if first_idx == 0 {
				None
			} else {
				let first_point = Readable::read(reader)?;
				let second_point_slice: [u8; 33] = Readable::read(reader)?;
				if second_point_slice[0..32] == [0; 32] && second_point_slice[32] == 0 {
					Some((first_idx, first_point, None))
				} else {
					Some((first_idx, first_point, Some(unwrap_obj!(PublicKey::from_slice(&second_point_slice)))))
				}
			}
		};

		let on_holder_tx_csv: u16 = Readable::read(reader)?;

		let commitment_secrets = Readable::read(reader)?;

		macro_rules! read_htlc_in_commitment {
			() => {
				{
					let offered: bool = Readable::read(reader)?;
					let amount_msat: u64 = Readable::read(reader)?;
					let cltv_expiry: u32 = Readable::read(reader)?;
					let payment_hash: PaymentHash = Readable::read(reader)?;
					let transaction_output_index: Option<u32> = Readable::read(reader)?;

					HTLCOutputInCommitment {
						offered, amount_msat, cltv_expiry, payment_hash, transaction_output_index
					}
				}
			}
		}

		let counterparty_claimable_outpoints_len: u64 = Readable::read(reader)?;
		let mut counterparty_claimable_outpoints = HashMap::with_capacity(cmp::min(counterparty_claimable_outpoints_len as usize, MAX_ALLOC_SIZE / 64));
		for _ in 0..counterparty_claimable_outpoints_len {
			let txid: Txid = Readable::read(reader)?;
			let htlcs_count: u64 = Readable::read(reader)?;
			let mut htlcs = Vec::with_capacity(cmp::min(htlcs_count as usize, MAX_ALLOC_SIZE / 32));
			for _ in 0..htlcs_count {
				htlcs.push((read_htlc_in_commitment!(), <Option<HTLCSource> as Readable>::read(reader)?.map(|o: HTLCSource| Box::new(o))));
			}
			if let Some(_) = counterparty_claimable_outpoints.insert(txid, htlcs) {
				return Err(DecodeError::InvalidValue);
			}
		}

		let counterparty_commitment_txn_on_chain_len: u64 = Readable::read(reader)?;
		let mut counterparty_commitment_txn_on_chain = HashMap::with_capacity(cmp::min(counterparty_commitment_txn_on_chain_len as usize, MAX_ALLOC_SIZE / 32));
		for _ in 0..counterparty_commitment_txn_on_chain_len {
			let txid: Txid = Readable::read(reader)?;
			let commitment_number = <U48 as Readable>::read(reader)?.0;
			if let Some(_) = counterparty_commitment_txn_on_chain.insert(txid, commitment_number) {
				return Err(DecodeError::InvalidValue);
			}
		}

		let counterparty_hash_commitment_number_len: u64 = Readable::read(reader)?;
		let mut counterparty_hash_commitment_number = HashMap::with_capacity(cmp::min(counterparty_hash_commitment_number_len as usize, MAX_ALLOC_SIZE / 32));
		for _ in 0..counterparty_hash_commitment_number_len {
			let payment_hash: PaymentHash = Readable::read(reader)?;
			let commitment_number = <U48 as Readable>::read(reader)?.0;
			if let Some(_) = counterparty_hash_commitment_number.insert(payment_hash, commitment_number) {
				return Err(DecodeError::InvalidValue);
			}
		}

		macro_rules! read_holder_tx {
			() => {
				{
					let txid = Readable::read(reader)?;
					let revocation_key = Readable::read(reader)?;
					let a_htlc_key = Readable::read(reader)?;
					let b_htlc_key = Readable::read(reader)?;
					let delayed_payment_key = Readable::read(reader)?;
					let per_commitment_point = Readable::read(reader)?;
					let feerate_per_kw: u32 = Readable::read(reader)?;

					let htlcs_len: u64 = Readable::read(reader)?;
					let mut htlcs = Vec::with_capacity(cmp::min(htlcs_len as usize, MAX_ALLOC_SIZE / 128));
					for _ in 0..htlcs_len {
						let htlc = read_htlc_in_commitment!();
						let sigs = match <u8 as Readable>::read(reader)? {
							0 => None,
							1 => Some(Readable::read(reader)?),
							_ => return Err(DecodeError::InvalidValue),
						};
						htlcs.push((htlc, sigs, Readable::read(reader)?));
					}

					HolderSignedTx {
						txid,
						revocation_key, a_htlc_key, b_htlc_key, delayed_payment_key, per_commitment_point, feerate_per_kw,
						htlc_outputs: htlcs
					}
				}
			}
		}

		let prev_holder_signed_commitment_tx = match <u8 as Readable>::read(reader)? {
			0 => None,
			1 => {
				Some(read_holder_tx!())
			},
			_ => return Err(DecodeError::InvalidValue),
		};
		let current_holder_commitment_tx = read_holder_tx!();

		let current_counterparty_commitment_number = <U48 as Readable>::read(reader)?.0;
		let current_holder_commitment_number = <U48 as Readable>::read(reader)?.0;

		let payment_preimages_len: u64 = Readable::read(reader)?;
		let mut payment_preimages = HashMap::with_capacity(cmp::min(payment_preimages_len as usize, MAX_ALLOC_SIZE / 32));
		for _ in 0..payment_preimages_len {
			let preimage: PaymentPreimage = Readable::read(reader)?;
			let hash = PaymentHash(Sha256::hash(&preimage.0[..]).into_inner());
			if let Some(_) = payment_preimages.insert(hash, preimage) {
				return Err(DecodeError::InvalidValue);
			}
		}

		let pending_monitor_events_len: u64 = Readable::read(reader)?;
		let mut pending_monitor_events = Vec::with_capacity(cmp::min(pending_monitor_events_len as usize, MAX_ALLOC_SIZE / (32 + 8*3)));
		for _ in 0..pending_monitor_events_len {
			let ev = match <u8 as Readable>::read(reader)? {
				0 => MonitorEvent::HTLCEvent(Readable::read(reader)?),
				1 => MonitorEvent::CommitmentTxBroadcasted(funding_info.0),
				_ => return Err(DecodeError::InvalidValue)
			};
			pending_monitor_events.push(ev);
		}

		let pending_events_len: u64 = Readable::read(reader)?;
		let mut pending_events = Vec::with_capacity(cmp::min(pending_events_len as usize, MAX_ALLOC_SIZE / mem::size_of::<Event>()));
		for _ in 0..pending_events_len {
			if let Some(event) = MaybeReadable::read(reader)? {
				pending_events.push(event);
			}
		}

		let best_block = BestBlock::new(Readable::read(reader)?, Readable::read(reader)?);

		let waiting_threshold_conf_len: u64 = Readable::read(reader)?;
		let mut onchain_events_awaiting_threshold_conf = Vec::with_capacity(cmp::min(waiting_threshold_conf_len as usize, MAX_ALLOC_SIZE / 128));
		for _ in 0..waiting_threshold_conf_len {
			let txid = Readable::read(reader)?;
			let height = Readable::read(reader)?;
			let event = match <u8 as Readable>::read(reader)? {
				0 => {
					let htlc_source = Readable::read(reader)?;
					let hash = Readable::read(reader)?;
					OnchainEvent::HTLCUpdate {
						htlc_update: (htlc_source, hash)
					}
				},
				1 => {
					let descriptor = Readable::read(reader)?;
					OnchainEvent::MaturingOutput {
						descriptor
					}
				},
				_ => return Err(DecodeError::InvalidValue),
			};
			onchain_events_awaiting_threshold_conf.push(OnchainEventEntry { txid, height, event });
		}

		let outputs_to_watch_len: u64 = Readable::read(reader)?;
		let mut outputs_to_watch = HashMap::with_capacity(cmp::min(outputs_to_watch_len as usize, MAX_ALLOC_SIZE / (mem::size_of::<Txid>() + mem::size_of::<u32>() + mem::size_of::<Vec<Script>>())));
		for _ in 0..outputs_to_watch_len {
			let txid = Readable::read(reader)?;
			let outputs_len: u64 = Readable::read(reader)?;
			let mut outputs = Vec::with_capacity(cmp::min(outputs_len as usize, MAX_ALLOC_SIZE / (mem::size_of::<u32>() + mem::size_of::<Script>())));
			for _ in 0..outputs_len {
				outputs.push((Readable::read(reader)?, Readable::read(reader)?));
			}
			if let Some(_) = outputs_to_watch.insert(txid, outputs) {
				return Err(DecodeError::InvalidValue);
			}
		}
		let onchain_tx_handler = ReadableArgs::read(reader, keys_manager)?;

		let lockdown_from_offchain = Readable::read(reader)?;
		let holder_tx_signed = Readable::read(reader)?;

		let mut secp_ctx = Secp256k1::new();
		secp_ctx.seeded_randomize(&keys_manager.get_secure_random_bytes());

		Ok((best_block.block_hash(), ChannelMonitor {
			inner: Mutex::new(ChannelMonitorImpl {
				latest_update_id,
				commitment_transaction_number_obscure_factor,

				destination_script,
				broadcasted_holder_revokable_script,
				counterparty_payment_script,
				shutdown_script,

				channel_keys_id,
				holder_revocation_basepoint,
				funding_info,
				current_counterparty_commitment_txid,
				prev_counterparty_commitment_txid,

				counterparty_tx_cache,
				funding_redeemscript,
				channel_value_satoshis,
				their_cur_revocation_points,

				on_holder_tx_csv,

				commitment_secrets,
				counterparty_claimable_outpoints,
				counterparty_commitment_txn_on_chain,
				counterparty_hash_commitment_number,

				prev_holder_signed_commitment_tx,
				current_holder_commitment_tx,
				current_counterparty_commitment_number,
				current_holder_commitment_number,

				payment_preimages,
				pending_monitor_events,
				pending_events,

				onchain_events_awaiting_threshold_conf,
				outputs_to_watch,

				onchain_tx_handler,

				lockdown_from_offchain,
				holder_tx_signed,

				best_block,

				secp_ctx,
			}),
		}))
	}
}

#[cfg(test)]
mod tests {
	use bitcoin::blockdata::script::{Script, Builder};
	use bitcoin::blockdata::opcodes;
	use bitcoin::blockdata::transaction::{Transaction, TxIn, TxOut, SigHashType};
	use bitcoin::blockdata::transaction::OutPoint as BitcoinOutPoint;
	use bitcoin::util::bip143;
	use bitcoin::hashes::Hash;
	use bitcoin::hashes::sha256::Hash as Sha256;
	use bitcoin::hashes::hex::FromHex;
	use bitcoin::hash_types::Txid;
	use bitcoin::network::constants::Network;
	use hex;
	use chain::channelmonitor::ChannelMonitor;
	use chain::transaction::OutPoint;
	use ln::{PaymentPreimage, PaymentHash};
	use ln::channelmanager::BestBlock;
	use ln::onchaintx::{OnchainTxHandler, InputDescriptors};
	use ln::chan_utils;
	use ln::chan_utils::{HTLCOutputInCommitment, ChannelPublicKeys, ChannelTransactionParameters, HolderCommitmentTransaction, CounterpartyChannelTransactionParameters};
	use util::test_utils::{TestLogger, TestBroadcaster, TestFeeEstimator};
	use bitcoin::secp256k1::key::{SecretKey,PublicKey};
	use bitcoin::secp256k1::Secp256k1;
	use std::sync::{Arc, Mutex};
	use chain::keysinterface::InMemorySigner;

	#[test]
	fn test_prune_preimages() {
		let secp_ctx = Secp256k1::new();
		let logger = Arc::new(TestLogger::new());
		let broadcaster = Arc::new(TestBroadcaster{txn_broadcasted: Mutex::new(Vec::new())});
		let fee_estimator = Arc::new(TestFeeEstimator { sat_per_kw: 253 });

		let dummy_key = PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[42; 32]).unwrap());
		let dummy_tx = Transaction { version: 0, lock_time: 0, input: Vec::new(), output: Vec::new() };

		let mut preimages = Vec::new();
		{
			for i in 0..20 {
				let preimage = PaymentPreimage([i; 32]);
				let hash = PaymentHash(Sha256::hash(&preimage.0[..]).into_inner());
				preimages.push((preimage, hash));
			}
		}

		macro_rules! preimages_slice_to_htlc_outputs {
			($preimages_slice: expr) => {
				{
					let mut res = Vec::new();
					for (idx, preimage) in $preimages_slice.iter().enumerate() {
						res.push((HTLCOutputInCommitment {
							offered: true,
							amount_msat: 0,
							cltv_expiry: 0,
							payment_hash: preimage.1.clone(),
							transaction_output_index: Some(idx as u32),
						}, None));
					}
					res
				}
			}
		}
		macro_rules! preimages_to_holder_htlcs {
			($preimages_slice: expr) => {
				{
					let mut inp = preimages_slice_to_htlc_outputs!($preimages_slice);
					let res: Vec<_> = inp.drain(..).map(|e| { (e.0, None, e.1) }).collect();
					res
				}
			}
		}

		macro_rules! test_preimages_exist {
			($preimages_slice: expr, $monitor: expr) => {
				for preimage in $preimages_slice {
					assert!($monitor.inner.lock().unwrap().payment_preimages.contains_key(&preimage.1));
				}
			}
		}

		let keys = InMemorySigner::new(
			&secp_ctx,
			SecretKey::from_slice(&[41; 32]).unwrap(),
			SecretKey::from_slice(&[41; 32]).unwrap(),
			SecretKey::from_slice(&[41; 32]).unwrap(),
			SecretKey::from_slice(&[41; 32]).unwrap(),
			SecretKey::from_slice(&[41; 32]).unwrap(),
			[41; 32],
			0,
			[0; 32]
		);

		let counterparty_pubkeys = ChannelPublicKeys {
			funding_pubkey: PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[44; 32]).unwrap()),
			revocation_basepoint: PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[45; 32]).unwrap()),
			payment_point: PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[46; 32]).unwrap()),
			delayed_payment_basepoint: PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[47; 32]).unwrap()),
			htlc_basepoint: PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[48; 32]).unwrap())
		};
		let funding_outpoint = OutPoint { txid: Default::default(), index: u16::max_value() };
		let channel_parameters = ChannelTransactionParameters {
			holder_pubkeys: keys.holder_channel_pubkeys.clone(),
			holder_selected_contest_delay: 66,
			is_outbound_from_holder: true,
			counterparty_parameters: Some(CounterpartyChannelTransactionParameters {
				pubkeys: counterparty_pubkeys,
				selected_contest_delay: 67,
			}),
			funding_outpoint: Some(funding_outpoint),
		};
		// Prune with one old state and a holder commitment tx holding a few overlaps with the
		// old state.
		let best_block = BestBlock::from_genesis(Network::Testnet);
		let monitor = ChannelMonitor::new(Secp256k1::new(), keys,
		                                  &PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[42; 32]).unwrap()), 0, &Script::new(),
		                                  (OutPoint { txid: Txid::from_slice(&[43; 32]).unwrap(), index: 0 }, Script::new()),
		                                  &channel_parameters,
		                                  Script::new(), 46, 0,
		                                  HolderCommitmentTransaction::dummy(), best_block);

		monitor.provide_latest_holder_commitment_tx(HolderCommitmentTransaction::dummy(), preimages_to_holder_htlcs!(preimages[0..10])).unwrap();
		let dummy_txid = dummy_tx.txid();
		monitor.provide_latest_counterparty_commitment_tx(dummy_txid, preimages_slice_to_htlc_outputs!(preimages[5..15]), 281474976710655, dummy_key, &logger);
		monitor.provide_latest_counterparty_commitment_tx(dummy_txid, preimages_slice_to_htlc_outputs!(preimages[15..20]), 281474976710654, dummy_key, &logger);
		monitor.provide_latest_counterparty_commitment_tx(dummy_txid, preimages_slice_to_htlc_outputs!(preimages[17..20]), 281474976710653, dummy_key, &logger);
		monitor.provide_latest_counterparty_commitment_tx(dummy_txid, preimages_slice_to_htlc_outputs!(preimages[18..20]), 281474976710652, dummy_key, &logger);
		for &(ref preimage, ref hash) in preimages.iter() {
			monitor.provide_payment_preimage(hash, preimage, &broadcaster, &fee_estimator, &logger);
		}

		// Now provide a secret, pruning preimages 10-15
		let mut secret = [0; 32];
		secret[0..32].clone_from_slice(&hex::decode("7cc854b54e3e0dcdb010d7a3fee464a9687be6e8db3be6854c475621e007a5dc").unwrap());
		monitor.provide_secret(281474976710655, secret.clone()).unwrap();
		assert_eq!(monitor.inner.lock().unwrap().payment_preimages.len(), 15);
		test_preimages_exist!(&preimages[0..10], monitor);
		test_preimages_exist!(&preimages[15..20], monitor);

		// Now provide a further secret, pruning preimages 15-17
		secret[0..32].clone_from_slice(&hex::decode("c7518c8ae4660ed02894df8976fa1a3659c1a8b4b5bec0c4b872abeba4cb8964").unwrap());
		monitor.provide_secret(281474976710654, secret.clone()).unwrap();
		assert_eq!(monitor.inner.lock().unwrap().payment_preimages.len(), 13);
		test_preimages_exist!(&preimages[0..10], monitor);
		test_preimages_exist!(&preimages[17..20], monitor);

		// Now update holder commitment tx info, pruning only element 18 as we still care about the
		// previous commitment tx's preimages too
		monitor.provide_latest_holder_commitment_tx(HolderCommitmentTransaction::dummy(), preimages_to_holder_htlcs!(preimages[0..5])).unwrap();
		secret[0..32].clone_from_slice(&hex::decode("2273e227a5b7449b6e70f1fb4652864038b1cbf9cd7c043a7d6456b7fc275ad8").unwrap());
		monitor.provide_secret(281474976710653, secret.clone()).unwrap();
		assert_eq!(monitor.inner.lock().unwrap().payment_preimages.len(), 12);
		test_preimages_exist!(&preimages[0..10], monitor);
		test_preimages_exist!(&preimages[18..20], monitor);

		// But if we do it again, we'll prune 5-10
		monitor.provide_latest_holder_commitment_tx(HolderCommitmentTransaction::dummy(), preimages_to_holder_htlcs!(preimages[0..3])).unwrap();
		secret[0..32].clone_from_slice(&hex::decode("27cddaa5624534cb6cb9d7da077cf2b22ab21e9b506fd4998a51d54502e99116").unwrap());
		monitor.provide_secret(281474976710652, secret.clone()).unwrap();
		assert_eq!(monitor.inner.lock().unwrap().payment_preimages.len(), 5);
		test_preimages_exist!(&preimages[0..5], monitor);
	}

	#[test]
	fn test_claim_txn_weight_computation() {
		// We test Claim txn weight, knowing that we want expected weigth and
		// not actual case to avoid sigs and time-lock delays hell variances.

		let secp_ctx = Secp256k1::new();
		let privkey = SecretKey::from_slice(&hex::decode("0101010101010101010101010101010101010101010101010101010101010101").unwrap()[..]).unwrap();
		let pubkey = PublicKey::from_secret_key(&secp_ctx, &privkey);
		let mut sum_actual_sigs = 0;

		macro_rules! sign_input {
			($sighash_parts: expr, $idx: expr, $amount: expr, $input_type: expr, $sum_actual_sigs: expr) => {
				let htlc = HTLCOutputInCommitment {
					offered: if *$input_type == InputDescriptors::RevokedOfferedHTLC || *$input_type == InputDescriptors::OfferedHTLC { true } else { false },
					amount_msat: 0,
					cltv_expiry: 2 << 16,
					payment_hash: PaymentHash([1; 32]),
					transaction_output_index: Some($idx as u32),
				};
				let redeem_script = if *$input_type == InputDescriptors::RevokedOutput { chan_utils::get_revokeable_redeemscript(&pubkey, 256, &pubkey) } else { chan_utils::get_htlc_redeemscript_with_explicit_keys(&htlc, &pubkey, &pubkey, &pubkey) };
				let sighash = hash_to_message!(&$sighash_parts.signature_hash($idx, &redeem_script, $amount, SigHashType::All)[..]);
				let sig = secp_ctx.sign(&sighash, &privkey);
				$sighash_parts.access_witness($idx).push(sig.serialize_der().to_vec());
				$sighash_parts.access_witness($idx)[0].push(SigHashType::All as u8);
				sum_actual_sigs += $sighash_parts.access_witness($idx)[0].len();
				if *$input_type == InputDescriptors::RevokedOutput {
					$sighash_parts.access_witness($idx).push(vec!(1));
				} else if *$input_type == InputDescriptors::RevokedOfferedHTLC || *$input_type == InputDescriptors::RevokedReceivedHTLC {
					$sighash_parts.access_witness($idx).push(pubkey.clone().serialize().to_vec());
				} else if *$input_type == InputDescriptors::ReceivedHTLC {
					$sighash_parts.access_witness($idx).push(vec![0]);
				} else {
					$sighash_parts.access_witness($idx).push(PaymentPreimage([1; 32]).0.to_vec());
				}
				$sighash_parts.access_witness($idx).push(redeem_script.into_bytes());
				println!("witness[0] {}", $sighash_parts.access_witness($idx)[0].len());
				println!("witness[1] {}", $sighash_parts.access_witness($idx)[1].len());
				println!("witness[2] {}", $sighash_parts.access_witness($idx)[2].len());
			}
		}

		let script_pubkey = Builder::new().push_opcode(opcodes::all::OP_RETURN).into_script();
		let txid = Txid::from_hex("56944c5d3f98413ef45cf54545538103cc9f298e0575820ad3591376e2e0f65d").unwrap();

		// Justice tx with 1 to_holder, 2 revoked offered HTLCs, 1 revoked received HTLCs
		let mut claim_tx = Transaction { version: 0, lock_time: 0, input: Vec::new(), output: Vec::new() };
		for i in 0..4 {
			claim_tx.input.push(TxIn {
				previous_output: BitcoinOutPoint {
					txid,
					vout: i,
				},
				script_sig: Script::new(),
				sequence: 0xfffffffd,
				witness: Vec::new(),
			});
		}
		claim_tx.output.push(TxOut {
			script_pubkey: script_pubkey.clone(),
			value: 0,
		});
		let base_weight = claim_tx.get_weight();
		let inputs_des = vec![InputDescriptors::RevokedOutput, InputDescriptors::RevokedOfferedHTLC, InputDescriptors::RevokedOfferedHTLC, InputDescriptors::RevokedReceivedHTLC];
		{
			let mut sighash_parts = bip143::SigHashCache::new(&mut claim_tx);
			for (idx, inp) in inputs_des.iter().enumerate() {
				sign_input!(sighash_parts, idx, 0, inp, sum_actual_sigs);
			}
		}
		assert_eq!(base_weight + OnchainTxHandler::<InMemorySigner>::get_witnesses_weight(&inputs_des[..]),  claim_tx.get_weight() + /* max_length_sig */ (73 * inputs_des.len() - sum_actual_sigs));

		// Claim tx with 1 offered HTLCs, 3 received HTLCs
		claim_tx.input.clear();
		sum_actual_sigs = 0;
		for i in 0..4 {
			claim_tx.input.push(TxIn {
				previous_output: BitcoinOutPoint {
					txid,
					vout: i,
				},
				script_sig: Script::new(),
				sequence: 0xfffffffd,
				witness: Vec::new(),
			});
		}
		let base_weight = claim_tx.get_weight();
		let inputs_des = vec![InputDescriptors::OfferedHTLC, InputDescriptors::ReceivedHTLC, InputDescriptors::ReceivedHTLC, InputDescriptors::ReceivedHTLC];
		{
			let mut sighash_parts = bip143::SigHashCache::new(&mut claim_tx);
			for (idx, inp) in inputs_des.iter().enumerate() {
				sign_input!(sighash_parts, idx, 0, inp, sum_actual_sigs);
			}
		}
		assert_eq!(base_weight + OnchainTxHandler::<InMemorySigner>::get_witnesses_weight(&inputs_des[..]),  claim_tx.get_weight() + /* max_length_sig */ (73 * inputs_des.len() - sum_actual_sigs));

		// Justice tx with 1 revoked HTLC-Success tx output
		claim_tx.input.clear();
		sum_actual_sigs = 0;
		claim_tx.input.push(TxIn {
			previous_output: BitcoinOutPoint {
				txid,
				vout: 0,
			},
			script_sig: Script::new(),
			sequence: 0xfffffffd,
			witness: Vec::new(),
		});
		let base_weight = claim_tx.get_weight();
		let inputs_des = vec![InputDescriptors::RevokedOutput];
		{
			let mut sighash_parts = bip143::SigHashCache::new(&mut claim_tx);
			for (idx, inp) in inputs_des.iter().enumerate() {
				sign_input!(sighash_parts, idx, 0, inp, sum_actual_sigs);
			}
		}
		assert_eq!(base_weight + OnchainTxHandler::<InMemorySigner>::get_witnesses_weight(&inputs_des[..]), claim_tx.get_weight() + /* max_length_isg */ (73 * inputs_des.len() - sum_actual_sigs));
	}

	// Further testing is done in the ChannelManager integration tests.
}
