// This file is Copyright its original authors, visible in version control
// history.
//
// This file is licensed under the Apache License, Version 2.0 <LICENSE-APACHE
// or http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your option.
// You may not use this file except in accordance with one or both of these
// licenses.

//! The logic to monitor for on-chain transactions and create the relevant claim responses lives
//! here.
//!
//! ChannelMonitor objects are generated by ChannelManager in response to relevant
//! messages/actions, and MUST be persisted to disk (and, preferably, remotely) before progress can
//! be made in responding to certain messages, see [`chain::Watch`] for more.
//!
//! Note that ChannelMonitors are an important part of the lightning trust model and a copy of the
//! latest ChannelMonitor must always be actively monitoring for chain updates (and no out-of-date
//! ChannelMonitors should do so). Thus, if you're building rust-lightning into an HSM or other
//! security-domain-separated system design, you should consider having multiple paths for
//! ChannelMonitors to get out of the HSM and onto monitoring devices.

use bitcoin::blockdata::block::{Block, BlockHeader};
use bitcoin::blockdata::transaction::{TxOut,Transaction};
use bitcoin::blockdata::script::{Script, Builder};
use bitcoin::blockdata::opcodes;

use bitcoin::hashes::Hash;
use bitcoin::hashes::sha256::Hash as Sha256;
use bitcoin::hash_types::{Txid, BlockHash, WPubkeyHash};

use bitcoin::secp256k1::{Secp256k1,Signature};
use bitcoin::secp256k1::key::{SecretKey,PublicKey};
use bitcoin::secp256k1;

use ln::{PaymentHash, PaymentPreimage};
use ln::msgs::DecodeError;
use ln::chan_utils;
use ln::chan_utils::{CounterpartyCommitmentSecrets, HTLCOutputInCommitment, HTLCType, ChannelTransactionParameters, HolderCommitmentTransaction};
use ln::channelmanager::HTLCSource;
use chain;
use chain::{BestBlock, WatchedOutput};
use chain::chaininterface::{BroadcasterInterface, FeeEstimator};
use chain::transaction::{OutPoint, TransactionData};
use chain::keysinterface::{SpendableOutputDescriptor, StaticPaymentOutputDescriptor, DelayedPaymentOutputDescriptor, Sign, KeysInterface};
use chain::onchaintx::OnchainTxHandler;
use chain::package::{CounterpartyOfferedHTLCOutput, CounterpartyReceivedHTLCOutput, HolderFundingOutput, HolderHTLCOutput, PackageSolvingData, PackageTemplate, RevokedOutput, RevokedHTLCOutput};
use chain::Filter;
use util::logger::Logger;
use util::ser::{Readable, ReadableArgs, MaybeReadable, Writer, Writeable, U48, OptionDeserWrapper};
use util::byte_utils;
use util::events::Event;

use prelude::*;
use core::{cmp, mem};
use io::{self, Error};
use core::ops::Deref;
use sync::Mutex;

/// An update generated by the underlying Channel itself which contains some new information the
/// ChannelMonitor should be made aware of.
#[cfg_attr(any(test, fuzzing, feature = "_test_utils"), derive(PartialEq))]
#[derive(Clone)]
#[must_use]
pub struct ChannelMonitorUpdate {
	pub(crate) updates: Vec<ChannelMonitorUpdateStep>,
	/// The sequence number of this update. Updates *must* be replayed in-order according to this
	/// sequence number (and updates may panic if they are not). The update_id values are strictly
	/// increasing and increase by one for each new update, with one exception specified below.
	///
	/// This sequence number is also used to track up to which points updates which returned
	/// ChannelMonitorUpdateErr::TemporaryFailure have been applied to all copies of a given
	/// ChannelMonitor when ChannelManager::channel_monitor_updated is called.
	///
	/// The only instance where update_id values are not strictly increasing is the case where we
	/// allow post-force-close updates with a special update ID of [`CLOSED_CHANNEL_UPDATE_ID`]. See
	/// its docs for more details.
	pub update_id: u64,
}

/// If:
///    (1) a channel has been force closed and
///    (2) we receive a preimage from a forward link that allows us to spend an HTLC output on
///        this channel's (the backward link's) broadcasted commitment transaction
/// then we allow the `ChannelManager` to send a `ChannelMonitorUpdate` with this update ID,
/// with the update providing said payment preimage. No other update types are allowed after
/// force-close.
pub const CLOSED_CHANNEL_UPDATE_ID: u64 = core::u64::MAX;

impl Writeable for ChannelMonitorUpdate {
	fn write<W: Writer>(&self, w: &mut W) -> Result<(), io::Error> {
		write_ver_prefix!(w, SERIALIZATION_VERSION, MIN_SERIALIZATION_VERSION);
		self.update_id.write(w)?;
		(self.updates.len() as u64).write(w)?;
		for update_step in self.updates.iter() {
			update_step.write(w)?;
		}
		write_tlv_fields!(w, {});
		Ok(())
	}
}
impl Readable for ChannelMonitorUpdate {
	fn read<R: io::Read>(r: &mut R) -> Result<Self, DecodeError> {
		let _ver = read_ver_prefix!(r, SERIALIZATION_VERSION);
		let update_id: u64 = Readable::read(r)?;
		let len: u64 = Readable::read(r)?;
		let mut updates = Vec::with_capacity(cmp::min(len as usize, MAX_ALLOC_SIZE / ::core::mem::size_of::<ChannelMonitorUpdateStep>()));
		for _ in 0..len {
			if let Some(upd) = MaybeReadable::read(r)? {
				updates.push(upd);
			}
		}
		read_tlv_fields!(r, {});
		Ok(Self { update_id, updates })
	}
}

/// An event to be processed by the ChannelManager.
#[derive(Clone, PartialEq)]
pub enum MonitorEvent {
	/// A monitor event containing an HTLCUpdate.
	HTLCEvent(HTLCUpdate),

	/// A monitor event that the Channel's commitment transaction was confirmed.
	CommitmentTxConfirmed(OutPoint),

	/// Indicates a [`ChannelMonitor`] update has completed. See
	/// [`ChannelMonitorUpdateErr::TemporaryFailure`] for more information on how this is used.
	///
	/// [`ChannelMonitorUpdateErr::TemporaryFailure`]: super::ChannelMonitorUpdateErr::TemporaryFailure
	UpdateCompleted {
		/// The funding outpoint of the [`ChannelMonitor`] that was updated
		funding_txo: OutPoint,
		/// The Update ID from [`ChannelMonitorUpdate::update_id`] which was applied or
		/// [`ChannelMonitor::get_latest_update_id`].
		///
		/// Note that this should only be set to a given update's ID if all previous updates for the
		/// same [`ChannelMonitor`] have been applied and persisted.
		monitor_update_id: u64,
	},

	/// Indicates a [`ChannelMonitor`] update has failed. See
	/// [`ChannelMonitorUpdateErr::PermanentFailure`] for more information on how this is used.
	///
	/// [`ChannelMonitorUpdateErr::PermanentFailure`]: super::ChannelMonitorUpdateErr::PermanentFailure
	UpdateFailed(OutPoint),
}
impl_writeable_tlv_based_enum_upgradable!(MonitorEvent,
	// Note that UpdateCompleted and UpdateFailed are currently never serialized to disk as they are
	// generated only in ChainMonitor
	(0, UpdateCompleted) => {
		(0, funding_txo, required),
		(2, monitor_update_id, required),
	},
;
	(2, HTLCEvent),
	(4, CommitmentTxConfirmed),
	(6, UpdateFailed),
);

/// Simple structure sent back by `chain::Watch` when an HTLC from a forward channel is detected on
/// chain. Used to update the corresponding HTLC in the backward channel. Failing to pass the
/// preimage claim backward will lead to loss of funds.
#[derive(Clone, PartialEq)]
pub struct HTLCUpdate {
	pub(crate) payment_hash: PaymentHash,
	pub(crate) payment_preimage: Option<PaymentPreimage>,
	pub(crate) source: HTLCSource,
	pub(crate) onchain_value_satoshis: Option<u64>,
}
impl_writeable_tlv_based!(HTLCUpdate, {
	(0, payment_hash, required),
	(1, onchain_value_satoshis, option),
	(2, source, required),
	(4, payment_preimage, option),
});

/// If an HTLC expires within this many blocks, don't try to claim it in a shared transaction,
/// instead claiming it in its own individual transaction.
pub(crate) const CLTV_SHARED_CLAIM_BUFFER: u32 = 12;
/// If an HTLC expires within this many blocks, force-close the channel to broadcast the
/// HTLC-Success transaction.
/// In other words, this is an upper bound on how many blocks we think it can take us to get a
/// transaction confirmed (and we use it in a few more, equivalent, places).
pub(crate) const CLTV_CLAIM_BUFFER: u32 = 18;
/// Number of blocks by which point we expect our counterparty to have seen new blocks on the
/// network and done a full update_fail_htlc/commitment_signed dance (+ we've updated all our
/// copies of ChannelMonitors, including watchtowers). We could enforce the contract by failing
/// at CLTV expiration height but giving a grace period to our peer may be profitable for us if he
/// can provide an over-late preimage. Nevertheless, grace period has to be accounted in our
/// CLTV_EXPIRY_DELTA to be secure. Following this policy we may decrease the rate of channel failures
/// due to expiration but increase the cost of funds being locked longuer in case of failure.
/// This delay also cover a low-power peer being slow to process blocks and so being behind us on
/// accurate block height.
/// In case of onchain failure to be pass backward we may see the last block of ANTI_REORG_DELAY
/// with at worst this delay, so we are not only using this value as a mercy for them but also
/// us as a safeguard to delay with enough time.
pub(crate) const LATENCY_GRACE_PERIOD_BLOCKS: u32 = 3;
/// Number of blocks we wait on seeing a HTLC output being solved before we fail corresponding
/// inbound HTLCs. This prevents us from failing backwards and then getting a reorg resulting in us
/// losing money.
///
/// Note that this is a library-wide security assumption. If a reorg deeper than this number of
/// blocks occurs, counterparties may be able to steal funds or claims made by and balances exposed
/// by a  [`ChannelMonitor`] may be incorrect.
// We also use this delay to be sure we can remove our in-flight claim txn from bump candidates buffer.
// It may cause spurious generation of bumped claim txn but that's alright given the outpoint is already
// solved by a previous claim tx. What we want to avoid is reorg evicting our claim tx and us not
// keep bumping another claim tx to solve the outpoint.
pub const ANTI_REORG_DELAY: u32 = 6;
/// Number of blocks before confirmation at which we fail back an un-relayed HTLC or at which we
/// refuse to accept a new HTLC.
///
/// This is used for a few separate purposes:
/// 1) if we've received an MPP HTLC to us and it expires within this many blocks and we are
///    waiting on additional parts (or waiting on the preimage for any HTLC from the user), we will
///    fail this HTLC,
/// 2) if we receive an HTLC within this many blocks of its expiry (plus one to avoid a race
///    condition with the above), we will fail this HTLC without telling the user we received it,
///
/// (1) is all about protecting us - we need enough time to update the channel state before we hit
/// CLTV_CLAIM_BUFFER, at which point we'd go on chain to claim the HTLC with the preimage.
///
/// (2) is the same, but with an additional buffer to avoid accepting an HTLC which is immediately
/// in a race condition between the user connecting a block (which would fail it) and the user
/// providing us the preimage (which would claim it).
pub(crate) const HTLC_FAIL_BACK_BUFFER: u32 = CLTV_CLAIM_BUFFER + LATENCY_GRACE_PERIOD_BLOCKS;

// TODO(devrandom) replace this with HolderCommitmentTransaction
#[derive(Clone, PartialEq)]
struct HolderSignedTx {
	/// txid of the transaction in tx, just used to make comparison faster
	txid: Txid,
	revocation_key: PublicKey,
	a_htlc_key: PublicKey,
	b_htlc_key: PublicKey,
	delayed_payment_key: PublicKey,
	per_commitment_point: PublicKey,
	htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Signature>, Option<HTLCSource>)>,
	to_self_value_sat: u64,
	feerate_per_kw: u32,
}
impl_writeable_tlv_based!(HolderSignedTx, {
	(0, txid, required),
	// Note that this is filled in with data from OnchainTxHandler if it's missing.
	// For HolderSignedTx objects serialized with 0.0.100+, this should be filled in.
	(1, to_self_value_sat, (default_value, u64::max_value())),
	(2, revocation_key, required),
	(4, a_htlc_key, required),
	(6, b_htlc_key, required),
	(8, delayed_payment_key, required),
	(10, per_commitment_point, required),
	(12, feerate_per_kw, required),
	(14, htlc_outputs, vec_type)
});

/// We use this to track static counterparty commitment transaction data and to generate any
/// justice or 2nd-stage preimage/timeout transactions.
#[derive(PartialEq)]
struct CounterpartyCommitmentParameters {
	counterparty_delayed_payment_base_key: PublicKey,
	counterparty_htlc_base_key: PublicKey,
	on_counterparty_tx_csv: u16,
}

impl Writeable for CounterpartyCommitmentParameters {
	fn write<W: Writer>(&self, w: &mut W) -> Result<(), io::Error> {
		w.write_all(&byte_utils::be64_to_array(0))?;
		write_tlv_fields!(w, {
			(0, self.counterparty_delayed_payment_base_key, required),
			(2, self.counterparty_htlc_base_key, required),
			(4, self.on_counterparty_tx_csv, required),
		});
		Ok(())
	}
}
impl Readable for CounterpartyCommitmentParameters {
	fn read<R: io::Read>(r: &mut R) -> Result<Self, DecodeError> {
		let counterparty_commitment_transaction = {
			// Versions prior to 0.0.100 had some per-HTLC state stored here, which is no longer
			// used. Read it for compatibility.
			let per_htlc_len: u64 = Readable::read(r)?;
			for _  in 0..per_htlc_len {
				let _txid: Txid = Readable::read(r)?;
				let htlcs_count: u64 = Readable::read(r)?;
				for _ in 0..htlcs_count {
					let _htlc: HTLCOutputInCommitment = Readable::read(r)?;
				}
			}

			let mut counterparty_delayed_payment_base_key = OptionDeserWrapper(None);
			let mut counterparty_htlc_base_key = OptionDeserWrapper(None);
			let mut on_counterparty_tx_csv: u16 = 0;
			read_tlv_fields!(r, {
				(0, counterparty_delayed_payment_base_key, required),
				(2, counterparty_htlc_base_key, required),
				(4, on_counterparty_tx_csv, required),
			});
			CounterpartyCommitmentParameters {
				counterparty_delayed_payment_base_key: counterparty_delayed_payment_base_key.0.unwrap(),
				counterparty_htlc_base_key: counterparty_htlc_base_key.0.unwrap(),
				on_counterparty_tx_csv,
			}
		};
		Ok(counterparty_commitment_transaction)
	}
}

/// An entry for an [`OnchainEvent`], stating the block height when the event was observed and the
/// transaction causing it.
///
/// Used to determine when the on-chain event can be considered safe from a chain reorganization.
#[derive(PartialEq)]
struct OnchainEventEntry {
	txid: Txid,
	height: u32,
	event: OnchainEvent,
}

impl OnchainEventEntry {
	fn confirmation_threshold(&self) -> u32 {
		let mut conf_threshold = self.height + ANTI_REORG_DELAY - 1;
		match self.event {
			OnchainEvent::MaturingOutput {
				descriptor: SpendableOutputDescriptor::DelayedPaymentOutput(ref descriptor)
			} => {
				// A CSV'd transaction is confirmable in block (input height) + CSV delay, which means
				// it's broadcastable when we see the previous block.
				conf_threshold = cmp::max(conf_threshold, self.height + descriptor.to_self_delay as u32 - 1);
			},
			OnchainEvent::FundingSpendConfirmation { on_local_output_csv: Some(csv), .. } |
			OnchainEvent::HTLCSpendConfirmation { on_to_local_output_csv: Some(csv), .. } => {
				// A CSV'd transaction is confirmable in block (input height) + CSV delay, which means
				// it's broadcastable when we see the previous block.
				conf_threshold = cmp::max(conf_threshold, self.height + csv as u32 - 1);
			},
			_ => {},
		}
		conf_threshold
	}

	fn has_reached_confirmation_threshold(&self, best_block: &BestBlock) -> bool {
		best_block.height() >= self.confirmation_threshold()
	}
}

/// Upon discovering of some classes of onchain tx by ChannelMonitor, we may have to take actions on it
/// once they mature to enough confirmations (ANTI_REORG_DELAY)
#[derive(PartialEq)]
enum OnchainEvent {
	/// An outbound HTLC failing after a transaction is confirmed. Used
	///  * when an outbound HTLC output is spent by us after the HTLC timed out
	///  * an outbound HTLC which was not present in the commitment transaction which appeared
	///    on-chain (either because it was not fully committed to or it was dust).
	/// Note that this is *not* used for preimage claims, as those are passed upstream immediately,
	/// appearing only as an `HTLCSpendConfirmation`, below.
	HTLCUpdate {
		source: HTLCSource,
		payment_hash: PaymentHash,
		onchain_value_satoshis: Option<u64>,
		/// None in the second case, above, ie when there is no relevant output in the commitment
		/// transaction which appeared on chain.
		input_idx: Option<u32>,
	},
	MaturingOutput {
		descriptor: SpendableOutputDescriptor,
	},
	/// A spend of the funding output, either a commitment transaction or a cooperative closing
	/// transaction.
	FundingSpendConfirmation {
		/// The CSV delay for the output of the funding spend transaction (implying it is a local
		/// commitment transaction, and this is the delay on the to_self output).
		on_local_output_csv: Option<u16>,
	},
	/// A spend of a commitment transaction HTLC output, set in the cases where *no* `HTLCUpdate`
	/// is constructed. This is used when
	///  * an outbound HTLC is claimed by our counterparty with a preimage, causing us to
	///    immediately claim the HTLC on the inbound edge and track the resolution here,
	///  * an inbound HTLC is claimed by our counterparty (with a timeout),
	///  * an inbound HTLC is claimed by us (with a preimage).
	///  * a revoked-state HTLC transaction was broadcasted, which was claimed by the revocation
	///    signature.
	HTLCSpendConfirmation {
		input_idx: u32,
		/// If the claim was made by either party with a preimage, this is filled in
		preimage: Option<PaymentPreimage>,
		/// If the claim was made by us on an inbound HTLC against a local commitment transaction,
		/// we set this to the output CSV value which we will have to wait until to spend the
		/// output (and generate a SpendableOutput event).
		on_to_local_output_csv: Option<u16>,
	},
}

impl Writeable for OnchainEventEntry {
	fn write<W: Writer>(&self, writer: &mut W) -> Result<(), io::Error> {
		write_tlv_fields!(writer, {
			(0, self.txid, required),
			(2, self.height, required),
			(4, self.event, required),
		});
		Ok(())
	}
}

impl MaybeReadable for OnchainEventEntry {
	fn read<R: io::Read>(reader: &mut R) -> Result<Option<Self>, DecodeError> {
		let mut txid = Default::default();
		let mut height = 0;
		let mut event = None;
		read_tlv_fields!(reader, {
			(0, txid, required),
			(2, height, required),
			(4, event, ignorable),
		});
		if let Some(ev) = event {
			Ok(Some(Self { txid, height, event: ev }))
		} else {
			Ok(None)
		}
	}
}

impl_writeable_tlv_based_enum_upgradable!(OnchainEvent,
	(0, HTLCUpdate) => {
		(0, source, required),
		(1, onchain_value_satoshis, option),
		(2, payment_hash, required),
		(3, input_idx, option),
	},
	(1, MaturingOutput) => {
		(0, descriptor, required),
	},
	(3, FundingSpendConfirmation) => {
		(0, on_local_output_csv, option),
	},
	(5, HTLCSpendConfirmation) => {
		(0, input_idx, required),
		(2, preimage, option),
		(4, on_to_local_output_csv, option),
	},

);

#[cfg_attr(any(test, fuzzing, feature = "_test_utils"), derive(PartialEq))]
#[derive(Clone)]
pub(crate) enum ChannelMonitorUpdateStep {
	LatestHolderCommitmentTXInfo {
		commitment_tx: HolderCommitmentTransaction,
		htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Signature>, Option<HTLCSource>)>,
	},
	LatestCounterpartyCommitmentTXInfo {
		commitment_txid: Txid,
		htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Box<HTLCSource>>)>,
		commitment_number: u64,
		their_revocation_point: PublicKey,
	},
	PaymentPreimage {
		payment_preimage: PaymentPreimage,
	},
	CommitmentSecret {
		idx: u64,
		secret: [u8; 32],
	},
	/// Used to indicate that the no future updates will occur, and likely that the latest holder
	/// commitment transaction(s) should be broadcast, as the channel has been force-closed.
	ChannelForceClosed {
		/// If set to false, we shouldn't broadcast the latest holder commitment transaction as we
		/// think we've fallen behind!
		should_broadcast: bool,
	},
	ShutdownScript {
		scriptpubkey: Script,
	},
}

impl ChannelMonitorUpdateStep {
	fn variant_name(&self) -> &'static str {
		match self {
			ChannelMonitorUpdateStep::LatestHolderCommitmentTXInfo { .. } => "LatestHolderCommitmentTXInfo",
			ChannelMonitorUpdateStep::LatestCounterpartyCommitmentTXInfo { .. } => "LatestCounterpartyCommitmentTXInfo",
			ChannelMonitorUpdateStep::PaymentPreimage { .. } => "PaymentPreimage",
			ChannelMonitorUpdateStep::CommitmentSecret { .. } => "CommitmentSecret",
			ChannelMonitorUpdateStep::ChannelForceClosed { .. } => "ChannelForceClosed",
			ChannelMonitorUpdateStep::ShutdownScript { .. } => "ShutdownScript",
		}
	}
}

impl_writeable_tlv_based_enum_upgradable!(ChannelMonitorUpdateStep,
	(0, LatestHolderCommitmentTXInfo) => {
		(0, commitment_tx, required),
		(2, htlc_outputs, vec_type),
	},
	(1, LatestCounterpartyCommitmentTXInfo) => {
		(0, commitment_txid, required),
		(2, commitment_number, required),
		(4, their_revocation_point, required),
		(6, htlc_outputs, vec_type),
	},
	(2, PaymentPreimage) => {
		(0, payment_preimage, required),
	},
	(3, CommitmentSecret) => {
		(0, idx, required),
		(2, secret, required),
	},
	(4, ChannelForceClosed) => {
		(0, should_broadcast, required),
	},
	(5, ShutdownScript) => {
		(0, scriptpubkey, required),
	},
);

/// Details about the balance(s) available for spending once the channel appears on chain.
///
/// See [`ChannelMonitor::get_claimable_balances`] for more details on when these will or will not
/// be provided.
#[derive(Clone, Debug, PartialEq, Eq)]
#[cfg_attr(test, derive(PartialOrd, Ord))]
pub enum Balance {
	/// The channel is not yet closed (or the commitment or closing transaction has not yet
	/// appeared in a block). The given balance is claimable (less on-chain fees) if the channel is
	/// force-closed now.
	ClaimableOnChannelClose {
		/// The amount available to claim, in satoshis, excluding the on-chain fees which will be
		/// required to do so.
		claimable_amount_satoshis: u64,
	},
	/// The channel has been closed, and the given balance is ours but awaiting confirmations until
	/// we consider it spendable.
	ClaimableAwaitingConfirmations {
		/// The amount available to claim, in satoshis, possibly excluding the on-chain fees which
		/// were spent in broadcasting the transaction.
		claimable_amount_satoshis: u64,
		/// The height at which an [`Event::SpendableOutputs`] event will be generated for this
		/// amount.
		confirmation_height: u32,
	},
	/// The channel has been closed, and the given balance should be ours but awaiting spending
	/// transaction confirmation. If the spending transaction does not confirm in time, it is
	/// possible our counterparty can take the funds by broadcasting an HTLC timeout on-chain.
	///
	/// Once the spending transaction confirms, before it has reached enough confirmations to be
	/// considered safe from chain reorganizations, the balance will instead be provided via
	/// [`Balance::ClaimableAwaitingConfirmations`].
	ContentiousClaimable {
		/// The amount available to claim, in satoshis, excluding the on-chain fees which will be
		/// required to do so.
		claimable_amount_satoshis: u64,
		/// The height at which the counterparty may be able to claim the balance if we have not
		/// done so.
		timeout_height: u32,
	},
	/// HTLCs which we sent to our counterparty which are claimable after a timeout (less on-chain
	/// fees) if the counterparty does not know the preimage for the HTLCs. These are somewhat
	/// likely to be claimed by our counterparty before we do.
	MaybeClaimableHTLCAwaitingTimeout {
		/// The amount available to claim, in satoshis, excluding the on-chain fees which will be
		/// required to do so.
		claimable_amount_satoshis: u64,
		/// The height at which we will be able to claim the balance if our counterparty has not
		/// done so.
		claimable_height: u32,
	},
}

/// An HTLC which has been irrevocably resolved on-chain, and has reached ANTI_REORG_DELAY.
#[derive(PartialEq)]
struct IrrevocablyResolvedHTLC {
	input_idx: u32,
	/// Only set if the HTLC claim was ours using a payment preimage
	payment_preimage: Option<PaymentPreimage>,
}

impl_writeable_tlv_based!(IrrevocablyResolvedHTLC, {
	(0, input_idx, required),
	(2, payment_preimage, option),
});

/// A ChannelMonitor handles chain events (blocks connected and disconnected) and generates
/// on-chain transactions to ensure no loss of funds occurs.
///
/// You MUST ensure that no ChannelMonitors for a given channel anywhere contain out-of-date
/// information and are actively monitoring the chain.
///
/// Pending Events or updated HTLCs which have not yet been read out by
/// get_and_clear_pending_monitor_events or get_and_clear_pending_events are serialized to disk and
/// reloaded at deserialize-time. Thus, you must ensure that, when handling events, all events
/// gotten are fully handled before re-serializing the new state.
///
/// Note that the deserializer is only implemented for (BlockHash, ChannelMonitor), which
/// tells you the last block hash which was block_connect()ed. You MUST rescan any blocks along
/// the "reorg path" (ie disconnecting blocks until you find a common ancestor from both the
/// returned block hash and the the current chain and then reconnecting blocks to get to the
/// best chain) upon deserializing the object!
pub struct ChannelMonitor<Signer: Sign> {
	#[cfg(test)]
	pub(crate) inner: Mutex<ChannelMonitorImpl<Signer>>,
	#[cfg(not(test))]
	inner: Mutex<ChannelMonitorImpl<Signer>>,
}

pub(crate) struct ChannelMonitorImpl<Signer: Sign> {
	latest_update_id: u64,
	commitment_transaction_number_obscure_factor: u64,

	destination_script: Script,
	broadcasted_holder_revokable_script: Option<(Script, PublicKey, PublicKey)>,
	counterparty_payment_script: Script,
	shutdown_script: Option<Script>,

	channel_keys_id: [u8; 32],
	holder_revocation_basepoint: PublicKey,
	funding_info: (OutPoint, Script),
	current_counterparty_commitment_txid: Option<Txid>,
	prev_counterparty_commitment_txid: Option<Txid>,

	counterparty_commitment_params: CounterpartyCommitmentParameters,
	funding_redeemscript: Script,
	channel_value_satoshis: u64,
	// first is the idx of the first of the two revocation points
	their_cur_revocation_points: Option<(u64, PublicKey, Option<PublicKey>)>,

	on_holder_tx_csv: u16,

	commitment_secrets: CounterpartyCommitmentSecrets,
	/// The set of outpoints in each counterparty commitment transaction. We always need at least
	/// the payment hash from `HTLCOutputInCommitment` to claim even a revoked commitment
	/// transaction broadcast as we need to be able to construct the witness script in all cases.
	counterparty_claimable_outpoints: HashMap<Txid, Vec<(HTLCOutputInCommitment, Option<Box<HTLCSource>>)>>,
	/// We cannot identify HTLC-Success or HTLC-Timeout transactions by themselves on the chain.
	/// Nor can we figure out their commitment numbers without the commitment transaction they are
	/// spending. Thus, in order to claim them via revocation key, we track all the counterparty
	/// commitment transactions which we find on-chain, mapping them to the commitment number which
	/// can be used to derive the revocation key and claim the transactions.
	counterparty_commitment_txn_on_chain: HashMap<Txid, u64>,
	/// Cache used to make pruning of payment_preimages faster.
	/// Maps payment_hash values to commitment numbers for counterparty transactions for non-revoked
	/// counterparty transactions (ie should remain pretty small).
	/// Serialized to disk but should generally not be sent to Watchtowers.
	counterparty_hash_commitment_number: HashMap<PaymentHash, u64>,

	// We store two holder commitment transactions to avoid any race conditions where we may update
	// some monitors (potentially on watchtowers) but then fail to update others, resulting in the
	// various monitors for one channel being out of sync, and us broadcasting a holder
	// transaction for which we have deleted claim information on some watchtowers.
	prev_holder_signed_commitment_tx: Option<HolderSignedTx>,
	current_holder_commitment_tx: HolderSignedTx,

	// Used just for ChannelManager to make sure it has the latest channel data during
	// deserialization
	current_counterparty_commitment_number: u64,
	// Used just for ChannelManager to make sure it has the latest channel data during
	// deserialization
	current_holder_commitment_number: u64,

	payment_preimages: HashMap<PaymentHash, PaymentPreimage>,

	// Note that `MonitorEvent`s MUST NOT be generated during update processing, only generated
	// during chain data processing. This prevents a race in `ChainMonitor::update_channel` (and
	// presumably user implementations thereof as well) where we update the in-memory channel
	// object, then before the persistence finishes (as it's all under a read-lock), we return
	// pending events to the user or to the relevant `ChannelManager`. Then, on reload, we'll have
	// the pre-event state here, but have processed the event in the `ChannelManager`.
	// Note that because the `event_lock` in `ChainMonitor` is only taken in
	// block/transaction-connected events and *not* during block/transaction-disconnected events,
	// we further MUST NOT generate events during block/transaction-disconnection.
	pending_monitor_events: Vec<MonitorEvent>,

	pending_events: Vec<Event>,

	// Used to track on-chain events (i.e., transactions part of channels confirmed on chain) on
	// which to take actions once they reach enough confirmations. Each entry includes the
	// transaction's id and the height when the transaction was confirmed on chain.
	onchain_events_awaiting_threshold_conf: Vec<OnchainEventEntry>,

	// If we get serialized out and re-read, we need to make sure that the chain monitoring
	// interface knows about the TXOs that we want to be notified of spends of. We could probably
	// be smart and derive them from the above storage fields, but its much simpler and more
	// Obviously Correct (tm) if we just keep track of them explicitly.
	outputs_to_watch: HashMap<Txid, Vec<(u32, Script)>>,

	#[cfg(test)]
	pub onchain_tx_handler: OnchainTxHandler<Signer>,
	#[cfg(not(test))]
	onchain_tx_handler: OnchainTxHandler<Signer>,

	// This is set when the Channel[Manager] generated a ChannelMonitorUpdate which indicated the
	// channel has been force-closed. After this is set, no further holder commitment transaction
	// updates may occur, and we panic!() if one is provided.
	lockdown_from_offchain: bool,

	// Set once we've signed a holder commitment transaction and handed it over to our
	// OnchainTxHandler. After this is set, no future updates to our holder commitment transactions
	// may occur, and we fail any such monitor updates.
	//
	// In case of update rejection due to a locally already signed commitment transaction, we
	// nevertheless store update content to track in case of concurrent broadcast by another
	// remote monitor out-of-order with regards to the block view.
	holder_tx_signed: bool,

	// If a spend of the funding output is seen, we set this to true and reject any further
	// updates. This prevents any further changes in the offchain state no matter the order
	// of block connection between ChannelMonitors and the ChannelManager.
	funding_spend_seen: bool,

	funding_spend_confirmed: Option<Txid>,
	/// The set of HTLCs which have been either claimed or failed on chain and have reached
	/// the requisite confirmations on the claim/fail transaction (either ANTI_REORG_DELAY or the
	/// spending CSV for revocable outputs).
	htlcs_resolved_on_chain: Vec<IrrevocablyResolvedHTLC>,

	// We simply modify best_block in Channel's block_connected so that serialization is
	// consistent but hopefully the users' copy handles block_connected in a consistent way.
	// (we do *not*, however, update them in update_monitor to ensure any local user copies keep
	// their best_block from its state and not based on updated copies that didn't run through
	// the full block_connected).
	best_block: BestBlock,

	secp_ctx: Secp256k1<secp256k1::All>, //TODO: dedup this a bit...
}

/// Transaction outputs to watch for on-chain spends.
pub type TransactionOutputs = (Txid, Vec<(u32, TxOut)>);

#[cfg(any(test, fuzzing, feature = "_test_utils"))]
/// Used only in testing and fuzzing to check serialization roundtrips don't change the underlying
/// object
impl<Signer: Sign> PartialEq for ChannelMonitor<Signer> {
	fn eq(&self, other: &Self) -> bool {
		let inner = self.inner.lock().unwrap();
		let other = other.inner.lock().unwrap();
		inner.eq(&other)
	}
}

#[cfg(any(test, fuzzing, feature = "_test_utils"))]
/// Used only in testing and fuzzing to check serialization roundtrips don't change the underlying
/// object
impl<Signer: Sign> PartialEq for ChannelMonitorImpl<Signer> {
	fn eq(&self, other: &Self) -> bool {
		if self.latest_update_id != other.latest_update_id ||
			self.commitment_transaction_number_obscure_factor != other.commitment_transaction_number_obscure_factor ||
			self.destination_script != other.destination_script ||
			self.broadcasted_holder_revokable_script != other.broadcasted_holder_revokable_script ||
			self.counterparty_payment_script != other.counterparty_payment_script ||
			self.channel_keys_id != other.channel_keys_id ||
			self.holder_revocation_basepoint != other.holder_revocation_basepoint ||
			self.funding_info != other.funding_info ||
			self.current_counterparty_commitment_txid != other.current_counterparty_commitment_txid ||
			self.prev_counterparty_commitment_txid != other.prev_counterparty_commitment_txid ||
			self.counterparty_commitment_params != other.counterparty_commitment_params ||
			self.funding_redeemscript != other.funding_redeemscript ||
			self.channel_value_satoshis != other.channel_value_satoshis ||
			self.their_cur_revocation_points != other.their_cur_revocation_points ||
			self.on_holder_tx_csv != other.on_holder_tx_csv ||
			self.commitment_secrets != other.commitment_secrets ||
			self.counterparty_claimable_outpoints != other.counterparty_claimable_outpoints ||
			self.counterparty_commitment_txn_on_chain != other.counterparty_commitment_txn_on_chain ||
			self.counterparty_hash_commitment_number != other.counterparty_hash_commitment_number ||
			self.prev_holder_signed_commitment_tx != other.prev_holder_signed_commitment_tx ||
			self.current_counterparty_commitment_number != other.current_counterparty_commitment_number ||
			self.current_holder_commitment_number != other.current_holder_commitment_number ||
			self.current_holder_commitment_tx != other.current_holder_commitment_tx ||
			self.payment_preimages != other.payment_preimages ||
			self.pending_monitor_events != other.pending_monitor_events ||
			self.pending_events.len() != other.pending_events.len() || // We trust events to round-trip properly
			self.onchain_events_awaiting_threshold_conf != other.onchain_events_awaiting_threshold_conf ||
			self.outputs_to_watch != other.outputs_to_watch ||
			self.lockdown_from_offchain != other.lockdown_from_offchain ||
			self.holder_tx_signed != other.holder_tx_signed ||
			self.funding_spend_seen != other.funding_spend_seen ||
			self.funding_spend_confirmed != other.funding_spend_confirmed ||
			self.htlcs_resolved_on_chain != other.htlcs_resolved_on_chain
		{
			false
		} else {
			true
		}
	}
}

impl<Signer: Sign> Writeable for ChannelMonitor<Signer> {
	fn write<W: Writer>(&self, writer: &mut W) -> Result<(), Error> {
		self.inner.lock().unwrap().write(writer)
	}
}

// These are also used for ChannelMonitorUpdate, above.
const SERIALIZATION_VERSION: u8 = 1;
const MIN_SERIALIZATION_VERSION: u8 = 1;

impl<Signer: Sign> Writeable for ChannelMonitorImpl<Signer> {
	fn write<W: Writer>(&self, writer: &mut W) -> Result<(), Error> {
		write_ver_prefix!(writer, SERIALIZATION_VERSION, MIN_SERIALIZATION_VERSION);

		self.latest_update_id.write(writer)?;

		// Set in initial Channel-object creation, so should always be set by now:
		U48(self.commitment_transaction_number_obscure_factor).write(writer)?;

		self.destination_script.write(writer)?;
		if let Some(ref broadcasted_holder_revokable_script) = self.broadcasted_holder_revokable_script {
			writer.write_all(&[0; 1])?;
			broadcasted_holder_revokable_script.0.write(writer)?;
			broadcasted_holder_revokable_script.1.write(writer)?;
			broadcasted_holder_revokable_script.2.write(writer)?;
		} else {
			writer.write_all(&[1; 1])?;
		}

		self.counterparty_payment_script.write(writer)?;
		match &self.shutdown_script {
			Some(script) => script.write(writer)?,
			None => Script::new().write(writer)?,
		}

		self.channel_keys_id.write(writer)?;
		self.holder_revocation_basepoint.write(writer)?;
		writer.write_all(&self.funding_info.0.txid[..])?;
		writer.write_all(&byte_utils::be16_to_array(self.funding_info.0.index))?;
		self.funding_info.1.write(writer)?;
		self.current_counterparty_commitment_txid.write(writer)?;
		self.prev_counterparty_commitment_txid.write(writer)?;

		self.counterparty_commitment_params.write(writer)?;
		self.funding_redeemscript.write(writer)?;
		self.channel_value_satoshis.write(writer)?;

		match self.their_cur_revocation_points {
			Some((idx, pubkey, second_option)) => {
				writer.write_all(&byte_utils::be48_to_array(idx))?;
				writer.write_all(&pubkey.serialize())?;
				match second_option {
					Some(second_pubkey) => {
						writer.write_all(&second_pubkey.serialize())?;
					},
					None => {
						writer.write_all(&[0; 33])?;
					},
				}
			},
			None => {
				writer.write_all(&byte_utils::be48_to_array(0))?;
			},
		}

		writer.write_all(&byte_utils::be16_to_array(self.on_holder_tx_csv))?;

		self.commitment_secrets.write(writer)?;

		macro_rules! serialize_htlc_in_commitment {
			($htlc_output: expr) => {
				writer.write_all(&[$htlc_output.offered as u8; 1])?;
				writer.write_all(&byte_utils::be64_to_array($htlc_output.amount_msat))?;
				writer.write_all(&byte_utils::be32_to_array($htlc_output.cltv_expiry))?;
				writer.write_all(&$htlc_output.payment_hash.0[..])?;
				$htlc_output.transaction_output_index.write(writer)?;
			}
		}

		writer.write_all(&byte_utils::be64_to_array(self.counterparty_claimable_outpoints.len() as u64))?;
		for (ref txid, ref htlc_infos) in self.counterparty_claimable_outpoints.iter() {
			writer.write_all(&txid[..])?;
			writer.write_all(&byte_utils::be64_to_array(htlc_infos.len() as u64))?;
			for &(ref htlc_output, ref htlc_source) in htlc_infos.iter() {
				serialize_htlc_in_commitment!(htlc_output);
				htlc_source.as_ref().map(|b| b.as_ref()).write(writer)?;
			}
		}

		writer.write_all(&byte_utils::be64_to_array(self.counterparty_commitment_txn_on_chain.len() as u64))?;
		for (ref txid, commitment_number) in self.counterparty_commitment_txn_on_chain.iter() {
			writer.write_all(&txid[..])?;
			writer.write_all(&byte_utils::be48_to_array(*commitment_number))?;
		}

		writer.write_all(&byte_utils::be64_to_array(self.counterparty_hash_commitment_number.len() as u64))?;
		for (ref payment_hash, commitment_number) in self.counterparty_hash_commitment_number.iter() {
			writer.write_all(&payment_hash.0[..])?;
			writer.write_all(&byte_utils::be48_to_array(*commitment_number))?;
		}

		if let Some(ref prev_holder_tx) = self.prev_holder_signed_commitment_tx {
			writer.write_all(&[1; 1])?;
			prev_holder_tx.write(writer)?;
		} else {
			writer.write_all(&[0; 1])?;
		}

		self.current_holder_commitment_tx.write(writer)?;

		writer.write_all(&byte_utils::be48_to_array(self.current_counterparty_commitment_number))?;
		writer.write_all(&byte_utils::be48_to_array(self.current_holder_commitment_number))?;

		writer.write_all(&byte_utils::be64_to_array(self.payment_preimages.len() as u64))?;
		for payment_preimage in self.payment_preimages.values() {
			writer.write_all(&payment_preimage.0[..])?;
		}

		writer.write_all(&(self.pending_monitor_events.iter().filter(|ev| match ev {
			MonitorEvent::HTLCEvent(_) => true,
			MonitorEvent::CommitmentTxConfirmed(_) => true,
			_ => false,
		}).count() as u64).to_be_bytes())?;
		for event in self.pending_monitor_events.iter() {
			match event {
				MonitorEvent::HTLCEvent(upd) => {
					0u8.write(writer)?;
					upd.write(writer)?;
				},
				MonitorEvent::CommitmentTxConfirmed(_) => 1u8.write(writer)?,
				_ => {}, // Covered in the TLV writes below
			}
		}

		writer.write_all(&byte_utils::be64_to_array(self.pending_events.len() as u64))?;
		for event in self.pending_events.iter() {
			event.write(writer)?;
		}

		self.best_block.block_hash().write(writer)?;
		writer.write_all(&byte_utils::be32_to_array(self.best_block.height()))?;

		writer.write_all(&byte_utils::be64_to_array(self.onchain_events_awaiting_threshold_conf.len() as u64))?;
		for ref entry in self.onchain_events_awaiting_threshold_conf.iter() {
			entry.write(writer)?;
		}

		(self.outputs_to_watch.len() as u64).write(writer)?;
		for (txid, idx_scripts) in self.outputs_to_watch.iter() {
			txid.write(writer)?;
			(idx_scripts.len() as u64).write(writer)?;
			for (idx, script) in idx_scripts.iter() {
				idx.write(writer)?;
				script.write(writer)?;
			}
		}
		self.onchain_tx_handler.write(writer)?;

		self.lockdown_from_offchain.write(writer)?;
		self.holder_tx_signed.write(writer)?;

		write_tlv_fields!(writer, {
			(1, self.funding_spend_confirmed, option),
			(3, self.htlcs_resolved_on_chain, vec_type),
			(5, self.pending_monitor_events, vec_type),
			(7, self.funding_spend_seen, required),
		});

		Ok(())
	}
}

impl<Signer: Sign> ChannelMonitor<Signer> {
	pub(crate) fn new(secp_ctx: Secp256k1<secp256k1::All>, keys: Signer, shutdown_script: Option<Script>,
	                  on_counterparty_tx_csv: u16, destination_script: &Script, funding_info: (OutPoint, Script),
	                  channel_parameters: &ChannelTransactionParameters,
	                  funding_redeemscript: Script, channel_value_satoshis: u64,
	                  commitment_transaction_number_obscure_factor: u64,
	                  initial_holder_commitment_tx: HolderCommitmentTransaction,
	                  best_block: BestBlock) -> ChannelMonitor<Signer> {

		assert!(commitment_transaction_number_obscure_factor <= (1 << 48));
		let payment_key_hash = WPubkeyHash::hash(&keys.pubkeys().payment_point.serialize());
		let counterparty_payment_script = Builder::new().push_opcode(opcodes::all::OP_PUSHBYTES_0).push_slice(&payment_key_hash[..]).into_script();

		let counterparty_channel_parameters = channel_parameters.counterparty_parameters.as_ref().unwrap();
		let counterparty_delayed_payment_base_key = counterparty_channel_parameters.pubkeys.delayed_payment_basepoint;
		let counterparty_htlc_base_key = counterparty_channel_parameters.pubkeys.htlc_basepoint;
		let counterparty_commitment_params = CounterpartyCommitmentParameters { counterparty_delayed_payment_base_key, counterparty_htlc_base_key, on_counterparty_tx_csv };

		let channel_keys_id = keys.channel_keys_id();
		let holder_revocation_basepoint = keys.pubkeys().revocation_basepoint;

		// block for Rust 1.34 compat
		let (holder_commitment_tx, current_holder_commitment_number) = {
			let trusted_tx = initial_holder_commitment_tx.trust();
			let txid = trusted_tx.txid();

			let tx_keys = trusted_tx.keys();
			let holder_commitment_tx = HolderSignedTx {
				txid,
				revocation_key: tx_keys.revocation_key,
				a_htlc_key: tx_keys.broadcaster_htlc_key,
				b_htlc_key: tx_keys.countersignatory_htlc_key,
				delayed_payment_key: tx_keys.broadcaster_delayed_payment_key,
				per_commitment_point: tx_keys.per_commitment_point,
				htlc_outputs: Vec::new(), // There are never any HTLCs in the initial commitment transactions
				to_self_value_sat: initial_holder_commitment_tx.to_broadcaster_value_sat(),
				feerate_per_kw: trusted_tx.feerate_per_kw(),
			};
			(holder_commitment_tx, trusted_tx.commitment_number())
		};

		let onchain_tx_handler =
			OnchainTxHandler::new(destination_script.clone(), keys,
			channel_parameters.clone(), initial_holder_commitment_tx, secp_ctx.clone());

		let mut outputs_to_watch = HashMap::new();
		outputs_to_watch.insert(funding_info.0.txid, vec![(funding_info.0.index as u32, funding_info.1.clone())]);

		ChannelMonitor {
			inner: Mutex::new(ChannelMonitorImpl {
				latest_update_id: 0,
				commitment_transaction_number_obscure_factor,

				destination_script: destination_script.clone(),
				broadcasted_holder_revokable_script: None,
				counterparty_payment_script,
				shutdown_script,

				channel_keys_id,
				holder_revocation_basepoint,
				funding_info,
				current_counterparty_commitment_txid: None,
				prev_counterparty_commitment_txid: None,

				counterparty_commitment_params,
				funding_redeemscript,
				channel_value_satoshis,
				their_cur_revocation_points: None,

				on_holder_tx_csv: counterparty_channel_parameters.selected_contest_delay,

				commitment_secrets: CounterpartyCommitmentSecrets::new(),
				counterparty_claimable_outpoints: HashMap::new(),
				counterparty_commitment_txn_on_chain: HashMap::new(),
				counterparty_hash_commitment_number: HashMap::new(),

				prev_holder_signed_commitment_tx: None,
				current_holder_commitment_tx: holder_commitment_tx,
				current_counterparty_commitment_number: 1 << 48,
				current_holder_commitment_number,

				payment_preimages: HashMap::new(),
				pending_monitor_events: Vec::new(),
				pending_events: Vec::new(),

				onchain_events_awaiting_threshold_conf: Vec::new(),
				outputs_to_watch,

				onchain_tx_handler,

				lockdown_from_offchain: false,
				holder_tx_signed: false,
				funding_spend_seen: false,
				funding_spend_confirmed: None,
				htlcs_resolved_on_chain: Vec::new(),

				best_block,

				secp_ctx,
			}),
		}
	}

	#[cfg(test)]
	fn provide_secret(&self, idx: u64, secret: [u8; 32]) -> Result<(), &'static str> {
		self.inner.lock().unwrap().provide_secret(idx, secret)
	}

	/// Informs this monitor of the latest counterparty (ie non-broadcastable) commitment transaction.
	/// The monitor watches for it to be broadcasted and then uses the HTLC information (and
	/// possibly future revocation/preimage information) to claim outputs where possible.
	/// We cache also the mapping hash:commitment number to lighten pruning of old preimages by watchtowers.
	pub(crate) fn provide_latest_counterparty_commitment_tx<L: Deref>(
		&self,
		txid: Txid,
		htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Box<HTLCSource>>)>,
		commitment_number: u64,
		their_revocation_point: PublicKey,
		logger: &L,
	) where L::Target: Logger {
		self.inner.lock().unwrap().provide_latest_counterparty_commitment_tx(
			txid, htlc_outputs, commitment_number, their_revocation_point, logger)
	}

	#[cfg(test)]
	fn provide_latest_holder_commitment_tx(
		&self, holder_commitment_tx: HolderCommitmentTransaction,
		htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Signature>, Option<HTLCSource>)>,
	) -> Result<(), ()> {
		self.inner.lock().unwrap().provide_latest_holder_commitment_tx(holder_commitment_tx, htlc_outputs).map_err(|_| ())
	}

	#[cfg(test)]
	pub(crate) fn provide_payment_preimage<B: Deref, F: Deref, L: Deref>(
		&self,
		payment_hash: &PaymentHash,
		payment_preimage: &PaymentPreimage,
		broadcaster: &B,
		fee_estimator: &F,
		logger: &L,
	) where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().provide_payment_preimage(
			payment_hash, payment_preimage, broadcaster, fee_estimator, logger)
	}

	pub(crate) fn broadcast_latest_holder_commitment_txn<B: Deref, L: Deref>(
		&self,
		broadcaster: &B,
		logger: &L,
	) where
		B::Target: BroadcasterInterface,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().broadcast_latest_holder_commitment_txn(broadcaster, logger)
	}

	/// Updates a ChannelMonitor on the basis of some new information provided by the Channel
	/// itself.
	///
	/// panics if the given update is not the next update by update_id.
	pub fn update_monitor<B: Deref, F: Deref, L: Deref>(
		&self,
		updates: &ChannelMonitorUpdate,
		broadcaster: &B,
		fee_estimator: &F,
		logger: &L,
	) -> Result<(), ()>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().update_monitor(updates, broadcaster, fee_estimator, logger)
	}

	/// Gets the update_id from the latest ChannelMonitorUpdate which was applied to this
	/// ChannelMonitor.
	pub fn get_latest_update_id(&self) -> u64 {
		self.inner.lock().unwrap().get_latest_update_id()
	}

	/// Gets the funding transaction outpoint of the channel this ChannelMonitor is monitoring for.
	pub fn get_funding_txo(&self) -> (OutPoint, Script) {
		self.inner.lock().unwrap().get_funding_txo().clone()
	}

	/// Gets a list of txids, with their output scripts (in the order they appear in the
	/// transaction), which we must learn about spends of via block_connected().
	pub fn get_outputs_to_watch(&self) -> Vec<(Txid, Vec<(u32, Script)>)> {
		self.inner.lock().unwrap().get_outputs_to_watch()
			.iter().map(|(txid, outputs)| (*txid, outputs.clone())).collect()
	}

	/// Loads the funding txo and outputs to watch into the given `chain::Filter` by repeatedly
	/// calling `chain::Filter::register_output` and `chain::Filter::register_tx` until all outputs
	/// have been registered.
	pub fn load_outputs_to_watch<F: Deref>(&self, filter: &F) where F::Target: chain::Filter {
		let lock = self.inner.lock().unwrap();
		filter.register_tx(&lock.get_funding_txo().0.txid, &lock.get_funding_txo().1);
		for (txid, outputs) in lock.get_outputs_to_watch().iter() {
			for (index, script_pubkey) in outputs.iter() {
				assert!(*index <= u16::max_value() as u32);
				filter.register_output(WatchedOutput {
					block_hash: None,
					outpoint: OutPoint { txid: *txid, index: *index as u16 },
					script_pubkey: script_pubkey.clone(),
				});
			}
		}
	}

	/// Get the list of HTLCs who's status has been updated on chain. This should be called by
	/// ChannelManager via [`chain::Watch::release_pending_monitor_events`].
	pub fn get_and_clear_pending_monitor_events(&self) -> Vec<MonitorEvent> {
		self.inner.lock().unwrap().get_and_clear_pending_monitor_events()
	}

	/// Gets the list of pending events which were generated by previous actions, clearing the list
	/// in the process.
	///
	/// This is called by ChainMonitor::get_and_clear_pending_events() and is equivalent to
	/// EventsProvider::get_and_clear_pending_events() except that it requires &mut self as we do
	/// no internal locking in ChannelMonitors.
	pub fn get_and_clear_pending_events(&self) -> Vec<Event> {
		self.inner.lock().unwrap().get_and_clear_pending_events()
	}

	pub(crate) fn get_min_seen_secret(&self) -> u64 {
		self.inner.lock().unwrap().get_min_seen_secret()
	}

	pub(crate) fn get_cur_counterparty_commitment_number(&self) -> u64 {
		self.inner.lock().unwrap().get_cur_counterparty_commitment_number()
	}

	pub(crate) fn get_cur_holder_commitment_number(&self) -> u64 {
		self.inner.lock().unwrap().get_cur_holder_commitment_number()
	}

	/// Used by ChannelManager deserialization to broadcast the latest holder state if its copy of
	/// the Channel was out-of-date. You may use it to get a broadcastable holder toxic tx in case of
	/// fallen-behind, i.e when receiving a channel_reestablish with a proof that our counterparty side knows
	/// a higher revocation secret than the holder commitment number we are aware of. Broadcasting these
	/// transactions are UNSAFE, as they allow counterparty side to punish you. Nevertheless you may want to
	/// broadcast them if counterparty don't close channel with his higher commitment transaction after a
	/// substantial amount of time (a month or even a year) to get back funds. Best may be to contact
	/// out-of-band the other node operator to coordinate with him if option is available to you.
	/// In any-case, choice is up to the user.
	pub fn get_latest_holder_commitment_txn<L: Deref>(&self, logger: &L) -> Vec<Transaction>
	where L::Target: Logger {
		self.inner.lock().unwrap().get_latest_holder_commitment_txn(logger)
	}

	/// Unsafe test-only version of get_latest_holder_commitment_txn used by our test framework
	/// to bypass HolderCommitmentTransaction state update lockdown after signature and generate
	/// revoked commitment transaction.
	#[cfg(any(test, feature = "unsafe_revoked_tx_signing"))]
	pub fn unsafe_get_latest_holder_commitment_txn<L: Deref>(&self, logger: &L) -> Vec<Transaction>
	where L::Target: Logger {
		self.inner.lock().unwrap().unsafe_get_latest_holder_commitment_txn(logger)
	}

	/// Processes transactions in a newly connected block, which may result in any of the following:
	/// - update the monitor's state against resolved HTLCs
	/// - punish the counterparty in the case of seeing a revoked commitment transaction
	/// - force close the channel and claim/timeout incoming/outgoing HTLCs if near expiration
	/// - detect settled outputs for later spending
	/// - schedule and bump any in-flight claims
	///
	/// Returns any new outputs to watch from `txdata`; after called, these are also included in
	/// [`get_outputs_to_watch`].
	///
	/// [`get_outputs_to_watch`]: #method.get_outputs_to_watch
	pub fn block_connected<B: Deref, F: Deref, L: Deref>(
		&self,
		header: &BlockHeader,
		txdata: &TransactionData,
		height: u32,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) -> Vec<TransactionOutputs>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().block_connected(
			header, txdata, height, broadcaster, fee_estimator, logger)
	}

	/// Determines if the disconnected block contained any transactions of interest and updates
	/// appropriately.
	pub fn block_disconnected<B: Deref, F: Deref, L: Deref>(
		&self,
		header: &BlockHeader,
		height: u32,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().block_disconnected(
			header, height, broadcaster, fee_estimator, logger)
	}

	/// Processes transactions confirmed in a block with the given header and height, returning new
	/// outputs to watch. See [`block_connected`] for details.
	///
	/// Used instead of [`block_connected`] by clients that are notified of transactions rather than
	/// blocks. See [`chain::Confirm`] for calling expectations.
	///
	/// [`block_connected`]: Self::block_connected
	pub fn transactions_confirmed<B: Deref, F: Deref, L: Deref>(
		&self,
		header: &BlockHeader,
		txdata: &TransactionData,
		height: u32,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) -> Vec<TransactionOutputs>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().transactions_confirmed(
			header, txdata, height, broadcaster, fee_estimator, logger)
	}

	/// Processes a transaction that was reorganized out of the chain.
	///
	/// Used instead of [`block_disconnected`] by clients that are notified of transactions rather
	/// than blocks. See [`chain::Confirm`] for calling expectations.
	///
	/// [`block_disconnected`]: Self::block_disconnected
	pub fn transaction_unconfirmed<B: Deref, F: Deref, L: Deref>(
		&self,
		txid: &Txid,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().transaction_unconfirmed(
			txid, broadcaster, fee_estimator, logger);
	}

	/// Updates the monitor with the current best chain tip, returning new outputs to watch. See
	/// [`block_connected`] for details.
	///
	/// Used instead of [`block_connected`] by clients that are notified of transactions rather than
	/// blocks. See [`chain::Confirm`] for calling expectations.
	///
	/// [`block_connected`]: Self::block_connected
	pub fn best_block_updated<B: Deref, F: Deref, L: Deref>(
		&self,
		header: &BlockHeader,
		height: u32,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) -> Vec<TransactionOutputs>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.inner.lock().unwrap().best_block_updated(
			header, height, broadcaster, fee_estimator, logger)
	}

	/// Returns the set of txids that should be monitored for re-organization out of the chain.
	pub fn get_relevant_txids(&self) -> Vec<Txid> {
		let inner = self.inner.lock().unwrap();
		let mut txids: Vec<Txid> = inner.onchain_events_awaiting_threshold_conf
			.iter()
			.map(|entry| entry.txid)
			.chain(inner.onchain_tx_handler.get_relevant_txids().into_iter())
			.collect();
		txids.sort_unstable();
		txids.dedup();
		txids
	}

	/// Gets the latest best block which was connected either via the [`chain::Listen`] or
	/// [`chain::Confirm`] interfaces.
	pub fn current_best_block(&self) -> BestBlock {
		self.inner.lock().unwrap().best_block.clone()
	}

	/// Gets the balances in this channel which are either claimable by us if we were to
	/// force-close the channel now or which are claimable on-chain (possibly awaiting
	/// confirmation).
	///
	/// Any balances in the channel which are available on-chain (excluding on-chain fees) are
	/// included here until an [`Event::SpendableOutputs`] event has been generated for the
	/// balance, or until our counterparty has claimed the balance and accrued several
	/// confirmations on the claim transaction.
	///
	/// Note that the balances available when you or your counterparty have broadcasted revoked
	/// state(s) may not be fully captured here.
	// TODO, fix that ^
	///
	/// See [`Balance`] for additional details on the types of claimable balances which
	/// may be returned here and their meanings.
	pub fn get_claimable_balances(&self) -> Vec<Balance> {
		let mut res = Vec::new();
		let us = self.inner.lock().unwrap();

		let mut confirmed_txid = us.funding_spend_confirmed;
		let mut pending_commitment_tx_conf_thresh = None;
		let funding_spend_pending = us.onchain_events_awaiting_threshold_conf.iter().find_map(|event| {
			if let OnchainEvent::FundingSpendConfirmation { .. } = event.event {
				Some((event.txid, event.confirmation_threshold()))
			} else { None }
		});
		if let Some((txid, conf_thresh)) = funding_spend_pending {
			debug_assert!(us.funding_spend_confirmed.is_none(),
				"We have a pending funding spend awaiting anti-reorg confirmation, we can't have confirmed it already!");
			confirmed_txid = Some(txid);
			pending_commitment_tx_conf_thresh = Some(conf_thresh);
		}

		macro_rules! walk_htlcs {
			($holder_commitment: expr, $htlc_iter: expr) => {
				for htlc in $htlc_iter {
					if let Some(htlc_input_idx) = htlc.transaction_output_index {
						if let Some(conf_thresh) = us.onchain_events_awaiting_threshold_conf.iter().find_map(|event| {
							if let OnchainEvent::MaturingOutput { descriptor: SpendableOutputDescriptor::DelayedPaymentOutput(descriptor) } = &event.event {
								if descriptor.outpoint.index as u32 == htlc_input_idx { Some(event.confirmation_threshold()) } else { None }
							} else { None }
						}) {
							debug_assert!($holder_commitment);
							res.push(Balance::ClaimableAwaitingConfirmations {
								claimable_amount_satoshis: htlc.amount_msat / 1000,
								confirmation_height: conf_thresh,
							});
						} else if us.htlcs_resolved_on_chain.iter().any(|v| v.input_idx == htlc_input_idx) {
							// Funding transaction spends should be fully confirmed by the time any
							// HTLC transactions are resolved, unless we're talking about a holder
							// commitment tx, whose resolution is delayed until the CSV timeout is
							// reached, even though HTLCs may be resolved after only
							// ANTI_REORG_DELAY confirmations.
							debug_assert!($holder_commitment || us.funding_spend_confirmed.is_some());
						} else if htlc.offered == $holder_commitment {
							// If the payment was outbound, check if there's an HTLCUpdate
							// indicating we have spent this HTLC with a timeout, claiming it back
							// and awaiting confirmations on it.
							let htlc_update_pending = us.onchain_events_awaiting_threshold_conf.iter().find_map(|event| {
								if let OnchainEvent::HTLCUpdate { input_idx: Some(input_idx), .. } = event.event {
									if input_idx == htlc_input_idx { Some(event.confirmation_threshold()) } else { None }
								} else { None }
							});
							if let Some(conf_thresh) = htlc_update_pending {
								res.push(Balance::ClaimableAwaitingConfirmations {
									claimable_amount_satoshis: htlc.amount_msat / 1000,
									confirmation_height: conf_thresh,
								});
							} else {
								res.push(Balance::MaybeClaimableHTLCAwaitingTimeout {
									claimable_amount_satoshis: htlc.amount_msat / 1000,
									claimable_height: htlc.cltv_expiry,
								});
							}
						} else if us.payment_preimages.get(&htlc.payment_hash).is_some() {
							// Otherwise (the payment was inbound), only expose it as claimable if
							// we know the preimage.
							// Note that if there is a pending claim, but it did not use the
							// preimage, we lost funds to our counterparty! We will then continue
							// to show it as ContentiousClaimable until ANTI_REORG_DELAY.
							let htlc_spend_pending = us.onchain_events_awaiting_threshold_conf.iter().find_map(|event| {
								if let OnchainEvent::HTLCSpendConfirmation { input_idx, preimage, .. } = event.event {
									if input_idx == htlc_input_idx {
										Some((event.confirmation_threshold(), preimage.is_some()))
									} else { None }
								} else { None }
							});
							if let Some((conf_thresh, true)) = htlc_spend_pending {
								res.push(Balance::ClaimableAwaitingConfirmations {
									claimable_amount_satoshis: htlc.amount_msat / 1000,
									confirmation_height: conf_thresh,
								});
							} else {
								res.push(Balance::ContentiousClaimable {
									claimable_amount_satoshis: htlc.amount_msat / 1000,
									timeout_height: htlc.cltv_expiry,
								});
							}
						}
					}
				}
			}
		}

		if let Some(txid) = confirmed_txid {
			let mut found_commitment_tx = false;
			if Some(txid) == us.current_counterparty_commitment_txid || Some(txid) == us.prev_counterparty_commitment_txid {
				walk_htlcs!(false, us.counterparty_claimable_outpoints.get(&txid).unwrap().iter().map(|(a, _)| a));
				if let Some(conf_thresh) = pending_commitment_tx_conf_thresh {
					if let Some(value) = us.onchain_events_awaiting_threshold_conf.iter().find_map(|event| {
						if let OnchainEvent::MaturingOutput {
							descriptor: SpendableOutputDescriptor::StaticPaymentOutput(descriptor)
						} = &event.event {
							Some(descriptor.output.value)
						} else { None }
					}) {
						res.push(Balance::ClaimableAwaitingConfirmations {
							claimable_amount_satoshis: value,
							confirmation_height: conf_thresh,
						});
					} else {
						// If a counterparty commitment transaction is awaiting confirmation, we
						// should either have a StaticPaymentOutput MaturingOutput event awaiting
						// confirmation with the same height or have never met our dust amount.
					}
				}
				found_commitment_tx = true;
			} else if txid == us.current_holder_commitment_tx.txid {
				walk_htlcs!(true, us.current_holder_commitment_tx.htlc_outputs.iter().map(|(a, _, _)| a));
				if let Some(conf_thresh) = pending_commitment_tx_conf_thresh {
					res.push(Balance::ClaimableAwaitingConfirmations {
						claimable_amount_satoshis: us.current_holder_commitment_tx.to_self_value_sat,
						confirmation_height: conf_thresh,
					});
				}
				found_commitment_tx = true;
			} else if let Some(prev_commitment) = &us.prev_holder_signed_commitment_tx {
				if txid == prev_commitment.txid {
					walk_htlcs!(true, prev_commitment.htlc_outputs.iter().map(|(a, _, _)| a));
					if let Some(conf_thresh) = pending_commitment_tx_conf_thresh {
						res.push(Balance::ClaimableAwaitingConfirmations {
							claimable_amount_satoshis: prev_commitment.to_self_value_sat,
							confirmation_height: conf_thresh,
						});
					}
					found_commitment_tx = true;
				}
			}
			if !found_commitment_tx {
				if let Some(conf_thresh) = pending_commitment_tx_conf_thresh {
					// We blindly assume this is a cooperative close transaction here, and that
					// neither us nor our counterparty misbehaved. At worst we've under-estimated
					// the amount we can claim as we'll punish a misbehaving counterparty.
					res.push(Balance::ClaimableAwaitingConfirmations {
						claimable_amount_satoshis: us.current_holder_commitment_tx.to_self_value_sat,
						confirmation_height: conf_thresh,
					});
				}
			}
			// TODO: Add logic to provide claimable balances for counterparty broadcasting revoked
			// outputs.
		} else {
			let mut claimable_inbound_htlc_value_sat = 0;
			for (htlc, _, _) in us.current_holder_commitment_tx.htlc_outputs.iter() {
				if htlc.transaction_output_index.is_none() { continue; }
				if htlc.offered {
					res.push(Balance::MaybeClaimableHTLCAwaitingTimeout {
						claimable_amount_satoshis: htlc.amount_msat / 1000,
						claimable_height: htlc.cltv_expiry,
					});
				} else if us.payment_preimages.get(&htlc.payment_hash).is_some() {
					claimable_inbound_htlc_value_sat += htlc.amount_msat / 1000;
				}
			}
			res.push(Balance::ClaimableOnChannelClose {
				claimable_amount_satoshis: us.current_holder_commitment_tx.to_self_value_sat + claimable_inbound_htlc_value_sat,
			});
		}

		res
	}

	/// Gets the set of outbound HTLCs which are pending resolution in this channel.
	/// This is used to reconstruct pending outbound payments on restart in the ChannelManager.
	pub(crate) fn get_pending_outbound_htlcs(&self) -> HashMap<HTLCSource, HTLCOutputInCommitment> {
		let mut res = HashMap::new();
		let us = self.inner.lock().unwrap();

		macro_rules! walk_htlcs {
			($holder_commitment: expr, $htlc_iter: expr) => {
				for (htlc, source) in $htlc_iter {
					if us.htlcs_resolved_on_chain.iter().any(|v| Some(v.input_idx) == htlc.transaction_output_index) {
						// We should assert that funding_spend_confirmed is_some() here, but we
						// have some unit tests which violate HTLC transaction CSVs entirely and
						// would fail.
						// TODO: Once tests all connect transactions at consensus-valid times, we
						// should assert here like we do in `get_claimable_balances`.
					} else if htlc.offered == $holder_commitment {
						// If the payment was outbound, check if there's an HTLCUpdate
						// indicating we have spent this HTLC with a timeout, claiming it back
						// and awaiting confirmations on it.
						let htlc_update_confd = us.onchain_events_awaiting_threshold_conf.iter().any(|event| {
							if let OnchainEvent::HTLCUpdate { input_idx: Some(input_idx), .. } = event.event {
								// If the HTLC was timed out, we wait for ANTI_REORG_DELAY blocks
								// before considering it "no longer pending" - this matches when we
								// provide the ChannelManager an HTLC failure event.
								Some(input_idx) == htlc.transaction_output_index &&
									us.best_block.height() >= event.height + ANTI_REORG_DELAY - 1
							} else if let OnchainEvent::HTLCSpendConfirmation { input_idx, .. } = event.event {
								// If the HTLC was fulfilled with a preimage, we consider the HTLC
								// immediately non-pending, matching when we provide ChannelManager
								// the preimage.
								Some(input_idx) == htlc.transaction_output_index
							} else { false }
						});
						if !htlc_update_confd {
							res.insert(source.clone(), htlc.clone());
						}
					}
				}
			}
		}

		// We're only concerned with the confirmation count of HTLC transactions, and don't
		// actually care how many confirmations a commitment transaction may or may not have. Thus,
		// we look for either a FundingSpendConfirmation event or a funding_spend_confirmed.
		let confirmed_txid = us.funding_spend_confirmed.or_else(|| {
			us.onchain_events_awaiting_threshold_conf.iter().find_map(|event| {
				if let OnchainEvent::FundingSpendConfirmation { .. } = event.event {
					Some(event.txid)
				} else { None }
			})
		});
		if let Some(txid) = confirmed_txid {
			if Some(txid) == us.current_counterparty_commitment_txid || Some(txid) == us.prev_counterparty_commitment_txid {
				walk_htlcs!(false, us.counterparty_claimable_outpoints.get(&txid).unwrap().iter().filter_map(|(a, b)| {
					if let &Some(ref source) = b {
						Some((a, &**source))
					} else { None }
				}));
			} else if txid == us.current_holder_commitment_tx.txid {
				walk_htlcs!(true, us.current_holder_commitment_tx.htlc_outputs.iter().filter_map(|(a, _, c)| {
					if let Some(source) = c { Some((a, source)) } else { None }
				}));
			} else if let Some(prev_commitment) = &us.prev_holder_signed_commitment_tx {
				if txid == prev_commitment.txid {
					walk_htlcs!(true, prev_commitment.htlc_outputs.iter().filter_map(|(a, _, c)| {
						if let Some(source) = c { Some((a, source)) } else { None }
					}));
				}
			}
		} else {
			// If we have not seen a commitment transaction on-chain (ie the channel is not yet
			// closed), just examine the available counterparty commitment transactions. See docs
			// on `fail_unbroadcast_htlcs`, below, for justification.
			macro_rules! walk_counterparty_commitment {
				($txid: expr) => {
					if let Some(ref latest_outpoints) = us.counterparty_claimable_outpoints.get($txid) {
						for &(ref htlc, ref source_option) in latest_outpoints.iter() {
							if let &Some(ref source) = source_option {
								res.insert((**source).clone(), htlc.clone());
							}
						}
					}
				}
			}
			if let Some(ref txid) = us.current_counterparty_commitment_txid {
				walk_counterparty_commitment!(txid);
			}
			if let Some(ref txid) = us.prev_counterparty_commitment_txid {
				walk_counterparty_commitment!(txid);
			}
		}

		res
	}
}

/// Compares a broadcasted commitment transaction's HTLCs with those in the latest state,
/// failing any HTLCs which didn't make it into the broadcasted commitment transaction back
/// after ANTI_REORG_DELAY blocks.
///
/// We always compare against the set of HTLCs in counterparty commitment transactions, as those
/// are the commitment transactions which are generated by us. The off-chain state machine in
/// `Channel` will automatically resolve any HTLCs which were never included in a commitment
/// transaction when it detects channel closure, but it is up to us to ensure any HTLCs which were
/// included in a remote commitment transaction are failed back if they are not present in the
/// broadcasted commitment transaction.
///
/// Specifically, the removal process for HTLCs in `Channel` is always based on the counterparty
/// sending a `revoke_and_ack`, which causes us to clear `prev_counterparty_commitment_txid`. Thus,
/// as long as we examine both the current counterparty commitment transaction and, if it hasn't
/// been revoked yet, the previous one, we we will never "forget" to resolve an HTLC.
macro_rules! fail_unbroadcast_htlcs {
	($self: expr, $commitment_tx_type: expr, $commitment_tx_conf_height: expr, $confirmed_htlcs_list: expr, $logger: expr) => { {
		macro_rules! check_htlc_fails {
			($txid: expr, $commitment_tx: expr) => {
				if let Some(ref latest_outpoints) = $self.counterparty_claimable_outpoints.get($txid) {
					for &(ref htlc, ref source_option) in latest_outpoints.iter() {
						if let &Some(ref source) = source_option {
							// Check if the HTLC is present in the commitment transaction that was
							// broadcast, but not if it was below the dust limit, which we should
							// fail backwards immediately as there is no way for us to learn the
							// payment_preimage.
							// Note that if the dust limit were allowed to change between
							// commitment transactions we'd want to be check whether *any*
							// broadcastable commitment transaction has the HTLC in it, but it
							// cannot currently change after channel initialization, so we don't
							// need to here.
							let confirmed_htlcs_iter: &mut Iterator<Item = (&HTLCOutputInCommitment, Option<&HTLCSource>)> = &mut $confirmed_htlcs_list;
							let mut matched_htlc = false;
							for (ref broadcast_htlc, ref broadcast_source) in confirmed_htlcs_iter {
								if broadcast_htlc.transaction_output_index.is_some() && Some(&**source) == *broadcast_source {
									matched_htlc = true;
									break;
								}
							}
							if matched_htlc { continue; }
							$self.onchain_events_awaiting_threshold_conf.retain(|ref entry| {
								if entry.height != $commitment_tx_conf_height { return true; }
								match entry.event {
									OnchainEvent::HTLCUpdate { source: ref update_source, .. } => {
										*update_source != **source
									},
									_ => true,
								}
							});
							let entry = OnchainEventEntry {
								txid: *$txid,
								height: $commitment_tx_conf_height,
								event: OnchainEvent::HTLCUpdate {
									source: (**source).clone(),
									payment_hash: htlc.payment_hash.clone(),
									onchain_value_satoshis: Some(htlc.amount_msat / 1000),
									input_idx: None,
								},
							};
							log_trace!($logger, "Failing HTLC with payment_hash {} from {} counterparty commitment tx due to broadcast of {} commitment transaction, waiting for confirmation (at height {})",
								log_bytes!(htlc.payment_hash.0), $commitment_tx, $commitment_tx_type, entry.confirmation_threshold());
							$self.onchain_events_awaiting_threshold_conf.push(entry);
						}
					}
				}
			}
		}
		if let Some(ref txid) = $self.current_counterparty_commitment_txid {
			check_htlc_fails!(txid, "current");
		}
		if let Some(ref txid) = $self.prev_counterparty_commitment_txid {
			check_htlc_fails!(txid, "previous");
		}
	} }
}

impl<Signer: Sign> ChannelMonitorImpl<Signer> {
	/// Inserts a revocation secret into this channel monitor. Prunes old preimages if neither
	/// needed by holder commitment transactions HTCLs nor by counterparty ones. Unless we haven't already seen
	/// counterparty commitment transaction's secret, they are de facto pruned (we can use revocation key).
	fn provide_secret(&mut self, idx: u64, secret: [u8; 32]) -> Result<(), &'static str> {
		if let Err(()) = self.commitment_secrets.provide_secret(idx, secret) {
			return Err("Previous secret did not match new one");
		}

		// Prune HTLCs from the previous counterparty commitment tx so we don't generate failure/fulfill
		// events for now-revoked/fulfilled HTLCs.
		if let Some(txid) = self.prev_counterparty_commitment_txid.take() {
			for &mut (_, ref mut source) in self.counterparty_claimable_outpoints.get_mut(&txid).unwrap() {
				*source = None;
			}
		}

		if !self.payment_preimages.is_empty() {
			let cur_holder_signed_commitment_tx = &self.current_holder_commitment_tx;
			let prev_holder_signed_commitment_tx = self.prev_holder_signed_commitment_tx.as_ref();
			let min_idx = self.get_min_seen_secret();
			let counterparty_hash_commitment_number = &mut self.counterparty_hash_commitment_number;

			self.payment_preimages.retain(|&k, _| {
				for &(ref htlc, _, _) in cur_holder_signed_commitment_tx.htlc_outputs.iter() {
					if k == htlc.payment_hash {
						return true
					}
				}
				if let Some(prev_holder_commitment_tx) = prev_holder_signed_commitment_tx {
					for &(ref htlc, _, _) in prev_holder_commitment_tx.htlc_outputs.iter() {
						if k == htlc.payment_hash {
							return true
						}
					}
				}
				let contains = if let Some(cn) = counterparty_hash_commitment_number.get(&k) {
					if *cn < min_idx {
						return true
					}
					true
				} else { false };
				if contains {
					counterparty_hash_commitment_number.remove(&k);
				}
				false
			});
		}

		Ok(())
	}

	pub(crate) fn provide_latest_counterparty_commitment_tx<L: Deref>(&mut self, txid: Txid, htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Box<HTLCSource>>)>, commitment_number: u64, their_revocation_point: PublicKey, logger: &L) where L::Target: Logger {
		// TODO: Encrypt the htlc_outputs data with the single-hash of the commitment transaction
		// so that a remote monitor doesn't learn anything unless there is a malicious close.
		// (only maybe, sadly we cant do the same for local info, as we need to be aware of
		// timeouts)
		for &(ref htlc, _) in &htlc_outputs {
			self.counterparty_hash_commitment_number.insert(htlc.payment_hash, commitment_number);
		}

		log_trace!(logger, "Tracking new counterparty commitment transaction with txid {} at commitment number {} with {} HTLC outputs", txid, commitment_number, htlc_outputs.len());
		self.prev_counterparty_commitment_txid = self.current_counterparty_commitment_txid.take();
		self.current_counterparty_commitment_txid = Some(txid);
		self.counterparty_claimable_outpoints.insert(txid, htlc_outputs.clone());
		self.current_counterparty_commitment_number = commitment_number;
		//TODO: Merge this into the other per-counterparty-transaction output storage stuff
		match self.their_cur_revocation_points {
			Some(old_points) => {
				if old_points.0 == commitment_number + 1 {
					self.their_cur_revocation_points = Some((old_points.0, old_points.1, Some(their_revocation_point)));
				} else if old_points.0 == commitment_number + 2 {
					if let Some(old_second_point) = old_points.2 {
						self.their_cur_revocation_points = Some((old_points.0 - 1, old_second_point, Some(their_revocation_point)));
					} else {
						self.their_cur_revocation_points = Some((commitment_number, their_revocation_point, None));
					}
				} else {
					self.their_cur_revocation_points = Some((commitment_number, their_revocation_point, None));
				}
			},
			None => {
				self.their_cur_revocation_points = Some((commitment_number, their_revocation_point, None));
			}
		}
		let mut htlcs = Vec::with_capacity(htlc_outputs.len());
		for htlc in htlc_outputs {
			if htlc.0.transaction_output_index.is_some() {
				htlcs.push(htlc.0);
			}
		}
	}

	/// Informs this monitor of the latest holder (ie broadcastable) commitment transaction. The
	/// monitor watches for timeouts and may broadcast it if we approach such a timeout. Thus, it
	/// is important that any clones of this channel monitor (including remote clones) by kept
	/// up-to-date as our holder commitment transaction is updated.
	/// Panics if set_on_holder_tx_csv has never been called.
	fn provide_latest_holder_commitment_tx(&mut self, holder_commitment_tx: HolderCommitmentTransaction, htlc_outputs: Vec<(HTLCOutputInCommitment, Option<Signature>, Option<HTLCSource>)>) -> Result<(), &'static str> {
		// block for Rust 1.34 compat
		let mut new_holder_commitment_tx = {
			let trusted_tx = holder_commitment_tx.trust();
			let txid = trusted_tx.txid();
			let tx_keys = trusted_tx.keys();
			self.current_holder_commitment_number = trusted_tx.commitment_number();
			HolderSignedTx {
				txid,
				revocation_key: tx_keys.revocation_key,
				a_htlc_key: tx_keys.broadcaster_htlc_key,
				b_htlc_key: tx_keys.countersignatory_htlc_key,
				delayed_payment_key: tx_keys.broadcaster_delayed_payment_key,
				per_commitment_point: tx_keys.per_commitment_point,
				htlc_outputs,
				to_self_value_sat: holder_commitment_tx.to_broadcaster_value_sat(),
				feerate_per_kw: trusted_tx.feerate_per_kw(),
			}
		};
		self.onchain_tx_handler.provide_latest_holder_tx(holder_commitment_tx);
		mem::swap(&mut new_holder_commitment_tx, &mut self.current_holder_commitment_tx);
		self.prev_holder_signed_commitment_tx = Some(new_holder_commitment_tx);
		if self.holder_tx_signed {
			return Err("Latest holder commitment signed has already been signed, update is rejected");
		}
		Ok(())
	}

	/// Provides a payment_hash->payment_preimage mapping. Will be automatically pruned when all
	/// commitment_tx_infos which contain the payment hash have been revoked.
	fn provide_payment_preimage<B: Deref, F: Deref, L: Deref>(&mut self, payment_hash: &PaymentHash, payment_preimage: &PaymentPreimage, broadcaster: &B, fee_estimator: &F, logger: &L)
	where B::Target: BroadcasterInterface,
		    F::Target: FeeEstimator,
		    L::Target: Logger,
	{
		self.payment_preimages.insert(payment_hash.clone(), payment_preimage.clone());

		// If the channel is force closed, try to claim the output from this preimage.
		// First check if a counterparty commitment transaction has been broadcasted:
		macro_rules! claim_htlcs {
			($commitment_number: expr, $txid: expr) => {
				let htlc_claim_reqs = self.get_counterparty_htlc_output_claim_reqs($commitment_number, $txid, None);
				self.onchain_tx_handler.update_claims_view(&Vec::new(), htlc_claim_reqs, self.best_block.height(), self.best_block.height(), broadcaster, fee_estimator, logger);
			}
		}
		if let Some(txid) = self.current_counterparty_commitment_txid {
			if let Some(commitment_number) = self.counterparty_commitment_txn_on_chain.get(&txid) {
				claim_htlcs!(*commitment_number, txid);
				return;
			}
		}
		if let Some(txid) = self.prev_counterparty_commitment_txid {
			if let Some(commitment_number) = self.counterparty_commitment_txn_on_chain.get(&txid) {
				claim_htlcs!(*commitment_number, txid);
				return;
			}
		}

		// Then if a holder commitment transaction has been seen on-chain, broadcast transactions
		// claiming the HTLC output from each of the holder commitment transactions.
		// Note that we can't just use `self.holder_tx_signed`, because that only covers the case where
		// *we* sign a holder commitment transaction, not when e.g. a watchtower broadcasts one of our
		// holder commitment transactions.
		if self.broadcasted_holder_revokable_script.is_some() {
			// Assume that the broadcasted commitment transaction confirmed in the current best
			// block. Even if not, its a reasonable metric for the bump criteria on the HTLC
			// transactions.
			let (claim_reqs, _) = self.get_broadcasted_holder_claims(&self.current_holder_commitment_tx, self.best_block.height());
			self.onchain_tx_handler.update_claims_view(&Vec::new(), claim_reqs, self.best_block.height(), self.best_block.height(), broadcaster, fee_estimator, logger);
			if let Some(ref tx) = self.prev_holder_signed_commitment_tx {
				let (claim_reqs, _) = self.get_broadcasted_holder_claims(&tx, self.best_block.height());
				self.onchain_tx_handler.update_claims_view(&Vec::new(), claim_reqs, self.best_block.height(), self.best_block.height(), broadcaster, fee_estimator, logger);
			}
		}
	}

	pub(crate) fn broadcast_latest_holder_commitment_txn<B: Deref, L: Deref>(&mut self, broadcaster: &B, logger: &L)
		where B::Target: BroadcasterInterface,
					L::Target: Logger,
	{
		for tx in self.get_latest_holder_commitment_txn(logger).iter() {
			log_info!(logger, "Broadcasting local {}", log_tx!(tx));
			broadcaster.broadcast_transaction(tx);
		}
		self.pending_monitor_events.push(MonitorEvent::CommitmentTxConfirmed(self.funding_info.0));
	}

	pub fn update_monitor<B: Deref, F: Deref, L: Deref>(&mut self, updates: &ChannelMonitorUpdate, broadcaster: &B, fee_estimator: &F, logger: &L) -> Result<(), ()>
	where B::Target: BroadcasterInterface,
		    F::Target: FeeEstimator,
		    L::Target: Logger,
	{
		log_info!(logger, "Applying update to monitor {}, bringing update_id from {} to {} with {} changes.",
			log_funding_info!(self), self.latest_update_id, updates.update_id, updates.updates.len());
		// ChannelMonitor updates may be applied after force close if we receive a
		// preimage for a broadcasted commitment transaction HTLC output that we'd
		// like to claim on-chain. If this is the case, we no longer have guaranteed
		// access to the monitor's update ID, so we use a sentinel value instead.
		if updates.update_id == CLOSED_CHANNEL_UPDATE_ID {
			assert_eq!(updates.updates.len(), 1);
			match updates.updates[0] {
				ChannelMonitorUpdateStep::PaymentPreimage { .. } => {},
				_ => {
					log_error!(logger, "Attempted to apply post-force-close ChannelMonitorUpdate of type {}", updates.updates[0].variant_name());
					panic!("Attempted to apply post-force-close ChannelMonitorUpdate that wasn't providing a payment preimage");
				},
			}
		} else if self.latest_update_id + 1 != updates.update_id {
			panic!("Attempted to apply ChannelMonitorUpdates out of order, check the update_id before passing an update to update_monitor!");
		}
		let mut ret = Ok(());
		for update in updates.updates.iter() {
			match update {
				ChannelMonitorUpdateStep::LatestHolderCommitmentTXInfo { commitment_tx, htlc_outputs } => {
					log_trace!(logger, "Updating ChannelMonitor with latest holder commitment transaction info");
					if self.lockdown_from_offchain { panic!(); }
					if let Err(e) = self.provide_latest_holder_commitment_tx(commitment_tx.clone(), htlc_outputs.clone()) {
						log_error!(logger, "Providing latest holder commitment transaction failed/was refused:");
						log_error!(logger, "    {}", e);
						ret = Err(());
					}
				}
				ChannelMonitorUpdateStep::LatestCounterpartyCommitmentTXInfo { commitment_txid, htlc_outputs, commitment_number, their_revocation_point } => {
					log_trace!(logger, "Updating ChannelMonitor with latest counterparty commitment transaction info");
					self.provide_latest_counterparty_commitment_tx(*commitment_txid, htlc_outputs.clone(), *commitment_number, *their_revocation_point, logger)
				},
				ChannelMonitorUpdateStep::PaymentPreimage { payment_preimage } => {
					log_trace!(logger, "Updating ChannelMonitor with payment preimage");
					self.provide_payment_preimage(&PaymentHash(Sha256::hash(&payment_preimage.0[..]).into_inner()), &payment_preimage, broadcaster, fee_estimator, logger)
				},
				ChannelMonitorUpdateStep::CommitmentSecret { idx, secret } => {
					log_trace!(logger, "Updating ChannelMonitor with commitment secret");
					if let Err(e) = self.provide_secret(*idx, *secret) {
						log_error!(logger, "Providing latest counterparty commitment secret failed/was refused:");
						log_error!(logger, "    {}", e);
						ret = Err(());
					}
				},
				ChannelMonitorUpdateStep::ChannelForceClosed { should_broadcast } => {
					log_trace!(logger, "Updating ChannelMonitor: channel force closed, should broadcast: {}", should_broadcast);
					self.lockdown_from_offchain = true;
					if *should_broadcast {
						self.broadcast_latest_holder_commitment_txn(broadcaster, logger);
					} else if !self.holder_tx_signed {
						log_error!(logger, "You have a toxic holder commitment transaction avaible in channel monitor, read comment in ChannelMonitor::get_latest_holder_commitment_txn to be informed of manual action to take");
					} else {
						// If we generated a MonitorEvent::CommitmentTxConfirmed, the ChannelManager
						// will still give us a ChannelForceClosed event with !should_broadcast, but we
						// shouldn't print the scary warning above.
						log_info!(logger, "Channel off-chain state closed after we broadcasted our latest commitment transaction.");
					}
				},
				ChannelMonitorUpdateStep::ShutdownScript { scriptpubkey } => {
					log_trace!(logger, "Updating ChannelMonitor with shutdown script");
					if let Some(shutdown_script) = self.shutdown_script.replace(scriptpubkey.clone()) {
						panic!("Attempted to replace shutdown script {} with {}", shutdown_script, scriptpubkey);
					}
				},
			}
		}
		self.latest_update_id = updates.update_id;

		if ret.is_ok() && self.funding_spend_seen {
			log_error!(logger, "Refusing Channel Monitor Update as counterparty attempted to update commitment after funding was spent");
			Err(())
		} else { ret }
	}

	pub fn get_latest_update_id(&self) -> u64 {
		self.latest_update_id
	}

	pub fn get_funding_txo(&self) -> &(OutPoint, Script) {
		&self.funding_info
	}

	pub fn get_outputs_to_watch(&self) -> &HashMap<Txid, Vec<(u32, Script)>> {
		// If we've detected a counterparty commitment tx on chain, we must include it in the set
		// of outputs to watch for spends of, otherwise we're likely to lose user funds. Because
		// its trivial to do, double-check that here.
		for (txid, _) in self.counterparty_commitment_txn_on_chain.iter() {
			self.outputs_to_watch.get(txid).expect("Counterparty commitment txn which have been broadcast should have outputs registered");
		}
		&self.outputs_to_watch
	}

	pub fn get_and_clear_pending_monitor_events(&mut self) -> Vec<MonitorEvent> {
		let mut ret = Vec::new();
		mem::swap(&mut ret, &mut self.pending_monitor_events);
		ret
	}

	pub fn get_and_clear_pending_events(&mut self) -> Vec<Event> {
		let mut ret = Vec::new();
		mem::swap(&mut ret, &mut self.pending_events);
		ret
	}

	/// Can only fail if idx is < get_min_seen_secret
	fn get_secret(&self, idx: u64) -> Option<[u8; 32]> {
		self.commitment_secrets.get_secret(idx)
	}

	pub(crate) fn get_min_seen_secret(&self) -> u64 {
		self.commitment_secrets.get_min_seen_secret()
	}

	pub(crate) fn get_cur_counterparty_commitment_number(&self) -> u64 {
		self.current_counterparty_commitment_number
	}

	pub(crate) fn get_cur_holder_commitment_number(&self) -> u64 {
		self.current_holder_commitment_number
	}

	/// Attempts to claim a counterparty commitment transaction's outputs using the revocation key and
	/// data in counterparty_claimable_outpoints. Will directly claim any HTLC outputs which expire at a
	/// height > height + CLTV_SHARED_CLAIM_BUFFER. In any case, will install monitoring for
	/// HTLC-Success/HTLC-Timeout transactions.
	/// Return updates for HTLC pending in the channel and failed automatically by the broadcast of
	/// revoked counterparty commitment tx
	fn check_spend_counterparty_transaction<L: Deref>(&mut self, tx: &Transaction, height: u32, logger: &L) -> (Vec<PackageTemplate>, TransactionOutputs) where L::Target: Logger {
		// Most secp and related errors trying to create keys means we have no hope of constructing
		// a spend transaction...so we return no transactions to broadcast
		let mut claimable_outpoints = Vec::new();
		let mut watch_outputs = Vec::new();

		let commitment_txid = tx.txid(); //TODO: This is gonna be a performance bottleneck for watchtowers!
		let per_commitment_option = self.counterparty_claimable_outpoints.get(&commitment_txid);

		macro_rules! ignore_error {
			( $thing : expr ) => {
				match $thing {
					Ok(a) => a,
					Err(_) => return (claimable_outpoints, (commitment_txid, watch_outputs))
				}
			};
		}

		let commitment_number = 0xffffffffffff - ((((tx.input[0].sequence as u64 & 0xffffff) << 3*8) | (tx.lock_time as u64 & 0xffffff)) ^ self.commitment_transaction_number_obscure_factor);
		if commitment_number >= self.get_min_seen_secret() {
			let secret = self.get_secret(commitment_number).unwrap();
			let per_commitment_key = ignore_error!(SecretKey::from_slice(&secret));
			let per_commitment_point = PublicKey::from_secret_key(&self.secp_ctx, &per_commitment_key);
			let revocation_pubkey = ignore_error!(chan_utils::derive_public_revocation_key(&self.secp_ctx, &per_commitment_point, &self.holder_revocation_basepoint));
			let delayed_key = ignore_error!(chan_utils::derive_public_key(&self.secp_ctx, &PublicKey::from_secret_key(&self.secp_ctx, &per_commitment_key), &self.counterparty_commitment_params.counterparty_delayed_payment_base_key));

			let revokeable_redeemscript = chan_utils::get_revokeable_redeemscript(&revocation_pubkey, self.counterparty_commitment_params.on_counterparty_tx_csv, &delayed_key);
			let revokeable_p2wsh = revokeable_redeemscript.to_v0_p2wsh();

			// First, process non-htlc outputs (to_holder & to_counterparty)
			for (idx, outp) in tx.output.iter().enumerate() {
				if outp.script_pubkey == revokeable_p2wsh {
					let revk_outp = RevokedOutput::build(per_commitment_point, self.counterparty_commitment_params.counterparty_delayed_payment_base_key, self.counterparty_commitment_params.counterparty_htlc_base_key, per_commitment_key, outp.value, self.counterparty_commitment_params.on_counterparty_tx_csv);
					let justice_package = PackageTemplate::build_package(commitment_txid, idx as u32, PackageSolvingData::RevokedOutput(revk_outp), height + self.counterparty_commitment_params.on_counterparty_tx_csv as u32, true, height);
					claimable_outpoints.push(justice_package);
				}
			}

			// Then, try to find revoked htlc outputs
			if let Some(ref per_commitment_data) = per_commitment_option {
				for (_, &(ref htlc, _)) in per_commitment_data.iter().enumerate() {
					if let Some(transaction_output_index) = htlc.transaction_output_index {
						if transaction_output_index as usize >= tx.output.len() ||
								tx.output[transaction_output_index as usize].value != htlc.amount_msat / 1000 {
							return (claimable_outpoints, (commitment_txid, watch_outputs)); // Corrupted per_commitment_data, fuck this user
						}
						let revk_htlc_outp = RevokedHTLCOutput::build(per_commitment_point, self.counterparty_commitment_params.counterparty_delayed_payment_base_key, self.counterparty_commitment_params.counterparty_htlc_base_key, per_commitment_key, htlc.amount_msat / 1000, htlc.clone(), self.onchain_tx_handler.channel_transaction_parameters.opt_anchors.is_some());
						let justice_package = PackageTemplate::build_package(commitment_txid, transaction_output_index, PackageSolvingData::RevokedHTLCOutput(revk_htlc_outp), htlc.cltv_expiry, true, height);
						claimable_outpoints.push(justice_package);
					}
				}
			}

			// Last, track onchain revoked commitment transaction and fail backward outgoing HTLCs as payment path is broken
			if !claimable_outpoints.is_empty() || per_commitment_option.is_some() { // ie we're confident this is actually ours
				// We're definitely a counterparty commitment transaction!
				log_error!(logger, "Got broadcast of revoked counterparty commitment transaction, going to generate general spend tx with {} inputs", claimable_outpoints.len());
				for (idx, outp) in tx.output.iter().enumerate() {
					watch_outputs.push((idx as u32, outp.clone()));
				}
				self.counterparty_commitment_txn_on_chain.insert(commitment_txid, commitment_number);

				fail_unbroadcast_htlcs!(self, "revoked counterparty", height, [].iter().map(|a| *a), logger);
			}
		} else if let Some(per_commitment_data) = per_commitment_option {
			// While this isn't useful yet, there is a potential race where if a counterparty
			// revokes a state at the same time as the commitment transaction for that state is
			// confirmed, and the watchtower receives the block before the user, the user could
			// upload a new ChannelMonitor with the revocation secret but the watchtower has
			// already processed the block, resulting in the counterparty_commitment_txn_on_chain entry
			// not being generated by the above conditional. Thus, to be safe, we go ahead and
			// insert it here.
			for (idx, outp) in tx.output.iter().enumerate() {
				watch_outputs.push((idx as u32, outp.clone()));
			}
			self.counterparty_commitment_txn_on_chain.insert(commitment_txid, commitment_number);

			log_info!(logger, "Got broadcast of non-revoked counterparty commitment transaction {}", commitment_txid);
			fail_unbroadcast_htlcs!(self, "counterparty", height, per_commitment_data.iter().map(|(a, b)| (a, b.as_ref().map(|b| b.as_ref()))), logger);

			let htlc_claim_reqs = self.get_counterparty_htlc_output_claim_reqs(commitment_number, commitment_txid, Some(tx));
			for req in htlc_claim_reqs {
				claimable_outpoints.push(req);
			}

		}
		(claimable_outpoints, (commitment_txid, watch_outputs))
	}

	fn get_counterparty_htlc_output_claim_reqs(&self, commitment_number: u64, commitment_txid: Txid, tx: Option<&Transaction>) -> Vec<PackageTemplate> {
		let mut claimable_outpoints = Vec::new();
		if let Some(htlc_outputs) = self.counterparty_claimable_outpoints.get(&commitment_txid) {
			if let Some(revocation_points) = self.their_cur_revocation_points {
				let revocation_point_option =
					// If the counterparty commitment tx is the latest valid state, use their latest
					// per-commitment point
					if revocation_points.0 == commitment_number { Some(&revocation_points.1) }
					else if let Some(point) = revocation_points.2.as_ref() {
						// If counterparty commitment tx is the state previous to the latest valid state, use
						// their previous per-commitment point (non-atomicity of revocation means it's valid for
						// them to temporarily have two valid commitment txns from our viewpoint)
						if revocation_points.0 == commitment_number + 1 { Some(point) } else { None }
					} else { None };
				if let Some(revocation_point) = revocation_point_option {
					for (_, &(ref htlc, _)) in htlc_outputs.iter().enumerate() {
						if let Some(transaction_output_index) = htlc.transaction_output_index {
							if let Some(transaction) = tx {
								if transaction_output_index as usize >= transaction.output.len() ||
									transaction.output[transaction_output_index as usize].value != htlc.amount_msat / 1000 {
										return claimable_outpoints; // Corrupted per_commitment_data, fuck this user
									}
							}
							let preimage = if htlc.offered { if let Some(p) = self.payment_preimages.get(&htlc.payment_hash) { Some(*p) } else { None } } else { None };
							if preimage.is_some() || !htlc.offered {
								let counterparty_htlc_outp = if htlc.offered { PackageSolvingData::CounterpartyOfferedHTLCOutput(CounterpartyOfferedHTLCOutput::build(*revocation_point, self.counterparty_commitment_params.counterparty_delayed_payment_base_key, self.counterparty_commitment_params.counterparty_htlc_base_key, preimage.unwrap(), htlc.clone())) } else { PackageSolvingData::CounterpartyReceivedHTLCOutput(CounterpartyReceivedHTLCOutput::build(*revocation_point, self.counterparty_commitment_params.counterparty_delayed_payment_base_key, self.counterparty_commitment_params.counterparty_htlc_base_key, htlc.clone())) };
								let aggregation = if !htlc.offered { false } else { true };
								let counterparty_package = PackageTemplate::build_package(commitment_txid, transaction_output_index, counterparty_htlc_outp, htlc.cltv_expiry,aggregation, 0);
								claimable_outpoints.push(counterparty_package);
							}
						}
					}
				}
			}
		}
		claimable_outpoints
	}

	/// Attempts to claim a counterparty HTLC-Success/HTLC-Timeout's outputs using the revocation key
	fn check_spend_counterparty_htlc<L: Deref>(&mut self, tx: &Transaction, commitment_number: u64, height: u32, logger: &L) -> (Vec<PackageTemplate>, Option<TransactionOutputs>) where L::Target: Logger {
		let htlc_txid = tx.txid();
		if tx.input.len() != 1 || tx.output.len() != 1 || tx.input[0].witness.len() != 5 {
			return (Vec::new(), None)
		}

		macro_rules! ignore_error {
			( $thing : expr ) => {
				match $thing {
					Ok(a) => a,
					Err(_) => return (Vec::new(), None)
				}
			};
		}

		let secret = if let Some(secret) = self.get_secret(commitment_number) { secret } else { return (Vec::new(), None); };
		let per_commitment_key = ignore_error!(SecretKey::from_slice(&secret));
		let per_commitment_point = PublicKey::from_secret_key(&self.secp_ctx, &per_commitment_key);

		log_error!(logger, "Got broadcast of revoked counterparty HTLC transaction, spending {}:{}", htlc_txid, 0);
		let revk_outp = RevokedOutput::build(per_commitment_point, self.counterparty_commitment_params.counterparty_delayed_payment_base_key, self.counterparty_commitment_params.counterparty_htlc_base_key, per_commitment_key, tx.output[0].value, self.counterparty_commitment_params.on_counterparty_tx_csv);
		let justice_package = PackageTemplate::build_package(htlc_txid, 0, PackageSolvingData::RevokedOutput(revk_outp), height + self.counterparty_commitment_params.on_counterparty_tx_csv as u32, true, height);
		let claimable_outpoints = vec!(justice_package);
		let outputs = vec![(0, tx.output[0].clone())];
		(claimable_outpoints, Some((htlc_txid, outputs)))
	}

	// Returns (1) `PackageTemplate`s that can be given to the OnChainTxHandler, so that the handler can
	// broadcast transactions claiming holder HTLC commitment outputs and (2) a holder revokable
	// script so we can detect whether a holder transaction has been seen on-chain.
	fn get_broadcasted_holder_claims(&self, holder_tx: &HolderSignedTx, conf_height: u32) -> (Vec<PackageTemplate>, Option<(Script, PublicKey, PublicKey)>) {
		let mut claim_requests = Vec::with_capacity(holder_tx.htlc_outputs.len());

		let redeemscript = chan_utils::get_revokeable_redeemscript(&holder_tx.revocation_key, self.on_holder_tx_csv, &holder_tx.delayed_payment_key);
		let broadcasted_holder_revokable_script = Some((redeemscript.to_v0_p2wsh(), holder_tx.per_commitment_point.clone(), holder_tx.revocation_key.clone()));

		for &(ref htlc, _, _) in holder_tx.htlc_outputs.iter() {
			if let Some(transaction_output_index) = htlc.transaction_output_index {
				let htlc_output = if htlc.offered {
						HolderHTLCOutput::build_offered(htlc.amount_msat, htlc.cltv_expiry)
					} else {
						let payment_preimage = if let Some(preimage) = self.payment_preimages.get(&htlc.payment_hash) {
							preimage.clone()
						} else {
							// We can't build an HTLC-Success transaction without the preimage
							continue;
						};
						HolderHTLCOutput::build_accepted(payment_preimage, htlc.amount_msat)
					};
				let htlc_package = PackageTemplate::build_package(holder_tx.txid, transaction_output_index, PackageSolvingData::HolderHTLCOutput(htlc_output), htlc.cltv_expiry, false, conf_height);
				claim_requests.push(htlc_package);
			}
		}

		(claim_requests, broadcasted_holder_revokable_script)
	}

	// Returns holder HTLC outputs to watch and react to in case of spending.
	fn get_broadcasted_holder_watch_outputs(&self, holder_tx: &HolderSignedTx, commitment_tx: &Transaction) -> Vec<(u32, TxOut)> {
		let mut watch_outputs = Vec::with_capacity(holder_tx.htlc_outputs.len());
		for &(ref htlc, _, _) in holder_tx.htlc_outputs.iter() {
			if let Some(transaction_output_index) = htlc.transaction_output_index {
				watch_outputs.push((transaction_output_index, commitment_tx.output[transaction_output_index as usize].clone()));
			}
		}
		watch_outputs
	}

	/// Attempts to claim any claimable HTLCs in a commitment transaction which was not (yet)
	/// revoked using data in holder_claimable_outpoints.
	/// Should not be used if check_spend_revoked_transaction succeeds.
	/// Returns None unless the transaction is definitely one of our commitment transactions.
	fn check_spend_holder_transaction<L: Deref>(&mut self, tx: &Transaction, height: u32, logger: &L) -> Option<(Vec<PackageTemplate>, TransactionOutputs)> where L::Target: Logger {
		let commitment_txid = tx.txid();
		let mut claim_requests = Vec::new();
		let mut watch_outputs = Vec::new();

		macro_rules! append_onchain_update {
			($updates: expr, $to_watch: expr) => {
				claim_requests = $updates.0;
				self.broadcasted_holder_revokable_script = $updates.1;
				watch_outputs.append(&mut $to_watch);
			}
		}

		// HTLCs set may differ between last and previous holder commitment txn, in case of one them hitting chain, ensure we cancel all HTLCs backward
		let mut is_holder_tx = false;

		if self.current_holder_commitment_tx.txid == commitment_txid {
			is_holder_tx = true;
			log_info!(logger, "Got broadcast of latest holder commitment tx {}, searching for available HTLCs to claim", commitment_txid);
			let res = self.get_broadcasted_holder_claims(&self.current_holder_commitment_tx, height);
			let mut to_watch = self.get_broadcasted_holder_watch_outputs(&self.current_holder_commitment_tx, tx);
			append_onchain_update!(res, to_watch);
			fail_unbroadcast_htlcs!(self, "latest holder", height, self.current_holder_commitment_tx.htlc_outputs.iter().map(|(a, _, c)| (a, c.as_ref())), logger);
		} else if let &Some(ref holder_tx) = &self.prev_holder_signed_commitment_tx {
			if holder_tx.txid == commitment_txid {
				is_holder_tx = true;
				log_info!(logger, "Got broadcast of previous holder commitment tx {}, searching for available HTLCs to claim", commitment_txid);
				let res = self.get_broadcasted_holder_claims(holder_tx, height);
				let mut to_watch = self.get_broadcasted_holder_watch_outputs(holder_tx, tx);
				append_onchain_update!(res, to_watch);
				fail_unbroadcast_htlcs!(self, "previous holder", height, holder_tx.htlc_outputs.iter().map(|(a, _, c)| (a, c.as_ref())), logger);
			}
		}

		if is_holder_tx {
			Some((claim_requests, (commitment_txid, watch_outputs)))
		} else {
			None
		}
	}

	pub fn get_latest_holder_commitment_txn<L: Deref>(&mut self, logger: &L) -> Vec<Transaction> where L::Target: Logger {
		log_debug!(logger, "Getting signed latest holder commitment transaction!");
		self.holder_tx_signed = true;
		let commitment_tx = self.onchain_tx_handler.get_fully_signed_holder_tx(&self.funding_redeemscript);
		let txid = commitment_tx.txid();
		let mut holder_transactions = vec![commitment_tx];
		for htlc in self.current_holder_commitment_tx.htlc_outputs.iter() {
			if let Some(vout) = htlc.0.transaction_output_index {
				let preimage = if !htlc.0.offered {
					if let Some(preimage) = self.payment_preimages.get(&htlc.0.payment_hash) { Some(preimage.clone()) } else {
						// We can't build an HTLC-Success transaction without the preimage
						continue;
					}
				} else if htlc.0.cltv_expiry > self.best_block.height() + 1 {
					// Don't broadcast HTLC-Timeout transactions immediately as they don't meet the
					// current locktime requirements on-chain. We will broadcast them in
					// `block_confirmed` when `should_broadcast_holder_commitment_txn` returns true.
					// Note that we add + 1 as transactions are broadcastable when they can be
					// confirmed in the next block.
					continue;
				} else { None };
				if let Some(htlc_tx) = self.onchain_tx_handler.get_fully_signed_htlc_tx(
					&::bitcoin::OutPoint { txid, vout }, &preimage) {
					holder_transactions.push(htlc_tx);
				}
			}
		}
		// We throw away the generated waiting_first_conf data as we aren't (yet) confirmed and we don't actually know what the caller wants to do.
		// The data will be re-generated and tracked in check_spend_holder_transaction if we get a confirmation.
		holder_transactions
	}

	#[cfg(any(test,feature = "unsafe_revoked_tx_signing"))]
	/// Note that this includes possibly-locktimed-in-the-future transactions!
	fn unsafe_get_latest_holder_commitment_txn<L: Deref>(&mut self, logger: &L) -> Vec<Transaction> where L::Target: Logger {
		log_debug!(logger, "Getting signed copy of latest holder commitment transaction!");
		let commitment_tx = self.onchain_tx_handler.get_fully_signed_copy_holder_tx(&self.funding_redeemscript);
		let txid = commitment_tx.txid();
		let mut holder_transactions = vec![commitment_tx];
		for htlc in self.current_holder_commitment_tx.htlc_outputs.iter() {
			if let Some(vout) = htlc.0.transaction_output_index {
				let preimage = if !htlc.0.offered {
					if let Some(preimage) = self.payment_preimages.get(&htlc.0.payment_hash) { Some(preimage.clone()) } else {
						// We can't build an HTLC-Success transaction without the preimage
						continue;
					}
				} else { None };
				if let Some(htlc_tx) = self.onchain_tx_handler.unsafe_get_fully_signed_htlc_tx(
					&::bitcoin::OutPoint { txid, vout }, &preimage) {
					holder_transactions.push(htlc_tx);
				}
			}
		}
		holder_transactions
	}

	pub fn block_connected<B: Deref, F: Deref, L: Deref>(&mut self, header: &BlockHeader, txdata: &TransactionData, height: u32, broadcaster: B, fee_estimator: F, logger: L) -> Vec<TransactionOutputs>
		where B::Target: BroadcasterInterface,
		      F::Target: FeeEstimator,
					L::Target: Logger,
	{
		let block_hash = header.block_hash();
		self.best_block = BestBlock::new(block_hash, height);

		self.transactions_confirmed(header, txdata, height, broadcaster, fee_estimator, logger)
	}

	fn best_block_updated<B: Deref, F: Deref, L: Deref>(
		&mut self,
		header: &BlockHeader,
		height: u32,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) -> Vec<TransactionOutputs>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		let block_hash = header.block_hash();

		if height > self.best_block.height() {
			self.best_block = BestBlock::new(block_hash, height);
			self.block_confirmed(height, vec![], vec![], vec![], &broadcaster, &fee_estimator, &logger)
		} else if block_hash != self.best_block.block_hash() {
			self.best_block = BestBlock::new(block_hash, height);
			self.onchain_events_awaiting_threshold_conf.retain(|ref entry| entry.height <= height);
			self.onchain_tx_handler.block_disconnected(height + 1, broadcaster, fee_estimator, logger);
			Vec::new()
		} else { Vec::new() }
	}

	fn transactions_confirmed<B: Deref, F: Deref, L: Deref>(
		&mut self,
		header: &BlockHeader,
		txdata: &TransactionData,
		height: u32,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) -> Vec<TransactionOutputs>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		let txn_matched = self.filter_block(txdata);
		for tx in &txn_matched {
			let mut output_val = 0;
			for out in tx.output.iter() {
				if out.value > 21_000_000_0000_0000 { panic!("Value-overflowing transaction provided to block connected"); }
				output_val += out.value;
				if output_val > 21_000_000_0000_0000 { panic!("Value-overflowing transaction provided to block connected"); }
			}
		}

		let block_hash = header.block_hash();

		let mut watch_outputs = Vec::new();
		let mut claimable_outpoints = Vec::new();
		for tx in &txn_matched {
			if tx.input.len() == 1 {
				// Assuming our keys were not leaked (in which case we're screwed no matter what),
				// commitment transactions and HTLC transactions will all only ever have one input,
				// which is an easy way to filter out any potential non-matching txn for lazy
				// filters.
				let prevout = &tx.input[0].previous_output;
				if prevout.txid == self.funding_info.0.txid && prevout.vout == self.funding_info.0.index as u32 {
					let mut balance_spendable_csv = None;
					log_info!(logger, "Channel {} closed by funding output spend in txid {}.",
						log_bytes!(self.funding_info.0.to_channel_id()), tx.txid());
					self.funding_spend_seen = true;
					if (tx.input[0].sequence >> 8*3) as u8 == 0x80 && (tx.lock_time >> 8*3) as u8 == 0x20 {
						let (mut new_outpoints, new_outputs) = self.check_spend_counterparty_transaction(&tx, height, &logger);
						if !new_outputs.1.is_empty() {
							watch_outputs.push(new_outputs);
						}
						claimable_outpoints.append(&mut new_outpoints);
						if new_outpoints.is_empty() {
							if let Some((mut new_outpoints, new_outputs)) = self.check_spend_holder_transaction(&tx, height, &logger) {
								if !new_outputs.1.is_empty() {
									watch_outputs.push(new_outputs);
								}
								claimable_outpoints.append(&mut new_outpoints);
								balance_spendable_csv = Some(self.on_holder_tx_csv);
							}
						}
					}
					let txid = tx.txid();
					self.onchain_events_awaiting_threshold_conf.push(OnchainEventEntry {
						txid,
						height: height,
						event: OnchainEvent::FundingSpendConfirmation {
							on_local_output_csv: balance_spendable_csv,
						},
					});
				} else {
					if let Some(&commitment_number) = self.counterparty_commitment_txn_on_chain.get(&prevout.txid) {
						let (mut new_outpoints, new_outputs_option) = self.check_spend_counterparty_htlc(&tx, commitment_number, height, &logger);
						claimable_outpoints.append(&mut new_outpoints);
						if let Some(new_outputs) = new_outputs_option {
							watch_outputs.push(new_outputs);
						}
					}
				}
			}
			// While all commitment/HTLC-Success/HTLC-Timeout transactions have one input, HTLCs
			// can also be resolved in a few other ways which can have more than one output. Thus,
			// we call is_resolving_htlc_output here outside of the tx.input.len() == 1 check.
			self.is_resolving_htlc_output(&tx, height, &logger);

			self.is_paying_spendable_output(&tx, height, &logger);
		}

		if height > self.best_block.height() {
			self.best_block = BestBlock::new(block_hash, height);
		}

		self.block_confirmed(height, txn_matched, watch_outputs, claimable_outpoints, &broadcaster, &fee_estimator, &logger)
	}

	/// Update state for new block(s)/transaction(s) confirmed. Note that the caller must update
	/// `self.best_block` before calling if a new best blockchain tip is available. More
	/// concretely, `self.best_block` must never be at a lower height than `conf_height`, avoiding
	/// complexity especially in `OnchainTx::update_claims_view`.
	///
	/// `conf_height` should be set to the height at which any new transaction(s)/block(s) were
	/// confirmed at, even if it is not the current best height.
	fn block_confirmed<B: Deref, F: Deref, L: Deref>(
		&mut self,
		conf_height: u32,
		txn_matched: Vec<&Transaction>,
		mut watch_outputs: Vec<TransactionOutputs>,
		mut claimable_outpoints: Vec<PackageTemplate>,
		broadcaster: &B,
		fee_estimator: &F,
		logger: &L,
	) -> Vec<TransactionOutputs>
	where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		log_trace!(logger, "Processing {} matched transactions for block at height {}.", txn_matched.len(), conf_height);
		debug_assert!(self.best_block.height() >= conf_height);

		let should_broadcast = self.should_broadcast_holder_commitment_txn(logger);
		if should_broadcast {
			let funding_outp = HolderFundingOutput::build(self.funding_redeemscript.clone());
			let commitment_package = PackageTemplate::build_package(self.funding_info.0.txid.clone(), self.funding_info.0.index as u32, PackageSolvingData::HolderFundingOutput(funding_outp), self.best_block.height(), false, self.best_block.height());
			claimable_outpoints.push(commitment_package);
			self.pending_monitor_events.push(MonitorEvent::CommitmentTxConfirmed(self.funding_info.0));
			let commitment_tx = self.onchain_tx_handler.get_fully_signed_holder_tx(&self.funding_redeemscript);
			self.holder_tx_signed = true;
			// Because we're broadcasting a commitment transaction, we should construct the package
			// assuming it gets confirmed in the next block. Sadly, we have code which considers
			// "not yet confirmed" things as discardable, so we cannot do that here.
			let (mut new_outpoints, _) = self.get_broadcasted_holder_claims(&self.current_holder_commitment_tx, self.best_block.height());
			let new_outputs = self.get_broadcasted_holder_watch_outputs(&self.current_holder_commitment_tx, &commitment_tx);
			if !new_outputs.is_empty() {
				watch_outputs.push((self.current_holder_commitment_tx.txid.clone(), new_outputs));
			}
			claimable_outpoints.append(&mut new_outpoints);
		}

		// Find which on-chain events have reached their confirmation threshold.
		let onchain_events_awaiting_threshold_conf =
			self.onchain_events_awaiting_threshold_conf.drain(..).collect::<Vec<_>>();
		let mut onchain_events_reaching_threshold_conf = Vec::new();
		for entry in onchain_events_awaiting_threshold_conf {
			if entry.has_reached_confirmation_threshold(&self.best_block) {
				onchain_events_reaching_threshold_conf.push(entry);
			} else {
				self.onchain_events_awaiting_threshold_conf.push(entry);
			}
		}

		// Used to check for duplicate HTLC resolutions.
		#[cfg(debug_assertions)]
		let unmatured_htlcs: Vec<_> = self.onchain_events_awaiting_threshold_conf
			.iter()
			.filter_map(|entry| match &entry.event {
				OnchainEvent::HTLCUpdate { source, .. } => Some(source),
				_ => None,
			})
			.collect();
		#[cfg(debug_assertions)]
		let mut matured_htlcs = Vec::new();

		// Produce actionable events from on-chain events having reached their threshold.
		for entry in onchain_events_reaching_threshold_conf.drain(..) {
			match entry.event {
				OnchainEvent::HTLCUpdate { ref source, payment_hash, onchain_value_satoshis, input_idx } => {
					// Check for duplicate HTLC resolutions.
					#[cfg(debug_assertions)]
					{
						debug_assert!(
							unmatured_htlcs.iter().find(|&htlc| htlc == &source).is_none(),
							"An unmature HTLC transaction conflicts with a maturing one; failed to \
							 call either transaction_unconfirmed for the conflicting transaction \
							 or block_disconnected for a block containing it.");
						debug_assert!(
							matured_htlcs.iter().find(|&htlc| htlc == source).is_none(),
							"A matured HTLC transaction conflicts with a maturing one; failed to \
							 call either transaction_unconfirmed for the conflicting transaction \
							 or block_disconnected for a block containing it.");
						matured_htlcs.push(source.clone());
					}

					log_debug!(logger, "HTLC {} failure update has got enough confirmations to be passed upstream", log_bytes!(payment_hash.0));
					self.pending_monitor_events.push(MonitorEvent::HTLCEvent(HTLCUpdate {
						payment_hash,
						payment_preimage: None,
						source: source.clone(),
						onchain_value_satoshis,
					}));
					if let Some(idx) = input_idx {
						self.htlcs_resolved_on_chain.push(IrrevocablyResolvedHTLC { input_idx: idx, payment_preimage: None });
					}
				},
				OnchainEvent::MaturingOutput { descriptor } => {
					log_debug!(logger, "Descriptor {} has got enough confirmations to be passed upstream", log_spendable!(descriptor));
					self.pending_events.push(Event::SpendableOutputs {
						outputs: vec![descriptor]
					});
				},
				OnchainEvent::HTLCSpendConfirmation { input_idx, preimage, .. } => {
					self.htlcs_resolved_on_chain.push(IrrevocablyResolvedHTLC { input_idx, payment_preimage: preimage });
				},
				OnchainEvent::FundingSpendConfirmation { .. } => {
					self.funding_spend_confirmed = Some(entry.txid);
				},
			}
		}

		self.onchain_tx_handler.update_claims_view(&txn_matched, claimable_outpoints, conf_height, self.best_block.height(), broadcaster, fee_estimator, logger);

		// Determine new outputs to watch by comparing against previously known outputs to watch,
		// updating the latter in the process.
		watch_outputs.retain(|&(ref txid, ref txouts)| {
			let idx_and_scripts = txouts.iter().map(|o| (o.0, o.1.script_pubkey.clone())).collect();
			self.outputs_to_watch.insert(txid.clone(), idx_and_scripts).is_none()
		});
		#[cfg(test)]
		{
		        // If we see a transaction for which we registered outputs previously,
			// make sure the registered scriptpubkey at the expected index match
			// the actual transaction output one. We failed this case before #653.
			for tx in &txn_matched {
				if let Some(outputs) = self.get_outputs_to_watch().get(&tx.txid()) {
					for idx_and_script in outputs.iter() {
						assert!((idx_and_script.0 as usize) < tx.output.len());
						assert_eq!(tx.output[idx_and_script.0 as usize].script_pubkey, idx_and_script.1);
					}
				}
			}
		}
		watch_outputs
	}

	pub fn block_disconnected<B: Deref, F: Deref, L: Deref>(&mut self, header: &BlockHeader, height: u32, broadcaster: B, fee_estimator: F, logger: L)
		where B::Target: BroadcasterInterface,
		      F::Target: FeeEstimator,
		      L::Target: Logger,
	{
		log_trace!(logger, "Block {} at height {} disconnected", header.block_hash(), height);

		//We may discard:
		//- htlc update there as failure-trigger tx (revoked commitment tx, non-revoked commitment tx, HTLC-timeout tx) has been disconnected
		//- maturing spendable output has transaction paying us has been disconnected
		self.onchain_events_awaiting_threshold_conf.retain(|ref entry| entry.height < height);

		self.onchain_tx_handler.block_disconnected(height, broadcaster, fee_estimator, logger);

		self.best_block = BestBlock::new(header.prev_blockhash, height - 1);
	}

	fn transaction_unconfirmed<B: Deref, F: Deref, L: Deref>(
		&mut self,
		txid: &Txid,
		broadcaster: B,
		fee_estimator: F,
		logger: L,
	) where
		B::Target: BroadcasterInterface,
		F::Target: FeeEstimator,
		L::Target: Logger,
	{
		self.onchain_events_awaiting_threshold_conf.retain(|ref entry| entry.txid != *txid);
		self.onchain_tx_handler.transaction_unconfirmed(txid, broadcaster, fee_estimator, logger);
	}

	/// Filters a block's `txdata` for transactions spending watched outputs or for any child
	/// transactions thereof.
	fn filter_block<'a>(&self, txdata: &TransactionData<'a>) -> Vec<&'a Transaction> {
		let mut matched_txn = HashSet::new();
		txdata.iter().filter(|&&(_, tx)| {
			let mut matches = self.spends_watched_output(tx);
			for input in tx.input.iter() {
				if matches { break; }
				if matched_txn.contains(&input.previous_output.txid) {
					matches = true;
				}
			}
			if matches {
				matched_txn.insert(tx.txid());
			}
			matches
		}).map(|(_, tx)| *tx).collect()
	}

	/// Checks if a given transaction spends any watched outputs.
	fn spends_watched_output(&self, tx: &Transaction) -> bool {
		for input in tx.input.iter() {
			if let Some(outputs) = self.get_outputs_to_watch().get(&input.previous_output.txid) {
				for (idx, _script_pubkey) in outputs.iter() {
					if *idx == input.previous_output.vout {
						#[cfg(test)]
						{
						        // If the expected script is a known type, check that the witness
						        // appears to be spending the correct type (ie that the match would
						        // actually succeed in BIP 158/159-style filters).
						        if _script_pubkey.is_v0_p2wsh() {
						                assert_eq!(&bitcoin::Address::p2wsh(&Script::from(input.witness.last().unwrap().clone()), bitcoin::Network::Bitcoin).script_pubkey(), _script_pubkey);
						        } else if _script_pubkey.is_v0_p2wpkh() {
						                assert_eq!(&bitcoin::Address::p2wpkh(&bitcoin::PublicKey::from_slice(&input.witness.last().unwrap()).unwrap(), bitcoin::Network::Bitcoin).unwrap().script_pubkey(), _script_pubkey);
						        } else { panic!(); }
						}
						return true;
					}
				}
			}
		}

		false
	}

	fn should_broadcast_holder_commitment_txn<L: Deref>(&self, logger: &L) -> bool where L::Target: Logger {
		// We need to consider all HTLCs which are:
		//  * in any unrevoked counterparty commitment transaction, as they could broadcast said
		//    transactions and we'd end up in a race, or
		//  * are in our latest holder commitment transaction, as this is the thing we will
		//    broadcast if we go on-chain.
		// Note that we consider HTLCs which were below dust threshold here - while they don't
		// strictly imply that we need to fail the channel, we need to go ahead and fail them back
		// to the source, and if we don't fail the channel we will have to ensure that the next
		// updates that peer sends us are update_fails, failing the channel if not. It's probably
		// easier to just fail the channel as this case should be rare enough anyway.
		let height = self.best_block.height();
		macro_rules! scan_commitment {
			($htlcs: expr, $holder_tx: expr) => {
				for ref htlc in $htlcs {
					// For inbound HTLCs which we know the preimage for, we have to ensure we hit the
					// chain with enough room to claim the HTLC without our counterparty being able to
					// time out the HTLC first.
					// For outbound HTLCs which our counterparty hasn't failed/claimed, our primary
					// concern is being able to claim the corresponding inbound HTLC (on another
					// channel) before it expires. In fact, we don't even really care if our
					// counterparty here claims such an outbound HTLC after it expired as long as we
					// can still claim the corresponding HTLC. Thus, to avoid needlessly hitting the
					// chain when our counterparty is waiting for expiration to off-chain fail an HTLC
					// we give ourselves a few blocks of headroom after expiration before going
					// on-chain for an expired HTLC.
					// Note that, to avoid a potential attack whereby a node delays claiming an HTLC
					// from us until we've reached the point where we go on-chain with the
					// corresponding inbound HTLC, we must ensure that outbound HTLCs go on chain at
					// least CLTV_CLAIM_BUFFER blocks prior to the inbound HTLC.
					//  aka outbound_cltv + LATENCY_GRACE_PERIOD_BLOCKS == height - CLTV_CLAIM_BUFFER
					//      inbound_cltv == height + CLTV_CLAIM_BUFFER
					//      outbound_cltv + LATENCY_GRACE_PERIOD_BLOCKS + CLTV_CLAIM_BUFFER <= inbound_cltv - CLTV_CLAIM_BUFFER
					//      LATENCY_GRACE_PERIOD_BLOCKS + 2*CLTV_CLAIM_BUFFER <= inbound_cltv - outbound_cltv
					//      CLTV_EXPIRY_DELTA <= inbound_cltv - outbound_cltv (by check in ChannelManager::decode_update_add_htlc_onion)
					//      LATENCY_GRACE_PERIOD_BLOCKS + 2*CLTV_CLAIM_BUFFER <= CLTV_EXPIRY_DELTA
					//  The final, above, condition is checked for statically in channelmanager
					//  with CHECK_CLTV_EXPIRY_SANITY_2.
					let htlc_outbound = $holder_tx == htlc.offered;
					if ( htlc_outbound && htlc.cltv_expiry + LATENCY_GRACE_PERIOD_BLOCKS <= height) ||
					   (!htlc_outbound && htlc.cltv_expiry <= height + CLTV_CLAIM_BUFFER && self.payment_preimages.contains_key(&htlc.payment_hash)) {
						log_info!(logger, "Force-closing channel due to {} HTLC timeout, HTLC expiry is {}", if htlc_outbound { "outbound" } else { "inbound "}, htlc.cltv_expiry);
						return true;
					}
				}
			}
		}

		scan_commitment!(self.current_holder_commitment_tx.htlc_outputs.iter().map(|&(ref a, _, _)| a), true);

		if let Some(ref txid) = self.current_counterparty_commitment_txid {
			if let Some(ref htlc_outputs) = self.counterparty_claimable_outpoints.get(txid) {
				scan_commitment!(htlc_outputs.iter().map(|&(ref a, _)| a), false);
			}
		}
		if let Some(ref txid) = self.prev_counterparty_commitment_txid {
			if let Some(ref htlc_outputs) = self.counterparty_claimable_outpoints.get(txid) {
				scan_commitment!(htlc_outputs.iter().map(|&(ref a, _)| a), false);
			}
		}

		false
	}

	/// Check if any transaction broadcasted is resolving HTLC output by a success or timeout on a holder
	/// or counterparty commitment tx, if so send back the source, preimage if found and payment_hash of resolved HTLC
	fn is_resolving_htlc_output<L: Deref>(&mut self, tx: &Transaction, height: u32, logger: &L) where L::Target: Logger {
		'outer_loop: for input in &tx.input {
			let mut payment_data = None;
			let revocation_sig_claim = (input.witness.len() == 3 && HTLCType::scriptlen_to_htlctype(input.witness[2].len()) == Some(HTLCType::OfferedHTLC) && input.witness[1].len() == 33)
				|| (input.witness.len() == 3 && HTLCType::scriptlen_to_htlctype(input.witness[2].len()) == Some(HTLCType::AcceptedHTLC) && input.witness[1].len() == 33);
			let accepted_preimage_claim = input.witness.len() == 5 && HTLCType::scriptlen_to_htlctype(input.witness[4].len()) == Some(HTLCType::AcceptedHTLC);
			#[cfg(not(fuzzing))]
			let accepted_timeout_claim = input.witness.len() == 3 && HTLCType::scriptlen_to_htlctype(input.witness[2].len()) == Some(HTLCType::AcceptedHTLC) && !revocation_sig_claim;
			let offered_preimage_claim = input.witness.len() == 3 && HTLCType::scriptlen_to_htlctype(input.witness[2].len()) == Some(HTLCType::OfferedHTLC) && !revocation_sig_claim;
			#[cfg(not(fuzzing))]
			let offered_timeout_claim = input.witness.len() == 5 && HTLCType::scriptlen_to_htlctype(input.witness[4].len()) == Some(HTLCType::OfferedHTLC);

			let mut payment_preimage = PaymentPreimage([0; 32]);
			if accepted_preimage_claim {
				payment_preimage.0.copy_from_slice(&input.witness[3]);
			} else if offered_preimage_claim {
				payment_preimage.0.copy_from_slice(&input.witness[1]);
			}

			macro_rules! log_claim {
				($tx_info: expr, $holder_tx: expr, $htlc: expr, $source_avail: expr) => {
					let outbound_htlc = $holder_tx == $htlc.offered;
					// HTLCs must either be claimed by a matching script type or through the
					// revocation path:
					#[cfg(not(fuzzing))] // Note that the fuzzer is not bound by pesky things like "signatures"
					debug_assert!(!$htlc.offered || offered_preimage_claim || offered_timeout_claim || revocation_sig_claim);
					#[cfg(not(fuzzing))] // Note that the fuzzer is not bound by pesky things like "signatures"
					debug_assert!($htlc.offered || accepted_preimage_claim || accepted_timeout_claim || revocation_sig_claim);
					// Further, only exactly one of the possible spend paths should have been
					// matched by any HTLC spend:
					#[cfg(not(fuzzing))] // Note that the fuzzer is not bound by pesky things like "signatures"
					debug_assert_eq!(accepted_preimage_claim as u8 + accepted_timeout_claim as u8 +
					                 offered_preimage_claim as u8 + offered_timeout_claim as u8 +
					                 revocation_sig_claim as u8, 1);
					if ($holder_tx && revocation_sig_claim) ||
							(outbound_htlc && !$source_avail && (accepted_preimage_claim || offered_preimage_claim)) {
						log_error!(logger, "Input spending {} ({}:{}) in {} resolves {} HTLC with payment hash {} with {}!",
							$tx_info, input.previous_output.txid, input.previous_output.vout, tx.txid(),
							if outbound_htlc { "outbound" } else { "inbound" }, log_bytes!($htlc.payment_hash.0),
							if revocation_sig_claim { "revocation sig" } else { "preimage claim after we'd passed the HTLC resolution back" });
					} else {
						log_info!(logger, "Input spending {} ({}:{}) in {} resolves {} HTLC with payment hash {} with {}",
							$tx_info, input.previous_output.txid, input.previous_output.vout, tx.txid(),
							if outbound_htlc { "outbound" } else { "inbound" }, log_bytes!($htlc.payment_hash.0),
							if revocation_sig_claim { "revocation sig" } else if accepted_preimage_claim || offered_preimage_claim { "preimage" } else { "timeout" });
					}
				}
			}

			macro_rules! check_htlc_valid_counterparty {
				($counterparty_txid: expr, $htlc_output: expr) => {
					if let Some(txid) = $counterparty_txid {
						for &(ref pending_htlc, ref pending_source) in self.counterparty_claimable_outpoints.get(&txid).unwrap() {
							if pending_htlc.payment_hash == $htlc_output.payment_hash && pending_htlc.amount_msat == $htlc_output.amount_msat {
								if let &Some(ref source) = pending_source {
									log_claim!("revoked counterparty commitment tx", false, pending_htlc, true);
									payment_data = Some(((**source).clone(), $htlc_output.payment_hash, $htlc_output.amount_msat));
									break;
								}
							}
						}
					}
				}
			}

			macro_rules! scan_commitment {
				($htlcs: expr, $tx_info: expr, $holder_tx: expr) => {
					for (ref htlc_output, source_option) in $htlcs {
						if Some(input.previous_output.vout) == htlc_output.transaction_output_index {
							if let Some(ref source) = source_option {
								log_claim!($tx_info, $holder_tx, htlc_output, true);
								// We have a resolution of an HTLC either from one of our latest
								// holder commitment transactions or an unrevoked counterparty commitment
								// transaction. This implies we either learned a preimage, the HTLC
								// has timed out, or we screwed up. In any case, we should now
								// resolve the source HTLC with the original sender.
								payment_data = Some(((*source).clone(), htlc_output.payment_hash, htlc_output.amount_msat));
							} else if !$holder_tx {
								check_htlc_valid_counterparty!(self.current_counterparty_commitment_txid, htlc_output);
								if payment_data.is_none() {
									check_htlc_valid_counterparty!(self.prev_counterparty_commitment_txid, htlc_output);
								}
							}
							if payment_data.is_none() {
								log_claim!($tx_info, $holder_tx, htlc_output, false);
								let outbound_htlc = $holder_tx == htlc_output.offered;
								if !outbound_htlc || revocation_sig_claim {
									self.onchain_events_awaiting_threshold_conf.push(OnchainEventEntry {
										txid: tx.txid(), height,
										event: OnchainEvent::HTLCSpendConfirmation {
											input_idx: input.previous_output.vout,
											preimage: if accepted_preimage_claim || offered_preimage_claim {
												Some(payment_preimage) } else { None },
											// If this is a payment to us (!outbound_htlc, above),
											// wait for the CSV delay before dropping the HTLC from
											// claimable balance if the claim was an HTLC-Success
											// transaction.
											on_to_local_output_csv: if accepted_preimage_claim {
												Some(self.on_holder_tx_csv) } else { None },
										},
									});
								} else {
									// Outbound claims should always have payment_data, unless
									// we've already failed the HTLC as the commitment transaction
									// which was broadcasted was revoked. In that case, we should
									// spend the HTLC output here immediately, and expose that fact
									// as a Balance, something which we do not yet do.
									// TODO: Track the above as claimable!
								}
								continue 'outer_loop;
							}
						}
					}
				}
			}

			if input.previous_output.txid == self.current_holder_commitment_tx.txid {
				scan_commitment!(self.current_holder_commitment_tx.htlc_outputs.iter().map(|&(ref a, _, ref b)| (a, b.as_ref())),
					"our latest holder commitment tx", true);
			}
			if let Some(ref prev_holder_signed_commitment_tx) = self.prev_holder_signed_commitment_tx {
				if input.previous_output.txid == prev_holder_signed_commitment_tx.txid {
					scan_commitment!(prev_holder_signed_commitment_tx.htlc_outputs.iter().map(|&(ref a, _, ref b)| (a, b.as_ref())),
						"our previous holder commitment tx", true);
				}
			}
			if let Some(ref htlc_outputs) = self.counterparty_claimable_outpoints.get(&input.previous_output.txid) {
				scan_commitment!(htlc_outputs.iter().map(|&(ref a, ref b)| (a, (b.as_ref().clone()).map(|boxed| &**boxed))),
					"counterparty commitment tx", false);
			}

			// Check that scan_commitment, above, decided there is some source worth relaying an
			// HTLC resolution backwards to and figure out whether we learned a preimage from it.
			if let Some((source, payment_hash, amount_msat)) = payment_data {
				if accepted_preimage_claim {
					if !self.pending_monitor_events.iter().any(
						|update| if let &MonitorEvent::HTLCEvent(ref upd) = update { upd.source == source } else { false }) {
						self.onchain_events_awaiting_threshold_conf.push(OnchainEventEntry {
							txid: tx.txid(),
							height,
							event: OnchainEvent::HTLCSpendConfirmation {
								input_idx: input.previous_output.vout,
								preimage: Some(payment_preimage),
								on_to_local_output_csv: None,
							},
						});
						self.pending_monitor_events.push(MonitorEvent::HTLCEvent(HTLCUpdate {
							source,
							payment_preimage: Some(payment_preimage),
							payment_hash,
							onchain_value_satoshis: Some(amount_msat / 1000),
						}));
					}
				} else if offered_preimage_claim {
					if !self.pending_monitor_events.iter().any(
						|update| if let &MonitorEvent::HTLCEvent(ref upd) = update {
							upd.source == source
						} else { false }) {
						self.onchain_events_awaiting_threshold_conf.push(OnchainEventEntry {
							txid: tx.txid(),
							height,
							event: OnchainEvent::HTLCSpendConfirmation {
								input_idx: input.previous_output.vout,
								preimage: Some(payment_preimage),
								on_to_local_output_csv: None,
							},
						});
						self.pending_monitor_events.push(MonitorEvent::HTLCEvent(HTLCUpdate {
							source,
							payment_preimage: Some(payment_preimage),
							payment_hash,
							onchain_value_satoshis: Some(amount_msat / 1000),
						}));
					}
				} else {
					self.onchain_events_awaiting_threshold_conf.retain(|ref entry| {
						if entry.height != height { return true; }
						match entry.event {
							OnchainEvent::HTLCUpdate { source: ref htlc_source, .. } => {
								*htlc_source != source
							},
							_ => true,
						}
					});
					let entry = OnchainEventEntry {
						txid: tx.txid(),
						height,
						event: OnchainEvent::HTLCUpdate {
							source, payment_hash,
							onchain_value_satoshis: Some(amount_msat / 1000),
							input_idx: Some(input.previous_output.vout),
						},
					};
					log_info!(logger, "Failing HTLC with payment_hash {} timeout by a spend tx, waiting for confirmation (at height {})", log_bytes!(payment_hash.0), entry.confirmation_threshold());
					self.onchain_events_awaiting_threshold_conf.push(entry);
				}
			}
		}
	}

	/// Check if any transaction broadcasted is paying fund back to some address we can assume to own
	fn is_paying_spendable_output<L: Deref>(&mut self, tx: &Transaction, height: u32, logger: &L) where L::Target: Logger {
		let mut spendable_output = None;
		for (i, outp) in tx.output.iter().enumerate() { // There is max one spendable output for any channel tx, including ones generated by us
			if i > ::core::u16::MAX as usize {
				// While it is possible that an output exists on chain which is greater than the
				// 2^16th output in a given transaction, this is only possible if the output is not
				// in a lightning transaction and was instead placed there by some third party who
				// wishes to give us money for no reason.
				// Namely, any lightning transactions which we pre-sign will never have anywhere
				// near 2^16 outputs both because such transactions must have ~2^16 outputs who's
				// scripts are not longer than one byte in length and because they are inherently
				// non-standard due to their size.
				// Thus, it is completely safe to ignore such outputs, and while it may result in
				// us ignoring non-lightning fund to us, that is only possible if someone fills
				// nearly a full block with garbage just to hit this case.
				continue;
			}
			if outp.script_pubkey == self.destination_script {
				spendable_output =  Some(SpendableOutputDescriptor::StaticOutput {
					outpoint: OutPoint { txid: tx.txid(), index: i as u16 },
					output: outp.clone(),
				});
				break;
			}
			if let Some(ref broadcasted_holder_revokable_script) = self.broadcasted_holder_revokable_script {
				if broadcasted_holder_revokable_script.0 == outp.script_pubkey {
					spendable_output =  Some(SpendableOutputDescriptor::DelayedPaymentOutput(DelayedPaymentOutputDescriptor {
						outpoint: OutPoint { txid: tx.txid(), index: i as u16 },
						per_commitment_point: broadcasted_holder_revokable_script.1,
						to_self_delay: self.on_holder_tx_csv,
						output: outp.clone(),
						revocation_pubkey: broadcasted_holder_revokable_script.2.clone(),
						channel_keys_id: self.channel_keys_id,
						channel_value_satoshis: self.channel_value_satoshis,
					}));
					break;
				}
			}
			if self.counterparty_payment_script == outp.script_pubkey {
				spendable_output = Some(SpendableOutputDescriptor::StaticPaymentOutput(StaticPaymentOutputDescriptor {
					outpoint: OutPoint { txid: tx.txid(), index: i as u16 },
					output: outp.clone(),
					channel_keys_id: self.channel_keys_id,
					channel_value_satoshis: self.channel_value_satoshis,
				}));
				break;
			}
			if self.shutdown_script.as_ref() == Some(&outp.script_pubkey) {
				spendable_output = Some(SpendableOutputDescriptor::StaticOutput {
					outpoint: OutPoint { txid: tx.txid(), index: i as u16 },
					output: outp.clone(),
				});
				break;
			}
		}
		if let Some(spendable_output) = spendable_output {
			let entry = OnchainEventEntry {
				txid: tx.txid(),
				height: height,
				event: OnchainEvent::MaturingOutput { descriptor: spendable_output.clone() },
			};
			log_info!(logger, "Received spendable output {}, spendable at height {}", log_spendable!(spendable_output), entry.confirmation_threshold());
			self.onchain_events_awaiting_threshold_conf.push(entry);
		}
	}
}

impl<Signer: Sign, T: Deref, F: Deref, L: Deref> chain::Listen for (ChannelMonitor<Signer>, T, F, L)
where
	T::Target: BroadcasterInterface,
	F::Target: FeeEstimator,
	L::Target: Logger,
{
	fn block_connected(&self, block: &Block, height: u32) {
		let txdata: Vec<_> = block.txdata.iter().enumerate().collect();
		self.0.block_connected(&block.header, &txdata, height, &*self.1, &*self.2, &*self.3);
	}

	fn block_disconnected(&self, header: &BlockHeader, height: u32) {
		self.0.block_disconnected(header, height, &*self.1, &*self.2, &*self.3);
	}
}

impl<Signer: Sign, T: Deref, F: Deref, L: Deref> chain::Confirm for (ChannelMonitor<Signer>, T, F, L)
where
	T::Target: BroadcasterInterface,
	F::Target: FeeEstimator,
	L::Target: Logger,
{
	fn transactions_confirmed(&self, header: &BlockHeader, txdata: &TransactionData, height: u32) {
		self.0.transactions_confirmed(header, txdata, height, &*self.1, &*self.2, &*self.3);
	}

	fn transaction_unconfirmed(&self, txid: &Txid) {
		self.0.transaction_unconfirmed(txid, &*self.1, &*self.2, &*self.3);
	}

	fn best_block_updated(&self, header: &BlockHeader, height: u32) {
		self.0.best_block_updated(header, height, &*self.1, &*self.2, &*self.3);
	}

	fn get_relevant_txids(&self) -> Vec<Txid> {
		self.0.get_relevant_txids()
	}
}

const MAX_ALLOC_SIZE: usize = 64*1024;

impl<'a, Signer: Sign, K: KeysInterface<Signer = Signer>> ReadableArgs<&'a K>
		for (BlockHash, ChannelMonitor<Signer>) {
	fn read<R: io::Read>(reader: &mut R, keys_manager: &'a K) -> Result<Self, DecodeError> {
		macro_rules! unwrap_obj {
			($key: expr) => {
				match $key {
					Ok(res) => res,
					Err(_) => return Err(DecodeError::InvalidValue),
				}
			}
		}

		let _ver = read_ver_prefix!(reader, SERIALIZATION_VERSION);

		let latest_update_id: u64 = Readable::read(reader)?;
		let commitment_transaction_number_obscure_factor = <U48 as Readable>::read(reader)?.0;

		let destination_script = Readable::read(reader)?;
		let broadcasted_holder_revokable_script = match <u8 as Readable>::read(reader)? {
			0 => {
				let revokable_address = Readable::read(reader)?;
				let per_commitment_point = Readable::read(reader)?;
				let revokable_script = Readable::read(reader)?;
				Some((revokable_address, per_commitment_point, revokable_script))
			},
			1 => { None },
			_ => return Err(DecodeError::InvalidValue),
		};
		let counterparty_payment_script = Readable::read(reader)?;
		let shutdown_script = {
			let script = <Script as Readable>::read(reader)?;
			if script.is_empty() { None } else { Some(script) }
		};

		let channel_keys_id = Readable::read(reader)?;
		let holder_revocation_basepoint = Readable::read(reader)?;
		// Technically this can fail and serialize fail a round-trip, but only for serialization of
		// barely-init'd ChannelMonitors that we can't do anything with.
		let outpoint = OutPoint {
			txid: Readable::read(reader)?,
			index: Readable::read(reader)?,
		};
		let funding_info = (outpoint, Readable::read(reader)?);
		let current_counterparty_commitment_txid = Readable::read(reader)?;
		let prev_counterparty_commitment_txid = Readable::read(reader)?;

		let counterparty_commitment_params = Readable::read(reader)?;
		let funding_redeemscript = Readable::read(reader)?;
		let channel_value_satoshis = Readable::read(reader)?;

		let their_cur_revocation_points = {
			let first_idx = <U48 as Readable>::read(reader)?.0;
			if first_idx == 0 {
				None
			} else {
				let first_point = Readable::read(reader)?;
				let second_point_slice: [u8; 33] = Readable::read(reader)?;
				if second_point_slice[0..32] == [0; 32] && second_point_slice[32] == 0 {
					Some((first_idx, first_point, None))
				} else {
					Some((first_idx, first_point, Some(unwrap_obj!(PublicKey::from_slice(&second_point_slice)))))
				}
			}
		};

		let on_holder_tx_csv: u16 = Readable::read(reader)?;

		let commitment_secrets = Readable::read(reader)?;

		macro_rules! read_htlc_in_commitment {
			() => {
				{
					let offered: bool = Readable::read(reader)?;
					let amount_msat: u64 = Readable::read(reader)?;
					let cltv_expiry: u32 = Readable::read(reader)?;
					let payment_hash: PaymentHash = Readable::read(reader)?;
					let transaction_output_index: Option<u32> = Readable::read(reader)?;

					HTLCOutputInCommitment {
						offered, amount_msat, cltv_expiry, payment_hash, transaction_output_index
					}
				}
			}
		}

		let counterparty_claimable_outpoints_len: u64 = Readable::read(reader)?;
		let mut counterparty_claimable_outpoints = HashMap::with_capacity(cmp::min(counterparty_claimable_outpoints_len as usize, MAX_ALLOC_SIZE / 64));
		for _ in 0..counterparty_claimable_outpoints_len {
			let txid: Txid = Readable::read(reader)?;
			let htlcs_count: u64 = Readable::read(reader)?;
			let mut htlcs = Vec::with_capacity(cmp::min(htlcs_count as usize, MAX_ALLOC_SIZE / 32));
			for _ in 0..htlcs_count {
				htlcs.push((read_htlc_in_commitment!(), <Option<HTLCSource> as Readable>::read(reader)?.map(|o: HTLCSource| Box::new(o))));
			}
			if let Some(_) = counterparty_claimable_outpoints.insert(txid, htlcs) {
				return Err(DecodeError::InvalidValue);
			}
		}

		let counterparty_commitment_txn_on_chain_len: u64 = Readable::read(reader)?;
		let mut counterparty_commitment_txn_on_chain = HashMap::with_capacity(cmp::min(counterparty_commitment_txn_on_chain_len as usize, MAX_ALLOC_SIZE / 32));
		for _ in 0..counterparty_commitment_txn_on_chain_len {
			let txid: Txid = Readable::read(reader)?;
			let commitment_number = <U48 as Readable>::read(reader)?.0;
			if let Some(_) = counterparty_commitment_txn_on_chain.insert(txid, commitment_number) {
				return Err(DecodeError::InvalidValue);
			}
		}

		let counterparty_hash_commitment_number_len: u64 = Readable::read(reader)?;
		let mut counterparty_hash_commitment_number = HashMap::with_capacity(cmp::min(counterparty_hash_commitment_number_len as usize, MAX_ALLOC_SIZE / 32));
		for _ in 0..counterparty_hash_commitment_number_len {
			let payment_hash: PaymentHash = Readable::read(reader)?;
			let commitment_number = <U48 as Readable>::read(reader)?.0;
			if let Some(_) = counterparty_hash_commitment_number.insert(payment_hash, commitment_number) {
				return Err(DecodeError::InvalidValue);
			}
		}

		let mut prev_holder_signed_commitment_tx: Option<HolderSignedTx> =
			match <u8 as Readable>::read(reader)? {
				0 => None,
				1 => {
					Some(Readable::read(reader)?)
				},
				_ => return Err(DecodeError::InvalidValue),
			};
		let mut current_holder_commitment_tx: HolderSignedTx = Readable::read(reader)?;

		let current_counterparty_commitment_number = <U48 as Readable>::read(reader)?.0;
		let current_holder_commitment_number = <U48 as Readable>::read(reader)?.0;

		let payment_preimages_len: u64 = Readable::read(reader)?;
		let mut payment_preimages = HashMap::with_capacity(cmp::min(payment_preimages_len as usize, MAX_ALLOC_SIZE / 32));
		for _ in 0..payment_preimages_len {
			let preimage: PaymentPreimage = Readable::read(reader)?;
			let hash = PaymentHash(Sha256::hash(&preimage.0[..]).into_inner());
			if let Some(_) = payment_preimages.insert(hash, preimage) {
				return Err(DecodeError::InvalidValue);
			}
		}

		let pending_monitor_events_len: u64 = Readable::read(reader)?;
		let mut pending_monitor_events = Some(
			Vec::with_capacity(cmp::min(pending_monitor_events_len as usize, MAX_ALLOC_SIZE / (32 + 8*3))));
		for _ in 0..pending_monitor_events_len {
			let ev = match <u8 as Readable>::read(reader)? {
				0 => MonitorEvent::HTLCEvent(Readable::read(reader)?),
				1 => MonitorEvent::CommitmentTxConfirmed(funding_info.0),
				_ => return Err(DecodeError::InvalidValue)
			};
			pending_monitor_events.as_mut().unwrap().push(ev);
		}

		let pending_events_len: u64 = Readable::read(reader)?;
		let mut pending_events = Vec::with_capacity(cmp::min(pending_events_len as usize, MAX_ALLOC_SIZE / mem::size_of::<Event>()));
		for _ in 0..pending_events_len {
			if let Some(event) = MaybeReadable::read(reader)? {
				pending_events.push(event);
			}
		}

		let best_block = BestBlock::new(Readable::read(reader)?, Readable::read(reader)?);

		let waiting_threshold_conf_len: u64 = Readable::read(reader)?;
		let mut onchain_events_awaiting_threshold_conf = Vec::with_capacity(cmp::min(waiting_threshold_conf_len as usize, MAX_ALLOC_SIZE / 128));
		for _ in 0..waiting_threshold_conf_len {
			if let Some(val) = MaybeReadable::read(reader)? {
				onchain_events_awaiting_threshold_conf.push(val);
			}
		}

		let outputs_to_watch_len: u64 = Readable::read(reader)?;
		let mut outputs_to_watch = HashMap::with_capacity(cmp::min(outputs_to_watch_len as usize, MAX_ALLOC_SIZE / (mem::size_of::<Txid>() + mem::size_of::<u32>() + mem::size_of::<Vec<Script>>())));
		for _ in 0..outputs_to_watch_len {
			let txid = Readable::read(reader)?;
			let outputs_len: u64 = Readable::read(reader)?;
			let mut outputs = Vec::with_capacity(cmp::min(outputs_len as usize, MAX_ALLOC_SIZE / (mem::size_of::<u32>() + mem::size_of::<Script>())));
			for _ in 0..outputs_len {
				outputs.push((Readable::read(reader)?, Readable::read(reader)?));
			}
			if let Some(_) = outputs_to_watch.insert(txid, outputs) {
				return Err(DecodeError::InvalidValue);
			}
		}
		let onchain_tx_handler: OnchainTxHandler<Signer> = ReadableArgs::read(reader, keys_manager)?;

		let lockdown_from_offchain = Readable::read(reader)?;
		let holder_tx_signed = Readable::read(reader)?;

		if let Some(prev_commitment_tx) = prev_holder_signed_commitment_tx.as_mut() {
			let prev_holder_value = onchain_tx_handler.get_prev_holder_commitment_to_self_value();
			if prev_holder_value.is_none() { return Err(DecodeError::InvalidValue); }
			if prev_commitment_tx.to_self_value_sat == u64::max_value() {
				prev_commitment_tx.to_self_value_sat = prev_holder_value.unwrap();
			} else if prev_commitment_tx.to_self_value_sat != prev_holder_value.unwrap() {
				return Err(DecodeError::InvalidValue);
			}
		}

		let cur_holder_value = onchain_tx_handler.get_cur_holder_commitment_to_self_value();
		if current_holder_commitment_tx.to_self_value_sat == u64::max_value() {
			current_holder_commitment_tx.to_self_value_sat = cur_holder_value;
		} else if current_holder_commitment_tx.to_self_value_sat != cur_holder_value {
			return Err(DecodeError::InvalidValue);
		}

		let mut funding_spend_confirmed = None;
		let mut htlcs_resolved_on_chain = Some(Vec::new());
		let mut funding_spend_seen = Some(false);
		read_tlv_fields!(reader, {
			(1, funding_spend_confirmed, option),
			(3, htlcs_resolved_on_chain, vec_type),
			(5, pending_monitor_events, vec_type),
			(7, funding_spend_seen, option),
		});

		let mut secp_ctx = Secp256k1::new();
		secp_ctx.seeded_randomize(&keys_manager.get_secure_random_bytes());

		Ok((best_block.block_hash(), ChannelMonitor {
			inner: Mutex::new(ChannelMonitorImpl {
				latest_update_id,
				commitment_transaction_number_obscure_factor,

				destination_script,
				broadcasted_holder_revokable_script,
				counterparty_payment_script,
				shutdown_script,

				channel_keys_id,
				holder_revocation_basepoint,
				funding_info,
				current_counterparty_commitment_txid,
				prev_counterparty_commitment_txid,

				counterparty_commitment_params,
				funding_redeemscript,
				channel_value_satoshis,
				their_cur_revocation_points,

				on_holder_tx_csv,

				commitment_secrets,
				counterparty_claimable_outpoints,
				counterparty_commitment_txn_on_chain,
				counterparty_hash_commitment_number,

				prev_holder_signed_commitment_tx,
				current_holder_commitment_tx,
				current_counterparty_commitment_number,
				current_holder_commitment_number,

				payment_preimages,
				pending_monitor_events: pending_monitor_events.unwrap(),
				pending_events,

				onchain_events_awaiting_threshold_conf,
				outputs_to_watch,

				onchain_tx_handler,

				lockdown_from_offchain,
				holder_tx_signed,
				funding_spend_seen: funding_spend_seen.unwrap(),
				funding_spend_confirmed,
				htlcs_resolved_on_chain: htlcs_resolved_on_chain.unwrap(),

				best_block,

				secp_ctx,
			}),
		}))
	}
}

#[cfg(test)]
mod tests {
	use bitcoin::blockdata::block::BlockHeader;
	use bitcoin::blockdata::script::{Script, Builder};
	use bitcoin::blockdata::opcodes;
	use bitcoin::blockdata::transaction::{Transaction, TxIn, TxOut, SigHashType};
	use bitcoin::blockdata::transaction::OutPoint as BitcoinOutPoint;
	use bitcoin::util::bip143;
	use bitcoin::hashes::Hash;
	use bitcoin::hashes::sha256::Hash as Sha256;
	use bitcoin::hashes::hex::FromHex;
	use bitcoin::hash_types::{BlockHash, Txid};
	use bitcoin::network::constants::Network;
	use bitcoin::secp256k1::key::{SecretKey,PublicKey};
	use bitcoin::secp256k1::Secp256k1;

	use hex;

	use super::ChannelMonitorUpdateStep;
	use ::{check_added_monitors, check_closed_broadcast, check_closed_event, check_spends, get_local_commitment_txn, get_monitor, get_route_and_payment_hash, unwrap_send_err};
	use chain::{BestBlock, Confirm};
	use chain::channelmonitor::ChannelMonitor;
	use chain::package::{weight_offered_htlc, weight_received_htlc, weight_revoked_offered_htlc, weight_revoked_received_htlc, WEIGHT_REVOKED_OUTPUT};
	use chain::transaction::OutPoint;
	use chain::keysinterface::InMemorySigner;
	use ln::{PaymentPreimage, PaymentHash};
	use ln::chan_utils;
	use ln::chan_utils::{HTLCOutputInCommitment, ChannelPublicKeys, ChannelTransactionParameters, HolderCommitmentTransaction, CounterpartyChannelTransactionParameters};
	use ln::channelmanager::PaymentSendFailure;
	use ln::features::InitFeatures;
	use ln::functional_test_utils::*;
	use ln::script::ShutdownScript;
	use util::errors::APIError;
	use util::events::{ClosureReason, MessageSendEventsProvider};
	use util::test_utils::{TestLogger, TestBroadcaster, TestFeeEstimator};
	use util::ser::{ReadableArgs, Writeable};
	use sync::{Arc, Mutex};
	use io;
	use prelude::*;

	fn do_test_funding_spend_refuses_updates(use_local_txn: bool) {
		// Previously, monitor updates were allowed freely even after a funding-spend transaction
		// confirmed. This would allow a race condition where we could receive a payment (including
		// the counterparty revoking their broadcasted state!) and accept it without recourse as
		// long as the ChannelMonitor receives the block first, the full commitment update dance
		// occurs after the block is connected, and before the ChannelManager receives the block.
		// Obviously this is an incredibly contrived race given the counterparty would be risking
		// their full channel balance for it, but its worth fixing nonetheless as it makes the
		// potential ChannelMonitor states simpler to reason about.
		//
		// This test checks said behavior, as well as ensuring a ChannelMonitorUpdate with multiple
		// updates is handled correctly in such conditions.
		let chanmon_cfgs = create_chanmon_cfgs(3);
		let node_cfgs = create_node_cfgs(3, &chanmon_cfgs);
		let node_chanmgrs = create_node_chanmgrs(3, &node_cfgs, &[None, None, None]);
		let nodes = create_network(3, &node_cfgs, &node_chanmgrs);
		let channel = create_announced_chan_between_nodes(
			&nodes, 0, 1, InitFeatures::known(), InitFeatures::known());
		create_announced_chan_between_nodes(
			&nodes, 1, 2, InitFeatures::known(), InitFeatures::known());

		// Rebalance somewhat
		send_payment(&nodes[0], &[&nodes[1]], 10_000_000);

		// First route two payments for testing at the end
		let payment_preimage_1 = route_payment(&nodes[0], &[&nodes[1], &nodes[2]], 1_000_000).0;
		let payment_preimage_2 = route_payment(&nodes[0], &[&nodes[1], &nodes[2]], 1_000_000).0;

		let local_txn = get_local_commitment_txn!(nodes[1], channel.2);
		assert_eq!(local_txn.len(), 1);
		let remote_txn = get_local_commitment_txn!(nodes[0], channel.2);
		assert_eq!(remote_txn.len(), 3); // Commitment and two HTLC-Timeouts
		check_spends!(remote_txn[1], remote_txn[0]);
		check_spends!(remote_txn[2], remote_txn[0]);
		let broadcast_tx = if use_local_txn { &local_txn[0] } else { &remote_txn[0] };

		// Connect a commitment transaction, but only to the ChainMonitor/ChannelMonitor. The
		// channel is now closed, but the ChannelManager doesn't know that yet.
		let new_header = BlockHeader {
			version: 2, time: 0, bits: 0, nonce: 0,
			prev_blockhash: nodes[0].best_block_info().0,
			merkle_root: Default::default() };
		let conf_height = nodes[0].best_block_info().1 + 1;
		nodes[1].chain_monitor.chain_monitor.transactions_confirmed(&new_header,
			&[(0, broadcast_tx)], conf_height);

		let (_, pre_update_monitor) = <(BlockHash, ChannelMonitor<InMemorySigner>)>::read(
						&mut io::Cursor::new(&get_monitor!(nodes[1], channel.2).encode()),
						&nodes[1].keys_manager.backing).unwrap();

		// If the ChannelManager tries to update the channel, however, the ChainMonitor will pass
		// the update through to the ChannelMonitor which will refuse it (as the channel is closed).
		let (route, payment_hash, _, payment_secret) = get_route_and_payment_hash!(nodes[1], nodes[0], 100_000);
		unwrap_send_err!(nodes[1].node.send_payment(&route, payment_hash, &Some(payment_secret)),
			true, APIError::ChannelUnavailable { ref err },
			assert!(err.contains("ChannelMonitor storage failure")));
		check_added_monitors!(nodes[1], 2); // After the failure we generate a close-channel monitor update
		check_closed_broadcast!(nodes[1], true);
		check_closed_event!(nodes[1], 1, ClosureReason::ProcessingError { err: "ChannelMonitor storage failure".to_string() });

		// Build a new ChannelMonitorUpdate which contains both the failing commitment tx update
		// and provides the claim preimages for the two pending HTLCs. The first update generates
		// an error, but the point of this test is to ensure the later updates are still applied.
		let monitor_updates = nodes[1].chain_monitor.monitor_updates.lock().unwrap();
		let mut replay_update = monitor_updates.get(&channel.2).unwrap().iter().rev().skip(1).next().unwrap().clone();
		assert_eq!(replay_update.updates.len(), 1);
		if let ChannelMonitorUpdateStep::LatestCounterpartyCommitmentTXInfo { .. } = replay_update.updates[0] {
		} else { panic!(); }
		replay_update.updates.push(ChannelMonitorUpdateStep::PaymentPreimage { payment_preimage: payment_preimage_1 });
		replay_update.updates.push(ChannelMonitorUpdateStep::PaymentPreimage { payment_preimage: payment_preimage_2 });

		let broadcaster = TestBroadcaster::new(Arc::clone(&nodes[1].blocks));
		assert!(
			pre_update_monitor.update_monitor(&replay_update, &&broadcaster, &&chanmon_cfgs[1].fee_estimator, &nodes[1].logger)
			.is_err());
		// Even though we error'd on the first update, we should still have generated an HTLC claim
		// transaction
		let txn_broadcasted = broadcaster.txn_broadcasted.lock().unwrap().split_off(0);
		assert!(txn_broadcasted.len() >= 2);
		let htlc_txn = txn_broadcasted.iter().filter(|tx| {
			assert_eq!(tx.input.len(), 1);
			tx.input[0].previous_output.txid == broadcast_tx.txid()
		}).collect::<Vec<_>>();
		assert_eq!(htlc_txn.len(), 2);
		check_spends!(htlc_txn[0], broadcast_tx);
		check_spends!(htlc_txn[1], broadcast_tx);
	}
	#[test]
	fn test_funding_spend_refuses_updates() {
		do_test_funding_spend_refuses_updates(true);
		do_test_funding_spend_refuses_updates(false);
	}

	#[test]
	fn test_prune_preimages() {
		let secp_ctx = Secp256k1::new();
		let logger = Arc::new(TestLogger::new());
		let broadcaster = Arc::new(TestBroadcaster{txn_broadcasted: Mutex::new(Vec::new()), blocks: Arc::new(Mutex::new(Vec::new()))});
		let fee_estimator = Arc::new(TestFeeEstimator { sat_per_kw: Mutex::new(253) });

		let dummy_key = PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[42; 32]).unwrap());
		let dummy_tx = Transaction { version: 0, lock_time: 0, input: Vec::new(), output: Vec::new() };

		let mut preimages = Vec::new();
		{
			for i in 0..20 {
				let preimage = PaymentPreimage([i; 32]);
				let hash = PaymentHash(Sha256::hash(&preimage.0[..]).into_inner());
				preimages.push((preimage, hash));
			}
		}

		macro_rules! preimages_slice_to_htlc_outputs {
			($preimages_slice: expr) => {
				{
					let mut res = Vec::new();
					for (idx, preimage) in $preimages_slice.iter().enumerate() {
						res.push((HTLCOutputInCommitment {
							offered: true,
							amount_msat: 0,
							cltv_expiry: 0,
							payment_hash: preimage.1.clone(),
							transaction_output_index: Some(idx as u32),
						}, None));
					}
					res
				}
			}
		}
		macro_rules! preimages_to_holder_htlcs {
			($preimages_slice: expr) => {
				{
					let mut inp = preimages_slice_to_htlc_outputs!($preimages_slice);
					let res: Vec<_> = inp.drain(..).map(|e| { (e.0, None, e.1) }).collect();
					res
				}
			}
		}

		macro_rules! test_preimages_exist {
			($preimages_slice: expr, $monitor: expr) => {
				for preimage in $preimages_slice {
					assert!($monitor.inner.lock().unwrap().payment_preimages.contains_key(&preimage.1));
				}
			}
		}

		let keys = InMemorySigner::new(
			&secp_ctx,
			SecretKey::from_slice(&[41; 32]).unwrap(),
			SecretKey::from_slice(&[41; 32]).unwrap(),
			SecretKey::from_slice(&[41; 32]).unwrap(),
			SecretKey::from_slice(&[41; 32]).unwrap(),
			SecretKey::from_slice(&[41; 32]).unwrap(),
			SecretKey::from_slice(&[41; 32]).unwrap(),
			[41; 32],
			0,
			[0; 32]
		);

		let counterparty_pubkeys = ChannelPublicKeys {
			funding_pubkey: PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[44; 32]).unwrap()),
			revocation_basepoint: PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[45; 32]).unwrap()),
			payment_point: PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[46; 32]).unwrap()),
			delayed_payment_basepoint: PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[47; 32]).unwrap()),
			htlc_basepoint: PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[48; 32]).unwrap())
		};
		let funding_outpoint = OutPoint { txid: Default::default(), index: u16::max_value() };
		let channel_parameters = ChannelTransactionParameters {
			holder_pubkeys: keys.holder_channel_pubkeys.clone(),
			holder_selected_contest_delay: 66,
			is_outbound_from_holder: true,
			counterparty_parameters: Some(CounterpartyChannelTransactionParameters {
				pubkeys: counterparty_pubkeys,
				selected_contest_delay: 67,
			}),
			funding_outpoint: Some(funding_outpoint),
			opt_anchors: None,
		};
		// Prune with one old state and a holder commitment tx holding a few overlaps with the
		// old state.
		let shutdown_pubkey = PublicKey::from_secret_key(&secp_ctx, &SecretKey::from_slice(&[42; 32]).unwrap());
		let best_block = BestBlock::from_genesis(Network::Testnet);
		let monitor = ChannelMonitor::new(Secp256k1::new(), keys,
		                                  Some(ShutdownScript::new_p2wpkh_from_pubkey(shutdown_pubkey).into_inner()), 0, &Script::new(),
		                                  (OutPoint { txid: Txid::from_slice(&[43; 32]).unwrap(), index: 0 }, Script::new()),
		                                  &channel_parameters,
		                                  Script::new(), 46, 0,
		                                  HolderCommitmentTransaction::dummy(), best_block);

		monitor.provide_latest_holder_commitment_tx(HolderCommitmentTransaction::dummy(), preimages_to_holder_htlcs!(preimages[0..10])).unwrap();
		let dummy_txid = dummy_tx.txid();
		monitor.provide_latest_counterparty_commitment_tx(dummy_txid, preimages_slice_to_htlc_outputs!(preimages[5..15]), 281474976710655, dummy_key, &logger);
		monitor.provide_latest_counterparty_commitment_tx(dummy_txid, preimages_slice_to_htlc_outputs!(preimages[15..20]), 281474976710654, dummy_key, &logger);
		monitor.provide_latest_counterparty_commitment_tx(dummy_txid, preimages_slice_to_htlc_outputs!(preimages[17..20]), 281474976710653, dummy_key, &logger);
		monitor.provide_latest_counterparty_commitment_tx(dummy_txid, preimages_slice_to_htlc_outputs!(preimages[18..20]), 281474976710652, dummy_key, &logger);
		for &(ref preimage, ref hash) in preimages.iter() {
			monitor.provide_payment_preimage(hash, preimage, &broadcaster, &fee_estimator, &logger);
		}

		// Now provide a secret, pruning preimages 10-15
		let mut secret = [0; 32];
		secret[0..32].clone_from_slice(&hex::decode("7cc854b54e3e0dcdb010d7a3fee464a9687be6e8db3be6854c475621e007a5dc").unwrap());
		monitor.provide_secret(281474976710655, secret.clone()).unwrap();
		assert_eq!(monitor.inner.lock().unwrap().payment_preimages.len(), 15);
		test_preimages_exist!(&preimages[0..10], monitor);
		test_preimages_exist!(&preimages[15..20], monitor);

		// Now provide a further secret, pruning preimages 15-17
		secret[0..32].clone_from_slice(&hex::decode("c7518c8ae4660ed02894df8976fa1a3659c1a8b4b5bec0c4b872abeba4cb8964").unwrap());
		monitor.provide_secret(281474976710654, secret.clone()).unwrap();
		assert_eq!(monitor.inner.lock().unwrap().payment_preimages.len(), 13);
		test_preimages_exist!(&preimages[0..10], monitor);
		test_preimages_exist!(&preimages[17..20], monitor);

		// Now update holder commitment tx info, pruning only element 18 as we still care about the
		// previous commitment tx's preimages too
		monitor.provide_latest_holder_commitment_tx(HolderCommitmentTransaction::dummy(), preimages_to_holder_htlcs!(preimages[0..5])).unwrap();
		secret[0..32].clone_from_slice(&hex::decode("2273e227a5b7449b6e70f1fb4652864038b1cbf9cd7c043a7d6456b7fc275ad8").unwrap());
		monitor.provide_secret(281474976710653, secret.clone()).unwrap();
		assert_eq!(monitor.inner.lock().unwrap().payment_preimages.len(), 12);
		test_preimages_exist!(&preimages[0..10], monitor);
		test_preimages_exist!(&preimages[18..20], monitor);

		// But if we do it again, we'll prune 5-10
		monitor.provide_latest_holder_commitment_tx(HolderCommitmentTransaction::dummy(), preimages_to_holder_htlcs!(preimages[0..3])).unwrap();
		secret[0..32].clone_from_slice(&hex::decode("27cddaa5624534cb6cb9d7da077cf2b22ab21e9b506fd4998a51d54502e99116").unwrap());
		monitor.provide_secret(281474976710652, secret.clone()).unwrap();
		assert_eq!(monitor.inner.lock().unwrap().payment_preimages.len(), 5);
		test_preimages_exist!(&preimages[0..5], monitor);
	}

	#[test]
	fn test_claim_txn_weight_computation() {
		// We test Claim txn weight, knowing that we want expected weigth and
		// not actual case to avoid sigs and time-lock delays hell variances.

		let secp_ctx = Secp256k1::new();
		let privkey = SecretKey::from_slice(&hex::decode("0101010101010101010101010101010101010101010101010101010101010101").unwrap()[..]).unwrap();
		let pubkey = PublicKey::from_secret_key(&secp_ctx, &privkey);

		macro_rules! sign_input {
			($sighash_parts: expr, $idx: expr, $amount: expr, $weight: expr, $sum_actual_sigs: expr, $opt_anchors: expr) => {
				let htlc = HTLCOutputInCommitment {
					offered: if *$weight == weight_revoked_offered_htlc($opt_anchors) || *$weight == weight_offered_htlc($opt_anchors) { true } else { false },
					amount_msat: 0,
					cltv_expiry: 2 << 16,
					payment_hash: PaymentHash([1; 32]),
					transaction_output_index: Some($idx as u32),
				};
				let redeem_script = if *$weight == WEIGHT_REVOKED_OUTPUT { chan_utils::get_revokeable_redeemscript(&pubkey, 256, &pubkey) } else { chan_utils::get_htlc_redeemscript_with_explicit_keys(&htlc, $opt_anchors, &pubkey, &pubkey, &pubkey) };
				let sighash = hash_to_message!(&$sighash_parts.signature_hash($idx, &redeem_script, $amount, SigHashType::All)[..]);
				let sig = secp_ctx.sign(&sighash, &privkey);
				$sighash_parts.access_witness($idx).push(sig.serialize_der().to_vec());
				$sighash_parts.access_witness($idx)[0].push(SigHashType::All as u8);
				$sum_actual_sigs += $sighash_parts.access_witness($idx)[0].len();
				if *$weight == WEIGHT_REVOKED_OUTPUT {
					$sighash_parts.access_witness($idx).push(vec!(1));
				} else if *$weight == weight_revoked_offered_htlc($opt_anchors) || *$weight == weight_revoked_received_htlc($opt_anchors) {
					$sighash_parts.access_witness($idx).push(pubkey.clone().serialize().to_vec());
				} else if *$weight == weight_received_htlc($opt_anchors) {
					$sighash_parts.access_witness($idx).push(vec![0]);
				} else {
					$sighash_parts.access_witness($idx).push(PaymentPreimage([1; 32]).0.to_vec());
				}
				$sighash_parts.access_witness($idx).push(redeem_script.into_bytes());
				println!("witness[0] {}", $sighash_parts.access_witness($idx)[0].len());
				println!("witness[1] {}", $sighash_parts.access_witness($idx)[1].len());
				println!("witness[2] {}", $sighash_parts.access_witness($idx)[2].len());
			}
		}

		let script_pubkey = Builder::new().push_opcode(opcodes::all::OP_RETURN).into_script();
		let txid = Txid::from_hex("56944c5d3f98413ef45cf54545538103cc9f298e0575820ad3591376e2e0f65d").unwrap();

		// Justice tx with 1 to_holder, 2 revoked offered HTLCs, 1 revoked received HTLCs
		for &opt_anchors in [false, true].iter() {
			let mut claim_tx = Transaction { version: 0, lock_time: 0, input: Vec::new(), output: Vec::new() };
			let mut sum_actual_sigs = 0;
			for i in 0..4 {
				claim_tx.input.push(TxIn {
					previous_output: BitcoinOutPoint {
						txid,
						vout: i,
					},
					script_sig: Script::new(),
					sequence: 0xfffffffd,
					witness: Vec::new(),
				});
			}
			claim_tx.output.push(TxOut {
				script_pubkey: script_pubkey.clone(),
				value: 0,
			});
			let base_weight = claim_tx.get_weight();
			let inputs_weight = vec![WEIGHT_REVOKED_OUTPUT, weight_revoked_offered_htlc(opt_anchors), weight_revoked_offered_htlc(opt_anchors), weight_revoked_received_htlc(opt_anchors)];
			let mut inputs_total_weight = 2; // count segwit flags
			{
				let mut sighash_parts = bip143::SigHashCache::new(&mut claim_tx);
				for (idx, inp) in inputs_weight.iter().enumerate() {
					sign_input!(sighash_parts, idx, 0, inp, sum_actual_sigs, opt_anchors);
					inputs_total_weight += inp;
				}
			}
			assert_eq!(base_weight + inputs_total_weight as usize,  claim_tx.get_weight() + /* max_length_sig */ (73 * inputs_weight.len() - sum_actual_sigs));
		}

		// Claim tx with 1 offered HTLCs, 3 received HTLCs
		for &opt_anchors in [false, true].iter() {
			let mut claim_tx = Transaction { version: 0, lock_time: 0, input: Vec::new(), output: Vec::new() };
			let mut sum_actual_sigs = 0;
			for i in 0..4 {
				claim_tx.input.push(TxIn {
					previous_output: BitcoinOutPoint {
						txid,
						vout: i,
					},
					script_sig: Script::new(),
					sequence: 0xfffffffd,
					witness: Vec::new(),
				});
			}
			claim_tx.output.push(TxOut {
				script_pubkey: script_pubkey.clone(),
				value: 0,
			});
			let base_weight = claim_tx.get_weight();
			let inputs_weight = vec![weight_offered_htlc(opt_anchors), weight_received_htlc(opt_anchors), weight_received_htlc(opt_anchors), weight_received_htlc(opt_anchors)];
			let mut inputs_total_weight = 2; // count segwit flags
			{
				let mut sighash_parts = bip143::SigHashCache::new(&mut claim_tx);
				for (idx, inp) in inputs_weight.iter().enumerate() {
					sign_input!(sighash_parts, idx, 0, inp, sum_actual_sigs, opt_anchors);
					inputs_total_weight += inp;
				}
			}
			assert_eq!(base_weight + inputs_total_weight as usize,  claim_tx.get_weight() + /* max_length_sig */ (73 * inputs_weight.len() - sum_actual_sigs));
		}

		// Justice tx with 1 revoked HTLC-Success tx output
		for &opt_anchors in [false, true].iter() {
			let mut claim_tx = Transaction { version: 0, lock_time: 0, input: Vec::new(), output: Vec::new() };
			let mut sum_actual_sigs = 0;
			claim_tx.input.push(TxIn {
				previous_output: BitcoinOutPoint {
					txid,
					vout: 0,
				},
				script_sig: Script::new(),
				sequence: 0xfffffffd,
				witness: Vec::new(),
			});
			claim_tx.output.push(TxOut {
				script_pubkey: script_pubkey.clone(),
				value: 0,
			});
			let base_weight = claim_tx.get_weight();
			let inputs_weight = vec![WEIGHT_REVOKED_OUTPUT];
			let mut inputs_total_weight = 2; // count segwit flags
			{
				let mut sighash_parts = bip143::SigHashCache::new(&mut claim_tx);
				for (idx, inp) in inputs_weight.iter().enumerate() {
					sign_input!(sighash_parts, idx, 0, inp, sum_actual_sigs, opt_anchors);
					inputs_total_weight += inp;
				}
			}
			assert_eq!(base_weight + inputs_total_weight as usize, claim_tx.get_weight() + /* max_length_isg */ (73 * inputs_weight.len() - sum_actual_sigs));
		}
	}

	// Further testing is done in the ChannelManager integration tests.
}
