use conv::ConvUtil;
use criterion::{black_box, Criterion};
use light_curve_feature::*;
use rand::prelude::*;
use rand_distr::StandardNormal;
use std::any::type_name;

pub fn bench_extractor<T>(c: &mut Criterion)
where
    T: Float + 'static,
    StandardNormal: Distribution<T>,
{
    const N: [usize; 2] = [100, 1000];

    let features: Vec<Feature<_>> = vec![
        Amplitude::default().into(),
        AndersonDarlingNormal::default().into(),
        BeyondNStd::default().into(),
        Cusum::default().into(),
        Eta::default().into(),
        EtaE::default().into(),
        ExcessVariance::default().into(),
        InterPercentileRange::default().into(),
        Kurtosis::default().into(),
        LinearFit::default().into(),
        LinearTrend::default().into(),
        MagnitudePercentageRatio::default().into(),
        MaximumSlope::default().into(),
        Mean::default().into(),
        MeanVariance::default().into(),
        Median::default().into(),
        MedianAbsoluteDeviation::default().into(),
        MedianBufferRangePercentage::default().into(),
        PercentAmplitude::default().into(),
        PercentDifferenceMagnitudePercentile::default().into(),
        ReducedChi2::default().into(),
        Skew::default().into(),
        StandardDeviation::default().into(),
        StetsonK::default().into(),
        WeightedMean::default().into(),
    ];

    let observation_count_vec: Vec<_> = (0..20)
        .map(|_| antifeatures::ObservationCount::default().into())
        .collect();

    let beyond_n_std_vec: Vec<_> = (1usize..21)
        .map(|i| BeyondNStd::new(i.value_as::<T>().unwrap() / T::ten()).into())
        .collect();

    let mut bins = Bins::default();
    bins.add_feature(StetsonK::default().into());

    let mut periodogram = Periodogram::default();
    periodogram.set_max_freq_factor(10.0);

    let names_fes: Vec<_> = features
        .iter()
        .map(|f| (f.get_names()[0], FeatureExtractor::new(vec![f.clone()])))
        .chain(std::iter::once((
            "all non-meta features",
            FeatureExtractor::new(features.clone()),
        )))
        .chain(std::iter::once((
            "multiple ObservationCount",
            FeatureExtractor::new(observation_count_vec.clone()),
        )))
        .chain(std::iter::once((
            "multiple BeyondNStd",
            FeatureExtractor::new(beyond_n_std_vec),
        )))
        .chain(std::iter::once((
            "Bins",
            FeatureExtractor::new(vec![bins.into()]),
        )))
        .chain(std::iter::once((
            "Periodogram",
            FeatureExtractor::new(vec![periodogram.into()]),
        )))
        .collect();

    for &n in N.iter() {
        let x = randspace(n);
        let y = randvec(n);
        let err = randvec(n);
        for (name, fe) in names_fes.iter() {
            c.bench_function(
                format!("FeatureExtractor {}: [{}; {}]", name, n, type_name::<T>()).as_str(),
                |b| {
                    b.iter(|| {
                        run(black_box(fe), black_box(&x), black_box(&y), black_box(&err)).unwrap();
                    });
                },
            );
        }
    }

    {
        let n = 10;
        let x = randspace(n);
        let y = randvec(n);
        let mut ts = TimeSeries::new_without_weight(&x, &y);
        let fe = FeatureExtractor::new(observation_count_vec);
        c.bench_function(
            format!("Multiple ObservationCount {}", type_name::<T>()).as_str(),
            |b| {
                b.iter(|| {
                    fe.eval(black_box(&mut ts)).unwrap();
                });
            },
        );
    }
}

fn run<T: Float>(
    fe: &FeatureExtractor<T, Feature<T>>,
    x: &[T],
    y: &[T],
    err: &[T],
) -> Result<Vec<T>, EvaluatorError> {
    let w: Vec<_> = err.iter().map(|&e| e.powi(-2)).collect();
    let mut ts = TimeSeries::new(&x, &y, &w);
    fe.eval(&mut ts)
}

fn randvec<T>(n: usize) -> Vec<T>
where
    T: Float,
    StandardNormal: Distribution<T>,
{
    (0..n)
        .map(|_| {
            let x: T = thread_rng().sample(StandardNormal);
            x
        })
        .collect()
}

fn randspace<T>(n: usize) -> Vec<T>
where
    T: Float,
    StandardNormal: Distribution<T>,
{
    let mut x = randvec::<T>(n);
    x[..].sort_unstable_by(|a, b| a.partial_cmp(b).unwrap());
    x
}
