use crate::{build_log_config, log_level_converter};
use ini::Ini;
use local_ipaddress;

pub struct ZymodConfig {
    pub mod_name: String,
    pub active: bool,
    pub agent_host: String,
    pub agent_port: u16,
    pub debug: u8,
    pub dry_run: i32,
    pub interval: u16,
}

#[allow(unused_must_use)]
pub fn load_zymod_config_parser(filename: &str, is_dry_run: i32, is_verbose: i32) -> ZymodConfig {
    let conf = Ini::load_from_file(filename).unwrap();

    let section = conf.section(Some("zymod")).expect("配置文件缺少section");

    let _agent_host = if section.get("agent_host").unwrap().to_string() == "localhost".to_string() {
        format!("http://{}", local_ipaddress::get().unwrap())
    } else {
        section.get("agent_host").unwrap().to_string()
    };

    let _debug = if is_verbose == -1 {
        section.get("debug").unwrap().parse::<u8>().unwrap()
    } else {
        4
    };

    let _dry_run = if is_dry_run == -1 {
        section.get("dry_run").unwrap().parse::<i32>().unwrap()
    } else {
        1
    };

    let conf_info = ZymodConfig {
        active: section.get("active").unwrap().parse::<bool>().unwrap(),
        agent_host: _agent_host,
        agent_port: section.get("agent_port").unwrap().parse::<u16>().unwrap(),
        debug: _debug,
        dry_run: _dry_run,
        interval: section.get("interval").unwrap().parse::<u16>().unwrap(),
        mod_name: section.get("mod_name").unwrap().to_string(),
    };

    let debug_value_to_log_level = log_level_converter(conf_info.debug as i8);

    let log_conf = build_log_config(debug_value_to_log_level);

    log4rs::init_config(log_conf);
    return conf_info;
}
