use std::error::Error;
use std::process::exit;
use std::{fs, thread, time::Duration};

use clap::{App, Arg, ArgMatches};
use log::*;
use log4rs::append::console::ConsoleAppender;
use log4rs::config::{Appender, Root};
use log4rs::encode::pattern::PatternEncoder;
use log4rs::Config;
use signal_hook::iterator::Signals;
use tokio::runtime::Builder;
use tonic::Request;

use zhiyan_rpc::zhi_yan_client::ZhiYanClient;
use zhiyan_rpc::ZhiYanRequest;

use crate::zymod_config::ZymodConfig;

mod zhiyan_rpc;
mod zymod_config;

pub mod floatutils;
pub mod stringutils;

pub fn string_to_f32_converter(value: &str) -> f32 {
    let change_value = (value).replace("\n", "").parse::<f32>().unwrap();
    return change_value;
}

pub fn log_level_converter(log_level: i8) -> LevelFilter {
    let number = log_level;
    match number {
        1 => LevelFilter::Error,
        2 => LevelFilter::Warn,
        3 => LevelFilter::Info,
        4 => LevelFilter::Debug,
        5 => LevelFilter::Trace,
        _ => LevelFilter::Debug,
    }
}

pub fn build_log_config(log_stdout_level: LevelFilter) -> Config {
    let stdout = ConsoleAppender::builder()
        .encoder(Box::new(PatternEncoder::new(
            "{d(%Y-%m-%d %H:%M:%S)} - {t} -{l} - {m}{n}",
        )))
        .build();

    let config = Config::builder()
        .appender(Appender::builder().build("stdout", Box::new(stdout)))
        .build(
            Root::builder()
                .appender("stdout")
                .build(log_stdout_level.clone()),
        )
        .unwrap();

    return config;
}

pub fn interrupt_from_keyboard_handler(signum: i32) -> Result<(), Box<dyn Error>> {
    let mut signals = Signals::new(&[signum])?;

    thread::spawn(move || {
        for _sig in signals.forever() {
            println!("检测到用户发送终止信号，退出程序中......");
            exit(1)
        }
    });

    Ok(())
}

pub fn path_exists(path: &str) -> bool {
    fs::metadata(path).is_ok()
}

pub struct Zymod {
    mod_conf: ZymodConfig,
}

impl Zymod {
    pub fn new(
        mod_conf_path: &str,
        is_dry_run_from_cmd_args: i32,
        is_verbose_from_cmd_args: i32,
    ) -> Self {
        if !path_exists(&*mod_conf_path) {
            log::error!("智眼模块配置文件:{}不存在", mod_conf_path);
            exit(1)
        }

        //-1:None 0:False 1:True
        let is_dry_run = if is_dry_run_from_cmd_args == 1 { 1 } else { -1 };
        let is_verbose = if is_verbose_from_cmd_args == 1 { 1 } else { -1 };

        return Zymod {
            mod_conf: zymod_config::load_zymod_config_parser(mod_conf_path, is_dry_run, is_verbose),
        };
    }

    pub fn build_help_parser(
        prog: &str,
        description: &'static str,
        version: &'static str,
        mod_config_file_path: &'static str,
        log_config_file_path: &'static str,
    ) -> App<'static, 'static> {
        let matches = App::new(prog)
            .version(version)
            .about(description)
            .arg(
                Arg::with_name("mod_conf")
                    .short("c")
                    .long("conf")
                    .help("指定智眼模块配置文件")
                    .default_value(mod_config_file_path)
                    .takes_value(true),
            )
            .arg(
                Arg::with_name("dry_run")
                    .short("n")
                    .long("dry-run")
                    .help("在不做任何更改的情况下试运行，通常和'-v'参数一起使用(1:True,0:False)")
                    .default_value("0")
                    .takes_value(true),
            )
            .arg(
                Arg::with_name("verbose")
                    .short("v")
                    .long("verbose")
                    .help("显示详细信息(1:True,0:False)")
                    .default_value("0")
                    .takes_value(true),
            )
            .arg(
                Arg::with_name("log")
                    .short("l")
                    .long("log_config")
                    .help("指定智眼CPU模块日志配置文件")
                    .default_value(log_config_file_path)
                    .takes_value(true),
            );

        return matches;
    }

    pub fn build_help(
        prog: &str,
        description: &'static str,
        version: &'static str,
        mod_config_file_path: &'static str,
        log_config_file_path: &'static str,
    ) -> ArgMatches<'static> {
        let parser = App::from(Zymod::build_help_parser(
            prog,
            description,
            version,
            mod_config_file_path,
            log_config_file_path,
        ));
        return parser.get_matches();
    }

    pub fn build_help_with_parser(parser: App<'static, 'static>) -> ArgMatches<'static> {
        let parser = App::from(parser);
        return parser.get_matches();
    }

    pub fn upload(&self, name: &str, datetime: i64, content: String) {
        log::info!("Name:{}", name);
        log::info!("Datetime:{}", datetime);
        log::info!("Content:{}", content);

        if self.mod_conf.dry_run == 1 {
            log::info!("zymod：在不做任何更改的情况下试运行.....");
        } else {
            log::info!("zymod：正在上传到agent.....");
            let run_time = Builder::new_multi_thread().enable_all().build().unwrap();
            let agent_address = format!(
                "{}:{}",
                self.mod_conf.agent_host,
                self.mod_conf.agent_port.to_string()
            );
            log::info!("zymod：agentURL:{}", agent_address);

            run_time.block_on(async {
                let mut client = ZhiYanClient::connect(agent_address).await.unwrap();

                let request = Request::new(ZhiYanRequest {
                    name: name.to_string(),
                    datetime: datetime.clone(),
                    content: content.clone(),
                });

                let response = client.zymod(request).await.unwrap();
                let upload_response = response.get_ref();
                log::info!(
                    "Code:{:?},Messages{:?}",
                    upload_response.code,
                    upload_response.message
                );
            });
        }
        thread::sleep(Duration::from_millis(self.mod_conf.interval.clone() as u64));
        return;
    }
}
