//Copyright (c) 2020-2022 Stefan Thesing
//
//This file is part of libzettels.
//
//libzettels is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//libzettels is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with libzettels. If not, see http://www.gnu.org/licenses/.

//! Module regarding the content of the Zettelkasten. All revolves around
//! the struct `Zettel`.

// --------------------------------------------------------------------------
// Imports
// --------------------------------------------------------------------------

use std::io::{BufRead, BufReader};
use std::fmt::Write as FmtWrite; 
use std::fs::File;
use std::path::{Path, PathBuf};
use std::collections::HashSet;

use backstage::error::Error;

// From parent module
use backstage::indexing::normalize_path;
use backstage::indexing::normalize_link;

// --------------------------------------------------------------------------
// Zettel
// --------------------------------------------------------------------------

/// A struct containing relevant data about a Zettel. 
///
/// It is created by reading the necessary information from a zettel file. 
/// Libzettels assumes all zettel file contain a YAML-header containing all 
/// or most of this information.
/// 
/// Libzettels was designed to deal with two kinds of zettel files:
/// - [Markdown-based](#markdown-based-zettel-file) zettel files
/// - [Image-based](#image-based-zettel-file) zettel files
///
/// but it is able to handle a lot of different formats, as long as the 
/// YAML-header is present and readable (see README).
/// 
/// # Markdown-based zettel file
/// A markdown-based zettel file is a markdown file with a YAML-header as
/// defined by
/// [pandoc](https://pandoc.org/MANUAL.html#extension-yaml_metadata_block)).
/// Most of the information concerning the interrelations with other Zettels is
/// read from the YAML-header. An exception is the field `links`, which is
/// generated by parsing markdown links in the file (only of the 
/// ["inline"](https://daringfireball.net/projects/markdown/syntax#link)
/// syntax).  
/// The YAML-metadata may contain additional information (like author, e.g.).
/// However, such additional data is ignored, here.
/// ## Example
/// ```yaml,no_run
/// ---
/// title:  'Some Zettel'
/// keywords: [example]
/// followups: [file2.md, file3.md]
/// ...
///
/// Here begins the Zettel's actual content, possibly containing links to 
/// [other Zettels](file2.md). These links are used for internal links within 
/// the Zettelkasten. External links can be achieved by [reference style][id] 
/// links. These are ignored by the Zettelkasten. Lorem ipsum…
///
/// [id]: https://daringfireball.net/projects/markdown/syntax#link
/// ```
/// # Image-based zettel file
/// Image-based zettel files are a way to integrate scans of handwritten 
/// notes into the Zettelkasten. To do this, a user copies the 
/// image file to the root directory of the Zettelkasten and creates an
/// accompanying text 
/// file containing the YAML-metadata about the interrelation to other zettels
/// and some sort of reference to the image file. For example, such a text file
/// could be a markdown file with a YAML-header and an 
/// [image link](https://daringfireball.net/projects/markdown/syntax#img) to 
/// the image file. Because the `links` field can not be automatically filled
/// with meaningful data, it needs to be set in the YAML (or it will be a
/// empty list).
/// ## Example
/// ```yaml,no_run
/// ---
/// title:  'An image-based Zettel'
/// keywords: [example]
/// followups: [file2.md, file3.md]
/// links: [file1.md]
/// ...
/// ![](imagefile.png)
/// ```
/// See the project's README for details and other file formats as zettel files.
#[derive(Debug, PartialEq, Serialize, Deserialize, Clone)]
pub struct Zettel {
    /// The title of the Zettel, as set in its YAML-metadata.
    #[serde(default = "default_title")]
    pub title: String,
    /// Other Zettels this one is pointing to via Markdown-syntax.
    #[serde(default = "default_links")]
    pub links: Vec<PathBuf>,
    /// Other Zettels that are considered followups to this one, as set in 
    /// the YAML-metadata.
    #[serde(default = "default_links")]
    pub followups: Vec<PathBuf>,
    /// Keywords applied to this Zettel, as set in the YAML-metadata.
    #[serde(default = "default_keywords")]
    pub keywords: Vec<String>,
}

// Functions providing default values for serde, in case as Zettel file
// doesn't contain the respective field. This is regularly used for links,
// since those are not part of the YAML-Header.
fn default_title() -> String { String::from("untitled") }
fn default_links() -> Vec<PathBuf> { vec![] } // links and followups
fn default_keywords() -> Vec<String> { vec![] }

impl Zettel {
    /// ## Extended API
    /// Creates a new Zettel by specifying its title. All other fields are 
    /// empty lists.
    /// # Example
    /// ```
    /// # use libzettels::Zettel;
    /// let z = Zettel::new("Example Zettel");
    ///
    /// assert_eq!(&z.title, "Example Zettel");
    /// assert!(&z.links.is_empty());
    /// assert!(&z.followups.is_empty());
    /// assert!(&z.keywords.is_empty());
    /// ```
    pub fn new<T: AsRef<str>>(title: T) -> Zettel {
        let title = title.as_ref();
        Zettel {
            title: title.to_string(),
            links: vec![],
            followups: vec![],
            keywords: vec![],
        }
    }
    
    /// ## Extended API
    /// Creates a new Zettel by deserializing it from a YAML-string.
    /// It needs references to filename and root directory in order to normalize
    /// the followup-links.
    /// Note: This only uses the metadata specified in the YAML-header of 
    /// the Zettel. Thus, the field `links` is not populated. For that,
    /// the markdown links need to be parsed.
    /// # Example
    /// ```rust, no_run
    /// # use libzettels::Zettel;
    /// # use std::path::Path;
    /// let rootdir = Path::new("examples/Zettelkasten");
    /// let zettelfile = rootdir.join("file1.md");
    /// let yaml = "---
    /// title: 'A Zettel'
    /// keywords: [example, yaml]
    /// followups: [file2.md]
    /// ...
    ///
    /// Here be contents.";
    /// let z = Zettel::from_yaml(yaml, rootdir, zettelfile.as_ref())
    ///         .expect("Something went wrong.");
    /// ```
    /// # Errors
    /// - [`Error::BadLink`](enum.Error.html#variant.BadLink) if an entry in
    ///   `followups` or `links` links to a file that doesn't exist
    ///   (wrapping an `std::io::Error` of the `NotFound` kind).
    /// - [`Error::Io`](enum.Error.html#variant.Io) wrapping several kinds of 
    ///   `std::io:Error`.
    /// - [`Error::NormalizePath`](enum.Error.html#variant.NormalizePath) if 
    ///   `zettelfile` or a link in `followups` or `links`can not be expressed 
    ///   relative to the root directory.
    /// - [`Error::BadHeader`](enum.Error.html#variant.BadHeader) when deserializing the
    ///   Zettel from YAML failed.
    pub fn from_yaml<P: AsRef<Path>>(yaml: &str, rootdir: P, zettelfile: P) 
        -> Result<Zettel, Error> {
        let rootdir = rootdir.as_ref();
        let zettelfile = zettelfile.as_ref();
        let zettelfile = normalize_path(rootdir, zettelfile)?;       //error::Error
        
        // just to be sure, just use the part of the str up to the first 
        // occurence of "..."
        let s: Vec<&str> = yaml.split("...").collect();
        let s = s[0];
        
        // Deserialize the Zettel
        let mut z: Zettel = match serde_yaml::from_str(s) {
            Ok(zettel) => Ok(zettel),
            // Wrap the YAML error into a BadHeader error
            Err(yaml_e) => Err(
                Error::BadHeader(
                    zettelfile.clone().to_path_buf(), 
                    yaml_e
                )
            ),
        }?;                                                 // BadHeader
        
        // The links in "followups" are relative to the file. They need to 
        // be normalized for the index.
        let mut normalized_followups = vec![];
        trace!("Normalizing followups of Zettel {:?}", zettelfile);
        for followup in z.followups.drain(..) {
            let nf = normalize_link(rootdir,
                                    &zettelfile,
                                    &followup)?;                 //error::Error 
            normalized_followups.push(nf);
        }
        z.followups = normalized_followups;        
        Ok(z)
    }
    
    /// ## Extended API
    /// Creates a new Zettel by deserializing it from a YAML-file.
    /// It needs a reference to the root directory in order to normalize
    /// the followup-links.
    /// # Example
    /// ```rust, no_run
    /// # use libzettels::Zettel;
    /// # use std::path::Path;
    /// let rootdir = Path::new("examples/Zettelkasten");
    /// let zettelfile = rootdir.join("file1.md");
    /// let z = Zettel::from_file(zettelfile.as_ref(), rootdir)
    ///         .expect("Something went wrong.");
    /// ```
    /// # Errors
    /// - [`Error::BadLink`](enum.Error.html#variant.BadLink) if an entry in
    ///   `followups` links to a file that doesn't exist 
    ///   (wrapping an `std::io::Error` of the `NotFound` kind).
    /// - [`Error::Io`](enum.Error.html#variant.Io) wrapping various kinds of 
    ///   `std::io::Error`. Most notably of the `InvalidData` kind, if the file
    ///   is a non-text file (e.g. an image).
    /// - [`Error::NormalizePath`](enum.Error.html#variant.NormalizePath) if 
    ///   `zettelfile` or a link in `followups``can not be expressed 
    ///   relative to the root directory.
    /// - [`Error::BadHeader`](enum.Error.html#variant.BadHeader) when deserializing the
    ///   Zettel from YAML failed.
    pub fn from_file<P: AsRef<Path>>(zettelfile: P, rootdir: P) 
            -> Result<Zettel, Error> {
        let zettelfile = zettelfile.as_ref();
        let rootdir = rootdir.as_ref();
        let file = File::open(zettelfile)?;  //io::Error
        // If I see this correctly, two types of `serde_yaml::Error`s 
        // should be prevented, here:
        // - serde_yaml::error::MoreThanOneDocument
        //   When yaml frontmatter doesn't end with "..." but with
        //   "---".
        // - serde_yaml::error::Message
        //   When the file doesn't contain any valid yaml. Most likely
        //   pure markdown.
        // For that reason, I only pass the first found yaml document to 
        // the from_yaml-Function.
        
        // Extract the yaml-metadata.
        let contents = get_first_yaml_document(file)?;  //io::Error
                                                        //InvalidData
        match contents {
            // No Yaml Frontmatter in file, return an empty Zettel
            None    => Ok(Zettel::new("untitled")), 
            // Pass the yaml on.
            Some(s) => Zettel::from_yaml(&s, rootdir, zettelfile), //error::Error
        }
    }
    
    /// ## Extended API
    /// Adds a link (specified by its path relative to the root directory) 
    /// to the Zettel. If the link is already present, it is not added a
    /// second time.
    /// # Example
    /// ```
    /// # use libzettels::Zettel;
    /// # use std::path::Path;
    /// let mut z = Zettel::new("Example Zettel");
    /// z.add_link(Path::new("anotherfile.md"));
    ///
    /// assert_eq!(&z.links, &vec![Path::new("anotherfile.md")]);
    /// ```
    pub fn add_link<P: AsRef<Path>>(&mut self, link: P) {
        let link = link.as_ref().to_path_buf();
        if !self.links.contains(&link) {
            self.links.push(link);
        }
    }
    
    /// ## Extended API
    /// Adds a followup (specified by its path relative to the root directory) 
    /// to the Zettel. If the followup is already present, it is not added a
    /// second time.
    /// # Example
    /// ```
    /// # use libzettels::Zettel;
    /// # use std::path::Path;
    /// let mut z = Zettel::new("Example Zettel");
    /// z.add_followup(Path::new("anotherfile.md"));
    ///
    /// assert_eq!(&z.followups, &vec![Path::new("anotherfile.md")]);
    /// ```
    pub fn add_followup<P: AsRef<Path>>(&mut self, followup: P) {
        let followup = followup.as_ref().to_path_buf();
        if !self.followups.contains(&followup) {
            self.followups.push(followup);
        }
    }
    
    /// ## Extended API
    /// Adds a keyword to the Zettel. If the keyword is already present, it is not 
    /// added a second time.
    /// # Example
    /// ```
    /// # use libzettels::Zettel;
    /// let mut z = Zettel::new("Example Zettel");
    /// z.add_keyword("foo");
    ///
    /// assert_eq!(&z.keywords, &vec![String::from("foo")]);
    /// ```
    pub fn add_keyword<T: AsRef<str>>(&mut self, keyword: T) {
        let keyword = keyword.as_ref().to_string();
        if !self.keywords.contains(&keyword) {
            self.keywords.push(keyword);
        }
    }
    
    /// ## Extended API
    /// Checks whether or not the zettel links to one other zettel
    /// out of a list specified by `searched_links` (or to all of them, if the
    /// parameter `all` is true).
    /// # Example
    /// ```
    /// # use libzettels::Zettel;
    /// # use std::path::PathBuf;
    /// # use std::collections::HashSet;
    /// let mut z = Zettel::new("Example Zettel");
    /// z.add_link(PathBuf::from("file1.md"));
    /// let mut searched_links = HashSet::new();
    /// searched_links.insert(PathBuf::from("file1.md"));
    /// searched_links.insert(PathBuf::from("file2.md"));
    /// 
    /// assert!(z.links_to(&searched_links, false));
    /// assert!(!z.links_to(&searched_links, true));
    /// ```
    pub fn links_to(&self, searched_links: &HashSet<PathBuf>, all: bool) 
                -> bool {
        if all {
            // All of searched_links need to be in zettel.links
            for link in searched_links {
                // so if it is NOT in, return false
                if !self.links.contains(&link) {
                    return false;
                }
            }
            // still here? So all are in.
            return true;
        } else {
            // Just one of the searched_links need to be in zettel.links
            for link in searched_links {
                // so if it is in, return true
                if self.links.contains(&link) {
                    return true;
                }
            }
            // No match, so:
            return false;
        }
    }
}

/// Reads the first YAML document in the opened file handle.
/// Returns `None` if the files contains no YAML.
/// # Error
/// - [`std::io::Error`](https://doc.rust-lang.org/std/io/struct.Error.html)
///   of the `InvalidData` kind if the file is non-text (e.g. an image) or 
///   an SVG file.
fn get_first_yaml_document(opened_zettelfile: File) 
                                    -> Result<Option<String>, std::io::Error> {
    let buf_reader = BufReader::new(opened_zettelfile);
    let mut within_yaml = false; // A sort of punchcard, see below
    let mut s = String::new();
    
    // Preparation for checking for svg files
    let mut xml = false;
    let mut linecount = 0;
    
    for line in buf_reader.lines() {
        // If we fail to read lines, the opened_zettelfile is probably not text,
        // but some other so called binary type, like an image e.g.
        // If so, we propagate the error.
        let line = match line {
            Ok(l) => l,
            Err(e) => return Err(e),                        //std::io::Error
                                                            //InvalidData   
        };
        
        // Check for svg files
        // Is it xml?
        linecount += 1;
        if linecount == 1 && line.trim().starts_with("<?xml") {
            xml = true;
        }
        // also svg?
        if linecount > 1 && xml && line.trim().starts_with("<svg") {
            //if so, do the same we'd to for other image formats:
            return Err(
                std::io::Error::new(std::io::ErrorKind::InvalidData, 
                "File is SVG"));
        }
        
        if line == "---" && !within_yaml { // Haven't punched in yet
            // Should be safe to use expect, here...
            writeln!(&mut s, "{}", line)
                .expect("Failed to write the line we just read from a text\
                         file to a string.");
            within_yaml = true;            // Punch in. We're inside the yaml  
        } else if within_yaml && (line == "..." || line == "---") {
            // We've reached the end of the yaml. Let's stop.
            break;
        } else if within_yaml {
            // we're inside the yaml. Let's record the current line.
            writeln!(&mut s, "{}", line)
                .expect("Failed to write the line we just read from a text\
                         file to a string.");
        }
    }
    match s.len() {
        0 => Ok(None),
        _ => Ok(Some(s)),
    }
}


// --------------------------------------------------------------------------
#[cfg(test)]
mod tests {
    extern crate tempfile;
    use self::tempfile::tempdir;
    use super::*;
    use examples::*;
    
    #[test]
    fn test_default_title() {
        assert_eq!(default_title(), "untitled");
    }
    
    #[test]
    fn test_default_links() {
        assert!(default_links().is_empty());
        let mut d = default_links();
        d.push(PathBuf::from("dummy.txt"));
    }
    
    #[test]
    fn test_default_keywords() {
        assert!(default_keywords().is_empty());
        let mut d = default_keywords();
        d.push("dummy.txt".to_string());
    }
    
    
    #[test]
    fn test_zettel_new() {
        let z = Zettel::new("Example Zettel");
        assert_eq!(&z.title, &String::from("Example Zettel"));
        assert!(&z.links.is_empty());
        assert!(&z.followups.is_empty());
        assert!(&z.keywords.is_empty());
    }
    
    #[test]
    fn test_zettel_from_yaml() {
        let tmp_dir = tempdir().expect("Failed to setup temp dir");
        let dir = tmp_dir.path();
        generate_bare_examples(dir).expect("Failed to generate examples");
        let rootdir = dir.join("examples/Zettelkasten");
        let zettelfile = rootdir.join("file1.md");
        let yaml = "---
title: 'A Zettel'
keywords: [example, yaml]
followups: [file2.md]
...

Here be contents.";
        let z = Zettel::from_yaml(yaml, rootdir, zettelfile);
        assert!(z.is_ok());
    }
    
    #[test]
    fn test_zettel_from_file() {
        let tmp_dir = tempdir().expect("Failed to setup temp dir");
        let dir = tmp_dir.path();
        generate_bare_examples(dir).expect("Failed to generate examples");
        let rootdir = dir.join("examples/Zettelkasten");
        let zettelfile = rootdir.join("file1.md");
        let z = Zettel::from_file(zettelfile, rootdir);
        assert!(z.is_ok());
    }
    
    #[test]
    fn test_zettel_add_link() {
        let mut z = Zettel::new("Example Zettel");
        z.add_link(PathBuf::from("anotherfile.md"));
        
        assert_eq!(&z.links, &vec![PathBuf::from("anotherfile.md")]);
    }
    
    #[test]
    fn test_test_add_double_link() {
        let mut z = Zettel::new("Example Zettel");
        z.add_link(PathBuf::from("anotherfile.md"));
        // and again
        z.add_link(PathBuf::from("anotherfile.md"));
        // There still should be just one link
        assert_eq!(z.links.len(), 1);
    }
    
    
    #[test]
    fn test_zettel_add_followup() {
        let mut z = Zettel::new("Example Zettel");
        z.add_followup(PathBuf::from("anotherfile.md"));
        
        assert_eq!(&z.followups, &vec![PathBuf::from("anotherfile.md")]);
    }
    
    #[test]
    fn test_zettel_add_double_followup() {
        let mut z = Zettel::new("Example Zettel");
        z.add_followup(PathBuf::from("anotherfile.md"));
        z.add_followup(PathBuf::from("anotherfile.md"));
        assert_eq!(z.followups.len(), 1);
    }
    
    #[test]
    fn test_zettel_add_keyword() {
        let mut z = Zettel::new("Example Zettel");
        z.add_keyword("foo");
        
        assert_eq!(&z.keywords, &vec![String::from("foo")]);
    }
    
    #[test]
    fn test_zettel_add_double_keyword() {
        let mut z = Zettel::new("Example Zettel");
        z.add_keyword("foo");
        z.add_keyword("foo");
        
        assert_eq!(z.keywords.len(), 1);
    }
    
    #[test]
    fn test_get_first_yaml_document() {
        use std::io::Write;
        // Setup
        let tmp_dir = tempdir().expect("Failed to create tempdir.");
        let mut f1 = File::create(tmp_dir.path().join("file1.md"))
            .expect("Something went wrong with creating temporary file1 for 
            testing.");
        let mut f2 = File::create(tmp_dir.path().join("file2.md"))
            .expect("Something went wrong with creating temporary file2 for 
            testing.");
        let mut f3 = File::create(tmp_dir.path().join("file3.md"))
            .expect("Something went wrong with creating temporary file3 for 
            testing.");
        let mut f4 = File::create(tmp_dir.path().join("file4.md"))
            .expect("Something went wrong with creating temporary file4 for 
            testing.");
        writeln!(f1, "{}", "---
foo: bar
...
Lorem ipsum --- dolor
--- sit amet,
consectetur ... adipiscing
... elit, sed
---
...
").expect("Failed to write to file");
        writeln!(f2, "{}", "---
foo: bar
---
Lorem ipsum --- dolor
--- sit amet,
consectetur ... adipiscing
... elit, sed
---
...
").expect("Failed to write to file");
        writeln!(f3, "{}", "Lorem ipsum
---
foo: bar
---
Lorem ipsum --- dolor
--- sit amet,
consectetur ... adipiscing
... elit, sed
---
...
").expect("Failed to write to file");
        writeln!(f4, "{}", "Lorem ipsum dolor sit --- amet, consectetur adipiscing 
elit, sed do eiusmod tempor ... incididunt ut labore et dolore magna aliqua. Ut 
enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut .")
        .expect("Failed to write to file");
        let f1 = File::open(tmp_dir.path().join("file1.md"))
                .expect("Failed to open file.");
        let f2 = File::open(tmp_dir.path().join("file2.md"))
                .expect("Failed to open file.");
        let f3 = File::open(tmp_dir.path().join("file3.md"))
                .expect("Failed to open file.");
        let f4 = File::open(tmp_dir.path().join("file4.md"))
                .expect("Failed to open file.");
        // Test
        let s1 = get_first_yaml_document(f1)
            .expect("Failed to read from file f1");
        let s2 = get_first_yaml_document(f2)
            .expect("Failed to read from file f2");
        let s3 = get_first_yaml_document(f3)
            .expect("Failed to read from file f3");
        let s4 = get_first_yaml_document(f4)
            .expect("Failed to read from file f4");
        let s1 = s1.unwrap();
        let s2 = s2.unwrap();
        let s3 = s3.unwrap();
        assert_eq!(s1, "---\nfoo: bar\n");
        assert_eq!(s1, s2);
        assert_eq!(s1, s3);
        assert!(s4.is_none());
    }
    
    // ----------------------------------------------------------------------
    // invalid data aka error handling
    // ----------------------------------------------------------------------
    
    #[test]
    fn test_get_first_yaml_document_empty_file() {
        let tmp_dir = tempfile::tempdir()
            .expect("Failed to create tempdir");
        let path = tmp_dir.path().join("dummy.txt");
        std::fs::File::create(&path)
            .expect("Failed to create file.");
        let file = std::fs::File::open(&path)
            .expect("Failed to open file.");
        
        let outcome = get_first_yaml_document(file);
        assert!(outcome.is_ok());
        let outcome = outcome.unwrap();
        assert!(outcome.is_none());
    }
    
    #[test]
    fn test_get_first_yaml_document_image_file() {
        let tmp_dir = tempdir().expect("Failed to create tempdir");
        let image_file = tmp_dir.path().join("foo.png");
        // Generate an image
        let width: u32 = 10;
        let height: u32 = 10;
        let mut non_text = image::ImageBuffer::new(width, height);
        // color all pixels white
        for (_, _, pixel) in non_text.enumerate_pixels_mut() {
            *pixel = image::Rgb([255, 255, 255]);
        }
        // write it to where our confing file is supposed to be.
        non_text.save(&image_file).unwrap();

        let file = std::fs::File::open(&image_file)
            .expect("Failed to open file.");
        let outcome = get_first_yaml_document(file);
        assert!(outcome.is_err());
    }
    
    #[test]
    fn test_zettel_from_yaml_invalid_type() {
        let tmp_dir = tempdir().expect("Failed to setup temp dir");
        let dir = tmp_dir.path();
        generate_bare_examples(dir).expect("Failed to generate examples");
        let rootdir = dir.join("examples/Zettelkasten");
        let zettelfile = rootdir.join("file1.md");
        let yaml = "---
title: Test
keywords: 1
followups: [file2.md]
...

Here be contents.";
        let z = Zettel::from_yaml(yaml, rootdir, zettelfile);
        assert!(z.is_err());
        let e = z.unwrap_err();
        match e {
            Error::BadHeader(_, inner) => {
                let message = inner.to_string();
                assert!(message.contains("invalid type"));
            },
            _ => panic!("Expected a BadHeader error, got: {:#?}", e),
        }
    }
    
        #[test]
    fn test_zettel_from_yaml_invalid_followups() {
        let tmp_dir = tempdir().expect("Failed to setup temp dir");
        let dir = tmp_dir.path();
        generate_bare_examples(dir).expect("Failed to generate examples");
        let rootdir = dir.join("examples/Zettelkasten");
        let zettelfile = rootdir.join("file1.md");
        let yaml = "---
title: Test
keywords: []
followups: [file2.md,
file3.md]
...

Here be contents.";
        let z = Zettel::from_yaml(yaml, rootdir, zettelfile);
        assert!(z.is_err());
        let e = z.unwrap_err();
        match e {
            Error::BadHeader(_, inner) => {
                let message = inner.to_string();
                println!("{:?}", message);
                assert!(message.contains("simple key expected"));
            },
            _ => panic!("Expected a BadHeader error, got: {:#?}", e),
        }
    }

    #[test]
    fn test_zettel_from_yaml_duplicate_field() {
        let tmp_dir = tempdir().expect("Failed to setup temp dir");
        let dir = tmp_dir.path();
        generate_bare_examples(dir).expect("Failed to generate examples");
        let rootdir = dir.join("examples/Zettelkasten");
        let zettelfile = rootdir.join("file1.md");
        let yaml = "---
title: 'A Zettel'
title: 'A Zettel'
keywords: [example, yaml]
followups: [file2.md]
...

Here be contents.";
        let z = Zettel::from_yaml(yaml, rootdir, zettelfile);
        let e = z.unwrap_err();
        match e {
            Error::BadHeader(_, inner) => {
                let message = inner.to_string();
                assert!(message.contains("duplicate field `title`"));
            },
            _ => panic!("Expected a BadHeader error, got: {:#?}", e),
        }
    }
    
    #[test]
    fn test_zettel_from_file_non_existing_file() {
        let tmp_dir = tempdir().expect("Failed to setup temp dir");
        let dir = tmp_dir.path();
        generate_bare_examples(dir).expect("Failed to generate examples");
        let rootdir = dir.join("examples/Zettelkasten");
        let zettelfile = rootdir.join("foo.md"); //doesn't exist
        let z = Zettel::from_file(zettelfile, rootdir);
        assert!(z.is_err());
        let e = z.unwrap_err();
        match e {
            Error::Io(inner) => {
                match inner.kind() {
                    std::io::ErrorKind::NotFound => {
                        assert!(inner.to_string().contains("No such file or \
                                                            directory"));
                    },
                    _ => panic!("Expected a NotFound error, got: {:#?}", inner)
                }
            },
            _ => panic!("Expected a Io error, got: {:#?}", e),
        }
    }
    
    #[test]
    fn test_links_to() {
        let mut z = Zettel::new("Example Zettel");
        z.add_link(PathBuf::from("file1.md"));
        let mut searched_links = HashSet::new();
        searched_links.insert(PathBuf::from("file1.md"));
        searched_links.insert(PathBuf::from("file2.md"));
        
        assert!(z.links_to(&searched_links, false));
        assert!(!z.links_to(&searched_links, true));
    }
}