/*
    libxbku-common
    Copyright (C) 2021 AluminiumTech

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
 */

pub mod status;

pub enum KernelSupportType{
    LongTermSupport,
    ShortTermSupport,
    Custom
}

pub enum LinuxKernelType {
    Mainline,
    NonMainline,
    NonMainlineModified,
    Custom
}

/*
    An enum to represent different CPU Architectures.
 */
pub enum ProcessorArchitecture {
    X86,
    X64,
    ARM32,
    ARM32EL,
    ARM32HF,
    ARM64,
    S390X,
    IA32,
    IA64,
    PowerPC32,
    PowerPC64,
    MIPS32,
    MIPS64,
    Sparc64,
    RiscV64,
    Undefined,
    InvalidArchitecture
}

impl ProcessorArchitecture {
    pub fn convert_string_to_enum(input: &str) -> ProcessorArchitecture {

        /*      if input.to_string().to_lowercase().contains("ppc") && input.to_string().to_lowercase().contains("64")
              {
                   let input: &str = "ppc64";
              }
              else if input.to_string().to_lowercase().contains("ppc") && !input.to_string().to_lowercase().contains("64"){
                  let input: &str = "ppc32";
              }
      */
        return match input {
            "riscv64" => ProcessorArchitecture::RiscV64,
            "x86" | "i386" => ProcessorArchitecture::X86,
            "x64" | "amd64" | "x86-x64" => ProcessorArchitecture::X64,
            "ia32" => ProcessorArchitecture::IA32,
            "ia64"  => ProcessorArchitecture::IA64,
            "armel" | "arm32el" => ProcessorArchitecture::ARM32EL,
            "armv7" | "arm32" | "aarch32" | "armv7-a" | "armhf" => ProcessorArchitecture::ARM32HF,
            "aarch64" | "armv8" | "armv8-a" | "armv9" | "armv9-a" => ProcessorArchitecture::ARM64,
            "ppc32" | "ppc" | "powerpc32" => ProcessorArchitecture::PowerPC32,
            "ppc64" | "powerpc64" => ProcessorArchitecture::PowerPC64,
            "s390x" => ProcessorArchitecture::S390X,
            "mips" => ProcessorArchitecture::MIPS32,
            "mips64" => ProcessorArchitecture::MIPS64,
            "sparc64" => ProcessorArchitecture::Sparc64,
            _=> ProcessorArchitecture::Undefined
        }
    }

    /*
        Detects and returns the CPU Architecture of the machine where the code is being currently run on by comparing Rust's ARCH const.
     */
    pub fn get() -> ProcessorArchitecture {
        return self::ProcessorArchitecture::convert_string_to_enum(std::env::consts::ARCH);
    }
}