use super::HolidayState;
use crate::{places::Country, time::Time, Text};
use serde::Deserialize;

#[derive(Debug, Deserialize)]
/// A holiday event.
pub struct Holiday {
    /// Identifier for the holiday definition. Please note that this id
    /// is not unique, not even with a single year - the same holiday
    /// event may be returned multiple times because it is observed on a
    /// different day, or because it is scheduled in a different calendar
    /// (Hebrew or Muslim calendar) and happens multiple times within a
    /// Gregorian year. Use the uid attribute for purposes where you need a
    /// unique identifier.
    pub id: i32,

    /// Id for the shown holiday instance. The id is designed to be unique
    /// across all holiday instances generated by the timeanddate.com API
    /// services and respects different calendars and other reasons that
    /// may cause events to occur multiple times within one Gregorian year.
    pub uid: String,

    /// List of holiday/observance names.
    pub name: Vec<Text>,

    /// Date/time of the holiday instance. Most holidays do not have a specific
    /// time - in this case the time components will be skipped. Some
    /// special events like equinoxes and solstices include the exact time
    /// of the event as well, in this case the timestamp will be in local
    /// time zone (including time zone data) (countries with multiple
    /// timezones: local time in capital city).
    pub date: Time,

    /// Further information about the specific holiday. The Url points to
    /// the timeanddate.com web page.
    ///
    /// Example: <https://www.timeanddate.com/holidays/us/new-year-day>
    pub url: url::Url,

    /// Country of the holiday instance.
    pub country: Option<Country>,

    /// Summary of locations where this holiday instance is valid. Element is
    /// only present if the holiday instance does not affect the whole country.
    pub locations: Option<String>,

    /// States/subdivisions that are affected by this holiday instance. This
    /// element is only present if the holiday instance is not valid in the
    /// whole country.
    pub states: Option<Vec<HolidayState>>,

    /// List of holiday/observance descriptions.
    pub oneliner: Option<Vec<Text>>,

    /// Classification of the holiday. Most days have only one classification,
    /// but some have multiple types associated. This happens e.g. in
    /// conjunction with religious days that also are flag days.
    ///
    /// Example: National Holiday
    pub types: Option<Vec<String>>,
}
