/*
 * Copyright (c) 2020-2021 Thomas Kramer.
 *
 * This file is part of LibrEDA 
 * (see https://codeberg.org/libreda).
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */


#![cfg(test)]

use libreda_structural_verilog::{StructuralVerilogReader, StructuralVerilogWriter};
use libreda_db::prelude::{HierarchyBase, NetlistBase,
                          NetlistReader, NetlistWriter};

use std::fs::File;
use libreda_db::chip::Chip;

#[test]
fn test_read_netlist() {
    let library_reader = StructuralVerilogReader::new()
        .load_blackboxes(true);
    let reader = StructuralVerilogReader::new();

    let mut f_lib = File::open("./tests/test_data/standard_cells.v").unwrap();
    let mut f_netlist = File::open("./tests/test_data/my_chip_45_nl.v").unwrap();
    let mut netlist: Chip = library_reader.read_netlist(&mut f_lib).unwrap();

    let result = reader.read_into_netlist(&mut f_netlist, &mut netlist);

    assert!(result.is_ok())
}

#[test]
fn test_read_write_netlist() {
    let library_reader = StructuralVerilogReader::new()
        .load_blackboxes(true);
    let reader = StructuralVerilogReader::new();

    let mut f_lib = File::open("./tests/test_data/standard_cells.v").unwrap();
    let mut f_netlist = File::open("./tests/test_data/my_chip_45_nl.v").unwrap();
    let mut netlist: Chip = library_reader.read_netlist(&mut f_lib).unwrap();

    let result = reader.read_into_netlist(&mut f_netlist, &mut netlist);

    assert!(result.is_ok());

    let writer = StructuralVerilogWriter::new();
    let mut f_out = File::create("./tests/test_data/my_chip_45_nl_out.v").unwrap();
    let result = writer.write_netlist(&mut f_out, &netlist);

    assert!(result.is_ok());
}

#[test]
fn test_pin_reordering() {
    let library_reader = StructuralVerilogReader::new()
        .load_blackboxes(true);

    let reader = StructuralVerilogReader::new();

    let library_data = r#"
module INVX1 (Y, A);
    output Y;
    input A;

    INVX1 _0_ (
        .A(A),
        .Y(Y)
    );
endmodule
    "#;

    let data = r#"

module TOP();
    wire a;
    wire y;

    INVX1 inv_inst1 (
        .A(a),
        .Y(y)
    );
endmodule
    "#;
    let mut bytes = data.as_bytes();
    let mut library_bytes = library_data.as_bytes();

    let mut netlist: Chip = library_reader.read_netlist(&mut library_bytes).unwrap();

    // let mut netlist: Chip = reader.read_netlist(&mut (&mut library_bytes).chain(&mut bytes)).unwrap();

    reader.read_into_netlist(&mut bytes, &mut netlist ).unwrap();

    let top_cell = netlist.cell_by_name("TOP").unwrap();
    let inv_cell = netlist.cell_by_name("INVX1").unwrap();
    let inv_inst = netlist.cell_instance_by_name(&top_cell, "inv_inst1").unwrap();

    let pin_y = netlist.pin_by_name(&inv_cell, "Y").unwrap();

    assert_eq!(
        netlist.net_of_pin_instance(&netlist.pin_instance(&inv_inst, &pin_y)),
        netlist.net_by_name(&top_cell, "y")
    )
}


#[test]
fn test_assign_after_assign() {
    let library_reader = StructuralVerilogReader::new()
        .load_blackboxes(true);

    let reader = StructuralVerilogReader::new();

    let library_data = r#"
module INVX1 (Y, A);
    output Y;
    input A;

    INVX1 _0_ (
        .A(A),
        .Y(Y)
    );
endmodule
    "#;

    let data = r#"

module TOP();
    wire a;
    wire y;
    wire b;
    wire c;

    INVX1 inv_inst1 (
        .A(b),
        .Y(y)
    );

    assign c = a;
    assign b = c;
endmodule
    "#;
    let mut bytes = data.as_bytes();
    let mut library_bytes = library_data.as_bytes();

    let mut netlist: Chip = library_reader.read_netlist(&mut library_bytes).unwrap();

    reader.read_into_netlist(&mut bytes, &mut netlist ).unwrap();

    let top_cell = netlist.cell_by_name("TOP").unwrap();
    let inv_cell = netlist.cell_by_name("INVX1").unwrap();
    let inv_inst = netlist.cell_instance_by_name(&top_cell, "inv_inst1").unwrap();

    let pin_a = netlist.pin_by_name(&inv_cell, "A").unwrap();

    assert_eq!(
        netlist.net_of_pin_instance(&netlist.pin_instance(&inv_inst, &pin_a)),
        netlist.net_by_name(&top_cell, "a")
    )
}