// This file is part of librador-rs, a project to provide a safe Rust API
// for the EspoTek Labrador electronics lab board.
//
// Copyright 2021 Andrew Dona-Couch
//
// librador-rs is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// librador-rs is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//! An arbitrary waveform generator.
//!
//! The Labrador features two channels of a signal generator operating at
//! 1Msps.  Get a handle on one of the channels from your [`Labrador`]
//! instance with one of the `signal_generator_ch<n>` methods, then start
//! outputting a waveform.  Sine, square, triangle, and sawtooth waves
//! are provided by the convenience functions `send_<type>_wave`.  You can
//! configure an arbitrary waveform by passing samples in to `send_custom_wave`.
//!
//! # Example
//!
//! ```
//! # use librador::Labrador;
//! #
//! let labrador = match Labrador::find() {
//!     Err(_) => panic!("Unable to find Labrador..."),
//!     Ok(l) => l,
//! };
//!
//! let res = labrador.signal_generator_ch1().send_sine_wave(440., 2., 0.);
//! if let Err(_) = res {
//!     println!("Unable to send sine wave!");
//! }
//! ```
//!
//! [`Labrador`]: ../struct.Labrador.html

/// A handle on a signal generator channel.
///
/// Send a standard wave shape with `send_sine_wave` and the like.  Pass in
/// a buffer of samples to `send_custom_wave` to produce your own shape.
/// See the [module docs] for more information.
///
/// [module docs]: index.html
pub struct SignalGenerator<Ch> {
    _data: std::marker::PhantomData<Ch>,
}

impl<Ch: crate::Channel> SignalGenerator<Ch> {
    pub(crate) fn new() -> Self {
        SignalGenerator {
            _data: std::marker::PhantomData,
        }
    }

    /// Configure a signal generator channel to produce a sine wave.
    ///
    /// The given channel will start to produce a sine wave with the given `frequency`.
    ///
    /// The signal will oscillate between `offset` volts and `offset + amplitude` volts.
    /// The signal generator's amplifier uses the configurable power supply as a power source,
    /// so voltages are relative to the power supply's ground rail, and the peak voltage of
    /// `offset + amplitude` must be at least 1.5V lower than the configured power supply voltage.
    ///
    /// Returns `Err(UnsupportedVoltage)` if the specified voltage is not valid.
    pub fn send_sine_wave(
        &self,
        frequency: f64,
        amplitude: f64,
        offset: f64,
    ) -> Result<(), crate::UnsupportedVoltage> {
        let res = librador_sys::librador_send_sin_wave(Ch::CHANNEL, frequency, amplitude, offset);
        if res == -2 || res == -1 {
            return Err(crate::UnsupportedVoltage);
        }
        assert_eq!(res, 0, "Unhandled error in librador_send_sin_wave!");
        Ok(())
    }

    /// Configure a signal generator channel to produce a square wave.
    ///
    /// The given channel will start to produce a square wave with the given `frequency`.
    ///
    /// The signal will oscillate between `offset` volts and `offset + amplitude` volts.
    /// The signal generator's amplifier uses the configurable power supply as a power source,
    /// so voltages are relative to the power supply's ground rail, and the peak voltage of
    /// `offset + amplitude` must be at least 1.5V lower than the configured power supply voltage.
    ///
    /// Returns `Err(UnsupportedVoltage)` if the specified voltage is not valid.
    pub fn send_square_wave(
        &self,
        frequency: f64,
        amplitude: f64,
        offset: f64,
    ) -> Result<(), crate::UnsupportedVoltage> {
        let res =
            librador_sys::librador_send_square_wave(Ch::CHANNEL, frequency, amplitude, offset);
        if res == -2 || res == -1 {
            return Err(crate::UnsupportedVoltage);
        }
        assert_eq!(res, 0, "Unhandled error in librador_send_square_wave!");
        Ok(())
    }

    /// Configure a signal generator channel to produce a sawtooth wave.
    ///
    /// The given channel will start to produce a sawtooth wave with the given `frequency`.
    ///
    /// The signal will oscillate between `offset` volts and `offset + amplitude` volts.
    /// The signal generator's amplifier uses the configurable power supply as a power source,
    /// so voltages are relative to the power supply's ground rail, and the peak voltage of
    /// `offset + amplitude` must be at least 1.5V lower than the configured power supply voltage.
    ///
    /// Returns `Err(UnsupportedVoltage)` if the specified voltage is not valid.
    pub fn send_sawtooth_wave(
        &self,
        frequency: f64,
        amplitude: f64,
        offset: f64,
    ) -> Result<(), crate::UnsupportedVoltage> {
        let res =
            librador_sys::librador_send_sawtooth_wave(Ch::CHANNEL, frequency, amplitude, offset);
        if res == -2 || res == -1 {
            return Err(crate::UnsupportedVoltage);
        }
        assert_eq!(res, 0, "Unhandled error in librador_send_sawtooth_wave!");
        Ok(())
    }

    /// Configure a signal generator channel to produce a triangle wave.
    ///
    /// The given channel will start to produce a triangle wave with the given `frequency`.
    ///
    /// The signal will oscillate between `offset` volts and `offset + amplitude` volts.
    /// The signal generator's amplifier uses the configurable power supply as a power source,
    /// so voltages are relative to the power supply's ground rail, and the peak voltage of
    /// `offset + amplitude` must be at least 1.5V lower than the configured power supply voltage.
    ///
    /// Returns `Err(UnsupportedVoltage)` if the specified voltage is not valid.
    pub fn send_triangle_wave(
        &self,
        frequency: f64,
        amplitude: f64,
        offset: f64,
    ) -> Result<(), crate::UnsupportedVoltage> {
        let res =
            librador_sys::librador_send_triangle_wave(Ch::CHANNEL, frequency, amplitude, offset);
        if res == -2 || res == -1 {
            return Err(crate::UnsupportedVoltage);
        }
        assert_eq!(res, 0, "Unhandled error in librador_send_triangle_wave!");
        Ok(())
    }

    /// Configure a signal generator channel to produce a custom waveform.
    ///
    /// The given channel will start to produce a wave specified by the given samples.  The
    /// overall period of the custom waveform is `samples.len() * sample_duration`.
    ///
    /// The signal will oscillate between `offset` volts and `offset + amplitude` volts.
    /// The signal generator's amplifier uses the configurable power supply as a power source,
    /// so voltages are relative to the power supply's ground rail, and the peak voltage of
    /// `offset + amplitude` must be at least 1.5V lower than the configured power supply voltage.
    ///
    /// Returns `Err(UnsupportedVoltage)` if the specified voltage is not valid.
    pub fn send_custom_wave(
        &self,
        samples: &[u8],
        sample_duration: std::time::Duration,
        amplitude: f64,
        offset: f64,
    ) -> Result<(), crate::UnsupportedVoltage> {
        let mut copied = samples.iter().copied().collect::<Vec<u8>>();
        let res = {
            let sample_buffer = copied.as_mut_ptr();
            let num_samples = copied.len() as _;
            let usecs_between_samples = (sample_duration.as_nanos() as f64)
                / (std::time::Duration::from_micros(1).as_nanos() as f64);

            if num_samples > 512 {
                panic!("Too many samples (max 512)!");
                // TODO: not panic
            }

            // SAFETY: the `sample_buffer` points to at least `num_samples` bytes
            unsafe {
                librador_sys::librador_update_signal_gen_settings(
                    Ch::CHANNEL,
                    sample_buffer,
                    num_samples,
                    usecs_between_samples,
                    amplitude,
                    offset,
                )
            }
        };
        if res == -2 || res == -1 {
            return Err(crate::UnsupportedVoltage);
        }
        assert_eq!(
            res, 0,
            "Unhandled error in librador_update_signal_gen_settings!"
        );
        Ok(())
    }
}

/// Signal generator channel 1.
pub enum Ch1 {}
impl super::Channel for Ch1 {
    const CHANNEL: i32 = 1;
}

/// Signal generator channel 2.
pub enum Ch2 {}
impl super::Channel for Ch2 {
    const CHANNEL: i32 = 2;
}

/// Signal generator channel 3.
pub enum Ch3 {}
impl super::Channel for Ch3 {
    const CHANNEL: i32 = 3;
}

/// Signal generator channel 4.
pub enum Ch4 {}
impl super::Channel for Ch4 {
    const CHANNEL: i32 = 4;
}
