// This file is part of librador-rs, a project to provide a safe Rust API
// for the EspoTek Labrador electronics lab board.
//
// Copyright 2021 Andrew Dona-Couch
//
// librador-rs is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// librador-rs is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//! Labrador mode control.
//!
//! Since the Labrador multiplexes a variety of peripherals through a pair of buffers,
//! you must select which tools to use at a given time.  The mode select is used to pick
//! which peripherals are enabled.
//!
//! The structs in this module represent each of the possible modes of the device, and
//! the traits represent capabilities available in each of those modes.
pub(crate) mod sealed {
    pub trait Mode {}
    pub trait ModeTag: Mode {
        const TAG: i32;
    }
}

/// Modes that can be actively switched to by the user.
pub trait ModeChange: sealed::ModeTag {}

/// The mode supports Oscilloscope CH1.
pub trait HasOsc1: sealed::Mode {}
/// The mode supports Oscilloscope CH2.
pub trait HasOsc2: sealed::Mode {}
/// The mode supports Logic Analyzer CH1.
pub trait HasLogic1: sealed::Mode {}
/// The mode supports Logic Analyzer CH2.
pub trait HasLogic2: sealed::Mode {}
/// The mode supports the Multimeter.
pub trait HasMulti: sealed::Mode {}

/// Oscilloscope CH1 (375 ksps)
pub struct Mode0 {}
impl sealed::Mode for Mode0 {}
impl sealed::ModeTag for Mode0 {
    const TAG: i32 = 0;
}
impl ModeChange for Mode0 {}
impl HasOsc1 for Mode0 {}

/// Oscilloscope CH1 (375 ksps) & Logic Analyzer CH1 (375 ksps)
pub struct Mode1 {}
impl sealed::Mode for Mode1 {}
impl sealed::ModeTag for Mode1 {
    const TAG: i32 = 1;
}
impl ModeChange for Mode1 {}
impl HasOsc1 for Mode1 {}
impl HasLogic1 for Mode1 {}

/// Oscilloscope CH1 (375 ksps) & Oscilloscope CH2 (375 ksps)
pub struct Mode2 {}
impl sealed::Mode for Mode2 {}
impl sealed::ModeTag for Mode2 {
    const TAG: i32 = 2;
}
impl ModeChange for Mode2 {}
impl HasOsc1 for Mode2 {}
impl HasOsc2 for Mode2 {}

/// Logic Analyzer CH1 (375 ksps)
pub struct Mode3 {}
impl sealed::Mode for Mode3 {}
impl sealed::ModeTag for Mode3 {
    const TAG: i32 = 3;
}
impl ModeChange for Mode3 {}
impl HasLogic1 for Mode3 {}

/// Logic Analyzer CH1 (375 ksps) & Logic Analyzer CH2 (375 ksps)
pub struct Mode4 {}
impl sealed::Mode for Mode4 {}
impl sealed::ModeTag for Mode4 {
    const TAG: i32 = 4;
}
impl ModeChange for Mode4 {}
impl HasLogic1 for Mode4 {}
impl HasLogic2 for Mode4 {}

/// No peripherals activated.
pub struct Mode5 {}
impl sealed::Mode for Mode5 {}
impl sealed::ModeTag for Mode5 {
    const TAG: i32 = 5;
}
impl ModeChange for Mode5 {}

/// Oscilloscope CH1 (750 ksps)
pub struct Mode6 {}
impl sealed::Mode for Mode6 {}
impl sealed::ModeTag for Mode6 {
    const TAG: i32 = 6;
}
impl ModeChange for Mode6 {}
impl HasOsc1 for Mode6 {} // ... and it is double-speed ....

/// Multimeter (375 ksps, 12-bit)
pub struct Mode7 {}
impl sealed::Mode for Mode7 {}
impl sealed::ModeTag for Mode7 {
    const TAG: i32 = 7;
}
impl ModeChange for Mode7 {}
impl HasMulti for Mode7 {}

/// A pseudo-mode indicating that the USB interface has insufficient bandwidth for isochronous transfer.
pub struct NoIso {}
impl sealed::Mode for NoIso {}
