// This file is part of librador-rs, a project to provide a safe Rust API
// for the EspoTek Labrador electronics lab board.
//
// Copyright 2021 Andrew Dona-Couch
//
// librador-rs is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// librador-rs is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//! The two channel logic analyzer.
//!
//! The Labrador features two channels of a logic analyzer operating at
//! 3Msps.  Get a handle on one of the channels from your [`Labrador`]
//! instance with one of the `logic_analyzer_ch<n>` methods, then call
//! `data` to retrieve the latest digital data.
//!
//! You can adjust the data sampling configuration with the associated
//! [`SampleWindow`] instance -- just remember that `librador` internally
//! buffers only 1 minute of data.
//!
//! # Example
//!
//! ```
//! # use std::time::Duration;
//! # use librador::Labrador;
//! #
//! let labrador = match Labrador::find() {
//!     Err(_) => panic!("Unable to find Labrador..."),
//!     Ok(l) => l,
//! };
//!
//! println!("Setting device mode to 3...");
//! let labrador = labrador.set_mode::<librador::mode::Mode3>();
//!
//! println!("Configuring logic analyzer...");
//! let logic = {
//!     let mut logic = labrador.logic_analyzer_ch1();
//!     logic.window = Duration::from_millis(400);
//!     logic.period = Duration::from_millis(10);
//!     logic
//! };
//!
//! println!("Getting digital data...");
//! match logic.data() {
//!     None => println!("Unable to get data!"),
//!     Some(data) => println!("Data: {:?}", data),
//! }
//! ```
//!
//! [`Labrador`]: ../struct.Labrador.html
//! [`SampleWindow`]: ../struct.SampleWindow.html

/// A handle on a logic analyzer channel.
///
/// Try calling `data` on it.  See the [module docs] for more information.
///
/// [module docs]: index.html
pub struct LogicAnalyzer<'a, Ch> {
    _data: std::marker::PhantomData<&'a Ch>,
    window: super::SampleWindow,
}

impl<'a, Ch: crate::Channel> LogicAnalyzer<'a, Ch> {
    pub(crate) fn new() -> Self {
        LogicAnalyzer {
            _data: std::marker::PhantomData,
            window: super::SampleWindow::new(),
        }
    }

    /// Get the latest window of data from the logic analyzer.
    ///
    /// This fetches recent data streamed from the logic analyzer.  The window of
    /// time covered by the data begins `window` + `delay` ago, and lasts until
    /// `delay` ago.  There is a sample for each `period` in the window.
    ///
    /// # Examples
    ///
    /// ```
    /// # use std::time::Duration;
    /// # use librador::Labrador;
    /// #
    /// # let mut labrador = match Labrador::find() {
    /// #     Err(_) => panic!("Unable to find Labrador..."),
    /// #     Ok(l) => l,
    /// # };
    /// #
    /// let mut labrador = labrador.set_mode::<librador::mode::Mode3>();
    /// let mut logic = labrador.logic_analyzer_ch1();
    /// logic.window = Duration::from_millis(400);
    /// logic.period = Duration::from_millis(10);
    /// logic.delay = Duration::from_millis(0);
    /// let data = logic.data();
    ///
    /// match data {
    ///     None => println!("Unable to get data!"),
    ///     Some(data) => println!("Data: {:?}", data),
    /// }
    /// ```
    pub fn data(&self) -> Option<Vec<bool>> {
        let nanos_per_second = std::time::Duration::from_secs(1).as_nanos() as f64;

        let channel = Ch::CHANNEL;
        let window = self.window.window.as_nanos() as f64 / nanos_per_second;
        let freq = nanos_per_second / (self.window.period.as_nanos() as f64);
        let delay = self.window.delay.as_nanos() as f64 / nanos_per_second;

        println!(
            "Getting digital data: {}, {}, {}, {}",
            channel, window, freq, delay
        );

        let res = librador_sys::librador_get_digital_data(channel, window, freq, delay);

        // SAFETY: we trust librador/cxx to give us a legitimate pointer.
        let maybe_vec = unsafe { res.as_ref() };

        maybe_vec.map(|v| {
            let mut res: Vec<_> = v.iter().map(|v| *v != 0).collect();
            res.reverse();
            res
        })
    }
}

impl<'a, Ch: crate::Channel> std::ops::Deref for LogicAnalyzer<'a, Ch> {
    type Target = super::SampleWindow;
    fn deref(&self) -> &super::SampleWindow {
        &self.window
    }
}

impl<'a, Ch: crate::Channel> std::ops::DerefMut for LogicAnalyzer<'a, Ch> {
    fn deref_mut(&mut self) -> &mut super::SampleWindow {
        &mut self.window
    }
}

/// Logic analyzer channel 1.
pub enum Ch1 {}
impl super::Channel for Ch1 {
    const CHANNEL: i32 = 1;
}

/// Logic analyzer channel 2.
pub enum Ch2 {}
impl super::Channel for Ch2 {
    const CHANNEL: i32 = 2;
}
