// This file is part of librador-rs, a project to provide a safe Rust API
// for the EspoTek Labrador electronics lab board.
//
// Copyright 2021 Andrew Dona-Couch
//
// librador-rs is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// librador-rs is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//! Safe, Rusty driver for the EspoTek Labrador electronics lab board.
//!
//! This crate provides direct access to the Labrador test board by
//! wrapping the librador driver library in a safe, Rusty API.  The primary
//! entry point of the crate is the [`Labrador` struct].
//!
//! The Labrador multiplexes a variety of peripherals through two
//! data buffers, so before you start to use it, you need to configure
//! the appropriate mode.  See the [`mode` module] for more details.
//!
//! [`Labrador` struct]: struct.Labrador.html
//! [`mode` module]: mode/index.html
//!
//! # Examples
//!
//! ## Find the Labrador board and get the firmware version
//!
//! ```
//! use librador::Labrador;
//!
//! let mut labrador = match Labrador::find() {
//!     Err(_) => panic!("Unable to find Labrador..."),
//!     Ok(l) => l,
//! };
//!
//! let variant = labrador.firmware_variant();
//! let version = labrador.firmware_version();
//! println!("Labrador found, variant {}, version {}", variant, version);
//! ```
//!
//! ## Set a digital output pin
//!
//! ```
//! use librador::Labrador;
//!
//! let mut labrador = match Labrador::find() {
//!     Err(_) => panic!("Unable to find Labrador..."),
//!     Ok(l) => l,
//! };
//!
//! labrador.digital_out_ch1().set_high();
//! ```
//!
//! ## Get recent digital data from the logic analyzer
//!
//! ```
//! use std::time::Duration;
//! use librador::Labrador;
//!
//! let mut labrador = match Labrador::find() {
//!     Err(_) => panic!("Unable to find Labrador..."),
//!     Ok(l) => l,
//! };
//!
//! let mut labrador = labrador.set_mode::<librador::mode::Mode3>();
//! let mut logic1 = labrador.logic_analyzer_ch1();
//! let data = logic1.data();
//!
//! match data {
//!     None => println!("Unable to get data!"),
//!     Some(data) => println!("Data: {:?}", data),
//! }
//! ```

use std::sync::Once;

use digital::DigitalOut;
use logic::LogicAnalyzer;
use signal::SignalGenerator;

static INIT_LIB: Once = Once::new();

pub(crate) fn initialize_lib() {
    INIT_LIB.call_once(|| {
        match librador_sys::librador_init() {
            -1 => panic!("Unhandled error initializing librador!"), // TODO: return an error instead?
            1 => eprintln!("Attempted to initalize librador more than once!"), // TODO: use log
            0 => {}
            _ => panic!("Unknown error code from librador!"), // TODO: return an error instead?
        }
    });
}

/// A Labrador device.
///
/// Get a handle on the attached Labrador with `find()`, then set the mode of
/// the device and query for data.  See the crate-level documentation for an overview,
/// or the methods below for more details.
pub struct Labrador<Mode> {
    _mode: core::marker::PhantomData<Mode>,
}

impl Labrador<mode::Mode5> {
    /// Find the attached Labrador device.
    ///
    /// Internally librador searches for the Labrador on your USB endpoints.  If you
    /// have one connected, this should return `Ok`.
    ///
    /// If the driver is unable to reserve sufficient USB bandwidth for the sample data,
    /// this may return a `LabradorInitError::Bandwidth`.  This contains Labrador in a
    /// highly limited mode, `<Labrador<mode::NoIso>>`, which is usable for debug
    /// information and setting the various outputs, but cannot be used as an
    /// oscilloscope, multimeter, or logic analyzer.
    pub fn find() -> Result<Labrador<mode::Mode5>, LabradorInitError> {
        initialize_lib();

        match librador_sys::librador_setup_usb() {
            -420 => Err(LabradorInitError::Unspecified), // library not initialized
            -3 => Err(LabradorInitError::DeviceBusy),
            -2 => Err(LabradorInitError::DeviceNotFound),
            -1 => Err(LabradorInitError::Unspecified), // unable to initialize USB
            0 => Ok(Labrador::new()),
            n if n < -500 => Err(LabradorInitError::Bandwidth(Labrador::new())),
            _ => Err(LabradorInitError::Unspecified), // unknown error code
        }
    }
}

impl<Mode> Labrador<Mode> {
    pub(crate) fn new() -> Labrador<Mode> {
        Labrador {
            _mode: core::marker::PhantomData,
        }
    }

    /// Retrieve the firware version from the Labrador device.
    pub fn firmware_version(&self) -> u16 {
        librador_sys::librador_get_device_firmware_version()
    }

    /// Retrieve the firmware variant from the Labrador device.
    ///
    /// From the librador docs: "firmware variant 0x02 has a single endpoint and
    /// is meant for use with Unix-like systems, whereas variant 0x01 has multiple
    /// endpoints and is meant for use with Windows systems."
    pub fn firmware_variant(&self) -> u8 {
        librador_sys::librador_get_device_firmware_variant()
    }

    /// Adjust the programmable power supply voltage.
    ///
    /// The power supply is rated for the range 4.5V to 12V. If the provided voltage
    /// is not supported, this method returns `Err(UnsupportedVoltage)`.
    pub fn set_power_supply_voltage(&self, voltage: f64) -> Result<(), crate::UnsupportedVoltage> {
        let res = librador_sys::librador_set_power_supply_voltage(voltage);
        if res == -1 {
            return Err(crate::UnsupportedVoltage);
        }
        assert_eq!(
            res, 0,
            "Unhandled error in librador_set_power_supply_voltage!"
        );
        Ok(())
    }

    /// Get a reference to Digital Out CH1.
    pub fn digital_out_ch1(&self) -> DigitalOut<digital::Ch1> {
        DigitalOut::new()
    }

    /// Get a reference to Digital Out CH2.
    pub fn digital_out_ch2(&self) -> DigitalOut<digital::Ch2> {
        DigitalOut::new()
    }

    /// Get a reference to Digital Out CH3.
    pub fn digital_out_ch3(&self) -> DigitalOut<digital::Ch3> {
        DigitalOut::new()
    }

    /// Get a reference to Digital Out CH4.
    pub fn digital_out_ch4(&self) -> DigitalOut<digital::Ch4> {
        DigitalOut::new()
    }

    /// Get a reference to Signal Generator CH1.
    pub fn signal_generator_ch1(&self) -> SignalGenerator<signal::Ch1> {
        SignalGenerator::new()
    }

    /// Get a reference to Signal Generator CH2.
    pub fn signal_generator_ch2(&self) -> SignalGenerator<signal::Ch2> {
        SignalGenerator::new()
    }

    /// Get a reference to Signal Generator CH3.
    pub fn signal_generator_ch3(&self) -> SignalGenerator<signal::Ch3> {
        SignalGenerator::new()
    }

    /// Get a reference to Signal Generator CH4.
    pub fn signal_generator_ch4(&self) -> SignalGenerator<signal::Ch4> {
        SignalGenerator::new()
    }
}

impl<Mode: mode::ModeChange> Labrador<Mode> {
    /// Change the mode of the Labrador device.
    ///
    /// This method consumes the Labrador and returns a new one in the new mode.
    /// See the [`mode` module] for more details.
    ///
    /// [`mode` module]: mode/index.html
    pub fn set_mode<NewMode: mode::ModeChange>(self) -> Labrador<NewMode> {
        let res = librador_sys::librador_set_device_mode(NewMode::TAG);
        assert_eq!(res, 0, "Unhandled error in librador_set_device_mode!");

        Labrador::new()
    }
}

pub mod digital;
pub mod logic;
pub mod mode;
pub mod signal;

/// Common configuration of sampling windows.
///
/// The streaming data returned by the Labrador is downsampled based on the
/// configured parameters.  Configure these as appropriate for your application,
/// but remember that `librador` internally only buffers 1 minute of samples.
pub struct SampleWindow {
    /// The window of time to retrieve samples from -- the duration between
    /// the first sample returned and the last.
    ///
    /// Defaults to 50 microseconds.
    pub window: std::time::Duration,
    /// The sampling period.  One sample is returned per period for the length
    /// of the sampling window.
    ///
    /// Defaults to 2667 nanoseconds, or 375ksps.
    // TODO: a different default for the 3Msps logic analyzer???
    pub period: std::time::Duration,
    /// A delay applied to the sampling window.  This is the duration between
    /// the last sample returned and `now` -- there will be no samples returned
    /// newer than the `delay` duration ago.
    ///
    /// Setting this to a small value can help alleviate jitter from USB packets
    /// arriving out of order.
    ///
    /// Defaults to no delay.
    pub delay: std::time::Duration,
}

impl SampleWindow {
    /// Create a new `SampleWindow` with the default values.
    pub fn new() -> Self {
        SampleWindow {
            window: std::time::Duration::from_micros(50), // default data volume low, e.g. for example code
            period: std::time::Duration::from_nanos(2667), // default to no downsampling -- 2.667us is 375ksps
            delay: std::time::Duration::from_micros(0),    // default no delay
        }
    }
}

impl Default for SampleWindow {
    fn default() -> Self {
        Self::new()
    }
}

/// One of the Labrador's peripheral channels.
pub trait Channel {
    const CHANNEL: i32;
}

impl<Mode: mode::HasLogic1> Labrador<Mode> {
    /// Get a reference to Logic Analyzer CH1.
    pub fn logic_analyzer_ch1(&self) -> LogicAnalyzer<logic::Ch1> {
        LogicAnalyzer::new()
    }
}

impl<Mode: mode::HasLogic2> Labrador<Mode> {
    /// Get a reference to Logic Analyzer CH2.
    pub fn logic_analyzer_ch2(&self) -> LogicAnalyzer<logic::Ch2> {
        LogicAnalyzer::new()
    }
}

/// An error arising during initialization.
pub enum LabradorInitError {
    Unspecified, // TODO: should this be panic?
    DeviceBusy,
    DeviceNotFound,
    Bandwidth(Labrador<mode::NoIso>),
}

/// The specified voltage is not supported by the power supply.
pub struct UnsupportedVoltage;
