// This file is part of librador-rs, a project to provide a safe Rust API
// for the EspoTek Labrador electronics lab board.
//
// Copyright 2021 Andrew Dona-Couch
//
// librador-rs is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// librador-rs is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//! Four channels of digital outputs.
//!
//! The Labrador features four channels of general-purpose digital output.
//! Get a handle on one of the channels from your [`Labrador`] instance with
//! one of the `digital_out_ch<n>` methods, then call `set_high` and `set_low`
//! to your heart's content.
//!
//! # Example
//!
//! ```
//! # use librador::Labrador;
//! #
//! let labrador = match Labrador::find() {
//!     Err(_) => panic!("Unable to find Labrador..."),
//!     Ok(l) => l,
//! };
//!
//! println!("Setting digital out...");
//! labrador.digital_out_ch1().set_high();
//! labrador.digital_out_ch2().set_low();
//! ```
//!
//! [`Labrador`]: ../struct.Labrador.html

/// A handle on a digital output channel.
///
/// Try calling `set_high` and `set_low` on it.  See the [module docs] for
/// more information.
///
/// [module docs]: index.html
pub struct DigitalOut<Ch> {
    _data: std::marker::PhantomData<Ch>,
}

impl<Ch: crate::Channel> DigitalOut<Ch> {
    pub(crate) fn new() -> Self {
        DigitalOut {
            _data: std::marker::PhantomData,
        }
    }

    /// Sets the specified channel to a high (3.3V) level.
    pub fn set_high(&self) {
        let res = librador_sys::librador_set_digital_out(Ch::CHANNEL, true);
        assert_eq!(res, 0, "Unhandled error in librador_set_digital_out!");
    }

    /// Sets the specified channel to a low (0V) level.
    pub fn set_low(&self) {
        let res = librador_sys::librador_set_digital_out(Ch::CHANNEL, false);
        assert_eq!(res, 0, "Unhandled error in librador_set_digital_out!");
    }
}

/// Digital out channel 1.
pub enum Ch1 {}
impl super::Channel for Ch1 {
    const CHANNEL: i32 = 1;
}

/// Digital out channel 2.
pub enum Ch2 {}
impl super::Channel for Ch2 {
    const CHANNEL: i32 = 2;
}

/// Digital out channel 3.
pub enum Ch3 {}
impl super::Channel for Ch3 {
    const CHANNEL: i32 = 3;
}

/// Digital out channel 4.
pub enum Ch4 {}
impl super::Channel for Ch4 {
    const CHANNEL: i32 = 4;
}
