// This file is part of librador-rs, a project to provide a safe Rust API
// for the EspoTek Labrador electronics lab board.
//
// Copyright 2021 Andrew Dona-Couch
//
// librador-rs is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// librador-rs is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// Example of using Librador to generate a custom waveform.
///
/// Wire the signal generator to the oscilloscope before running:
/// - TODO!
use std::time::Duration;

use librador::Labrador;

const MAX_SAMPLES: u32 = 512;

fn main() {
    println!("Getting Labrador...");
    let labrador = match Labrador::find() {
        Err(_) => panic!("Unable to find Labrador..."),
        Ok(l) => l,
    };

    println!("Setting device mode to 5...");
    let labrador = labrador.set_mode::<librador::mode::Mode5>();

    let num_samples = MAX_SAMPLES;
    let wave_duration = Duration::from_secs(1);
    let sample_duration = wave_duration.div_f32(num_samples as _);

    println!("Generating custom waveform...");
    let samples = (0..num_samples)
        .into_iter()
        .map(|sample| {
            let progress = (sample as f64) / (num_samples as f64);
            let result = if progress < 0.1 {
                10. * progress
            } else {
                1. - progress * 0.9
            };
            (256. * result) as _
        })
        .collect::<Vec<_>>();

    let signal_generator = labrador.signal_generator_ch1();

    let res = signal_generator.send_custom_wave(
        &samples,
        sample_duration,
        1., // volts max
        0., // volts min
    );
    if res.is_err() {
        panic!("Unable to send custom wave to signal generator!");
    }

    // TODO: read back analog data
}
