// This file is part of librador-rs, a project to provide a safe Rust API
// for the EspoTek Labrador electronics lab board.
//
// Copyright 2021 Andrew Dona-Couch
//
// librador-rs is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// librador-rs is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// Example of using Librador to set digital outputs and read the logic analyzer.
///
/// Wire the digital outputs to the logic analyzer before running:
/// - Digital Output CH1 to Logic Analyzer CH1
/// - Digital Output CH2 to Logic Analyzer CH2
use std::iter::repeat;
use std::time::Duration;

use librador::Labrador;

fn main() {
    println!("Getting Labrador...");
    let labrador = match Labrador::find() {
        Err(_) => panic!("Unable to find Labrador..."),
        Ok(l) => l,
    };

    println!("Setting device mode to 4...");
    let labrador = labrador.set_mode::<librador::mode::Mode4>();

    println!("Configuring logic analyzers...");
    let logic1 = {
        let mut logic1 = labrador.logic_analyzer_ch1();
        logic1.window = Duration::from_millis(400);
        logic1.period = Duration::from_millis(10);
        logic1
    };
    let logic2 = {
        let mut logic2 = labrador.logic_analyzer_ch2();
        logic2.window = Duration::from_millis(400);
        logic2.period = Duration::from_millis(10);
        logic2
    };

    println!("Setting digital out...");
    labrador.digital_out_ch1().set_high();
    labrador.digital_out_ch2().set_low();

    println!("Waiting...");
    std::thread::sleep(std::time::Duration::from_millis(50));

    println!("Clearing digital out...");
    labrador.digital_out_ch1().set_low();
    labrador.digital_out_ch2().set_high();

    println!("Waiting...");
    std::thread::sleep(std::time::Duration::from_millis(50));

    println!("Setting digital out...");
    labrador.digital_out_ch1().set_high();
    labrador.digital_out_ch2().set_low();

    println!("Waiting...");
    std::thread::sleep(std::time::Duration::from_millis(150));

    println!("Clearing digital out...");
    labrador.digital_out_ch1().set_low();
    labrador.digital_out_ch2().set_high();

    println!("Waiting...");
    std::thread::sleep(std::time::Duration::from_millis(40));

    println!("Getting digital data...");
    let data1 = logic1.data().map(|data| {
        data.into_iter()
            .map(|v| if v { 1 } else { 0 })
            .collect::<Vec<_>>()
    });
    let data2 = logic2.data().map(|data| {
        data.into_iter()
            .map(|v| if v { 1 } else { 0 })
            .collect::<Vec<_>>()
    });

    match data1 {
        None => println!("Unable to get data!"),
        Some(data) => println!("Data1: {:?}", data),
    }

    let exp = repeat(0)
        .take(12)
        .chain(repeat(1).take(4))
        .chain(repeat(0).take(5))
        .chain(repeat(1).take(15))
        .chain(repeat(0).take(4))
        .collect::<Vec<u8>>();
    println!("Exp.:  {:?}", exp);

    match data2 {
        None => println!("Unable to get data!"),
        Some(data) => println!("Data2: {:?}", data),
    }

    let exp = repeat(0)
        .take(12)
        .chain(repeat(0).take(4))
        .chain(repeat(1).take(5))
        .chain(repeat(0).take(15))
        .chain(repeat(1).take(4))
        .collect::<Vec<u8>>();
    println!("Exp.:  {:?}", exp);
}
