// This file is part of librador-sys, Rust bindings to librador, the driver
// for the EspoTek Labrador electronics lab board.
//
// Copyright 2021 Andrew Dona-Couch
//
// librador-sys is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// librador-sys is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//! Raw bindings to `librador`, the driver API for the EspoTek Labrador electronics lab board.
//!
//! See the [Librador User Guide] for more information.
//!
//! [Librador User Guide]: https://docs.google.com/document/d/11Cu6602X_y5x0IwIBZnP7ImIx0jykX7AzToh83-7hCM/edit#heading=h.l4vc0gyrpbsq

extern crate libusb1_sys;

pub use ffi::*;

#[cxx::bridge]
mod ffi {
    unsafe extern "C++" {
        include!("librador.h");

        /// This function initialises the library.  It must be called before you do anything else.
        fn librador_init() -> i32;

        /// This function closes the library.  Do not make any other calls after calling this function.
        fn librador_exit() -> i32;

        /// This function sets up the USB stack.
        ///
        /// You must call this before calling any USB-Control or USB-Iso function.  If you get a return value of approximately -1000, it means you can call USB-Control but not USB-Iso functions.
        fn librador_setup_usb() -> i32;

        fn librador_reset_usb() -> i32;

        /// This function gets some status info from the device.
        ///
        /// It's mainly useful for testing/debugging purposes.
        fn librador_avr_debug() -> i32;

        /// This function gets the firmware version of the Labrador device, as a 16-bit unsigned int.
        ///
        /// The higher the number, the newer the firmware release.
        fn librador_get_device_firmware_version() -> u16;
        /// This function gets the firmware variant of the Labrador device, as an 8-bit unsigned int.
        ///
        /// Different numbers correspond to different variants of the same firmware release.  For example, firmware variant 0x02 has a single endpoint and is meant for use with Unix-like systems, whereas variant 0x01 has multiple endpoints and is meant for use with Windows systems.
        fn librador_get_device_firmware_variant() -> u8;

        /// Changes the Labrador's device mode.
        ///
        /// The device mode determines which functions are active at any point in time.  The valid modes are 0-7.  The table below shows which functions are active in each given mode.
        ///
        /// | MODE                         | 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 |
        /// |------------------------------|---|---|---|---|---|---|---|---|
        /// | Oscilloscope CH1 (375ksps)   | X | X | X |   |   |   |   |   |
        /// | Oscilloscope CH2 (375ksps)   |   |   | X |   |   |   |   |   |
        /// | Oscilloscope CH1 (750ksps)   |   |   |   |   |   |   | X |   |
        /// | Logic Analyzer CH1 (3MSPS)   |   | X |   | X | X |   |   |   |
        /// | Logic Analyzer CH2 (3MSPS)   |   |   |   |   | X |   |   |   |
        /// | Multimeter (375ksps, 12-bit) |   |   |   |   |   |   |   | X |
        fn librador_set_device_mode(mode: i32) -> i32;

        /// Sets the input gain on the oscilloscope.
        ///
        /// In order to achieve good accuracy over a wide range of inputs, a programmable gain amplifier (PGA) is used to amplify oscilloscope/multimeter input voltages in the analog domain before sampling.  This function lets you set the gain level of the PGA.
        ///
        /// Valid input values are 0.5, 1, 2, 4, 8, 16, 32, and 64.  If the value is too low, you will have high levels of quantisation noise.  If the value is too high, you will have clipping.  A value of 4 is recommended for most signals in the 0-3.5V range.  For larger voltages, set the gain lower or AC couple.
        fn librador_set_oscilloscope_gain(gain: f64) -> i32;

        /// This functions turns the digital outputs on or off.
        fn librador_set_digital_out(channel: i32, state_on: bool) -> i32;

        /// Changes the Labrador's power supply output voltage.
        fn librador_set_power_supply_voltage(voltage: f64) -> i32;

        /// This function sends a custom waveform to the Labrador's signal generator output.
        ///
        /// # Safety
        ///
        /// You must ensure that the `sample_buffer` argument points to a region of
        /// at least `num_samples` bytes.  The data is copied out, so it only needs to
        /// live until this call is complete.
        unsafe fn librador_update_signal_gen_settings(
            channel: i32,
            sample_buffer: *mut u8,
            num_samples: i32,
            usecs_between_samples: f64,
            amplitude: f64,
            offset: f64,
        ) -> i32;

        /// This function sends a sine wave to the Labrador's signal generator output.
        fn librador_send_sin_wave(channel: i32, frequency: f64, amplitude: f64, offset: f64)
            -> i32;

        /// This function sends a square wave to the Labrador's signal generator output.
        fn librador_send_square_wave(
            channel: i32,
            frequency: f64,
            amplitude: f64,
            offset: f64,
        ) -> i32;

        /// This function sends a sawtooth wave to the Labrador's signal generator output.
        fn librador_send_sawtooth_wave(
            channel: i32,
            frequency: f64,
            amplitude: f64,
            offset: f64,
        ) -> i32;

        /// This function sends a triangle wave to the Labrador's signal generator output.
        fn librador_send_triangle_wave(
            channel: i32,
            frequency: f64,
            amplitude: f64,
            offset: f64,
        ) -> i32;

        /// This function gets the most recent data streamed from a single channel of the oscilloscope or multimeter.
        ///
        /// The first sample contains the value of the stream delay_seconds seconds ago. The last sample contains the value of the stream (delay_seconds + timeWindow_seconds) seconds ago.  The samples in between are evenly spaced, temporally, at an interval of (1/sample_rate_Hz) seconds.  The filter_mode variable determines whether or not an anto-aliasing filter is used.  0 is for no filter, 1 is linear interpolation.
        fn librador_get_analog_data(
            channel: i32,
            time_window_seconds: f64,
            sample_rate_hz: f64,
            delay_seconds: f64,
            filter_mode: i32,
        ) -> *mut CxxVector<f64>;

        /// This function is similar to librador_get_analog_data(), but it will only return data that has been received since the last time this function was called.
        ///
        /// The timeWindow_max_seconds variable specifies the maximum size of the time window you want to look at.  Set it to inf if you want to read all data since the last call.  This is useful if you're logging data to a file.
        ///
        /// For example, let's assume you called this function once at t=10s with delay_seconds = 0.1 and timeWindow_max_seconds=5, and then again at and t=20s with delay_seconds = 0 and timeWindow_max_seconds=inf.
        /// The first call would return the data corresponding to the t=4.9s to t=9.9s, as the maximum time window is 5s.
        /// The second call would contain t=9.9s to t=19.9s.  It would not contain anything earlier than 9.9s since that data was returned by the last call, and it would not contain anything later than 19.9 since there is a 0.1s delay.
        ///
        /// Keep in mind that the API only stores the last 30 to 60 seconds worth of data, depending on whether 750kHz sampling is used.  Anything older than that cannot be accessed.
        fn librador_get_analog_data_sincelast(
            channel: i32,
            time_window_seconds: f64,
            sample_rate_hz: f64,
            delay_seconds: f64,
            filter_mode: i32,
        ) -> *mut CxxVector<f64>;

        /// This function is similar to librador_get_analog_data(), but it's used to access the logic analyzer inputs.
        ///
        /// Each unsigned char in the vector will have the value 0 or 1, depending on whether or not the input was driven high or low at the time of sampling.
        fn librador_get_digital_data(
            channel: i32,
            time_window_seconds: f64,
            sample_rate_hz: f64,
            delay_seconds: f64,
        ) -> *mut CxxVector<u8>;
    }
}
