use crate::change::{Local, LocalChange};
use crate::changestore::ChangeStore;
use crate::pristine::*;
use crate::record::{Algorithm, Builder};
use crate::working_copy::WorkingCopy;
use crate::*;
use chrono::*;

use std::sync::{Arc, RwLock};

mod add_file;
/*
mod change;
mod clone;
mod conflict;
mod file_conflicts;
mod filesystem;
mod missing_context;
mod partial;
mod performance;
mod rm_file;
mod rollback;
mod unrecord;
*/
mod text;

fn record_all_change<
    T: MutTxnT + Send + Sync + 'static,
    R: WorkingCopy + Send + Sync + 'static,
    P: ChangeStore + Clone + Send + Sync + 'static,
>(
    repo: Arc<R>,
    store: &P,
    txn: Arc<RwLock<T>>,
    channel: ChannelRef<T>,
    prefix: &str,
) -> Result<(Hash, LocalChange<Local>), anyhow::Error>
where
    R::Error: Send + Sync + 'static,
{
    let mut state = Builder::new();
    state.record(
        txn.clone(),
        Algorithm::default(),
        channel.clone(),
        repo,
        store,
        prefix,
        1,
    )?;

    let rec = state.finish();
    let changes = rec
        .actions
        .into_iter()
        .map(|rec| rec.globalize(&*txn.read().unwrap()).unwrap())
        .collect();
    let change0 = crate::change::Change::make_change(
        &*txn.read().unwrap(),
        &channel.clone(),
        changes,
        std::mem::take(&mut *rec.contents.lock().unwrap()),
        crate::change::ChangeHeader {
            message: "test".to_string(),
            authors: vec![],
            description: None,
            // Beware of changing the following line: two changes
            // doing the same thing will be equal. Sometimes we don't
            // want that, as in tests::unrecord::unrecord_double.
            timestamp: Utc::now(),
        },
        Vec::new(),
    )
    .unwrap();
    let hash = store.save_change(&change0)?;
    if log_enabled!(log::Level::Debug) {
        change0
            .write(
                store,
                Some(hash),
                |l, _p| format!("{}:{}", l.path, l.line),
                true,
                &mut std::io::stderr(),
            )
            .unwrap();
    }
    apply::apply_local_change(
        &mut *txn.write().unwrap(),
        &channel,
        &change0,
        &hash,
        &rec.updatables,
    )?;
    Ok((hash, change0))
}

fn record_all<T: MutTxnT, R: WorkingCopy, P: ChangeStore>(
    repo: Arc<R>,
    store: &P,
    txn: Arc<RwLock<T>>,
    channel: ChannelRef<T>,
    prefix: &str,
) -> Result<Hash, anyhow::Error>
where
    T: MutTxnT + Send + Sync + 'static,
    R: WorkingCopy + Send + Sync + 'static,
    P: ChangeStore + Clone + Send + Sync + 'static,
    R::Error: Send + Sync + 'static,
{
    let (hash, _) = record_all_change(repo, store, txn, channel, prefix).unwrap();
    Ok(hash)
}

fn record_all_output<
    T: MutTxnT + Send + Sync + 'static,
    R: WorkingCopy + Send + Sync + 'static,
    P: ChangeStore + Clone + Send + Sync + 'static,
>(
    repo: Arc<R>,
    changes: P,
    txn: Arc<RwLock<T>>,
    channel: ChannelRef<T>,
    prefix: &str,
) -> Result<Hash, anyhow::Error>
where
    T: MutTxnT + Send + Sync + 'static,
    R: WorkingCopy + Send + Sync + 'static,
    P: ChangeStore + Clone + Send + Sync + 'static,
    R::Error: Send + Sync + 'static,
{
    let hash = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), prefix)?;
    output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn,
        channel.clone(),
        "",
        true,
        None,
        1,
        0,
    )
    .unwrap();
    Ok(hash)
}
