use super::*;
use crate::working_copy::WorkingCopy;
use std::io::Write;

/// Add a file, write to it, then fork the branch and unrecord once on
/// one side.
#[test]
fn test() -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();
    repo.add_file("dir/file", b"a\nb\nc\nd\n".to_vec());

    let env = pristine::sanakirja::Pristine::new_anon()?;

    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));
    txn.write().unwrap().add_file("dir/file")?;

    let channel = (&mut *txn.write().unwrap()).open_or_create_channel("main")?;
    let _h0 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    use std::io::Write;
    repo.write_file("dir/file")?.write_all(b"a\nx\nb\nd\n")?;

    let h1 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    let channel2 = (&mut *txn.write().unwrap()).fork(&channel, "main2")?;
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h1)?;
    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    if !conflicts.is_empty() {
        panic!("conflicts = {:#?}", conflicts);
    }
    let mut buf = Vec::new();
    repo.read_file("dir/file", &mut buf)?;
    assert_eq!(std::str::from_utf8(&buf), Ok("a\nb\nc\nd\n"));

    debug_to_file(&*txn.read().unwrap(), &channel, "debug_un")?;
    debug_to_file(&*txn.read().unwrap(), &channel2, "debug_un2")?;
    commit(txn)?;

    Ok(())
}

#[test]
fn replace() -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();
    repo.add_file("dir/file", b"a\nb\nc\nd\n".to_vec());

    let env = pristine::sanakirja::Pristine::new_anon()?;

    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));
    txn.write().unwrap().add_file("dir/file")?;

    let channel = txn.write().unwrap().open_or_create_channel("main")?;
    let _h0 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    repo.write_file("dir/file")?.write_all(b"a\nx\ny\nd\n")?;

    let h1 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    let channel2 = txn.write().unwrap().fork(&channel, "main2")?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug_un0")?;
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h1)?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug_un1")?;
    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    if !conflicts.is_empty() {
        panic!("conflicts = {:#?}", conflicts);
    }
    let mut buf = Vec::new();
    repo.read_file("dir/file", &mut buf)?;
    assert_eq!(std::str::from_utf8(&buf), Ok("a\nb\nc\nd\n"));

    debug_to_file(&*txn.read().unwrap(), &channel, "debug_un")?;
    debug_to_file(&*txn.read().unwrap(), &channel2, "debug_un2")?;
    commit(txn)?;

    Ok(())
}

#[test]
fn file_move() -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();
    repo.add_file("file", b"a\nb\nc\nd\n".to_vec());

    let env = pristine::sanakirja::Pristine::new_anon()?;

    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));
    txn.write().unwrap().add_file("file")?;

    let channel = (&mut *txn.write().unwrap()).open_or_create_channel("main")?;
    let _h0 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    repo.rename("file", "dir/file")?;
    txn.write().unwrap().move_file("file", "dir/file")?;
    debug!("recording the move");
    let h1 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    debug_to_file(&*txn.read().unwrap(), &channel, "debug_un")?;
    debug!("unrecording the move");
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h1)?;

    debug_to_file(&*txn.read().unwrap(), &channel, "debug_un2")?;
    assert_eq!(
        crate::fs::iter_working_copy(&*txn.read().unwrap(), Inode::ROOT)
            .map(|n| n.unwrap().1)
            .collect::<Vec<_>>(),
        vec!["dir", "dir/file"]
    );
    assert_eq!(repo.list_files(), vec!["dir", "dir/file"]);

    output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    assert_eq!(
        crate::fs::iter_working_copy(&*txn.read().unwrap(), Inode::ROOT)
            .map(|n| n.unwrap().1)
            .collect::<Vec<_>>(),
        vec!["file"]
    );

    // Checking that unrecord doesn't delete `dir`, and moves `file`
    // back to the root.
    let mut files = repo.list_files();
    files.sort();
    assert_eq!(files, vec!["dir", "file"]);

    commit(txn)?;

    Ok(())
}

#[test]
fn reconnect_lines() -> Result<(), anyhow::Error> {
    reconnect_(false)
}

#[test]
fn reconnect_files() -> Result<(), anyhow::Error> {
    reconnect_(true)
}

fn reconnect_(delete_file: bool) -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let repo2 = Arc::new(working_copy::memory::Memory::new());
    let repo3 = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();
    repo.add_file("file", b"a\nb\nc\nd\n".to_vec());

    let env = pristine::sanakirja::Pristine::new_anon()?;
    let env2 = pristine::sanakirja::Pristine::new_anon()?;
    let env3 = pristine::sanakirja::Pristine::new_anon()?;

    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));
    let txn2 = Arc::new(RwLock::new(env2.mut_txn_begin().unwrap()));
    let txn3 = Arc::new(RwLock::new(env3.mut_txn_begin().unwrap()));
    txn.write().unwrap().add_file("file")?;

    let channel = (&mut *txn.write().unwrap()).open_or_create_channel("main")?;
    let h0 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    let channel2 = (&mut *txn2.write().unwrap()).open_or_create_channel("main")?;
    let channel3 = (&mut *txn3.write().unwrap()).open_or_create_channel("main")?;

    apply::apply_change(
        &changes,
        &mut *txn2.write().unwrap(),
        &mut *channel2.write().unwrap(),
        &h0,
    )?;
    output::output_repository_no_pending(
        repo2.clone(),
        &changes,
        txn2.clone(),
        channel2.clone(),
        "",
        true,
        None,
        1,
    )?;
    apply::apply_change(
        &changes,
        &mut *txn3.write().unwrap(),
        &mut *channel3.write().unwrap(),
        &h0,
    )?;
    output::output_repository_no_pending(
        repo3.clone(),
        &changes,
        txn3.clone(),
        channel3.clone(),
        "",
        true,
        None,
        1,
    )?;

    // This test removes a line (in h1), then replaces it with another
    // one (in h2), removes the pseudo-edges (output, below), and then
    // unrecords h2 to delete the connection. Test: do the
    // pseudo-edges reappear?

    ///////////
    if delete_file {
        repo.remove_path("file")?;
    } else {
        repo.write_file("file")?.write_all(b"a\nd\n")?;
    }
    record_all_output(
        repo.clone(),
        changes.clone(),
        txn.clone(),
        channel.clone(),
        "",
    )?;

    ///////////
    repo2.write_file("file")?.write_all(b"a\nb\nx\nc\nd\n")?;
    let h2 = record_all(repo2.clone(), &changes, txn2.clone(), channel2.clone(), "")?;

    repo2.write_file("file")?.write_all(b"a\nb\nx\nc\ny\nd\n")?;
    let h3 = record_all(repo2.clone(), &changes, txn2.clone(), channel2.clone(), "")?;

    ///////////
    apply::apply_change(
        &changes,
        &mut *txn.write().unwrap(),
        &mut *channel.write().unwrap(),
        &h2,
    )?;
    apply::apply_change(
        &changes,
        &mut *txn.write().unwrap(),
        &mut *channel.write().unwrap(),
        &h3,
    )?;
    output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;

    debug_to_file(&*txn.read().unwrap(), &channel, "debug_un")?;

    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h2)?;

    debug_to_file(&*txn.read().unwrap(), &channel, "debug_un2")?;
    Ok(())
}

#[test]
fn zombie_file_test() -> Result<(), anyhow::Error> {
    zombie_(None)
}

#[test]
fn zombie_lines_test() -> Result<(), anyhow::Error> {
    zombie_(Some(b"d\n"))
}

fn zombie_(file: Option<&[u8]>) -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let repo2 = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();
    repo.add_file("file", b"a\nb\nc\nd\n".to_vec());

    let env = pristine::sanakirja::Pristine::new_anon()?;
    let env2 = pristine::sanakirja::Pristine::new_anon()?;

    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));
    let txn2 = Arc::new(RwLock::new(env2.mut_txn_begin().unwrap()));
    txn.write().unwrap().add_file("file")?;

    let channel = (&mut *txn.write().unwrap()).open_or_create_channel("main")?;
    let h0 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;
    let channel2 = (&mut *txn2.write().unwrap()).open_or_create_channel("main")?;

    apply::apply_change(
        &changes,
        &mut *txn2.write().unwrap(),
        &mut *channel2.write().unwrap(),
        &h0,
    )?;
    output::output_repository_no_pending(
        repo2.clone(),
        &changes,
        txn2.clone(),
        channel2.clone(),
        "",
        true,
        None,
        1,
    )?;

    ///////////
    if let Some(file) = file {
        repo.write_file("file")?.write_all(file)?;
    } else {
        repo.remove_path("file")?;
    }
    let h1 = record_all_output(
        repo.clone(),
        changes.clone(),
        txn.clone(),
        channel.clone(),
        "",
    )?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug_a")?;

    ///////////

    repo2.write_file("file")?.write_all(b"a\nb\nx\nc\nd\n")?;
    let h2 = record_all_output(
        repo2.clone(),
        changes.clone(),
        txn2.clone(),
        channel2.clone(),
        "",
    )?;
    debug_to_file(&*txn2.read().unwrap(), &channel2, "debug_b")?;

    ///////////
    apply::apply_change(
        &changes,
        &mut *txn.write().unwrap(),
        &mut *channel.write().unwrap(),
        &h2,
    )?;

    debug_to_file(&*txn.read().unwrap(), &channel, "debug_un")?;
    debug!("unrecording");
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h2)?;
    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug_un2")?;
    let mut buf = Vec::new();
    if let Some(f) = file {
        if !conflicts.is_empty() {
            panic!("conflicts = {:#?}", conflicts)
        }
        repo.read_file("file", &mut buf)?;
        assert_eq!(&buf[..], f);
    } else {
        if conflicts.len() != 1 {
            panic!("conflicts = {:#?}", conflicts)
        }
        match conflicts[0] {
            Conflict::ZombieFile { ref path } => assert_eq!(path, "file"),
            ref c => panic!("c = {:#?}", c),
        }
    }

    let (alive_, reachable_) = check_alive(&*txn.read().unwrap(), &channel.read().unwrap().graph);
    if !alive_.is_empty() {
        panic!("alive: {:?}", alive_);
    }
    if !reachable_.is_empty() {
        panic!("reachable: {:?}", reachable_);
    }

    commit(txn)?;

    // Applying the symmetric.
    apply::apply_change(
        &changes,
        &mut *txn2.write().unwrap(),
        &mut *channel2.write().unwrap(),
        &h1,
    )?;
    debug_to_file(&*txn2.read().unwrap(), &channel2, "debug_un3")?;

    debug!("unrecording h1 = {:?}", h1);
    crate::unrecord::unrecord(&mut *txn2.write().unwrap(), &channel2, &changes, &h1)?;
    debug_to_file(&*txn2.read().unwrap(), &channel2, "debug_un4")?;
    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn2.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    if !conflicts.is_empty() {
        panic!("conflicts = {:#?}", conflicts)
    }
    Ok(())
}

// Should fail: we're resurrecting a file in a directory that doesn't
// exist any more.
#[test]
fn zombie_dir() -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();
    repo.add_file("a/b/c/d", b"a\nb\nc\nd\n".to_vec());

    let env = pristine::sanakirja::Pristine::new_anon()?;
    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));
    txn.write().unwrap().add_file("a/b/c/d")?;

    let channel = (&mut *txn.write().unwrap()).open_or_create_channel("main")?;
    record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    repo.remove_path("a/b/c/d")?;
    let h1 = record_all_output(
        repo.clone(),
        changes.clone(),
        txn.clone(),
        channel.clone(),
        "",
    )?;

    repo.remove_path("a/b")?;
    let _h2 = record_all_output(
        repo.clone(),
        changes.clone(),
        txn.clone(),
        channel.clone(),
        "",
    )?;
    output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    let files = repo.list_files();
    assert_eq!(files, &["a"]);
    debug!("files={:?}", files);

    debug_to_file(&*txn.read().unwrap(), &channel, "debug_un")?;
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h1)?;

    debug_to_file(&*txn.read().unwrap(), &channel, "debug_un2")?;

    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;

    match conflicts[0] {
        Conflict::ZombieFile { ref path } => assert_eq!(path, "a/b"),
        ref c => panic!("c = {:?}", c),
    }
    match conflicts[1] {
        Conflict::ZombieFile { ref path } => assert_eq!(path, "a/b/c"),
        ref c => panic!("c = {:?}", c),
    }

    let files = repo.list_files();
    debug!("files={:?}", files);
    assert_eq!(files, &["a", "a/b", "a/b/c", "a/b/c/d"]);

    let (alive_, reachable_) = check_alive(&*txn.read().unwrap(), &channel.read().unwrap().graph);
    if !alive_.is_empty() {
        panic!("alive: {:?}", alive_);
    }
    if !reachable_.is_empty() {
        panic!("reachable: {:?}", reachable_);
    }

    commit(txn)?;

    Ok(())
}

#[test]
fn nodep() -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();
    repo.add_file("dir/file", b"a\nb\nc\nd\n".to_vec());

    let env = pristine::sanakirja::Pristine::new_anon()?;

    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));
    txn.write().unwrap().add_file("dir/file")?;
    debug_inodes(&*txn.read().unwrap());

    let channel = (&mut *txn.write().unwrap()).open_or_create_channel("main")?;
    let h0 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    repo.write_file("dir/file")?.write_all(b"a\nx\nb\nd\n")?;

    let h1 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;
    debug_inodes(&*txn.read().unwrap());

    match crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h0) {
        Err(crate::unrecord::UnrecordError::ChangeIsDependedUpon { .. }) => {}
        _ => panic!("Should not be able to unrecord"),
    }

    debug_inodes(&*txn.read().unwrap());
    let channel2 = (&mut *txn.write().unwrap()).open_or_create_channel("main2")?;
    match crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel2, &changes, &h0) {
        Err(crate::unrecord::UnrecordError::ChangeNotInChannel { .. }) => {}
        _ => panic!("Should not be able to unrecord"),
    }

    for p in txn
        .read()
        .unwrap()
        .log(&*channel.read().unwrap(), 0)
        .unwrap()
    {
        debug!("p = {:?}", p);
    }

    debug_inodes(&*txn.read().unwrap());
    debug_to_file(&*txn.read().unwrap(), &channel, "debug")?;
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h1)?;

    for p in txn
        .read()
        .unwrap()
        .log(&*channel.read().unwrap(), 0)
        .unwrap()
    {
        debug!("p = {:?}", p);
    }

    debug_inodes(&*txn.read().unwrap());
    debug_to_file(&*txn.read().unwrap(), &channel, "debug2")?;
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h0)?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug3")?;

    output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;

    // Checking that unrecord doesn't delete files on the filesystem,
    // but updates the tree/revtree tables.
    let mut files = repo.list_files();
    files.sort();
    assert_eq!(files, &["dir", "dir/file"]);
    assert!(
        crate::fs::iter_working_copy(&*txn.read().unwrap(), Inode::ROOT)
            .next()
            .is_none()
    );
    commit(txn)?;

    Ok(())
}

#[test]
fn file_del() -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();

    let env = pristine::sanakirja::Pristine::new_anon()?;

    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));

    let channel = (&mut *txn.write().unwrap()).open_or_create_channel("main")?;

    repo.add_file("file", b"blabla".to_vec());
    txn.write().unwrap().add_file("file")?;
    let h0 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    repo.remove_path("file")?;
    let h = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    debug_to_file(&*txn.read().unwrap(), &channel, "debug")?;
    debug!("unrecord h");
    // Unrecording the deletion.
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h)?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug2")?;
    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    if !conflicts.is_empty() {
        panic!("conflicts = {:#?}", conflicts);
    }
    assert_eq!(repo.list_files(), vec!["file"]);

    // Unrecording the initial change.
    debug!("unrecord h0");
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h0)?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug3")?;
    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    if !conflicts.is_empty() {
        panic!("conflicts = {:#?}", conflicts);
    }
    let files = repo.list_files();
    // Unrecording a file addition shouldn't delete the file.
    assert_eq!(files, &["file"]);
    commit(txn)?;
    Ok(())
}

/// Unrecording a change that edits the file around a conflict marker.
#[test]
fn self_context() -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();

    let env = pristine::sanakirja::Pristine::new_anon()?;

    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));

    let mut channel = (&mut *txn.write().unwrap()).open_or_create_channel("main")?;

    repo.add_file("file", b"a\nb\n".to_vec());
    txn.write().unwrap().add_file("file")?;
    record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    let channel2 = (&mut *txn.write().unwrap()).fork(&channel, "main2")?;

    repo.write_file("file")?.write_all(b"a\nx\nb\n")?;
    record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;
    repo.write_file("file")?.write_all(b"a\ny\nb\n")?;
    let b = record_all(repo.clone(), &changes, txn.clone(), channel2.clone(), "")?;

    apply::apply_change(
        &changes,
        &mut *txn.write().unwrap(),
        &mut *channel.write().unwrap(),
        &b,
    )?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug")?;
    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    debug!("conflicts = {:#?}", conflicts);
    let mut buf = Vec::new();
    repo.read_file("file", &mut buf)?;
    debug!("buf = {:?}", std::str::from_utf8(&buf));
    assert_eq!(conflicts.len(), 1);
    match conflicts[0] {
        Conflict::Order { .. } => {}
        ref c => panic!("c = {:?}", c),
    }

    let mut buf = Vec::new();
    repo.read_file("file", &mut buf)?;
    let conflict: Vec<_> = std::str::from_utf8(&buf)?.lines().collect();
    {
        let mut w = repo.write_file("file")?;
        for l in conflict.iter() {
            if l.starts_with(">>>") {
                writeln!(w, "bla\n{}\nbli", l)?
            } else {
                writeln!(w, "{}", l)?
            }
        }
    }
    let c = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug2")?;

    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &mut channel, &changes, &c)?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug3")?;

    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    debug!("conflicts = {:#?}", conflicts);
    assert_eq!(conflicts.len(), 1);
    match conflicts[0] {
        Conflict::Order { .. } => {}
        ref c => panic!("c = {:?}", c),
    }

    let mut buf = Vec::new();
    repo.read_file("file", &mut buf)?;
    let mut conflict: Vec<_> = std::str::from_utf8(&buf)?.lines().collect();
    conflict.sort();
    assert_eq!(
        conflict,
        vec![
            "<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<",
            "================================",
            ">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>",
            "a",
            "b",
            "x",
            "y"
        ]
    );
    commit(txn)?;

    Ok(())
}

#[test]
fn rollback_lines() -> Result<(), anyhow::Error> {
    rollback_(false)
}

#[test]
fn rollback_file() -> Result<(), anyhow::Error> {
    rollback_(true)
}

fn rollback_(delete_file: bool) -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();

    let env = pristine::sanakirja::Pristine::new_anon()?;

    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));

    let channel = (&mut *txn.write().unwrap()).open_or_create_channel("main")?;

    // Write a-b-c
    repo.add_file("file", b"a\nb\nc\n".to_vec());
    txn.write().unwrap().add_file("file")?;
    record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    // Delete -b-
    if delete_file {
        repo.remove_path("file")?
    } else {
        repo.write_file("file")?.write_all(b"a\nd\n")?;
    }
    let h_del = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;

    // Rollback the deletion of -b-
    let p_del = changes.get_change(&h_del)?;
    debug!("p_del = {:#?}", p_del);
    let p_inv = p_del.inverse(
        &h_del,
        crate::change::ChangeHeader {
            authors: vec![],
            message: "rollback".to_string(),
            description: None,
            timestamp: chrono::Utc::now(),
        },
        Vec::new(),
    );
    let h_inv = changes.save_change(&p_inv)?;
    apply::apply_change(
        &changes,
        &mut *txn.write().unwrap(),
        &mut *channel.write().unwrap(),
        &h_inv,
    )?;
    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug")?;
    if !conflicts.is_empty() {
        panic!("conflicts = {:#?}", conflicts)
    }
    let mut buf = Vec::new();
    repo.read_file("file", &mut buf)?;
    assert_eq!(std::str::from_utf8(&buf), Ok("a\nb\nc\n"));

    // Unrecord the rollback
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h_inv)?;
    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug2")?;
    if !conflicts.is_empty() {
        panic!("conflicts = {:#?}", conflicts)
    }
    let mut buf = Vec::new();
    repo.read_file("file", &mut buf).unwrap();
    if delete_file {
        assert_eq!(std::str::from_utf8(&buf), Ok("a\nb\nc\n"));
    } else {
        assert_eq!(std::str::from_utf8(&buf), Ok("a\nd\n"));
    }

    commit(txn)?;

    Ok(())
}

/// Delete a line twice on two different channels, merge and unrecord
/// only one of them. Does the deleted edge reappear? It shouldn't.
#[test]
fn double_test() -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();

    let env = pristine::sanakirja::Pristine::new_anon()?;
    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));
    let channel = (&mut *txn.write().unwrap()).open_or_create_channel("main")?;
    let channel2 = (&mut *txn.write().unwrap()).open_or_create_channel("main2")?;

    repo.add_file("file", b"blabla\nblibli\nblublu\n".to_vec());
    txn.write().unwrap().add_file("file")?;
    let h0 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;
    debug!("h0 = {:?}", h0);

    apply::apply_change(
        &changes,
        &mut *txn.write().unwrap(),
        &mut *channel2.write().unwrap(),
        &h0,
    )?;

    // First deletion
    {
        let mut w = repo.write_file("file")?;
        writeln!(w, "blabla\nblublu")?;
    }
    let h1 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;
    debug!("h1 = {:?}", h1);

    debug_to_file(&*txn.read().unwrap(), &channel, "debug0")?;

    // Second deletion
    let h2 = record_all(repo.clone(), &changes, txn.clone(), channel2.clone(), "")?;
    debug!("h2 = {:?}", h2);

    // Both deletions together.
    debug!("applying");
    apply::apply_change(
        &changes,
        &mut *txn.write().unwrap(),
        &mut *channel.write().unwrap(),
        &h2,
    )?;

    debug_to_file(&*txn.read().unwrap(), &channel, "debug1a")?;
    debug_to_file(&*txn.read().unwrap(), &channel2, "debug1b")?;
    debug!("unrecord h");
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h2)?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug2")?;

    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    if !conflicts.is_empty() {
        panic!("conflicts = {:#?}", conflicts);
    }

    commit(txn)?;

    Ok(())
}

/// Same as `double` above, but with a (slightly) more convoluted change
/// dependency graph made by rolling the change back a few times.
#[test]
fn double_convoluted() -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();

    let env = pristine::sanakirja::Pristine::new_anon()?;
    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));
    let mut channel = (&mut *txn.write().unwrap()).open_or_create_channel("main")?;
    let mut channel2 = (&mut *txn.write().unwrap()).open_or_create_channel("main2")?;

    repo.add_file("file", b"blabla\nblibli\nblublu\n".to_vec());
    txn.write().unwrap().add_file("file")?;
    let h0 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;
    debug!("h0 = {:?}", h0);

    apply::apply_change(
        &changes,
        &mut *txn.write().unwrap(),
        &mut *channel2.write().unwrap(),
        &h0,
    )?;

    // First deletion
    {
        let mut w = repo.write_file("file")?;
        write!(w, "blabla\nblibli\n")?;
    }
    let h1 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;
    debug!("h1 = {:?}", h1);

    debug_to_file(&*txn.read().unwrap(), &channel, "debug0")?;

    // Second deletion
    {
        let mut w = repo.write_file("file")?;
        writeln!(w, "blabla")?;
    }
    let h2 = record_all(repo.clone(), &changes, txn.clone(), channel2.clone(), "")?;
    debug!("h2 = {:?}", h2);

    // Both deletions together, then unrecord on ~channel~.
    debug!("applying");
    apply::apply_change(
        &changes,
        &mut *txn.write().unwrap(),
        &mut *channel.write().unwrap(),
        &h2,
    )?;

    debug_to_file(&*txn.read().unwrap(), &channel, "debug1a")?;
    debug_to_file(&*txn.read().unwrap(), &channel2, "debug1b")?;
    debug!("unrecord h");
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &mut channel, &changes, &h2)?;
    debug_to_file(&*txn.read().unwrap(), &channel, "debug2")?;

    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    if !conflicts.is_empty() {
        panic!("conflicts = {:#?}", conflicts);
    }

    // Same on ~channel2~, but with a few extra layers of rollbacks in between.
    debug!("rolling back");
    apply::apply_change(
        &changes,
        &mut *txn.write().unwrap(),
        &mut *channel2.write().unwrap(),
        &h1,
    )?;
    let rollback = |h| {
        let p = changes.get_change(&h).unwrap();
        let p_inv = p.inverse(
            &h,
            crate::change::ChangeHeader {
                authors: vec![],
                message: "rollback".to_string(),
                description: None,
                timestamp: chrono::Utc::now(),
            },
            Vec::new(),
        );
        let h_inv = changes.save_change(&p_inv).unwrap();
        h_inv
    };
    let mut h = h2;
    for i in 0..6 {
        let r = rollback(h);
        apply::apply_change(
            &changes,
            &mut *txn.write().unwrap(),
            &mut *channel2.write().unwrap(),
            &r,
        )
        .unwrap();
        debug_to_file(&*txn.read().unwrap(), &channel2, format!("debug_{}", i))?;
        h = r
    }
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &mut channel2, &changes, &h1)?;
    debug_to_file(&*txn.read().unwrap(), &channel2, "debug_final")?;

    let conflicts = output::output_repository_no_pending(
        repo.clone(),
        &changes,
        txn.clone(),
        channel.clone(),
        "",
        true,
        None,
        1,
    )?;
    if !conflicts.is_empty() {
        panic!("conflicts = {:#?}", conflicts)
    }

    commit(txn)?;

    Ok(())
}

/// Delete the same file on two different channels, merge, unrecord each patch on the same channel. What happens to tree/revtree?
#[test]
fn double_file() -> Result<(), anyhow::Error> {
    env_logger::try_init().unwrap_or(());

    let repo = Arc::new(working_copy::memory::Memory::new());
    let changes = changestore::memory::Memory::new();

    let env = pristine::sanakirja::Pristine::new_anon()?;
    let txn = Arc::new(RwLock::new(env.mut_txn_begin().unwrap()));
    let channel = (&mut *txn.write().unwrap()).open_or_create_channel("main")?;
    let channel2 = (&mut *txn.write().unwrap()).open_or_create_channel("main2")?;

    repo.add_file("file", b"blabla\nblibli\nblublu\n".to_vec());
    txn.write().unwrap().add_file("file")?;
    let h0 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;
    debug!("h0 = {:?}", h0);

    apply::apply_change(
        &changes,
        &mut *txn.write().unwrap(),
        &mut *channel2.write().unwrap(),
        &h0,
    )?;

    // First deletion
    repo.remove_path("file")?;
    let h1 = record_all(repo.clone(), &changes, txn.clone(), channel.clone(), "")?;
    debug!("h1 = {:?}", h1);
    // Second deletion
    let h2 = record_all(repo.clone(), &changes, txn.clone(), channel2.clone(), "")?;
    debug!("h2 = {:?}", h2);

    // Both deletions together.
    debug!("applying");
    apply::apply_change(
        &changes,
        &mut *txn.write().unwrap(),
        &mut *channel.write().unwrap(),
        &h2,
    )?;

    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h1)?;
    crate::unrecord::unrecord(&mut *txn.write().unwrap(), &channel, &changes, &h2)?;

    let txn = txn.read().unwrap();
    let mut inodes = txn.iter_inodes().unwrap();
    assert!(inodes.next().is_some());
    assert!(inodes.next().is_none());
    Ok(())
}
