use chardetng::EncodingDetector;

use crate::pristine::InodeMetadata;
use crate::text_encoding::Encoding;

#[cfg(feature = "ondisk-repos")]
pub mod filesystem;
#[cfg(feature = "ondisk-repos")]
pub use filesystem::FileSystem;

pub mod memory;
pub use memory::Memory;

pub trait WorkingCopy {
    type Error: std::error::Error + Send;
    fn create_dir_all(&self, path: &str) -> Result<(), Self::Error>;
    fn file_metadata(&self, file: &str) -> Result<InodeMetadata, Self::Error>;
    fn read_file(&self, file: &str, buffer: &mut Vec<u8>) -> Result<(), Self::Error>;
    fn modified_time(&self, file: &str) -> Result<std::time::SystemTime, Self::Error>;
    fn remove_path(&self, name: &str) -> Result<(), Self::Error>;
    fn rename(&self, former: &str, new: &str) -> Result<(), Self::Error>;
    fn set_permissions(&self, name: &str, permissions: u16) -> Result<(), Self::Error>;

    type Writer: std::io::Write;
    fn write_file(&self, file: &str) -> Result<Self::Writer, Self::Error>;
    /// Read the file into the buffer
    ///
    /// Returns the file's text encoding or None if it was a binary file
    fn decode_file(
        &self,
        file: &str,
        buffer: &mut Vec<u8>,
    ) -> Result<Option<Encoding>, Self::Error> {
        let mut uncoded = Vec::new();
        self.read_file(&file, &mut uncoded)?;
        let mime = tree_magic::from_u8(&uncoded);
        debug!("mime = {:?}", mime);
        let encoding = if mime.starts_with("text/") {
            let mut detector = EncodingDetector::new();
            detector.feed(&uncoded, true);
            let encoding = detector.guess(None, true);
            debug!("guessed encoding = {:?}", encoding.name());
            let (_decoded, encoding, malformed) = encoding.decode(&uncoded);
            debug!("final encoding = {:?}", encoding.name());
            if !malformed {
                Some(Encoding(encoding))
            } else {
                warn!("text file was malformed");
                None
            }
        } else {
            None
        };
        buffer.append(&mut uncoded);
        Ok(encoding)
    }
}
