use std::fmt;

use serde::{de::Visitor, Deserialize, Serialize};

#[derive(Debug, PartialEq, Eq)]
pub struct Encoding(pub(crate) &'static encoding_rs::Encoding);

impl Encoding {
    pub(crate) fn for_label(label: &str) -> Encoding {
        Encoding(encoding_rs::Encoding::for_label_no_replacement(label.as_bytes()).unwrap())
    }

    pub(crate) fn label(&self) -> &str {
        self.0.name()
    }

    pub(crate) fn decode(&self, text: &[u8]) -> String {
        let (decoded, ..) = self.0.decode(&text);
        String::from_utf8(decoded.as_bytes().to_vec()).unwrap()
    }

    pub(crate) fn encode(&self, text: &str) -> Vec<u8> {
        let (encoded, ..) = self.0.encode(text);
        encoded.into_owned().to_vec()
    }
}

impl Clone for Encoding {
    fn clone(&self) -> Self {
        Encoding(self.0)
    }
}

impl Serialize for Encoding {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        serializer.serialize_str(self.label())
    }
}

struct EncodingVisitor;

impl<'de> Deserialize<'de> for Encoding {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        impl<'de> Visitor<'de> for EncodingVisitor {
            type Value = Encoding;

            fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
                formatter.write_str("a string label meeting the encoding standard https://encoding.spec.whatwg.org/#concept-encoding-get")
            }

            fn visit_str<E>(self, v: &str) -> Result<Self::Value, E>
            where
                E: serde::de::Error,
            {
                Ok(Encoding::for_label(v))
            }
        }

        deserializer.deserialize_str(EncodingVisitor)
    }
}
