use crate::{PdfObject, PdfObjectIdentifier, PdfStreamObject, PdfString, PdfUntypedDictionary};
use std::collections::HashMap;

pub struct PdfReaderPosition {
    index: usize,
}

impl PdfReaderPosition {
    #[must_use]
    pub const fn new() -> Self {
        Self { index: 0 }
    }
}

pub struct PdfParser<'a> {
    buffer: &'a [u8],
}

impl<'a> PdfParser<'a> {
    #[must_use]
    pub const fn new(buffer: &'a [u8]) -> PdfParser<'a> {
        PdfParser { buffer }
    }

    /// Advance the position to after the next newline.
    pub fn read_line(&self, position: &mut PdfReaderPosition) -> &[u8] {
        let start_position = position.index;

        let last_byte = loop {
            let current_byte = self.current_value(position);
            if current_byte == b'\n' || current_byte == b'\r' {
                break current_byte;
            } else if !self.advance_position(position) {
                break b'0';
            };
            // Continue.
        };

        self.advance_position(position);

        let slice = &self.buffer[start_position..position.index];

        if last_byte == b'\r' {
            self.advance_position_if_next(b'\n', position)
        }

        slice
    }

    fn advance_position(&self, position: &mut PdfReaderPosition) -> bool {
        if position.index < self.buffer.len() - 1 {
            position.index += 1;
            true
        } else {
            false
        }
    }

    fn advance_position_if_next(&self, next: u8, position: &mut PdfReaderPosition) {
        if position.index < self.buffer.len() - 1 && self.current_value(position) == next {
            position.index += 1;
        }
    }

    pub fn next_object(&self, position: &mut PdfReaderPosition) -> Option<PdfObject<'a>> {
        let next_word = self.next_word(position)?;
        if next_word == b"true" {
            return Some(PdfObject::Boolean(true));
        } else if next_word == b"false" {
            return Some(PdfObject::Boolean(false));
        } else if next_word == b"null" {
            return Some(PdfObject::Null);
        } else if next_word
            .iter()
            .enumerate()
            .all(|(idx, b)| b.is_ascii_digit() || (idx == 0 && matches!(b, b'+' | b'-')))
        {
            // TODO: use from_utf8 here?
            // TODO: Error handling (too big integer to fit in i32)
            let first_number = std::str::from_utf8(next_word).ok()?.parse::<i32>().ok()?;

            // This might be a object reference: "<obj> <gen> R".
            let index_after_first = position.index;
            if let Some(second_number) = self.parse_next::<u16>(position) {
                let third_word = self.next_word(position);
                if third_word == Some(b"R") {
                    // TODO: Check if numbers fits in u16
                    return Some(PdfObject::Reference(PdfObjectIdentifier::new(
                        first_number as u16,
                        second_number,
                    )));
                }
                // We might be reading an integer array.
                position.index = index_after_first;
            } else {
                position.index = index_after_first;
            }

            return Some(PdfObject::Integer(first_number));
        } else if next_word == b"/" {
            let current_byte = self.current_value(position);
            let name_word = if Self::is_whitespace(current_byte) || Self::is_delimiter(current_byte)
            {
                return Some(PdfObject::Name(
                    &self.buffer[position.index..position.index],
                ));
            } else {
                self.next_word(position)?
            };
            return if Self::is_delimiter(name_word[0]) {
                // TODO: Return error?
                None
            } else {
                Some(PdfObject::Name(name_word))
            };
        } else if next_word == b"[" {
            let mut array = Vec::new();
            // FIXME: Validate array ends with ']'
            while let Some(object) = self.next_object(position) {
                array.push(object);
            }
            return Some(PdfObject::Array(array));
        } else if next_word == b"<<" {
            let mut map = HashMap::new();
            // FIXME: Do not consume other type of object if it's out of place (a "non-name key")
            // FIXME: Validate dict ends with '>>'
            while let Some(PdfObject::Name(key)) = self.next_object(position) {
                let value = self.next_object(position)?;
                map.insert(key, value);
            }

            let index_after_dict = position.index;
            let next_word = self.next_word(position);
            if next_word == Some(b"stream") {
                if self.current_value(position) == b'\r' && self.buffer[position.index + 1] == b'\n'
                {
                    position.index += 2;
                } else if self.current_value(position) == b'\n' {
                    position.index += 1;
                } else {
                    unimplemented!("'stream' not followed by \r\n or \n");
                }

                if let Some(PdfObject::Integer(length)) = map.get(&b"Length"[..]) {
                    // Section 7.3.8: "The keyword stream that follows the stream dictionary shall
                    // be followed by an end-of-line marker consisting of either a CARRIAGE RETURN ´
                    // and a LINE FEED or just a LINE FEED, and not by a CARRIAGE RETURN alone. The
                    // sequence of bytes that make up a stream lie between the end-of-line marker
                    // following the stream keyword and the endstream keyword; the stream dictionary
                    // specifies the exact number of bytes. There should be an end-of-line marker
                    // after the data and before endstream; this marker shall not be included in the
                    // stream length. There shall not be any extra bytes, other than white space,
                    // between endstream and endobj."
                    // TODO: Verify stream length positive
                    let stream_length = *length as usize;
                    let bytes = &self.buffer[position.index..(position.index + stream_length)];
                    position.index += stream_length;
                    // TODO: Verify out of bounds
                    let stream_object = PdfStreamObject {
                        dictionary: PdfUntypedDictionary::new(map),
                        bytes,
                    };
                    let next_word = self.next_word(position)?;
                    if next_word == b"endstream" {
                        return Some(PdfObject::Stream(stream_object));
                    }
                    unimplemented!("Stream not ended with 'endstream'");
                } else {
                    unimplemented!("stream without /Length in dictionary");
                }
            } else {
                position.index = index_after_dict;
            }

            return Some(PdfObject::Dictionary(PdfUntypedDictionary::new(map)));
        } else if next_word == b"(" {
            // Start of string.
            let start_position = position.index;
            let mut last_was_slash = false;
            while (last_was_slash, self.current_value(position)) != (false, b')') {
                last_was_slash = self.current_value(position) == b'\\';
                if !self.advance_position(position) {
                    return None;
                }
            }
            let slice = &self.buffer[start_position..position.index];
            if !self.advance_position(position) {
                return None;
            }
            return Some(PdfObject::String(PdfString::new_literal(slice)));
        } else if next_word == b"<" {
            // Start of hexadecimal string.
            // TODO: Decode (on demand?)
            let start_position = position.index;
            while self.current_value(position) != b'>' {
                if !self.advance_position(position) {
                    return None;
                }
            }
            let slice = &self.buffer[start_position..position.index];
            if !self.advance_position(position) {
                return None;
            }
            return Some(PdfObject::String(PdfString::new_hexadecimal(slice)));
        }

        if let Ok(float_value) = std::str::from_utf8(next_word).ok()?.parse::<f32>() {
            return Some(PdfObject::Real(float_value));
        }

        None
    }

    pub fn next_indirect_object(
        &self,
        position: &mut PdfReaderPosition,
    ) -> Option<(PdfObjectIdentifier, PdfObject<'a>)> {
        let object_identifier = self.parse_next(position)?;
        let generation_number = self.parse_next(position)?;

        if self.next_word(position) != Some(b"obj") {
            return None;
        }
        let object = self.next_object(position)?;
        let w = self.next_word(position);
        if w != Some(b"endobj") {
            return None;
        }

        let object_identifier = PdfObjectIdentifier::new(object_identifier, generation_number);
        Some((object_identifier, object))
    }

    fn parse_next<T: core::str::FromStr>(&self, position: &mut PdfReaderPosition) -> Option<T> {
        let next_word = self.next_word(position)?;
        // TODO: Do not panic below. Avoid from_utf8?
        let next_str = std::str::from_utf8(next_word).ok()?;
        next_str.parse::<T>().ok()
    }

    /// Check if a byte represents a whitespace according to the PDF file format.
    ///
    /// See PDF 32000-1:2008, 7.2.2 Character Set.
    pub const fn is_whitespace(byte: u8) -> bool {
        matches!(byte, 0 | 9 | 10 | 12 | 13 | 32)
    }

    /// Check if a byte is a delimiter
    ///
    /// 7.2.2 Character Set: "The delimiter characters (, ), <, >, [, ], {, }, /, and % are special
    /// (LEFT PARENTHESIS (28h), RIGHT PARENTHESIS (29h), LESS-THAN SIGN (3Ch), GREATER-THAN SIGN
    /// (3Eh), LEFT SQUARE BRACKET (5Bh), RIGHT SQUARE BRACKET (5Dh), LEFT CURLY BRACE (7Bh), RIGHT
    /// CURLY BRACE (07Dh), SOLIDUS (2Fh) and PERCENT SIGN (25h), respectively). They delimit
    /// syntactic entities such as arrays, names, and comments. Any of these characters terminates
    /// the entity preceding it and is not included in the entity. Delimiter characters are allowed
    /// within the scope of a string when following the rules for composing strings; see 7.3.4.2,
    /// “Literal Strings”. The leading ( of a string does delimit a preceding entity and the closing
    /// ) of a string delimits the string’s end."
    const fn is_delimiter(byte: u8) -> bool {
        matches!(
            byte,
            b'(' | b')' | b'<' | b'>' | b'[' | b']' | b'{' | b'}' | b'/' | b'%'
        )
    }

    const fn at_eof(&self, position: &PdfReaderPosition) -> bool {
        position.index >= self.buffer.len()
    }

    const fn current_value(&self, position: &PdfReaderPosition) -> u8 {
        self.buffer[position.index]
    }

    /// Next word or None if no word follows (end of file or delimiter).
    pub fn next_word(&self, position: &mut PdfReaderPosition) -> Option<&'a [u8]> {
        loop {
            let current_char = self.current_value(position);
            if Self::is_whitespace(current_char) {
                if !self.advance_position(position) {
                    return None;
                }
            } else if current_char == b'%' {
                self.read_line(position);
                if self.at_eof(position) {
                    return None;
                }
            } else {
                break;
            }
        }

        let start_index = position.index;
        let start_char = self.current_value(position);
        let start_char_is_delimiter = Self::is_delimiter(start_char);
        if start_char_is_delimiter {
            if !self.advance_position(position) {
                // TODO: Return EOF error
                return None;
            }
            let next_char = self.current_value(position);
            if (start_char == b'<' && next_char == b'<')
                || (start_char == b'>' && next_char == b'>')
            {
                // The double delimiters << and >> are special - see section 7.2.3 of the PDF 1.7
                // specification.
                if !self.advance_position(position) {
                    // TODO: Return EOF error
                    return None;
                }
            }
            return Some(&self.buffer[start_index..position.index]);
        }

        loop {
            let current_char = self.current_value(position);
            if Self::is_whitespace(current_char) || Self::is_delimiter(current_char) {
                break;
            } else if !self.advance_position(position) {
                return None;
            }
        }

        Some(&self.buffer[start_index..position.index])
    }
}

#[cfg(test)]
mod tests {
    use crate::parse::{PdfParser, PdfReaderPosition};
    use crate::{
        PdfDocument, PdfDocumentData, PdfFormField, PdfObject, PdfObjectIdentifier, PdfString,
        PdfVersion,
    };

    #[test]
    fn parse_junk() {
        let pdf_bytes = b"hello world";
        let parse_result = PdfDocumentData::parse(pdf_bytes);
        assert_eq!(
            parse_result.err().unwrap(),
            "File not starting with '%PDF-'".to_string()
        );
    }

    #[test]
    fn parse_minimal() {
        fn assert_minimal_first_object(pdf: &PdfDocumentData) {
            if let Some(PdfObject::Dictionary(dict)) =
                pdf.objects.get(&PdfObjectIdentifier::new(1, 0))
            {
                assert_eq!(dict.map.len(), 2);
                assert_eq!(
                    dict.map.get(&b"Type"[..]),
                    Some(&PdfObject::Name(b"Catalog"))
                );
                assert_eq!(
                    dict.map.get(&b"Pages"[..]),
                    Some(&PdfObject::Reference(PdfObjectIdentifier::new(2, 0)))
                );
            } else {
                panic!("Object 1 0 was not parsed into a dictionary");
            }
        }

        fn assert_minimal_second_object(pdf: &PdfDocumentData) {
            if let Some(PdfObject::Dictionary(dict)) =
                pdf.objects.get(&PdfObjectIdentifier::new(2, 0))
            {
                assert_eq!(dict.map.len(), 4);
                assert_eq!(dict.map.get(&b"Type"[..]), Some(&PdfObject::Name(b"Pages")));
                assert_eq!(
                    dict.map.get(&b"Kids"[..]),
                    Some(&PdfObject::Array(vec![PdfObject::Reference(
                        PdfObjectIdentifier::new(3, 0)
                    )]))
                );
                assert_eq!(dict.map.get(&b"Count"[..]), Some(&PdfObject::Integer(1)));
                assert_eq!(
                    dict.map.get(&b"MediaBox"[..]),
                    Some(&PdfObject::Array(vec![
                        PdfObject::Integer(0),
                        PdfObject::Integer(0),
                        PdfObject::Integer(300),
                        PdfObject::Integer(144)
                    ]))
                );
            } else {
                panic!("Object 2 0 was not parsed into a dictionary");
            }
        }

        fn assert_minimal_third_object(pdf: &PdfDocumentData) {
            if let Some(PdfObject::Dictionary(dict)) =
                pdf.objects.get(&PdfObjectIdentifier::new(3, 0))
            {
                assert_eq!(dict.map.len(), 4);
                assert_eq!(dict.map.get(&b"Type"[..]), Some(&PdfObject::Name(b"Page")));
                assert_eq!(
                    dict.map.get(&b"Parent"[..]),
                    Some(&PdfObject::Reference(PdfObjectIdentifier::new(2, 0)))
                );
                assert_eq!(
                    dict.map.get(&b"Contents"[..]),
                    Some(&PdfObject::Reference(PdfObjectIdentifier::new(4, 0)))
                );
                if let Some(PdfObject::Dictionary(resources_dict)) = dict.map.get(&b"Resources"[..])
                {
                    assert_eq!(resources_dict.map.len(), 1);
                    if let Some(PdfObject::Dictionary(font_dict)) =
                        resources_dict.map.get(&b"Font"[..])
                    {
                        assert_eq!(font_dict.map.len(), 1);
                        if let Some(PdfObject::Dictionary(f1_dict)) = font_dict.map.get(&b"F1"[..])
                        {
                            assert_eq!(f1_dict.map.len(), 3);
                            assert_eq!(
                                f1_dict.map.get(&b"Type"[..]),
                                Some(&PdfObject::Name(b"Font"))
                            );
                            assert_eq!(
                                f1_dict.map.get(&b"Subtype"[..]),
                                Some(&PdfObject::Name(b"Type1"))
                            );
                            assert_eq!(
                                f1_dict.map.get(&b"BaseFont"[..]),
                                Some(&PdfObject::Name(b"Times-Roman"))
                            );
                        } else {
                            panic!("Failed to parse /Resources->/Font->/F1 dictionary");
                        }
                    } else {
                        panic!("Failed to parse /Resources->/Font dictionary");
                    }
                } else {
                    panic!("Failed to parse /Resources dictionary");
                }
            } else {
                panic!("Object 3 0 was not parsed into a dictionary");
            }
        }

        fn assert_minimal_fourth_object(pdf: &PdfDocumentData, unix_line_endings: bool) {
            if let Some(PdfObject::Stream(stream)) =
                pdf.objects.get(&PdfObjectIdentifier::new(4, 0))
            {
                assert_eq!(stream.dictionary.map.len(), 1);
                if unix_line_endings {
                    assert_eq!(stream.bytes.len(), 55);
                    assert_eq!(
                        stream.bytes,
                        b"  BT
    /F1 18 Tf
    0 0 Td
    (Hello World) Tj
  ET"
                    );
                } else {
                    assert_eq!(stream.bytes.len(), 59);
                    assert_eq!(
                        stream.bytes,
                        b"  BT\r
    /F1 18 Tf\r
    0 0 Td\r
    (Hello World) Tj\r
  ET"
                    );
                }
            } else {
                panic!("Object 4 0 was not parsed into a stream");
            }
        }

        fn assert_minimal_file(pdf_bytes: &[u8], unix_line_endings: bool) {
            // Minimal PDF file explained at https://brendanzagaeski.appspot.com/0004.html
            assert!(pdf_bytes.starts_with(b"%PDF-1.1"));

            let pdf = PdfDocumentData::parse(pdf_bytes).unwrap();
            assert_eq!(pdf.version, PdfVersion::Version11);
            assert_eq!(pdf.objects.len(), 4);

            assert_minimal_first_object(&pdf);
            assert_minimal_second_object(&pdf);
            assert_minimal_third_object(&pdf);
            assert_minimal_fourth_object(&pdf, unix_line_endings);

            assert_eq!(
                pdf.trailer.map.get(&b"Root"[..]),
                Some(&PdfObject::Reference(PdfObjectIdentifier::new(1, 0)))
            );
            assert_eq!(
                pdf.trailer.map.get(&b"Size"[..]),
                Some(&PdfObject::Integer(5))
            );
        }

        assert_minimal_file(include_bytes!("tests/assets/minimal.pdf"), true);
        assert_minimal_file(include_bytes!("tests/assets/minimal_crlf_l.pdf"), false);
    }

    #[test]
    fn parse_signed_signicat() {
        let pdf_bytes = include_bytes!("tests/assets/signed-by-signicat-example.pdf");
        let pdf = PdfDocumentData::parse(pdf_bytes).unwrap();
        assert_eq!(pdf.version, PdfVersion::Version17);
        assert_eq!(pdf.objects.len(), 110);

        let pdf = PdfDocument::parse(pdf_bytes).unwrap();
        let pdf_fields = pdf.catalog.interactive_form.unwrap().fields;
        assert_eq!(pdf_fields.len(), 1);
        if let PdfFormField::Signature(signature_field) = &pdf_fields[0] {
            if let Some(_signature) = &signature_field.signature {
                return;
            }
        }
        panic!("Failed parsing signature");
    }

    #[test]
    fn test_whitespace() {
        assert!(PdfParser::is_whitespace(b' '));
        assert!(PdfParser::is_whitespace(b'\n'));
        assert!(PdfParser::is_whitespace(b'\r'));
        assert!(!PdfParser::is_whitespace(b'a'));
        assert!(!PdfParser::is_whitespace(b'.'));
        assert!(!PdfParser::is_whitespace(b'!'));
    }

    #[test]
    fn test_read_line() {
        let bytes = b"123\n456";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(position.index, 0);
        parser.read_line(&mut position);
        assert_eq!(position.index, 4);
        assert_eq!(parser.current_value(&position), b'4');

        let bytes = b"123\r456";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        parser.read_line(&mut position);
        assert_eq!(position.index, 4);
        assert_eq!(parser.current_value(&position), b'4');

        let bytes = b"123\r\n456";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        parser.read_line(&mut position);
        assert_eq!(position.index, 5);
        assert_eq!(parser.current_value(&position), b'4');
    }

    #[test]
    fn test_next_word() {
        let bytes = b"obj ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(parser.next_word(&mut position), Some(&b"obj"[..]));

        let bytes = b"   obj<";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(parser.next_word(&mut position), Some(&b"obj"[..]));

        let bytes = b"   obj  ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(parser.next_word(&mut position), Some(&b"obj"[..]));

        let bytes = b"   obj<<";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(parser.next_word(&mut position), Some(&b"obj"[..]));
        assert_eq!(parser.next_word(&mut position), None);

        let bytes = b"   obj    endobj ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(parser.next_word(&mut position), Some(&b"obj"[..]));
        assert_eq!(parser.next_word(&mut position), Some(&b"endobj"[..]));
        assert_eq!(parser.next_word(&mut position), None);

        let bytes = b"  % a comment\n   obj    endobj ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(parser.next_word(&mut position), Some(&b"obj"[..]));
        assert_eq!(parser.next_word(&mut position), Some(&b"endobj"[..]));
        assert_eq!(parser.next_word(&mut position), None);

        let bytes = b" obj<< ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(parser.next_word(&mut position), Some(&b"obj"[..]));
        assert_eq!(parser.next_word(&mut position), Some(&b"<<"[..]));
        assert_eq!(parser.next_word(&mut position), None);

        let bytes = b" obj \n % a comment\n<< ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(parser.next_word(&mut position), Some(&b"obj"[..]));
        assert_eq!(parser.next_word(&mut position), Some(&b"<<"[..]));
        assert_eq!(parser.next_word(&mut position), None);
    }

    #[test]
    fn test_next_object() {
        let bytes = b"true ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::Boolean(true))
        );

        let bytes = b"false ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::Boolean(false))
        );

        let bytes = b"6.14 ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::Real(6.14))
        );

        let bytes = b"false false ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::Boolean(false))
        );
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::Boolean(false))
        );

        let bytes = b" 0 ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::Integer(0))
        );

        let bytes = b" -1 ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::Integer(-1))
        );

        let bytes = b" +1 ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::Integer(1))
        );

        let bytes = b" 2147483647 ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::Integer(2_147_483_647))
        );

        let bytes = b" -2147483648 ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::Integer(-2_147_483_648))
        );

        let bytes = b"/Type ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::Name(b"Type"))
        );

        let bytes = b" << /Type /Catalog /Value 1 >> << /Name /Value >> % A comment\n999 << /OtherName /OtherValue >> ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        if let Some(PdfObject::Dictionary(dictionary)) = parser.next_object(&mut position) {
            assert_eq!(dictionary.map.len(), 2);
            let entry = dictionary.map.get(&b"Type"[..]);
            assert_eq!(entry, Some(&PdfObject::Name(b"Catalog")));
            let entry = dictionary.map.get(&b"Value"[..]);
            assert_eq!(entry, Some(&PdfObject::Integer(1)));
        } else {
            panic!("Failed to parse dict");
        }
        if let Some(PdfObject::Dictionary(dictionary)) = parser.next_object(&mut position) {
            assert_eq!(dictionary.map.len(), 1);
            let entry = dictionary.map.get(&b"Name"[..]);
            assert_eq!(entry, Some(&PdfObject::Name(b"Value")));
        } else {
            panic!("Failed to parse dict");
        }
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::Integer(999))
        );
        if let Some(PdfObject::Dictionary(dictionary)) = parser.next_object(&mut position) {
            assert_eq!(dictionary.map.len(), 1);
            let entry = dictionary.map.get(&b"OtherName"[..]);
            assert_eq!(entry, Some(&PdfObject::Name(b"OtherValue")));
        } else {
            panic!("Failed to parse dict");
        }
    }

    #[test]
    fn test_parse_dictionaries() {
        let bytes = b"<</FT/Sig/T(Signature1)/V 1 0 R/F 132/Type/Annot/Subtype/Widget/Rect[0 0 0 0]/AP<</N 2 0 R>>/P 4 0 R/DR<<>>>> ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        if let Some(PdfObject::Dictionary(dictionary)) = parser.next_object(&mut position) {
            assert_eq!(dictionary.map.len(), 10);
            assert_eq!(
                dictionary.map.get(&b"FT"[..]),
                Some(&PdfObject::Name(b"Sig"))
            );
            assert_eq!(
                dictionary.map.get(&b"T"[..]),
                Some(&PdfObject::String(PdfString::new_literal(b"Signature1")))
            );
            assert_eq!(
                dictionary.map.get(&b"V"[..]),
                Some(&PdfObject::Reference(PdfObjectIdentifier::new(1, 0)))
            );
            assert_eq!(
                dictionary.map.get(&b"F"[..]),
                Some(&PdfObject::Integer(132))
            );
            assert_eq!(
                dictionary.map.get(&b"Type"[..]),
                Some(&PdfObject::Name(b"Annot"))
            );
        } else {
            panic!("Failed to parse dict");
        }

        let bytes = b"<</Contents <ff>/Reference 1>> ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        if let Some(PdfObject::Dictionary(dictionary)) = parser.next_object(&mut position) {
            assert_eq!(dictionary.map.len(), 2);
            assert_eq!(
                dictionary.map.get(&b"Contents"[..]),
                // TODO:
                Some(&PdfObject::String(PdfString::new_hexadecimal(b"ff")))
            );
            assert_eq!(
                dictionary.map.get(&b"Reference"[..]),
                Some(&PdfObject::Integer(1))
            );
        } else {
            panic!("Failed to parse dict");
        }
    }

    #[test]
    fn test_parse_dictionary_mapping_to_empty_name() {
        for bytes in [
            &b"<< /App << /Name / >> >> "[..],
            &b"<</App<</Name/>>>> "[..],
        ] {
            let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
            if let Some(PdfObject::Dictionary(dictionary)) = parser.next_object(&mut position) {
                assert_eq!(dictionary.map.len(), 1);
                if let Some(PdfObject::Dictionary(dictionary)) = dictionary.map.get(&b"App"[..]) {
                    assert_eq!(dictionary.map.len(), 1);
                    assert_eq!(
                        dictionary.map.get(&b"Name"[..]),
                        Some(&PdfObject::Name(b""))
                    );
                    return;
                }
            }
            panic!("Failed to parse dict");
        }
    }

    /// Test that "A dictionary entry whose value is null (see 7.3.9, 'Null object') shall be
    /// treated the same as if the entry does not exist".
    ///
    /// From 7.3.7 Dictionary objects:
    /// <https://www.adobe.com/content/dam/acom/en/devnet/pdf/pdfs/PDF32000_2008.pdf#page=26>
    ///
    /// TODO: Ensure this behaviour also when value is indirect reference to non-existing object.
    /// TODO: Perhaps by having this logic on a dictionary lookup method?
    #[test]
    #[ignore]
    fn test_parse_dictionary_with_null_value() {
        let bytes = b"<< /FirstKey 1 /SecondKey null /ThirdKey 3 >> ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        if let Some(PdfObject::Dictionary(dictionary)) = parser.next_object(&mut position) {
            assert_eq!(dictionary.map.len(), 2);
            assert_eq!(
                dictionary.map.get(&b"FirstKey"[..]),
                Some(&PdfObject::Integer(1))
            );
            assert_eq!(
                dictionary.map.get(&b"SecondKey"[..]),
                Some(&PdfObject::Integer(3))
            );
            return;
        }
        panic!("Failed to parse dict");
    }

    #[test]
    fn test_parse_null() {
        let bytes = b"null ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(parser.next_object(&mut position), Some(PdfObject::Null));
    }

    #[test]
    fn test_parse_names() {
        fn assert_parsing(from: &[u8], expected_name: &[u8]) {
            let (parser, mut position) = (PdfParser::new(from), PdfReaderPosition::new());
            assert_eq!(
                parser.next_object(&mut position),
                Some(PdfObject::Name(expected_name))
            );
        }

        assert_parsing(b"/Name1 ", b"Name1");
        assert_parsing(b"/ASomewhatLongerName ", b"ASomewhatLongerName");
        assert_parsing(
            b"/A;Name_With-Various***Characters? ",
            b"A;Name_With-Various***Characters?",
        );
        assert_parsing(b"/1.2 ", b"1.2");
        assert_parsing(b"/$$ ", b"$$");
        assert_parsing(b"/@pattern ", b"@pattern");
        assert_parsing(b"/.notdef ", b".notdef");
        // TODO: assert_parsing(b"/Lime#20Green ", b"Lime Green");
        // TODO: assert_parsing(b"/paired#28#29parentheses ", b"paired( )parentheses");
        // TODO: assert_parsing(b"/paired#28#29parentheses ", b"paired( )parentheses");
        // TODO: assert_parsing(b"/The_Key_of_F#23_Minor ", b"The_Key_of_F#_Minor");
        // TODO: assert_parsing(b"/A#42 ", b"AB");
        // TODO: assert_parsing(b"/ ", b"");
    }

    #[test]
    fn test_parse_strings() {
        let bytes = b"() ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::String(PdfString::new_literal(b"")))
        );

        let bytes = b"(hello, world) (second string)(third) ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::String(PdfString::new_literal(b"hello, world")))
        );
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::String(PdfString::new_literal(b"second string")))
        );
        assert_eq!(
            parser.next_object(&mut position),
            Some(PdfObject::String(PdfString::new_literal(b"third")))
        );

        let bytes = b"(hello \\(world\\) bye) ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(
            parser.next_object(&mut position),
            // TODO: Unescape backslash (on demand?)
            Some(PdfObject::String(PdfString::new_literal(
                b"hello \\(world\\) bye"
            )))
        );
    }

    #[test]
    fn test_parse_arrays() {
        let bytes = b"[549 6.14 false (Ralph) /SomeName ]";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        if let Some(PdfObject::Array(array)) = parser.next_object(&mut position) {
            assert_eq!(array.len(), 5);
            assert_eq!(array[0], PdfObject::Integer(549));
            assert_eq!(array[1], PdfObject::Real(6.14));
            assert_eq!(array[2], PdfObject::Boolean(false));
            assert_eq!(
                array[3],
                PdfObject::String(PdfString::new_literal(b"Ralph"))
            );
            assert_eq!(array[4], PdfObject::Name(b"SomeName"));
        } else {
            panic!("Failed to parse array");
        }

        let bytes = b"[] ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        if let Some(PdfObject::Array(array)) = parser.next_object(&mut position) {
            assert_eq!(array.len(), 0);
        } else {
            panic!("Failed to parse empty array");
        }
    }

    #[test]
    fn test_parse_streams() {
        let bytes = b"4 1 obj
  << /Length 55 >>
stream
  BT
    /F1 18 Tf
    0 0 Td
    (Hello World) Tj
  ET
endstream
endobj ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        let (parsed_id, object) = parser.next_indirect_object(&mut position).unwrap();
        assert_eq!(parsed_id, PdfObjectIdentifier::new(4, 1));
        if let PdfObject::Stream(stream) = object {
            assert_eq!(stream.dictionary.map.len(), 1);
            assert_eq!(stream.bytes.len(), 55);
            assert_eq!(
                stream.bytes,
                b"  BT
    /F1 18 Tf
    0 0 Td
    (Hello World) Tj
  ET"
            );
        } else {
            panic!("Failed parsing stream");
        }
    }

    #[test]
    fn test_parse_next() {
        let bytes = b"32 ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(parser.parse_next(&mut position), Some(32));

        let bytes = b" 32 ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(parser.parse_next(&mut position), Some(32));

        let bytes = b" 32 % a comment\n   33 34 ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        assert_eq!(parser.parse_next(&mut position), Some(32));
        assert_eq!(parser.parse_next(&mut position), Some(33));
        assert_eq!(parser.parse_next(&mut position), Some(34));
    }

    #[test]
    fn test_next_indirect_object() {
        let bytes = b"8 0 obj\n 77\nendobj ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        let id = PdfObjectIdentifier::new(8, 0);
        let object = PdfObject::Integer(77);
        assert_eq!(
            parser.next_indirect_object(&mut position),
            Some((id, object))
        );

        let bytes = b"7 2 obj 62 endobj ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        let parsed = parser.next_indirect_object(&mut position);
        let id = PdfObjectIdentifier::new(7, 2);
        let object = PdfObject::Integer(62);
        assert_eq!(parsed, Some((id, object)));

        let bytes = b"7 0 obj 62 endobj\n\n8 0 obj 63 endobj ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        let id = PdfObjectIdentifier::new(7, 0);
        let object = PdfObject::Integer(62);
        assert_eq!(
            parser.next_indirect_object(&mut position),
            Some((id, object))
        );
        let id = PdfObjectIdentifier::new(8, 0);
        let object = PdfObject::Integer(63);
        assert_eq!(
            parser.next_indirect_object(&mut position),
            Some((id, object))
        );

        let bytes = b"3 0 obj
  <<  /Type /Page
      /Parent 2 0 R
      /Resources
       << /Font
           << /F1
               << /Type /Font
                  /Subtype /Type1
                  /BaseFont /Times-Roman
               >>
           >>
       >>
      /Contents 4 1 R
  >>
endobj\n\n4 1 obj\n765\nendobj ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        let expected_id = PdfObjectIdentifier::new(3, 0);
        let (parsed_id, object) = parser.next_indirect_object(&mut position).unwrap();
        assert_eq!(parsed_id, expected_id);
        if let PdfObject::Dictionary(dict) = object {
            assert_eq!(dict.map.len(), 4);
            assert_eq!(dict.map.get(&b"Type"[..]), Some(&PdfObject::Name(b"Page")));
            assert_eq!(
                dict.map.get(&b"Parent"[..]),
                Some(&PdfObject::Reference(PdfObjectIdentifier::new(2, 0)))
            );
            assert_eq!(
                dict.map.get(&b"Contents"[..]),
                Some(&PdfObject::Reference(PdfObjectIdentifier::new(4, 1)))
            );
            if let Some(PdfObject::Dictionary(resources_dict)) = dict.map.get(&b"Resources"[..]) {
                assert_eq!(resources_dict.map.len(), 1);
                if let Some(PdfObject::Dictionary(font_dict)) = resources_dict.map.get(&b"Font"[..])
                {
                    assert_eq!(font_dict.map.len(), 1);
                    if let Some(PdfObject::Dictionary(f1_dict)) = font_dict.map.get(&b"F1"[..]) {
                        assert_eq!(f1_dict.map.len(), 3);
                        assert_eq!(
                            f1_dict.map.get(&b"Type"[..]),
                            Some(&PdfObject::Name(b"Font"))
                        );
                        assert_eq!(
                            f1_dict.map.get(&b"Subtype"[..]),
                            Some(&PdfObject::Name(b"Type1"))
                        );
                        assert_eq!(
                            f1_dict.map.get(&b"BaseFont"[..]),
                            Some(&PdfObject::Name(b"Times-Roman"))
                        );
                    } else {
                        panic!("Failed to parse /Resources->/Font->/F1 dictionary");
                    }
                } else {
                    panic!("Failed to parse /Resources->/Font dictionary");
                }
            } else {
                panic!("Failed to parse /Resources dictionary");
            }
        } else {
            panic!("Failed parsing dictionary");
        }
        let expected_id = PdfObjectIdentifier::new(4, 1);
        let (parsed_id, object) = parser.next_indirect_object(&mut position).unwrap();
        assert_eq!(parsed_id, expected_id);
        assert_eq!(object, PdfObject::Integer(765));
    }

    #[test]
    fn test_parsing_indirect_object() {
        let bytes = b"2 0 obj
  << /Type /Pages
     /Kids [3 1 R]
     /Count 1
     /MediaBox [1 2 300 144]
  >>
endobj ";
        let (parser, mut position) = (PdfParser::new(bytes), PdfReaderPosition::new());
        let (parsed_id, object) = parser.next_indirect_object(&mut position).unwrap();
        assert_eq!(parsed_id, PdfObjectIdentifier::new(2, 0));
        if let PdfObject::Dictionary(dict) = object {
            assert_eq!(dict.map.len(), 4);
            assert_eq!(dict.map.get(&b"Type"[..]), Some(&PdfObject::Name(b"Pages")));
            assert_eq!(
                dict.map.get(&b"Kids"[..]),
                Some(&PdfObject::Array(vec![PdfObject::Reference(
                    PdfObjectIdentifier::new(3, 1)
                )]))
            );
            assert_eq!(dict.map.get(&b"Count"[..]), Some(&PdfObject::Integer(1)));
            assert_eq!(
                dict.map.get(&b"MediaBox"[..]),
                Some(&PdfObject::Array(vec![
                    PdfObject::Integer(1),
                    PdfObject::Integer(2),
                    PdfObject::Integer(300),
                    PdfObject::Integer(144)
                ]))
            );
        } else {
            panic!("Failed to parse dict");
        }
    }
}
