use crate::{
    PdfDocumentData, PdfFormField, PdfInteractiveForm, PdfObject, PdfSignature, PdfSignatureField,
};

pub fn extract_interactive_forms(pdf: &PdfDocumentData) -> Option<PdfInteractiveForm> {
    if let Some(PdfObject::Dictionary(root)) = pdf.trailer.lookup(&b"Root"[..], pdf) {
        if let Some(PdfObject::Dictionary(acro_form)) = root.lookup(&b"AcroForm"[..], pdf) {
            if let Some(PdfObject::Array(field_objects)) = acro_form.lookup(&b"Fields"[..], pdf) {
                let form_fields = field_objects
                    .iter()
                    .filter_map(|field_object| {
                        let field_object = pdf.lookup(field_object);
                        if let PdfObject::Dictionary(field_dict) = field_object {
                            let field_type = match field_dict.lookup(&b"FT"[..], pdf) {
                                Some(PdfObject::Name(b"Btn")) => PdfFormField::Button,
                                Some(PdfObject::Name(b"Tx")) => PdfFormField::Text,
                                Some(PdfObject::Name(b"Ch")) => PdfFormField::Choice,
                                Some(PdfObject::Name(b"Sig")) => {
                                    if let Some(PdfObject::Dictionary(signature_dict)) =
                                        field_dict.lookup(&b"V"[..], pdf)
                                    {
                                        if let Some(PdfObject::String(signature_byte_string)) =
                                            signature_dict.lookup(&b"Contents"[..], pdf)
                                        {
                                            let signature_field = PdfSignatureField {
                                                signature: Some(PdfSignature {
                                                    contents: signature_byte_string
                                                        .decode_byte_string(),
                                                }),
                                            };
                                            return Some(PdfFormField::Signature(signature_field));
                                        }
                                    }
                                    PdfFormField::Signature(PdfSignatureField { signature: None })
                                }
                                _ => {
                                    // TODO: Report error?
                                    return None;
                                }
                            };

                            // let field_name = match field_dict.lookup(&b"FT"[..], pdf) {
                            // };
                            Some(field_type)
                        } else {
                            // TODO: Report error and ignore?
                            None
                        }
                    })
                    .collect();
                return Some(PdfInteractiveForm {
                    fields: form_fields,
                });
            }
        }
    }

    None
}
