//! `NcMenu*` methods and associated functions.

use super::{NcMenuItem, NcMenuSection};
use crate::{cstring, NcInput};
use core::ptr::null_mut;

#[allow(unused_imports)] // for doc comments
use crate::widgets::NcMenu;

mod menu;
mod options;

pub use menu::*;
pub use options::*;

/// # `NcMenuItem` Constructors
impl NcMenuItem {
    /// New NcMenuItem for [`NcMenu`].
    pub fn new(desc: &str, shortcut: NcInput) -> Self {
        let cs = cstring![desc];
        Self {
            // utf-8 menu item, NULL for horizontal separator
            desc: cs.into_raw(),

            // ´NcInput´ shortcut, all should be distinct
            shortcut,
        }
    }

    /// New empty NcMenuItem for [`NcMenu`].
    pub fn new_empty() -> Self {
        Self { desc: null_mut(), shortcut: NcInput::new_empty() }
    }
}

/// # `NcMenuSection` Constructors
///
// Must contain at least 1 NcMenuItem.
impl NcMenuSection {
    /// New `NcMenuSection` for [`NcMenu`].
    pub fn new(name: &str, items: &mut [NcMenuItem], shortcut: NcInput) -> Self {
        let cs = cstring![name];
        Self {
            // utf-8 name string
            name: cs.into_raw(),

            // array of itemcount `NcMenuItem`s
            items: items.as_mut_ptr(),

            //
            itemcount: items.len() as i32,

            // shortcut, will be underlined if present in name
            shortcut,
        }
    }

    /// New NcMenuSection separator for [`NcMenu`].
    ///
    pub fn new_separator() -> Self {
        Self { name: null_mut(), items: null_mut(), itemcount: 0, shortcut: NcInput::new_empty() }
    }
}
