pub use std::{
    error::Error as StdError,
    fmt::{Display, Formatter},
    path::PathBuf,
};

use crate::units::Error as UnitError;

use super::SensorSubFunctionType;

pub(super) type Result<T> = std::result::Result<T, Error>;

/// Error which can be returned from interacting with sensors.
#[allow(missing_docs)]
#[derive(Debug)]
pub enum Error {
    /// Error reading from sensor.
    Read {
        source: std::io::Error,
        path: PathBuf,
    },

    /// Error writing to sensor.
    Write {
        source: std::io::Error,
        path: PathBuf,
    },

    /// A UnitError occurred.
    UnitError { source: UnitError },

    /// You have insufficient rights. Try using the read only variant of whatever returned this error.
    InsufficientRights { path: PathBuf },

    /// The subfunction you requested is not supported by this sensor.
    SubtypeNotSupported { sub_type: SensorSubFunctionType },

    /// The sensor you tried to read from is faulty.
    FaultySensor,

    /// The sensor you tried to read from or write to is disabled.
    DisabledSensor,
}

impl StdError for Error {
    fn cause(&self) -> Option<&(dyn StdError + 'static)> {
        match self {
            Error::Read { source, .. } => Some(source),
            Error::Write { source, .. } => Some(source),
            Error::UnitError { source } => Some(source),
            Error::InsufficientRights { .. } => None,
            Error::SubtypeNotSupported { .. } => None,
            Error::FaultySensor => None,
            Error::DisabledSensor => None,
        }
    }
}

impl Display for Error {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            Error::Read { path, source } => write!(
                f,
                "Reading from sensor at {} failed: {}",
                path.display(),
                source
            ),
            Error::Write { path, source } => write!(
                f,
                "Writing to sensor at {} failed: {}",
                path.display(),
                source
            ),
            Error::UnitError { source } => write!(f, "Raw sensor error: {}", source),
            Error::InsufficientRights { path } => write!(
                f,
                "You have insufficient rights to read/write {}",
                path.display()
            ),
            Error::SubtypeNotSupported { sub_type } => {
                write!(f, "Sensor does not support the subtype {}", sub_type)
            }
            Error::FaultySensor => write!(f, "The sensor is faulty"),
            Error::DisabledSensor => write!(f, "The sensor is disabled"),
        }
    }
}

impl From<UnitError> for Error {
    fn from(raw_error: UnitError) -> Error {
        Error::UnitError { source: raw_error }
    }
}
