use crate::log_error;
use std::fmt::Debug;

/// Handles a given [`libcnb::Error`] in a consistent, Heroku specific, style.
///
/// This function is intended to be used inside [`libcnb::Buildpack::handle_error`].
///
/// It outputs generic libcnb errors in a consistent style using the [logging functions](log_error) from this
/// crate. Buildpack specific errors are handled by the passed custom handler.
///
/// # Example:
/// ```
/// use libcnb::build::{BuildContext, BuildResult};
/// use libcnb::Buildpack;
/// use libcnb::detect::{DetectContext, DetectResult};
/// use libcnb::generic::{GenericMetadata, GenericPlatform};
/// use libherokubuildpack::{handle_error_heroku, log_error};
///
/// #[derive(Debug)]
/// enum FooBuildpackError {
///     CannotExecuteFooBuildTool(std::io::Error),
///     InvalidFooDescriptorToml
/// }
///
/// fn handle_foo_buildpack_error(e: FooBuildpackError) -> i32 {
///     match e {
///         FooBuildpackError::InvalidFooDescriptorToml => {
///             log_error("Invalid foo.toml", "Your app's foo.toml is invalid!");
///             23
///         }
///         FooBuildpackError::CannotExecuteFooBuildTool(inner) => {
///             log_error("Cannot execute foo buildtool", format!("Cause: {}", &inner));
///             42
///         }
///     }
/// }
///
/// struct FooBuildpack;
///
/// impl Buildpack for FooBuildpack {
///     type Platform = GenericPlatform;
///     type Metadata = GenericMetadata;
///     type Error = FooBuildpackError;
///
///     // Omitted detect and build implementations...
///     # fn detect(&self, context: DetectContext<Self>) -> libcnb::Result<DetectResult, Self::Error> {
///     #     unimplemented!()
///     # }
///     #
///     # fn build(&self, context: BuildContext<Self>) -> libcnb::Result<BuildResult, Self::Error> {
///     #     unimplemented!()
///     # }
///
///     fn handle_error(&self, error: libcnb::Error<Self::Error>) -> i32 {
///         handle_error_heroku(handle_foo_buildpack_error, error)
///     }
/// }
/// ```
pub fn handle_error_heroku<F, E>(f: F, error: libcnb::Error<E>) -> i32
where
    E: Debug,
    F: Fn(E) -> i32,
{
    match error {
        libcnb::Error::BuildpackError(buildpack_error) => f(buildpack_error),
        libcnb_error => {
            log_error("Internal Buildpack Error", format!("{}", libcnb_error));
            INTERNAL_BUILDPACK_ERROR_EXIT_CODE
        }
    }
}

const INTERNAL_BUILDPACK_ERROR_EXIT_CODE: i32 = 100;

#[cfg(test)]
mod test {
    use super::handle_error_heroku;

    #[test]
    fn test() {
        let base_value: i32 = 23;
        let handler = |_| base_value * 2;

        let error = libcnb::Error::BuildpackError("()");

        assert_eq!(handle_error_heroku(handler, error), base_value * 2);
    }
}
