use crate::log_error;
use std::fmt::{Debug, Display};

/// A shared libcnb [`ErrorHandler`](libcnb::ErrorHandler) for Heroku buildpacks
///
/// It outputs generic libcnb errors in a consistent style using the [logging functions](log_error) from this
/// create. Buildpack specific errors are handled by a custom handler.
pub struct HerokuBuildpackErrorHandler<E> {
    buildpack_error_handler: Box<dyn Fn(E) -> i32>,
}

impl<E> HerokuBuildpackErrorHandler<E> {
    /// Constructs a new HerokuBuildpackErrorHandler with the given error handler for custom
    /// buildpack errors.
    ///
    /// # Example:
    /// ```
    /// use libherokubuildpack::{HerokuBuildpackErrorHandler, log_error};
    ///
    /// enum FooBuildpackError {
    ///     CannotExecuteFooBuildTool(std::io::Error),
    ///     InvalidFooDescriptorToml
    /// }
    ///
    /// fn handle_foo_buildpack_error(e: FooBuildpackError) -> i32 {
    ///     match e {
    ///         FooBuildpackError::InvalidFooDescriptorToml => {
    ///             log_error("Invalid foo.toml", "Your app's foo.toml is invalid!");
    ///             23
    ///         },
    ///         FooBuildpackError::CannotExecuteFooBuildTool(inner) => {
    ///             log_error("Cannot execute foo buildtool", format!("Cause: {}", &inner));
    ///             42
    ///         }
    ///     }
    /// }
    ///
    /// let libcnb_error_handler
    ///     = HerokuBuildpackErrorHandler::new(Box::new(handle_foo_buildpack_error));
    /// ```
    pub fn new(buildpack_error_handler: Box<dyn Fn(E) -> i32>) -> Self {
        HerokuBuildpackErrorHandler {
            buildpack_error_handler,
        }
    }
}

impl<E: Debug + Display> libcnb::ErrorHandler<E> for HerokuBuildpackErrorHandler<E> {
    fn handle_error(&self, error: libcnb::Error<E>) -> i32 {
        match error {
            libcnb::Error::BuildpackError(buildpack_error) => {
                (self.buildpack_error_handler)(buildpack_error)
            }
            libcnb_error => {
                log_error("Internal Buildpack Error", format!("{}", libcnb_error));
                INTERNAL_BUILDPACK_ERROR_EXIT_CODE
            }
        }
    }
}

const INTERNAL_BUILDPACK_ERROR_EXIT_CODE: i32 = 100;

#[cfg(test)]
mod test {
    use crate::HerokuBuildpackErrorHandler;
    use libcnb::ErrorHandler;

    #[test]
    fn test() {
        let base_value: i32 = 23;

        assert_eq!(
            HerokuBuildpackErrorHandler::new(Box::new(move |_| base_value * 2))
                .handle_error(libcnb::Error::BuildpackError("()")),
            46
        );
    }
}
