/*  Copyright (C) 2020 - 2022, J2 Innovations. */

#ifndef libhaystack.h
#define libhaystack.h

/* Warning, this file is autogenerated by Rust cbindgen. Don't modify this manually. */

#include <stdarg.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>

/**
 * Zinc encoding version
 */
#define VER 3.0

/**
 * Generic result type for API calls
 */
typedef enum ResultType {
  ERR = -1,
  FALSE = 0,
  TRUE = 1,
} ResultType;

/**
 * A Haystack Filter
 */
typedef struct Filter Filter;

/**
 *
 * The `Value` object is the universal type that can have any of supported
 * tag values specified by the [Project Haystack 4.0 spec](https://project-haystack.org/doc/docHaystack/Kinds).
 *
 * It is implemented as an Algebraic type (enum) than can be constructed
 * from build-in types, such as `bool` or `i32`, and dedicated Haystack types such as `Str`
 *
 * # Usage
 *
 * Creating a scalar Haystack number value
 * ```
 * use libhaystack::val::*;
 * use std::convert::TryFrom;
 *
 * let num = Value::make_number_unit(42.0, "°F");
 *
 * assert_eq!(num.is_number(), true);
 * assert_eq!(Number::try_from(&num).unwrap().unit, Some(String::from("°F")));
 * ```
 *
 * Creating complex structures such as a Dict
 * ```
 * use libhaystack::dict;
 * use libhaystack::val::*;
 *
 * let dict = Value::make_dict(dict!("strTagName" => Value::from("Str Tag value"),
 *                  "marker" => Value::make_marker()));
 * assert_eq!(dict.is_dict(), true);
 * ```
 *
 * Creating a Haystack Grid
 * ```
 * use libhaystack::dict;
 * use libhaystack::val::*;
 * use std::convert::TryFrom;
 *
 * let grid = Value::make_grid_from_dicts(vec![
 *     dict!("dis" => Value::from("First dict row"),
 *           "id" => Value::make_ref_gen()),
 *     dict!("dis" => Value::from("Second dict row"),
 *           "id" => Value::make_ref_gen())
 * ]);
 * assert_eq!(grid.is_grid(), true);
 * assert_eq!(Grid::try_from(&grid).unwrap().len(), 2);
 * ```
 *
 */
typedef struct Value Value;

/**
 * Get the latitude of a  [Coord](crate::val::Coord) [Value](crate::val::Value)
 * # Arguments
 * val A [Coord](crate::val::Coord) [Value](crate::val::Value)
 * # Returns
 * - value
 * - NAN if there was an error
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::coord::*;
 * # unsafe {
 * let val = haystack_value_make_coord(1.0, 2.0);
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_coord(val));
 * assert_eq!(haystack_value_get_coord_lat(val), 1.0);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
double haystack_value_get_coord_lat(const struct Value *val);

/**
 * Get the longitude of a  [Coord](crate::val::Coord) [Value](crate::val::Value)
 * # Arguments
 * val A [Coord](crate::val::Coord) [Value](crate::val::Value)
 * # Returns
 * - value
 * - NAN if there was an error
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::coord::*;
 * # unsafe {
 * let val = haystack_value_make_coord(1.0, 2.0);
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_coord(val));
 * assert_eq!(haystack_value_get_coord_long(val), 2.0);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
double haystack_value_get_coord_long(const struct Value *val);

/**
 * Get the year of a [Date](crate::val::Date) [Value](crate::val::Value)
 * # Arguments
 * val A [Date](crate::val::Date) [Value](crate::val::Value)
 * # Returns
 * - value
 * - -1 (u32::MAX) if there was an error
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::date::*;
 * # unsafe {
 * let val = haystack_value_make_date(2021, 8, 13);
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_date(val));
 * assert_eq!(haystack_value_get_date_year(val), 2021);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uint32_t haystack_value_get_date_year(const struct Value *val);

/**
 * Get the month of a [Date](crate::val::Date) [Value](crate::val::Value)
 * # Arguments
 * val A [Date](crate::val::Date) [Value](crate::val::Value)
 * # Returns
 * - value
 * - -1 (u32::MAX) if there was an error
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::date::*;
 * # unsafe {
 * let val = haystack_value_make_date(2021, 8, 13);
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_date(val));
 * assert_eq!(haystack_value_get_date_month(val), 8);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uint32_t haystack_value_get_date_month(const struct Value *val);

/**
 * Get the day of a [Date](crate::val::Date) [Value](crate::val::Value)
 * # Arguments
 * val A [Date](crate::val::Date) [Value](crate::val::Value)
 * # Returns
 * - value
 * - -1 (u32::MAX) if there was an error
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::date::*;
 * # unsafe {
 * let val = haystack_value_make_date(2021, 8, 13);
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_date(val));
 * assert_eq!(haystack_value_get_date_day(val), 13);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uint32_t haystack_value_get_date_day(const struct Value *val);

/**
 * Get the date of a [DateTime](crate::val::DateTime) [Value](crate::val::Value)
 * # Arguments
 * - val A [DateTime](crate::val::DateTime) [Value](crate::val::Value)
 * - utc True if need to return the utc variant, false returns the local variant
 * - result The Value to be updated with the result DateTime.
 * # Returns
 * - 1 (True) if the operation was successful, -1 otherwise in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::datetime::*;
 * # unsafe {
 * let date = haystack_value_make_date(2021, 8, 13);
 * let time = haystack_value_make_time(9, 45, 59);
 * # let date = Box::<Value>::leak(date.unwrap());
 * # let time = Box::<Value>::leak(time.unwrap());
 * let tz = std::ffi::CString::new("New_York").unwrap();
 * let val = haystack_value_make_tz_datetime(date, time, tz.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_datetime(val));
 * # let result = Box::into_raw(haystack_value_init());
 * assert_eq!(haystack_value_get_datetime_date(val, true, result), ResultType::TRUE);
 * # assert_eq!((*result).to_string(), (*date).to_string());
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_value_get_datetime_date(const struct Value *val,
                                                 bool utc,
                                                 struct Value *result);

/**
 * Get the time of a [Time](crate::val::Time) [Value](crate::val::Value)
 * # Arguments
 * - val A [DateTime](crate::val::DateTime) [Value](crate::val::Value)
 * - utc True if need to return the utc variant, false returns the local variant
 * - result The Value to be updated with the result Time.
 * # Returns
 * - 1 (True) if the operation was successful, -1 otherwise in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::datetime::*;
 * # unsafe {
 * let date = haystack_value_make_date(2021, 8, 13);
 * let time = haystack_value_make_time(9, 45, 59);
 * # let date = Box::<Value>::leak(date.unwrap());
 * # let time = Box::<Value>::leak(time.unwrap());
 * let tz = std::ffi::CString::new("New_York").unwrap();
 * let val = haystack_value_make_tz_datetime(date, time, tz.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_datetime(val));
 * # let result = Box::into_raw(haystack_value_init());
 * assert_eq!(haystack_value_get_datetime_time(val, true, result), ResultType::TRUE);
 * # assert_eq!((*result).to_string(), (*time).to_string());
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_value_get_datetime_time(const struct Value *val,
                                                 bool utc,
                                                 struct Value *result);

/**
 * Get the timezone of a [DateTime](crate::val::DateTime) [Value](crate::val::Value)
 * # Arguments
 * - val A [DateTime](crate::val::DateTime) [Value](crate::val::Value)
 * # Returns
 * The timezone name as a C String or `null` if there was an error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::datetime::*;
 * # unsafe {
 * let date = haystack_value_make_date(2021, 8, 13);
 * let time = haystack_value_make_time(9, 45, 59);
 * # let date = Box::<Value>::leak(date.unwrap());
 * # let time = Box::<Value>::leak(time.unwrap());
 * let tz = std::ffi::CString::new("New_York").unwrap();
 * let val = haystack_value_make_tz_datetime(date, time, tz.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_datetime(val));
 * let res = haystack_value_get_datetime_timezone(val) as *mut i8;
 * assert_eq!(std::ffi::CString::from_raw(res), tz);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
const char *haystack_value_get_datetime_timezone(const struct Value *val);

/**
 * Get number of elements of a [Dict](crate::val::Dict) [Value](crate::val::Value)
 * # Arguments
 * val A [Dict](crate::val::Dict) [Value](crate::val::Value)
 * # Returns
 * - number of elements
 * - -1 if there was an error getting the length, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::dict::*;
 * # unsafe {
 * let val = haystack_value_make_dict();
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_dict(val));
 * assert_eq!(haystack_value_get_dict_len(val), 0);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uintptr_t haystack_value_get_dict_len(const struct Value *val);

/**
 * Get the keys of a [Dict](crate::val::Dict) [Value](crate::val::Value)
 * as a [List](crate::val::List) [Value](crate::val::Value)
 * # Arguments
 * - val A [Dict](crate::val::Dict) [Value](crate::val::Value)
 * - result The Value to be updated with the result Str List.
 * # Returns
 * - 1 (True) if the operation was successful, -1 otherwise in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::dict::*;
 * # use crate::libhaystack::c_api::list::*;
 * # unsafe {
 * let dict = haystack_value_make_dict();
 * # let dict = Box::<Value>::leak(dict);
 * assert!(haystack_value_is_dict(dict));
 * let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * let key = std::ffi::CString::new("equip").unwrap();
 * haystack_value_insert_dict_entry(dict, key.as_ptr(), entry);
 * let keys = Box::into_raw(haystack_value_init());
 * assert_eq!(haystack_value_get_dict_keys(dict, keys), ResultType::TRUE);
 * assert_eq!(haystack_value_get_list_len(keys), 1);
 * let mut  result = std::ptr::null();
 * assert_eq!(haystack_value_get_list_entry_at(keys, 0, &mut result), ResultType::TRUE);
 * # assert_eq!((*result).to_string(), String::from("\"equip\""));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_value_get_dict_keys(const struct Value *val,
                                             struct Value *result);

/**
 * Insert an new element or updates existing key of a [Dict](crate::val::) [Value](crate::val::Value)
 * # Arguments
 * - val A [Dict](crate::val::Dict) [Value](crate::val::Value)
 * - key A C string for the dict key, must be valid Haystack key.
 * - entry A [Value](crate::val::Value) to be inserted into the dict
 * # Returns
 * - 1 (True) if the operation was successful, -1 otherwise in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::dict::*;
 * # unsafe {
 * let val = haystack_value_make_dict();
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_dict(val));
 * let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * let key = std::ffi::CString::new("equip").unwrap();
 * assert_eq!(haystack_value_insert_dict_entry(val, key.as_ptr(), entry), ResultType::TRUE);
 * assert_eq!(haystack_value_get_dict_len(val), 1);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_value_insert_dict_entry(struct Value *val,
                                                 const char *key,
                                                 const struct Value *entry);

/**
 * Get the element of a [Dict](crate::val::Dict) [Value](crate::val::Value) by the key
 * # Arguments
 * - val A [Dict](crate::val::Dict) [Value](crate::val::Value)
 * - result The Value to be updated.
 * # Returns
 * - 1 (True) if the operation was successful, 0 (false) if no element found, -1 otherwise in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::dict::*;
 * # unsafe {
 * let val = haystack_value_make_dict();
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_dict(val));
 * # let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * let key = std::ffi::CString::new("equip").unwrap();
 * haystack_value_insert_dict_entry(val, key.as_ptr(), entry);
 * assert_eq!(haystack_value_get_dict_len(val), 1);
 * let mut result = std::ptr::null();
 * assert_eq!(haystack_value_get_dict_entry(val, key.as_ptr(), &mut result), ResultType::TRUE);
 * assert_eq!(&*result, entry);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_value_get_dict_entry(struct Value *val,
                                              const char *key,
                                              const struct Value **result);

/**
 * Remove the element of a [Dict](crate::val::Dict) [Value](crate::val::Value) for the key, if it exists.
 * # Arguments
 * - val A [Dict](crate::val::Dict) [Value](crate::val::Value)
 * - key The key to be removed
 * # Returns
 * - 1 (True) if the operation was successful, -1 otherwise in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::dict::*;
 * # unsafe {
 * let val = haystack_value_make_dict();
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_dict(val));
 * # let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * let key = std::ffi::CString::new("equip").unwrap();
 * haystack_value_insert_dict_entry(val, key.as_ptr(), entry);
 * assert_eq!(haystack_value_get_dict_len(val), 1);
 * assert_eq!(haystack_value_remove_dict_entry(val, key.as_ptr()), ResultType::TRUE);
 * assert_eq!(haystack_value_get_dict_len(val), 0);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_value_remove_dict_entry(struct Value *val,
                                                 const char *key);

/**
 * Parses a [Filter](crate::filter::Filter) from  a C string.
 * # Arguments
 * val The Haystack filter as a C string.
 * # Returns
 * The Filter object or None if error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::filter::Filter;
 * # use crate::libhaystack::c_api::filter::*;
 * # unsafe {
 * let str = std::ffi::CString::new("site and dis==\"Some site\"").unwrap();
 * let filter = haystack_filter_parse(str.as_ptr());
 * # let filter = Box::<Filter>::into_raw(filter.unwrap());
 * assert_ne!(filter, std::ptr::null_mut());
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
struct Filter *haystack_filter_parse(const char *val);

/**
 * Uses a [Filter](crate::filter::Filter) to match against a [Dict](crate::val::Dict)
 * [Value](crate::val::Value).
 * # Arguments
 * - filter The Haystack filter.
 * - val The Dict Value.
 * # Returns
 * 1 (True) if filter matches, 0 (false) when no march, or -1 when error is encountered, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::filter::Filter;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::filter::*;
 * # use crate::libhaystack::c_api::dict::*;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let str = std::ffi::CString::new("equip").unwrap();
 * let filter = haystack_filter_parse(str.as_ptr());
 * # let filter = Box::<Filter>::into_raw(filter.unwrap());
 * let dict = haystack_value_make_dict();
 * # let dict = Box::<Value>::leak(dict);
 * let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * let key = std::ffi::CString::new("equip").unwrap();
 * haystack_value_insert_dict_entry(dict, key.as_ptr(), entry);
 * assert!(haystack_filter_match_dict(filter, dict) == ResultType::TRUE);
 *
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_filter_match_dict(const struct Filter *filter,
                                           const struct Value *dict);

/**
 * Uses a [Filter](crate::filter::Filter) to match against a [Grid](crate::val::Grid) [Value](crate::val::Value) and return
 * first matching record, or None if no matches found.
 * # Arguments
 * - filter The Haystack filter.
 * - val The Grid Value.
 * - result The Value to be updated with the result Dict.
 * # Returns
 * 1 (True) if filter matches, 0 (false) when no march, or -1 when error is encountered, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::list::*;
 * # use crate::libhaystack::c_api::grid::*;
 * # use crate::libhaystack::c_api::dict::*;
 * # use crate::libhaystack::filter::Filter;
 * # use crate::libhaystack::c_api::filter::*;
 * # use crate::libhaystack::c_api::zinc::*;
 * # unsafe {
 * // Make first row
 * let row1 = haystack_value_make_dict();
 * # let row1 = Box::<Value>::leak(row1);
 * let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * let key = std::ffi::CString::new("equip").unwrap();
 * haystack_value_insert_dict_entry(row1, key.as_ptr(), entry);
 *
 * // Make second row
 * let row2 = haystack_value_make_dict();
 * # let row2 = Box::<Value>::leak(row2);
 * let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * let key = std::ffi::CString::new("site").unwrap();
 * haystack_value_insert_dict_entry(row2, key.as_ptr(), entry);
 *
 * // Make a list of rows
 * let list = haystack_value_make_list();
 * # let list = Box::<Value>::leak(list);
 * haystack_value_push_list_entry(list, row1);
 * haystack_value_push_list_entry(list, row2);
 *
 * // Make the Grid
 * let grid = haystack_value_make_grid_from_rows(list);
 * # let grid = Box::<Value>::leak(grid.unwrap());
 *
 * // Make a filter
 * let str = std::ffi::CString::new("equip").unwrap();
 * let filter = haystack_filter_parse(str.as_ptr());
 * # let filter = Box::<Filter>::into_raw(filter.unwrap());
 * # let result = Box::into_raw(haystack_value_init());
 * assert_eq!(haystack_filter_first_match_in_grid(filter, grid, result), ResultType::TRUE);
 * # assert_eq!((*result).to_string(), (*row1).to_string());
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_filter_first_match_in_grid(const struct Filter *filter,
                                                    const struct Value *grid,
                                                    struct Value *result);

/**
 * Uses a [Filter](crate::filter::Filter) to match against a [Grid](crate::val::Grid) [Value](crate::val::Value) and return a
 * new [Grid](crate::val::Grid) [Value](crate::val::Value) with the matching records.
 * # Arguments
 * - filter The Haystack filter.
 * - val The Grid Value.
 * - result The Value to be updated with the result Grid.
 * # Returns
 * 1 (True) if filter matches, 0 (false) when no march, or -1 when error is encountered, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::{Value, Dict};
 * # use crate::libhaystack::dict;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::list::*;
 * # use crate::libhaystack::c_api::grid::*;
 * # use crate::libhaystack::c_api::dict::*;
 * # use crate::libhaystack::filter::Filter;
 * # use crate::libhaystack::c_api::filter::*;
 * # unsafe {
 * // Make a row
 * let dict = haystack_value_make_dict();
 * # let dict = Box::<Value>::leak(dict);
 * let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * let key = std::ffi::CString::new("equip").unwrap();
 * haystack_value_insert_dict_entry(dict, key.as_ptr(), entry);
 * // Make a list of rows
 * let list = haystack_value_make_list();
 * # let list = Box::<Value>::leak(list);
 * haystack_value_push_list_entry(list, dict);
 * // Make the Grid
 * let grid = haystack_value_make_grid_from_rows(list);
 * # let grid = Box::<Value>::leak(grid.unwrap());
 * // Make a filter
 * let str = std::ffi::CString::new("equip").unwrap();
 * let filter = haystack_filter_parse(str.as_ptr());
 * # let filter = Box::<Filter>::into_raw(filter.unwrap());
 * let mut result = Box::into_raw(haystack_value_init());
 * assert_eq!(haystack_filter_match_all_grid(filter, grid, result), ResultType::TRUE);
 * # let test = Value::make_grid_from_dicts(vec![dict!{"equip" => Value::Marker}]);
 * # assert_eq!((*result).to_string(), test.to_string());
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_filter_match_all_grid(const struct Filter *filter,
                                               const struct Value *grid,
                                               struct Value *result);

/**
 * Get number of rows of a [Grid](crate::val::Grid) [Value](crate::val::Value)
 * # Arguments
 * val A [Grid](crate::val::Grid) [Value](crate::val::Value)
 * # Returns
 * - number of elements
 * - -1 if there was an error getting the length
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::grid::*;
 * # unsafe {
 * let val = haystack_value_make_grid();
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_grid(val));
 * assert_eq!(haystack_value_get_grid_len(val), 0);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uintptr_t haystack_value_get_grid_len(const struct Value *val);

/**
 * Construct a [Grid](crate::val::Grid) [Value](crate::val::Value) from a [List](crate::val::List) [Value](crate::val::Value) of
 * [Dict](crate::val::Dict) [Value](crate::val::Value)s
 * # Arguments
 * - rows List](crate::val::List) [Value](crate::val::Value) of
 * [Dict](crate::val::Dict) [Value](crate::val::Value)s
 * # Returns
 * - The value pointer
 * - None if there was an error, in which case use [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::list::*;
 * # use crate::libhaystack::c_api::grid::*;
 * # use crate::libhaystack::c_api::dict::*;
 * # unsafe {
 * // Make a row
 * let dict = haystack_value_make_dict();
 * # let dict = Box::<Value>::leak(dict);
 * let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * let key = std::ffi::CString::new("equip").unwrap();
 * haystack_value_insert_dict_entry(dict, key.as_ptr(), entry);
 * // Make a list of rows
 * let list = haystack_value_make_list();
 * # let list = Box::<Value>::leak(list);
 * haystack_value_push_list_entry(list, dict);
 * // Make the Grid
 * let grid = haystack_value_make_grid_from_rows(list);
 * # let grid = Box::<Value>::leak(grid.unwrap());
 * assert!(grid.is_grid());
 * assert_eq!(haystack_value_get_grid_len(grid), 1);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_grid_from_rows(const struct Value *rows);

/**
 * Construct a [Grid](crate::val::Grid) [Value](crate::val::Value) from a [List](crate::val::List) [Value](crate::val::Value) of
 * [Dict](crate::val::Dict) [Value](crate::val::Value)s
 * # Arguments
 * - rows a [List](crate::val::List) [Value](crate::val::Value) of
 * [Dict](crate::val::Dict) [Value](crate::val::Value)s
 * # Returns
 * - The value pointer
 * - None if there was an error, in which case use [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::list::*;
 * # use crate::libhaystack::c_api::grid::*;
 * # use crate::libhaystack::c_api::dict::*;
 * # unsafe {
 * // Make a row
 * let dict = haystack_value_make_dict();
 * # let dict = Box::<Value>::leak(dict);
 * let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * let key = std::ffi::CString::new("equip").unwrap();
 * haystack_value_insert_dict_entry(dict, key.as_ptr(), entry);
 * // Make a list of rows
 * let list = haystack_value_make_list();
 * # let list = Box::<Value>::leak(list);
 * haystack_value_push_list_entry(list, dict);
 * // Make the Grid
 * let grid = haystack_value_make_grid_from_rows(list);
 * # let grid = Box::<Value>::leak(grid.unwrap());
 * assert!(grid.is_grid());
 * assert_eq!(haystack_value_get_grid_len(grid), 1);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_grid_from_rows_with_meta(const struct Value *rows,
                                                           const struct Value *meta);

/**
 * Get the row of a [Grid](crate::val::Grid) [Value](crate::val::Value) at the specified index
 * # Arguments
 * val A [Grid](crate::val::Grid) [Value](crate::val::Value)
 * # Returns
 * - The row at the specified index, or None if there was an error, in which case use [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::list::*;
 * # use crate::libhaystack::c_api::grid::*;
 * # use crate::libhaystack::c_api::dict::*;
 * # unsafe {
 * // Make a row
 * let dict = haystack_value_make_dict();
 * # let dict = Box::<Value>::leak(dict);
 * let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * let key = std::ffi::CString::new("equip").unwrap();
 * haystack_value_insert_dict_entry(dict, key.as_ptr(), entry);
 * // Make a list of rows
 * let list = haystack_value_make_list();
 * # let list = Box::<Value>::leak(list);
 * haystack_value_push_list_entry(list, dict);
 * // Make the Grid
 * let grid = haystack_value_make_grid_from_rows(list);
 * # let grid = Box::<Value>::leak(grid.unwrap());
 * let result = Box::into_raw(haystack_value_init());
 * assert_eq!(haystack_value_get_grid_row_at(grid, 0, result), ResultType::TRUE);
 * assert_eq!(*result, *dict);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_value_get_grid_row_at(struct Value *val,
                                               uintptr_t index,
                                               struct Value *result);

/**
 * Encodes a [Value](crate::val::Value) to a JSON string
 * # Arguments
 * - val Value to encode as a JSON string
 * # Return
 * a `C string` with the JSON encoded value
 * or `null` if there was an error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Safety
 * Panics on invalid input data
 */
const char *haystack_value_to_json_string(const struct Value *val);

/**
 * Decodes a [Value](crate::val::Value) from a JSON string
 * # Arguments
 * - val `C string` with the JSON encoded value
 * # Return
 * the decoded [Value](crate::val::Value) or `null` if there was an error,
 * in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_from_json_string(const char *input);

/**
 * Get number of elements of a [List](crate::val::List) [Value](crate::val::Value)
 * # Arguments
 * val A [List](crate::val::List) [Value](crate::val::Value)
 * # Returns
 * - number of elements
 * - -1 if there was an error getting the length
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::list::*;
 * # unsafe {
 * let val = haystack_value_make_list();
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_list(val));
 * assert_eq!(haystack_value_get_list_len(val), 0);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uintptr_t haystack_value_get_list_len(struct Value *val);

/**
 * Push an element to a [List](crate::val::List) [Value](crate::val::Value)
 * # Arguments
 * - val A [List](crate::val::List) [Value](crate::val::Value)
 * - entry A [Value](crate::val::Value) to be inserted into the list
 * # Returns
 * - 1 (True) if the operation was successful, -1 otherwise in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::list::*;
 * # unsafe {
 * let val = haystack_value_make_list();
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_list(val));
 * # let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * assert_eq!(haystack_value_push_list_entry(val, entry), ResultType::TRUE);
 * assert_eq!(haystack_value_get_list_len(val), 1);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_value_push_list_entry(struct Value *val,
                                               const struct Value *entry);

/**
 * Get the element of a [List](crate::val::List) [Value](crate::val::Value) at the specified index
 * # Arguments
 * - val A [List](crate::val::List) [Value](crate::val::Value)
 * - result The Value to be updated with the result.
 * # Returns
 * - 1 (True) if the operation was successful, -1 otherwise in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::list::*;
 * # unsafe {
 * let val = haystack_value_make_list();
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_list(val));
 * # let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * haystack_value_push_list_entry(val, entry);
 * let mut result = std::ptr::null();
 * assert_eq!(haystack_value_get_list_entry_at(val, 0, &mut result), ResultType::TRUE);
 * assert_eq!(*result, *entry);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_value_get_list_entry_at(const struct Value *val,
                                                 uintptr_t index,
                                                 const struct Value **result);

/**
 * Set the element of a [List](crate::val::List) [Value](crate::val::Value) at the specified index
 * # Arguments
 * - val A [List](crate::val::List) [Value](crate::val::Value)
 * - index The index were to inset the value at
 * - element  A  [Value](crate::val::Value)
 * # Returns
 * - 1 (True) if the operation was successful, -1 otherwise in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::list::*;
 * # unsafe {
 * let val = haystack_value_make_list();
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_list(val));
 * let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * haystack_value_push_list_entry(val, entry);
 * let mut result = std::ptr::null();
 * assert_eq!(haystack_value_get_list_entry_at(val, 0, &mut result), ResultType::TRUE);
 * assert_eq!(*result, *entry);
 * let entry = haystack_value_make_remove();
 * # let entry = Box::<Value>::leak(entry);
 * haystack_value_set_list_entry_at(val, 0, entry);
 * assert_eq!(haystack_value_get_list_entry_at(val, 0, &mut result), ResultType::TRUE);
 * assert_eq!(*result, *entry);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_value_set_list_entry_at(struct Value *val,
                                                 uintptr_t index,
                                                 struct Value *entry);

/**
 * Remove the element of a [List](crate::val::List) [Value](crate::val::Value) at the specified index
 * # Arguments
 * - val A [List](crate::val::List) [Value](crate::val::Value)
 * - index The index were to inset the value at
 * # Returns
 * - 1 (True) if the operation was successful, -1 otherwise in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::list::*;
 * # unsafe {
 * let val = haystack_value_make_list();
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_list(val));
 * let entry = haystack_value_make_marker();
 * # let entry = Box::<Value>::leak(entry);
 * haystack_value_push_list_entry(val, entry);
 * let mut result = std::ptr::null();
 * assert_eq!(haystack_value_get_list_entry_at(val, 0, &mut result), ResultType::TRUE);
 * haystack_value_remove_list_entry_at(val, 0);
 * assert_eq!(haystack_value_get_list_len(val), 0);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_value_remove_list_entry_at(struct Value *val,
                                                    uintptr_t index);

/**
 * Get value of a [Number](crate::val::Number) [Value](crate::val::Value)
 * # Arguments
 * val A [Number](crate::val::Number) [Value](crate::val::Value)
 * # Returns
 * - value
 * - NAN if there was an error getting the string length
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::number::*;
 * # unsafe {
 * let val = haystack_value_make_number(100.0);
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_number(val));
 * assert_eq!(haystack_value_get_number_value(val), 100.0);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
double haystack_value_get_number_value(const struct Value *val);

/**
 * True if [Number](crate::val::Number) [Value](crate::val::Value) has an unit
 * # Arguments
 * val A [Number](crate::val::Number) [Value](crate::val::Value) to test
 * # Returns
 * - 1 if has unit, 0 otherwise
 * - -1 if there was an error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::ResultType;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::number::*;
 * # unsafe {
 * let val = haystack_value_make_number(100.0);
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_number(val));
 * assert!(haystack_value_number_has_unit(val) == ResultType::FALSE);
 * let unit = std::ffi::CString::new("kwh").unwrap();
 * let val = haystack_value_make_number_with_unit(42.0, unit.as_ptr()).unwrap();
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_number(val));
 * assert!(haystack_value_number_has_unit(val) == ResultType::TRUE);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
enum ResultType haystack_value_number_has_unit(const struct Value *val);

/**
 * Get the character of a [Number](crate::val::Number) [Value](crate::val::Value) unit
 * # Arguments
 * val A [Number](crate::val::Number) [Value](crate::val::Value) to get the unit from.
 * # Returns
 * The C string or `null` if there was an error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::number::*;
 * let unit = std::ffi::CString::new("kwh").unwrap();
 * # unsafe {
 * let val = haystack_value_make_number_with_unit(42.0, unit.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_number(val));
 * let res = haystack_value_get_number_unit(val) as *mut i8;
 * assert_eq!(std::ffi::CString::from_raw(res), unit);
 * }
 * ```
 * # Safety
 * Panics on invalid input data
 */
const char *haystack_value_get_number_unit(const struct Value *val);

/**
 * Get number of character, without trailing zero, of a [Ref](crate::val::Ref) [Value](crate::val::Value)
 * # Arguments
 * val A [Ref](crate::val::Ref) [Value](crate::val::Value) to get the len from
 * # Returns
 * - the size of the string
 * - -1 (usize::MAX) if there was an error getting the string length
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::reference::*;
 * # unsafe {
 * let id = std::ffi::CString::new("someId").unwrap();
 * let val = haystack_value_make_ref(id.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_ref(val));
 * assert_eq!(haystack_value_get_ref_value_len(val), 6);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uintptr_t haystack_value_get_ref_value_len(const struct Value *val);

/**
 * Get the character of a [Ref](crate::val::Ref) [Value](crate::val::Value)
 * # Arguments
 * val A [Ref](crate::val::Ref) [Value](crate::val::Value) to get the C string from.
 * # Returns
 * The C string or `null` if there was an error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::reference::*;
 * # unsafe {
 * let id = std::ffi::CString::new("someId").unwrap();
 * let val = haystack_value_make_ref(id.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * let res = haystack_value_get_ref_value(val) as *mut i8;
 * assert_eq!(std::ffi::CString::from_raw(res), id);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
const char *haystack_value_get_ref_value(const struct Value *val);

/**
 * Get the character of a [Ref](crate::val::Ref) [Value](crate::val::Value)
 * # Arguments
 * val A [Ref](crate::val::Ref) [Value](crate::val::Value) to get the C string from.
 * # Returns
 * The C string or `null` if there was an error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::reference::*;
 * # unsafe {
 * let id = std::ffi::CString::new("someId").unwrap();
 * let dis = std::ffi::CString::new("Some Id").unwrap();
 * let val = haystack_value_make_ref_with_dis(id.as_ptr(), dis.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * let res = haystack_value_get_ref_value(val) as *mut i8;
 * assert_eq!(std::ffi::CString::from_raw(res), id);
 * let res = haystack_value_get_ref_dis(val) as *mut i8;
 * assert_eq!(std::ffi::CString::from_raw(res), dis);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
const char *haystack_value_get_ref_dis(const struct Value *val);

/**
 * Get number of character, without trailing zero, of a [Str](crate::val::Str) [Value](crate::val::Value)
 * # Arguments
 * val A [Str](crate::val::Str) [Value](crate::val::Value) to get the len from
 * # Returns
 * - the size of the string
 * - -1 (usize::MAX) if there was an error getting the string length, in which case use [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::str::*;
 * # unsafe {
 * let str = std::ffi::CString::new("foo bar").unwrap();
 * let val = haystack_value_make_str(str.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_str(val));
 * assert_eq!(haystack_value_get_str_len(val), 7);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uintptr_t haystack_value_get_str_len(const struct Value *val);

/**
 * Get the character of a [Str](crate::val::Str) [Value](crate::val::Value)
 * # Arguments
 * val A [Str](crate::val::Str) [Value](crate::val::Value) to get the C string from.
 * # Returns
 * The C string or `null` if there was an error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::str::*;
 * # unsafe {
 * let str = std::ffi::CString::new("foo bar").unwrap();
 * let val = haystack_value_make_str(str.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_str(val));
 * let res = haystack_value_get_str_value(val) as *mut i8;
 * assert_eq!(std::ffi::CString::from_raw(res), str);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
const char *haystack_value_get_str_value(const struct Value *val);

/**
 * Destructs and free a C Strings allocated by haystack functions
 * # Arguments
 * val C String pointer.
 * # Safety
 * Panics on invalid input data
 */
void haystack_string_destroy(char *val);

/**
 * Get number of character, without trailing zero, of a [Symbol](crate::val::Symbol) [Value](crate::val::Value)
 * # Arguments
 * val A [Symbol](crate::val::Symbol) [Value](crate::val::Value) to get the len from
 * # Returns
 * - the size of the string
 * - -1 (usize::MAX) if there was an error getting the string length
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::symbol::*;
 * # unsafe {
 * let symbol = std::ffi::CString::new("symbol").unwrap();
 * let val = haystack_value_make_symbol(symbol.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_symbol(val));
 * assert_eq!(haystack_value_get_symbol_value_len(val), 6);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uintptr_t haystack_value_get_symbol_value_len(const struct Value *val);

/**
 * Get the character of a [Symbol](crate::val::Symbol) [Value](crate::val::Value)
 * # Arguments
 * val A [Symbol](crate::val::Symbol) [Value](crate::val::Value) to get the C string from.
 * # Returns
 * The C string or `null` if there was an error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::symbol::*;
 * # unsafe {
 * let symbol = std::ffi::CString::new("symbol").unwrap();
 * let val = haystack_value_make_symbol(symbol.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * let res = haystack_value_get_symbol_value(val) as *mut i8;
 * assert_eq!(std::ffi::CString::from_raw(res), symbol);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
const char *haystack_value_get_symbol_value(const struct Value *val);

/**
 * Get the hour of a [Time](crate::val::Time) [Value](crate::val::Value)
 * # Arguments
 * val A [Time](crate::val::Time) [Value](crate::val::Value)
 * # Returns
 * - value
 * - -1 (u32::MAX) if there was an error
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::time::*;
 * # unsafe {
 * let val = haystack_value_make_time(12, 12, 59);
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_time(val));
 * assert_eq!(haystack_value_get_time_hour(val), 12);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uint32_t haystack_value_get_time_hour(const struct Value *val);

/**
 * Get the minutes of a [Time](crate::val::Time) [Value](crate::val::Value)
 * # Arguments
 * val A [Time](crate::val::Time) [Value](crate::val::Value)
 * # Returns
 * - value
 * - -1 (u32::MAX) if there was an error
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::time::*;
 * # unsafe {
 * let val = haystack_value_make_time(12, 15, 59);
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_time(val));
 * assert_eq!(haystack_value_get_time_minutes(val), 15);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uint32_t haystack_value_get_time_minutes(const struct Value *val);

/**
 * Get the seconds of a [Time](crate::val::Time) [Value](crate::val::Value)
 * # Arguments
 * val A [Time](crate::val::Time) [Value](crate::val::Value)
 * # Returns
 * - value
 * - -1 (u32::MAX) if there was an error
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::time::*;
 * # unsafe {
 * let val = haystack_value_make_time(12, 15, 59);
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_time(val));
 * assert_eq!(haystack_value_get_time_seconds(val), 59);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uint32_t haystack_value_get_time_seconds(const struct Value *val);

/**
 * Get the milliseconds of a [Time](crate::val::Time) [Value](crate::val::Value)
 * # Arguments
 * val A [Time](crate::val::Time) [Value](crate::val::Value)
 * # Returns
 * - value
 * - -1 (u32::MAX) if there was an error
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::time::*;
 * # unsafe {
 * let val = haystack_value_make_time_millis(12, 15, 59, 999);
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_time(val));
 * assert_eq!(haystack_value_get_time_millis(val), 999);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uint32_t haystack_value_get_time_millis(const struct Value *val);

/**
 * Get number of character, without trailing zero, of a [Uri](crate::val::Uri) [Value](crate::val::Value)
 * # Arguments
 * val A [Uri](crate::val::Uri) [Value](crate::val::Value) to get the len from
 * # Returns
 * - the size of the string
 * - -1 (usize::MAX) if there was an error getting the string length
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::uri::*;
 * # unsafe {
 * let uri = std::ffi::CString::new("/a/b/c").unwrap();
 * let val = haystack_value_make_uri(uri.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_uri(val));
 * assert_eq!(haystack_value_get_uri_value_len(val), 6);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
uintptr_t haystack_value_get_uri_value_len(const struct Value *val);

/**
 * Get the character of a [Uri](crate::val::Uri) [Value](crate::val::Value)
 * # Arguments
 * val A [Uri](crate::val::Uri) [Value](crate::val::Value) to get the C string from.
 * # Returns
 * The C string or `null` if there was an error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::uri::*;
 * # unsafe {
 * let uri = std::ffi::CString::new("/a/b/c").unwrap();
 * let val = haystack_value_make_uri(uri.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(haystack_value_is_uri(val));
 * let res = haystack_value_get_uri_value(val) as *mut i8;
 * assert_eq!(std::ffi::CString::from_raw(res), uri);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
const char *haystack_value_get_uri_value(const struct Value *val);

/**
 * Construct an empty [Value](crate::val::Value)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * let val = haystack_value_init();
 * # unsafe {
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_null(val))
 * # }
 * ```
 */
struct Value *haystack_value_init(void);

/**
 * Destructs and free a [Value](crate::val::Value)
 * # Arguments
 * val a [Value](crate::val::Value).
 *
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * let val = haystack_value_init();
 * # unsafe {
 * # let val = Box::<Value>::leak(val);
 * haystack_value_destroy(val);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
void haystack_value_destroy(struct Value *val);

/**
 * True if a null (empty) [Value](crate::val::Value)
 * # Arguments
 * val a [Value](crate::val::Value).
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * let val = haystack_value_init();
 * # unsafe {
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_null(val))
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_null(const struct Value *val);

/**
 * Construct a [Marker](crate::val::Marker) [Value](crate::val::Value)
 */
struct Value *haystack_value_make_marker(void);

/**
 * True if a [Marker](crate::val::Marker) [Value](crate::val::Value)
 * # Arguments
 * val a [Value](crate::val::Value).
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * let val = haystack_value_make_marker();
 * # unsafe {
 * # let val = Box::<Value>::leak(val);
 * assert!(haystack_value_is_marker(val))
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_marker(const struct Value *val);

/**
 * Construct a [NA](crate::val::Na) [Value](crate::val::Value)
 */
struct Value *haystack_value_make_na(void);

/**
 * True if a [NA](crate::val::Na) [Value](crate::val::Value)
 * # Arguments
 * val a [Value](crate::val::Value).
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * let val = haystack_value_make_na();
 * # unsafe {
 * # let val = Box::<Value>::leak(val);
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_na(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_na(const struct Value *val);

/**
 * Construct a [Remove](crate::val::Remove) [Value](crate::val::Value)
 */
struct Value *haystack_value_make_remove(void);

/**
 * True if a [Remove](crate::val::Remove) [Value](crate::val::Value)
 * # Arguments
 * val a [Value](crate::val::Value).
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * let val = haystack_value_make_remove();
 * # unsafe {
 * # let val = Box::<Value>::leak(val);
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_remove(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_remove(const struct Value *val);

/**
 * Construct a [Bool](crate::val::Bool) [Value](crate::val::Value)
 */
struct Value *haystack_value_make_bool(bool val);

/**
 * True if a [Bool](crate::val::Bool) [Value](crate::val::Value)
 * # Arguments
 * val a [Value](crate::val::Value).
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * let val = haystack_value_make_bool(true);
 * # unsafe {
 * # let val = Box::<Value>::leak(val);
 * assert!(!haystack_value_is_na(val));
 * assert!(haystack_value_is_bool(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_bool(const struct Value *val);

/**
 * Construct a [Number](crate::val::Number) [Value](crate::val::Value)
 */
struct Value *haystack_value_make_number(double val);

/**
 * True if a [Number](crate::val::Number) [Value](crate::val::Value)
 * # Arguments
 * val a [Value](crate::val::Value).
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * let val = haystack_value_make_number(12.0);
 * # unsafe {
 * # let val = Box::<Value>::leak(val);
 * assert!(!haystack_value_is_remove(val));
 * assert!(haystack_value_is_number(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_number(const struct Value *val);

/**
 * Construct a [Number](crate::val::Number) [Value](crate::val::Value) with an unit
 * # Arguments
 * - val a 64 bit double.
 * - a C string for the unit
 * # Returns
 * A Number [Value](crate::val::Value), if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let unit = std::ffi::CString::new("kwh").unwrap();
 * let val = haystack_value_make_number_with_unit(42.0, unit.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_number(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_number_with_unit(double val,
                                                   const char *unit);

/**
 * Construct a [Coord](crate::val::Coord) [Value](crate::val::Value)
 */
struct Value *haystack_value_make_coord(double lat, double long_);

/**
 * True if a [Coord](crate::val::Coord) [Value](crate::val::Value)
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * let val = haystack_value_make_coord(1.0, 2.0);
 * # unsafe {
 * # let val = Box::<Value>::leak(val);
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_coord(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_coord(const struct Value *val);

/**
 * Construct a [Str](crate::val::Str) [Value](crate::val::Value)
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_str(const char *val);

/**
 * True if a [Str](crate::val::Str) [Value](crate::val::Value)
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let str = std::ffi::CString::new("foo bar").unwrap();
 * let val = haystack_value_make_str(str.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_str(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_str(const struct Value *val);

/**
 * Construct a [Ref](crate::val::Ref) [Value](crate::val::Value)
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_ref(const char *val);

/**
 * Construct a [Ref](crate::val::Ref) [Value](crate::val::Value) with a display name
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_ref_with_dis(const char *val, const char *dis);

/**
 * True if a [Ref](crate::val::Ref) [Value](crate::val::Value)
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let id = std::ffi::CString::new("someId_ofSomething").unwrap();
 * let val = haystack_value_make_ref(id.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_ref(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_ref(const struct Value *val);

/**
 * Construct a [Uri](crate::val::Uri) [Value](crate::val::Value)
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_uri(const char *val);

/**
 * True if a [Uri](crate::val::Uri) [Value](crate::val::Value)
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let str = std::ffi::CString::new("/a/b/c").unwrap();
 * let val = haystack_value_make_uri(str.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_uri(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_uri(const struct Value *val);

/**
 * Construct a [Symbol](crate::val::Symbol) [Value](crate::val::Value)
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_symbol(const char *val);

/**
 * True if a [Symbol](crate::val::Symbol) [Value](crate::val::Value)
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let str = std::ffi::CString::new("def").unwrap();
 * let val = haystack_value_make_symbol(str.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_symbol(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_symbol(const struct Value *val);

/**
 * Construct a [XStr](crate::val::XStr) [Value](crate::val::Value)
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_xstr(const char *name, const char *data);

/**
 * True if a [XStr](crate::val::XStr) [Value](crate::val::Value)
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let typ = std::ffi::CString::new("Foo").unwrap();
 * let data = std::ffi::CString::new("bar").unwrap();
 * let val = haystack_value_make_xstr(typ.as_ptr(), data.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_xstr(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_xstr(const struct Value *val);

/**
 * Construct a [Time](crate::val::Time) [Value](crate::val::Value)
 */
struct Value *haystack_value_make_time(uint32_t hour, uint32_t min, uint32_t sec);

/**
 * Construct a [Time](crate::val::Time) [Value](crate::val::Value)
 */
struct Value *haystack_value_make_time_millis(uint32_t hour,
                                              uint32_t min,
                                              uint32_t sec,
                                              uint32_t milli);

/**
 * True if a [Time](crate::val::Time) [Value](crate::val::Value)
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let val = haystack_value_make_time(12, 12, 59);
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_time(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_time(const struct Value *val);

/**
 * Construct a [Date](crate::val::Date) [Value](crate::val::Value)
 */
struct Value *haystack_value_make_date(int32_t year, uint32_t month, uint32_t day);

/**
 * True if a [Date](crate::val::Date) [Value](crate::val::Value)
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let val = haystack_value_make_date(2021, 8, 13);
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_date(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_date(const struct Value *val);

/**
 * Construct a UTC [DateTime](crate::val::DateTime) [Value](crate::val::Value)
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_utc_datetime(struct Value *date, struct Value *time);

/**
 * Construct a Timezone [DateTime](crate::val::DateTime) [Value](crate::val::Value)
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_tz_datetime(struct Value *date,
                                              struct Value *time,
                                              const char *tz);

/**
 * True if a [DateTime](crate::val::DateTime) [Value](crate::val::Value)
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let date = haystack_value_make_date(2021, 8, 13);
 * let time = haystack_value_make_time(9, 45, 59);
 * # let date = Box::<Value>::leak(date.unwrap());
 * # let time = Box::<Value>::leak(time.unwrap());
 * let tz = std::ffi::CString::new("UTC").unwrap();
 * let val = haystack_value_make_tz_datetime(date, time, tz.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_datetime(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_datetime(const struct Value *val);

/**
 * Construct a empty [List](crate::val::List) [Value](crate::val::Value)
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_list(void);

/**
 * True if a [List](crate::val::List) [Value](crate::val::Value)
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let val = haystack_value_make_list();
 * # let val = Box::<Value>::leak(val);
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_list(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_list(const struct Value *val);

/**
 * Construct an empty [Dict](crate::val::Dict) [Value](crate::val::Value)
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_dict(void);

/**
 * True if a [Dict](crate::val::Dict) [Value](crate::val::Value)
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let val = haystack_value_make_dict();
 * # let val = Box::<Value>::leak(val);
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_dict(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_dict(const struct Value *val);

/**
 * Construct an empty [Grid](crate::val::Grid) [Value](crate::val::Value)
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_make_grid(void);

/**
 * True if a [Grid](crate::val::Grid) [Value](crate::val::Value)
 * # Returns
 * True, or False in which case if argument type mismatch an error is created that can be read using
 * [last_error_message](super::err::last_error_message)
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # unsafe {
 * let val = haystack_value_make_grid();
 * # let val = Box::<Value>::leak(val);
 * assert!(!haystack_value_is_null(val));
 * assert!(haystack_value_is_grid(val));
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
bool haystack_value_is_grid(const struct Value *val);

/**
 * Get the characters of a [XStr](crate::val::XStr) [Value](crate::val::Value) type
 * # Arguments
 * val A [XStr](crate::val::XStr) [Value](crate::val::Value) to get the C string from.
 * # Returns
 * The C string or `null` if there was an error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::xstr::*;
 * # unsafe {
 * let name = std::ffi::CString::new("Type").unwrap();
 * let data = std::ffi::CString::new("data").unwrap();
 * let val = haystack_value_make_xstr(name.as_ptr(), data.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * let res = haystack_value_get_xstr_type(val) as *mut i8;
 * assert_eq!(std::ffi::CString::from_raw(res), name);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
const char *haystack_value_get_xstr_type(const struct Value *val);

/**
 * Get the characters of a [XStr](crate::val::XStr) [Value](crate::val::Value)
 * # Arguments
 * val A [XStr](crate::val::XStr) [Value](crate::val::Value) to get the C string from.
 * # Returns
 * The C string or `null` if there was an error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::xstr::*;
 * # unsafe {
 * let name = std::ffi::CString::new("Type").unwrap();
 * let data = std::ffi::CString::new("data").unwrap();
 * let val = haystack_value_make_xstr(name.as_ptr(), data.as_ptr());
 * # let val = Box::<Value>::leak(val.unwrap());
 * let res = haystack_value_get_xstr_value(val) as *mut i8;
 * assert_eq!(std::ffi::CString::from_raw(res), data);
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
const char *haystack_value_get_xstr_value(const struct Value *val);

/**
 * Encodes a [Value](crate::val::Value) to a Zinc string
 * # Arguments
 * - val Value to encode as a zinc string
 * # Return
 * an utf-8 encoded `C string` with the Zinc encoded value
 * or `null` if there was an error, in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Safety
 * Panics on invalid input data
 */
const char *haystack_value_to_zinc_string(const struct Value *val);

/**
 * Decodes a [Value](crate::val::Value) from a Zinc string
 * # Arguments
 * - val Utf-8 encoded C string with the Zinc encoded value
 * # Return
 * the decoded [Value](crate::val::Value) or `null` if there was an error,
 * in which case the [last_error_message](super::err::last_error_message)
 * can be called to get the error message.
 * # Safety
 * Panics on invalid input data
 */
struct Value *haystack_value_from_zinc_string(const char *input);

/**
 * Retrieves last error message if any
 * # Example
 * ```rust
 * # use crate::libhaystack::val::Value;
 * # use crate::libhaystack::c_api::value::*;
 * # use crate::libhaystack::c_api::err::*;
 * # unsafe {
 * let val = haystack_value_make_str(std::ptr::null());
 * assert!(val.is_none());
 * assert_ne!(last_error_message(), std::ptr::null())
 * # }
 * ```
 * # Safety
 * Panics on invalid input data
 */
const char *last_error_message(void);

#endif /* libhaystack.h */
