/*
 * Copyright (c) 2021-2021 Thomas Kramer.
 *
 * This file is part of LibrEDA 
 * (see https://codeberg.org/libreda/liberty-io).
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */


use crate::ast::*;
use crate::stream_lexer::*;
use itertools::PeekingNext;
use std::io::Read;
use std::iter::FromIterator;
use std::str::FromStr;

/// An item of a liberty group.
#[derive(Debug, Clone)]
enum GroupItem {
    Group(Group),
    Attribute {
        name: String,
        values: Vec<Value>,
    },
    Define(Define),
}


/// Read a liberty library from a byte stream
pub fn read_liberty_bytes<R: Read>(reader: &mut R) -> Result<Group, ParserError>
{
    read_liberty_chars(
        reader.bytes().map(|b| b.unwrap() as char)
    )
}

/// Read a liberty library from an iterator over characters.
pub fn read_liberty_chars<I>(chars: I) -> Result<Group, ParserError>
    where I: Iterator<Item=char> {
    let mut line_num = 0;
    let mut char_num = 0; // Position on the line.

    // Count newlines.
    let line_count = chars.inspect(|&c| {
        char_num += 1;
        if c == '\n' {
            line_num += 1;
            char_num = 0;
        }
    });

    let result = read_liberty_impl(line_count);

    if result.is_err() {
        log::error!("Liberty error on line: {} (at {})", line_num, char_num);
    }

    result
}

fn read_liberty_impl<I>(chars: I) -> Result<Group, ParserError>
    where I: Iterator<Item=char>
{
    // Token stream.
    let mut tk = tokenize(chars);
    tk.advance();

    let item = read_group_item(&mut tk)?;
    match item {
        GroupItem::Group(g) => Ok(g),
        _ => Err(ParserError::Other("Library must start with a group."))
    }
}

fn read_value<I: Iterator<Item=char> + PeekingNext>(tk: &mut Tokenized<I>) -> Result<Value, ParserError>
{
    let s = tk.current_token_str().ok_or(ParserError::UnexpectedEndOfFile)?;
    let first_char = s.chars().next().ok_or(ParserError::UnexpectedEndOfFile)?;
    let last_char = s.chars().last().ok_or(ParserError::UnexpectedEndOfFile)?;
    if first_char.is_digit(10) && last_char.is_digit(10) {
        // Read a number.
        let f: f64 = tk.take_and_parse()?;
        Ok(Value::Float(f))
    } else {
        if first_char == '"' {
            // Strip away the quotes.
            let without_quotes = String::from_iter(s.chars().skip(1).take(s.len() - 2));
            tk.advance();
            Ok(Value::QuotedString(without_quotes))
        } else {
            let name = tk.take()?;

            if tk.test("[")? {
                let end: u32 = tk.take_and_parse()?;
                let start = if tk.test(":")? {
                    let start: u32 = tk.take_and_parse()?;
                    Some(start)
                } else {
                    None
                };
                tk.expect("]")?;
                if let Some(start) = start {
                    Ok(Value::Sliced(name, end, start))
                } else {
                    Ok(Value::Indexed(name, end))
                }
            } else {
                Ok(Value::String(name))
            }
        }
    }
}

/// Read a liberty group.
fn read_group_item<I: Iterator<Item=char> + PeekingNext>(tk: &mut Tokenized<I>) -> Result<GroupItem, ParserError>
{
    let name = tk.take()?;
    if tk.test("(")? {
        // Group or complex attribute.
        let mut args = vec![];
        while !tk.test(")")? {
            args.push(read_value(tk)?);
            if !tk.peeking_test(")")? {
                tk.expect(",")?;
            }
        }

        if tk.test("{")? {
            // It's a group.

            let mut group = Group::default();
            group.name = name;
            group.arguments = args;

            while !tk.test("}")? {
                // Recursively read group items.
                let item = read_group_item(tk)?;
                match item {
                    GroupItem::Group(g) => { group.groups.push(g) }
                    GroupItem::Attribute { name, values } =>
                        {
                            group.attributes.entry(name)
                                .or_insert(vec![])
                                .push(values);
                        }
                    GroupItem::Define(d) => { group.defines.push(d) }
                }
            }
            Ok(GroupItem::Group(group))
        } else {
            // It's a complex attribute or define statement.
            tk.test(";")?; // Consume an optional trailing semicolon.
            if name == "define" && args.len() == 3 {
                // Define statement.

                // Values must be names or quoted names:
                let attribute_name = match &args[0] {
                    Value::String(s) => s.clone(),
                    Value::QuotedString(s) => s.clone(),
                    a => a.to_string()
                };
                let group_name = match &args[1] {
                    Value::String(s) => s.clone(),
                    Value::QuotedString(s) => s.clone(),
                    a => a.to_string()
                };
                let attr_type = match &args[2] {
                    Value::String(s) => s.clone(),
                    Value::QuotedString(s) => s.clone(),
                    a => a.to_string()
                };

                // Parse the attribute type.
                let attribute_type = AttributeType::from_str(attr_type.as_str())
                    .map_err(|_| ParserError::UnexpectedToken("".to_string(), attr_type))?;

                Ok((GroupItem::Define(
                    Define {
                        attribute_name,
                        group_name,
                        attribute_type
                    }
                )))
            } else {
                // It's a complex attribute.
                Ok(GroupItem::Attribute {
                    name,
                    values: args,
                })
            }
        }
    } else if tk.test(":")? {
        // Simple attribute.
        let value = read_value(tk)?;
        tk.expect(";")?;

        Ok(GroupItem::Attribute {
            name,
            values: vec![value],
        })
    } else {
        Err(ParserError::UnexpectedToken("'(' | ':'".into(), tk.current_token().unwrap()))
    }
}


#[test]
fn test_read_liberty() {
    let data = r#"
library (myLib) {
    simple_attribute1: value;
    simple_attribute2: value;
    complex_attribute1 (value1, "value 2");

    // Single line comment // does not end here / also not here

    /* comment with ** * characters */

    cell (invx1) {
        simple_attribute1: value;
    }
}
"#;

    let result = read_liberty_chars(data.chars());

    dbg!(&result);

    assert!(result.is_ok());
}
