/*
 * Copyright (c) 2021-2021 Thomas Kramer.
 *
 * This file is part of LibrEDA 
 * (see https://codeberg.org/libreda/liberty-io).
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */


//! Input and output library for the Liberty format.
//! The Liberty format is used to describe characteristics of standard-cell libraries.
//!
//! The liberty library is represented as nested [`Group`] structures.
//!
//! # Example
//!
//! Read a liberty file:
//! ```
//! use liberty_io;
//! use std::fs::File;
//! use std::io::BufReader;
//!
//! // Create a buffered reader for faster reading.
//! let f = File::open("./tests/data/freepdk45/gscl45nm.lib").expect("Failed to open file.");
//! let mut buf = BufReader::new(f);
//!
//! // Read the file.
//! let read_result = liberty_io::read_liberty_bytes(&mut buf);
//! // Print the parsed library or error value.
//! // dbg!(&read_result);
//! // Abort the program if the library could not be read.
//! let library_group = read_result.expect("Failed to read library!");
//!
//! // Access the content.
//! assert_eq!(&library_group.name, "library");
//! assert_eq!(&library_group.arguments[0].to_string(), "gscl45nm");
//!
//! // List all cell names. (There's only DFFNEGX1 in the provided example file.)
//! println!("Library cells:");
//! for group in &library_group.groups {
//!     if group.name == "cell" {
//!         println!("* {}", &group.arguments[0]);
//!     }
//! }
//!
//! // There's some utility functions for accessing the library structure.
//! // Find a cell group by it's name.
//! let dffnegx1 = library_group.find_cell("DFFNEGX1");
//! assert!(dffnegx1.is_some());
//! ```
//!
//! [`Group`]: ast::Group
//!

#![deny(missing_docs)]

// TODO: Remove those once this crate stabilizes.
#![allow(unused)]

mod stream_lexer;
mod stream_lexer_v2;
mod parser;
mod parser_v2;
mod ast;
pub mod util;
mod ast_generated;
mod boolean;


pub use parser::{read_liberty_chars, read_liberty_bytes};
pub use ast::*;
