use phf::phf_map;

#[allow(dead_code)]
pub static POLYNOMIALS: phf::Map<u32, &'static [u8]> = phf_map! {
  // deg => primitive polynomial
  3u32 => &[0xB],         // (7, 4): x^3 + x + 1
  4u32 => &[0x13],        // (15, 11): x^4 + x + 1
  5u32 => &[0x25],        // (31, 26): x^5 + x^2 + 1
  6u32 => &[0x43],        // (63, 57): x^6 + x + 1
  7u32 => &[0x89],        // (127, 120): x^7 + x^3 + 1
  8u32 => &[0x01, 0x1D],  // (255, 247): x^8 + x^4 + x^3 + x^2 + 1
  9u32 => &[0x02, 0x11],  // (511, 502): x^9 + x^4 + 1
  10u32 => &[0x04, 0x09], // (1023,1013): x^10 + x^3 + 1
};

#[allow(dead_code)]
pub static ERROR_POS_TO_SYNDROME: phf::Map<u32, &'static [u32]> = phf_map! {
  3u32 => &[
    0x0000, 0x0005, 0x0007, 0x0006, 0x0003, 0x0004, 0x0002, 0x0001,
  ],
  4u32 => &[
    0x0000, 0x0009, 0x000D, 0x000F, 0x000E, 0x0007, 0x000A, 0x0005, 0x000B, 0x000C, 0x0006, 0x0003, 0x0008, 0x0004, 0x0002, 0x0001,
  ],
  5u32 => &[
    0x0000, 0x0012, 0x0009, 0x0016, 0x000B, 0x0017, 0x0019, 0x001E, 0x000F, 0x0015, 0x0018, 0x000C, 0x0006, 0x0003, 0x0013, 0x001B,
    0x001F, 0x001D, 0x001C, 0x000E, 0x0007, 0x0011, 0x001A, 0x000D, 0x0014, 0x000A, 0x0005, 0x0010, 0x0008, 0x0004, 0x0002, 0x0001,
  ],
  6u32 => &[
    0x0000, 0x0021, 0x0031, 0x0039, 0x003D, 0x003F, 0x003E, 0x001F, 0x002E, 0x0017, 0x002A, 0x0015, 0x002B, 0x0034, 0x001A, 0x000D,
    0x0027, 0x0032, 0x0019, 0x002D, 0x0037, 0x003A, 0x001D, 0x002F, 0x0036, 0x001B, 0x002C, 0x0016, 0x000B, 0x0024, 0x0012, 0x0009,
    0x0025, 0x0033, 0x0038, 0x001C, 0x000E, 0x0007, 0x0022, 0x0011, 0x0029, 0x0035, 0x003B, 0x003C, 0x001E, 0x000F, 0x0026, 0x0013,
    0x0028, 0x0014, 0x000A, 0x0005, 0x0023, 0x0030, 0x0018, 0x000C, 0x0006, 0x0003, 0x0020, 0x0010, 0x0008, 0x0004, 0x0002, 0x0001,
  ],
  7u32 => &[
    0x0000, 0x0044, 0x0022, 0x0011, 0x004C, 0x0026, 0x0013, 0x004D, 0x0062, 0x0031, 0x005C, 0x002E, 0x0017, 0x004F, 0x0063, 0x0075,
    0x007E, 0x003F, 0x005B, 0x0069, 0x0070, 0x0038, 0x001C, 0x000E, 0x0007, 0x0047, 0x0067, 0x0077, 0x007F, 0x007B, 0x0079, 0x0078,
    0x003C, 0x001E, 0x000F, 0x0043, 0x0065, 0x0076, 0x003B, 0x0059, 0x0068, 0x0034, 0x001A, 0x000D, 0x0042, 0x0021, 0x0054, 0x002A,
    0x0015, 0x004E, 0x0027, 0x0057, 0x006F, 0x0073, 0x007D, 0x007A, 0x003D, 0x005A, 0x002D, 0x0052, 0x0029, 0x0050, 0x0028, 0x0014,
    0x000A, 0x0005, 0x0046, 0x0023, 0x0055, 0x006E, 0x0037, 0x005F, 0x006B, 0x0071, 0x007C, 0x003E, 0x001F, 0x004B, 0x0061, 0x0074,
    0x003A, 0x001D, 0x004A, 0x0025, 0x0056, 0x002B, 0x0051, 0x006C, 0x0036, 0x001B, 0x0049, 0x0060, 0x0030, 0x0018, 0x000C, 0x0006,
    0x0003, 0x0045, 0x0066, 0x0033, 0x005D, 0x006A, 0x0035, 0x005E, 0x002F, 0x0053, 0x006D, 0x0072, 0x0039, 0x0058, 0x002C, 0x0016,
    0x000B, 0x0041, 0x0064, 0x0032, 0x0019, 0x0048, 0x0024, 0x0012, 0x0009, 0x0040, 0x0020, 0x0010, 0x0008, 0x0004, 0x0002, 0x0001,
  ],
  8u32 => &[
    0x0000, 0x008E, 0x0047, 0x00AD, 0x00D8, 0x006C, 0x0036, 0x001B, 0x0083, 0x00CF, 0x00E9, 0x00FA, 0x007D, 0x00B0, 0x0058, 0x002C,
    0x0016, 0x000B, 0x008B, 0x00CB, 0x00EB, 0x00FB, 0x00F3, 0x00F7, 0x00F5, 0x00F4, 0x007A, 0x003D, 0x0090, 0x0048, 0x0024, 0x0012,
    0x0009, 0x008A, 0x0045, 0x00AC, 0x0056, 0x002B, 0x009B, 0x00C3, 0x00EF, 0x00F9, 0x00F2, 0x0079, 0x00B2, 0x0059, 0x00A2, 0x0051,
    0x00A6, 0x0053, 0x00A7, 0x00DD, 0x00E0, 0x0070, 0x0038, 0x001C, 0x000E, 0x0007, 0x008D, 0x00C8, 0x0064, 0x0032, 0x0019, 0x0082,
    0x0041, 0x00AE, 0x0057, 0x00A5, 0x00DC, 0x006E, 0x0037, 0x0095, 0x00C4, 0x0062, 0x0031, 0x0096, 0x004B, 0x00AB, 0x00DB, 0x00E3,
    0x00FF, 0x00F1, 0x00F6, 0x007B, 0x00B3, 0x00D7, 0x00E5, 0x00FC, 0x007E, 0x003F, 0x0091, 0x00C6, 0x0063, 0x00BF, 0x00D1, 0x00E6,
    0x0073, 0x00B7, 0x00D5, 0x00E4, 0x0072, 0x0039, 0x0092, 0x0049, 0x00AA, 0x0055, 0x00A4, 0x0052, 0x0029, 0x009A, 0x004D, 0x00A8,
    0x0054, 0x002A, 0x0015, 0x0084, 0x0042, 0x0021, 0x009E, 0x004F, 0x00A9, 0x00DA, 0x006D, 0x00B8, 0x005C, 0x002E, 0x0017, 0x0085,
    0x00CC, 0x0066, 0x0033, 0x0097, 0x00C5, 0x00EC, 0x0076, 0x003B, 0x0093, 0x00C7, 0x00ED, 0x00F8, 0x007C, 0x003E, 0x001F, 0x0081,
    0x00CE, 0x0067, 0x00BD, 0x00D0, 0x0068, 0x0034, 0x001A, 0x000D, 0x0088, 0x0044, 0x0022, 0x0011, 0x0086, 0x0043, 0x00AF, 0x00D9,
    0x00E2, 0x0071, 0x00B6, 0x005B, 0x00A3, 0x00DF, 0x00E1, 0x00FE, 0x007F, 0x00B1, 0x00D6, 0x006B, 0x00BB, 0x00D3, 0x00E7, 0x00FD,
    0x00F0, 0x0078, 0x003C, 0x001E, 0x000F, 0x0089, 0x00CA, 0x0065, 0x00BC, 0x005E, 0x002F, 0x0099, 0x00C2, 0x0061, 0x00BE, 0x005F,
    0x00A1, 0x00DE, 0x006F, 0x00B9, 0x00D2, 0x0069, 0x00BA, 0x005D, 0x00A0, 0x0050, 0x0028, 0x0014, 0x000A, 0x0005, 0x008C, 0x0046,
    0x0023, 0x009F, 0x00C1, 0x00EE, 0x0077, 0x00B5, 0x00D4, 0x006A, 0x0035, 0x0094, 0x004A, 0x0025, 0x009C, 0x004E, 0x0027, 0x009D,
    0x00C0, 0x0060, 0x0030, 0x0018, 0x000C, 0x0006, 0x0003, 0x008F, 0x00C9, 0x00EA, 0x0075, 0x00B4, 0x005A, 0x002D, 0x0098, 0x004C,
    0x0026, 0x0013, 0x0087, 0x00CD, 0x00E8, 0x0074, 0x003A, 0x001D, 0x0080, 0x0040, 0x0020, 0x0010, 0x0008, 0x0004, 0x0002, 0x0001,
  ],
  9u32 => &[
    0x0000, 0x0108, 0x0084, 0x0042, 0x0021, 0x0118, 0x008C, 0x0046, 0x0023, 0x0119, 0x0184, 0x00C2, 0x0061, 0x0138, 0x009C, 0x004E,
    0x0027, 0x011B, 0x0185, 0x01CA, 0x00E5, 0x017A, 0x00BD, 0x0156, 0x00AB, 0x015D, 0x01A6, 0x00D3, 0x0161, 0x01B8, 0x00DC, 0x006E,
    0x0037, 0x0113, 0x0181, 0x01C8, 0x00E4, 0x0072, 0x0039, 0x0114, 0x008A, 0x0045, 0x012A, 0x0095, 0x0142, 0x00A1, 0x0158, 0x00AC,
    0x0056, 0x002B, 0x011D, 0x0186, 0x00C3, 0x0169, 0x01BC, 0x00DE, 0x006F, 0x013F, 0x0197, 0x01C3, 0x01E9, 0x01FC, 0x00FE, 0x007F,
    0x0137, 0x0193, 0x01C1, 0x01E8, 0x00F4, 0x007A, 0x003D, 0x0116, 0x008B, 0x014D, 0x01AE, 0x00D7, 0x0163, 0x01B9, 0x01D4, 0x00EA,
    0x0075, 0x0132, 0x0099, 0x0144, 0x00A2, 0x0051, 0x0120, 0x0090, 0x0048, 0x0024, 0x0012, 0x0009, 0x010C, 0x0086, 0x0043, 0x0129,
    0x019C, 0x00CE, 0x0067, 0x013B, 0x0195, 0x01C2, 0x00E1, 0x0178, 0x00BC, 0x005E, 0x002F, 0x011F, 0x0187, 0x01CB, 0x01ED, 0x01FE,
    0x00FF, 0x0177, 0x01B3, 0x01D1, 0x01E0, 0x00F0, 0x0078, 0x003C, 0x001E, 0x000F, 0x010F, 0x018F, 0x01CF, 0x01EF, 0x01FF, 0x01F7,
    0x01F3, 0x01F1, 0x01F0, 0x00F8, 0x007C, 0x003E, 0x001F, 0x0107, 0x018B, 0x01CD, 0x01EE, 0x00F7, 0x0173, 0x01B1, 0x01D0, 0x00E8,
    0x0074, 0x003A, 0x001D, 0x0106, 0x0083, 0x0149, 0x01AC, 0x00D6, 0x006B, 0x013D, 0x0196, 0x00CB, 0x016D, 0x01BE, 0x00DF, 0x0167,
    0x01BB, 0x01D5, 0x01E2, 0x00F1, 0x0170, 0x00B8, 0x005C, 0x002E, 0x0017, 0x0103, 0x0189, 0x01CC, 0x00E6, 0x0073, 0x0131, 0x0190,
    0x00C8, 0x0064, 0x0032, 0x0019, 0x0104, 0x0082, 0x0041, 0x0128, 0x0094, 0x004A, 0x0025, 0x011A, 0x008D, 0x014E, 0x00A7, 0x015B,
    0x01A5, 0x01DA, 0x00ED, 0x017E, 0x00BF, 0x0157, 0x01A3, 0x01D9, 0x01E4, 0x00F2, 0x0079, 0x0134, 0x009A, 0x004D, 0x012E, 0x0097,
    0x0143, 0x01A9, 0x01DC, 0x00EE, 0x0077, 0x0133, 0x0191, 0x01C0, 0x00E0, 0x0070, 0x0038, 0x001C, 0x000E, 0x0007, 0x010B, 0x018D,
    0x01CE, 0x00E7, 0x017B, 0x01B5, 0x01D2, 0x00E9, 0x017C, 0x00BE, 0x005F, 0x0127, 0x019B, 0x01C5, 0x01EA, 0x00F5, 0x0172, 0x00B9,
    0x0154, 0x00AA, 0x0055, 0x0122, 0x0091, 0x0140, 0x00A0, 0x0050, 0x0028, 0x0014, 0x000A, 0x0005, 0x010A, 0x0085, 0x014A, 0x00A5,
    0x015A, 0x00AD, 0x015E, 0x00AF, 0x015F, 0x01A7, 0x01DB, 0x01E5, 0x01FA, 0x00FD, 0x0176, 0x00BB, 0x0155, 0x01A2, 0x00D1, 0x0160,
    0x00B0, 0x0058, 0x002C, 0x0016, 0x000B, 0x010D, 0x018E, 0x00C7, 0x016B, 0x01BD, 0x01D6, 0x00EB, 0x017D, 0x01B6, 0x00DB, 0x0165,
    0x01BA, 0x00DD, 0x0166, 0x00B3, 0x0151, 0x01A0, 0x00D0, 0x0068, 0x0034, 0x001A, 0x000D, 0x010E, 0x0087, 0x014B, 0x01AD, 0x01DE,
    0x00EF, 0x017F, 0x01B7, 0x01D3, 0x01E1, 0x01F8, 0x00FC, 0x007E, 0x003F, 0x0117, 0x0183, 0x01C9, 0x01EC, 0x00F6, 0x007B, 0x0135,
    0x0192, 0x00C9, 0x016C, 0x00B6, 0x005B, 0x0125, 0x019A, 0x00CD, 0x016E, 0x00B7, 0x0153, 0x01A1, 0x01D8, 0x00EC, 0x0076, 0x003B,
    0x0115, 0x0182, 0x00C1, 0x0168, 0x00B4, 0x005A, 0x002D, 0x011E, 0x008F, 0x014F, 0x01AF, 0x01DF, 0x01E7, 0x01FB, 0x01F5, 0x01F2,
    0x00F9, 0x0174, 0x00BA, 0x005D, 0x0126, 0x0093, 0x0141, 0x01A8, 0x00D4, 0x006A, 0x0035, 0x0112, 0x0089, 0x014C, 0x00A6, 0x0053,
    0x0121, 0x0198, 0x00CC, 0x0066, 0x0033, 0x0111, 0x0180, 0x00C0, 0x0060, 0x0030, 0x0018, 0x000C, 0x0006, 0x0003, 0x0109, 0x018C,
    0x00C6, 0x0063, 0x0139, 0x0194, 0x00CA, 0x0065, 0x013A, 0x009D, 0x0146, 0x00A3, 0x0159, 0x01A4, 0x00D2, 0x0069, 0x013C, 0x009E,
    0x004F, 0x012F, 0x019F, 0x01C7, 0x01EB, 0x01FD, 0x01F6, 0x00FB, 0x0175, 0x01B2, 0x00D9, 0x0164, 0x00B2, 0x0059, 0x0124, 0x0092,
    0x0049, 0x012C, 0x0096, 0x004B, 0x012D, 0x019E, 0x00CF, 0x016F, 0x01BF, 0x01D7, 0x01E3, 0x01F9, 0x01F4, 0x00FA, 0x007D, 0x0136,
    0x009B, 0x0145, 0x01AA, 0x00D5, 0x0162, 0x00B1, 0x0150, 0x00A8, 0x0054, 0x002A, 0x0015, 0x0102, 0x0081, 0x0148, 0x00A4, 0x0052,
    0x0029, 0x011C, 0x008E, 0x0047, 0x012B, 0x019D, 0x01C6, 0x00E3, 0x0179, 0x01B4, 0x00DA, 0x006D, 0x013E, 0x009F, 0x0147, 0x01AB,
    0x01DD, 0x01E6, 0x00F3, 0x0171, 0x01B0, 0x00D8, 0x006C, 0x0036, 0x001B, 0x0105, 0x018A, 0x00C5, 0x016A, 0x00B5, 0x0152, 0x00A9,
    0x015C, 0x00AE, 0x0057, 0x0123, 0x0199, 0x01C4, 0x00E2, 0x0071, 0x0130, 0x0098, 0x004C, 0x0026, 0x0013, 0x0101, 0x0188, 0x00C4,
    0x0062, 0x0031, 0x0110, 0x0088, 0x0044, 0x0022, 0x0011, 0x0100, 0x0080, 0x0040, 0x0020, 0x0010, 0x0008, 0x0004, 0x0002, 0x0001,
  ],
  10u32 => &[
    0x0000, 0x0204, 0x0102, 0x0081, 0x0244, 0x0122, 0x0091, 0x024C, 0x0126, 0x0093, 0x024D, 0x0322, 0x0191, 0x02CC, 0x0166, 0x00B3,
    0x025D, 0x032A, 0x0195, 0x02CE, 0x0167, 0x02B7, 0x035F, 0x03AB, 0x03D1, 0x03EC, 0x01F6, 0x00FB, 0x0279, 0x0338, 0x019C, 0x00CE,
    0x0067, 0x0237, 0x031F, 0x038B, 0x03C1, 0x03E4, 0x01F2, 0x00F9, 0x0278, 0x013C, 0x009E, 0x004F, 0x0223, 0x0315, 0x038E, 0x01C7,
    0x02E7, 0x0377, 0x03BF, 0x03DB, 0x03E9, 0x03F0, 0x01F8, 0x00FC, 0x007E, 0x003F, 0x021B, 0x0309, 0x0380, 0x01C0, 0x00E0, 0x0070,
    0x0038, 0x001C, 0x000E, 0x0007, 0x0207, 0x0307, 0x0387, 0x03C7, 0x03E7, 0x03F7, 0x03FF, 0x03FB, 0x03F9, 0x03F8, 0x01FC, 0x00FE,
    0x007F, 0x023B, 0x0319, 0x0388, 0x01C4, 0x00E2, 0x0071, 0x023C, 0x011E, 0x008F, 0x0243, 0x0325, 0x0396, 0x01CB, 0x02E1, 0x0374,
    0x01BA, 0x00DD, 0x026A, 0x0135, 0x029E, 0x014F, 0x02A3, 0x0355, 0x03AE, 0x01D7, 0x02EF, 0x0373, 0x03BD, 0x03DA, 0x01ED, 0x02F2,
    0x0179, 0x02B8, 0x015C, 0x00AE, 0x0057, 0x022F, 0x0313, 0x038D, 0x03C2, 0x01E1, 0x02F4, 0x017A, 0x00BD, 0x025A, 0x012D, 0x0292,
    0x0149, 0x02A0, 0x0150, 0x00A8, 0x0054, 0x002A, 0x0015, 0x020E, 0x0107, 0x0287, 0x0347, 0x03A7, 0x03D7, 0x03EF, 0x03F3, 0x03FD,
    0x03FA, 0x01FD, 0x02FA, 0x017D, 0x02BA, 0x015D, 0x02AA, 0x0155, 0x02AE, 0x0157, 0x02AF, 0x0353, 0x03AD, 0x03D2, 0x01E9, 0x02F0,
    0x0178, 0x00BC, 0x005E, 0x002F, 0x0213, 0x030D, 0x0382, 0x01C1, 0x02E4, 0x0172, 0x00B9, 0x0258, 0x012C, 0x0096, 0x004B, 0x0221,
    0x0314, 0x018A, 0x00C5, 0x0266, 0x0133, 0x029D, 0x034A, 0x01A5, 0x02D6, 0x016B, 0x02B1, 0x035C, 0x01AE, 0x00D7, 0x026F, 0x0333,
    0x039D, 0x03CA, 0x01E5, 0x02F6, 0x017B, 0x02B9, 0x0358, 0x01AC, 0x00D6, 0x006B, 0x0231, 0x031C, 0x018E, 0x00C7, 0x0267, 0x0337,
    0x039F, 0x03CB, 0x03E1, 0x03F4, 0x01FA, 0x00FD, 0x027A, 0x013D, 0x029A, 0x014D, 0x02A2, 0x0151, 0x02AC, 0x0156, 0x00AB, 0x0251,
    0x032C, 0x0196, 0x00CB, 0x0261, 0x0334, 0x019A, 0x00CD, 0x0262, 0x0131, 0x029C, 0x014E, 0x00A7, 0x0257, 0x032F, 0x0393, 0x03CD,
    0x03E2, 0x01F1, 0x02FC, 0x017E, 0x00BF, 0x025B, 0x0329, 0x0390, 0x01C8, 0x00E4, 0x0072, 0x0039, 0x0218, 0x010C, 0x0086, 0x0043,
    0x0225, 0x0316, 0x018B, 0x02C1, 0x0364, 0x01B2, 0x00D9, 0x0268, 0x0134, 0x009A, 0x004D, 0x0222, 0x0111, 0x028C, 0x0146, 0x00A3,
    0x0255, 0x032E, 0x0197, 0x02CF, 0x0363, 0x03B5, 0x03DE, 0x01EF, 0x02F3, 0x037D, 0x03BA, 0x01DD, 0x02EA, 0x0175, 0x02BE, 0x015F,
    0x02AB, 0x0351, 0x03AC, 0x01D6, 0x00EB, 0x0271, 0x033C, 0x019E, 0x00CF, 0x0263, 0x0335, 0x039E, 0x01CF, 0x02E3, 0x0375, 0x03BE,
    0x01DF, 0x02EB, 0x0371, 0x03BC, 0x01DE, 0x00EF, 0x0273, 0x033D, 0x039A, 0x01CD, 0x02E2, 0x0171, 0x02BC, 0x015E, 0x00AF, 0x0253,
    0x032D, 0x0392, 0x01C9, 0x02E0, 0x0170, 0x00B8, 0x005C, 0x002E, 0x0017, 0x020F, 0x0303, 0x0385, 0x03C6, 0x01E3, 0x02F5, 0x037E,
    0x01BF, 0x02DB, 0x0369, 0x03B0, 0x01D8, 0x00EC, 0x0076, 0x003B, 0x0219, 0x0308, 0x0184, 0x00C2, 0x0061, 0x0234, 0x011A, 0x008D,
    0x0242, 0x0121, 0x0294, 0x014A, 0x00A5, 0x0256, 0x012B, 0x0291, 0x034C, 0x01A6, 0x00D3, 0x026D, 0x0332, 0x0199, 0x02C8, 0x0164,
    0x00B2, 0x0059, 0x0228, 0x0114, 0x008A, 0x0045, 0x0226, 0x0113, 0x028D, 0x0342, 0x01A1, 0x02D4, 0x016A, 0x00B5, 0x025E, 0x012F,
    0x0293, 0x034D, 0x03A2, 0x01D1, 0x02EC, 0x0176, 0x00BB, 0x0259, 0x0328, 0x0194, 0x00CA, 0x0065, 0x0236, 0x011B, 0x0289, 0x0340,
    0x01A0, 0x00D0, 0x0068, 0x0034, 0x001A, 0x000D, 0x0202, 0x0101, 0x0284, 0x0142, 0x00A1, 0x0254, 0x012A, 0x0095, 0x024E, 0x0127,
    0x0297, 0x034F, 0x03A3, 0x03D5, 0x03EE, 0x01F7, 0x02FF, 0x037B, 0x03B9, 0x03D8, 0x01EC, 0x00F6, 0x007B, 0x0239, 0x0318, 0x018C,
    0x00C6, 0x0063, 0x0235, 0x031E, 0x018F, 0x02C3, 0x0365, 0x03B6, 0x01DB, 0x02E9, 0x0370, 0x01B8, 0x00DC, 0x006E, 0x0037, 0x021F,
    0x030B, 0x0381, 0x03C4, 0x01E2, 0x00F1, 0x027C, 0x013E, 0x009F, 0x024B, 0x0321, 0x0394, 0x01CA, 0x00E5, 0x0276, 0x013B, 0x0299,
    0x0348, 0x01A4, 0x00D2, 0x0069, 0x0230, 0x0118, 0x008C, 0x0046, 0x0023, 0x0215, 0x030E, 0x0187, 0x02C7, 0x0367, 0x03B7, 0x03DF,
    0x03EB, 0x03F1, 0x03FC, 0x01FE, 0x00FF, 0x027B, 0x0339, 0x0398, 0x01CC, 0x00E6, 0x0073, 0x023D, 0x031A, 0x018D, 0x02C2, 0x0161,
    0x02B4, 0x015A, 0x00AD, 0x0252, 0x0129, 0x0290, 0x0148, 0x00A4, 0x0052, 0x0029, 0x0210, 0x0108, 0x0084, 0x0042, 0x0021, 0x0214,
    0x010A, 0x0085, 0x0246, 0x0123, 0x0295, 0x034E, 0x01A7, 0x02D7, 0x036F, 0x03B3, 0x03DD, 0x03EA, 0x01F5, 0x02FE, 0x017F, 0x02BB,
    0x0359, 0x03A8, 0x01D4, 0x00EA, 0x0075, 0x023E, 0x011F, 0x028B, 0x0341, 0x03A4, 0x01D2, 0x00E9, 0x0270, 0x0138, 0x009C, 0x004E,
    0x0027, 0x0217, 0x030F, 0x0383, 0x03C5, 0x03E6, 0x01F3, 0x02FD, 0x037A, 0x01BD, 0x02DA, 0x016D, 0x02B2, 0x0159, 0x02A8, 0x0154,
    0x00AA, 0x0055, 0x022E, 0x0117, 0x028F, 0x0343, 0x03A5, 0x03D6, 0x01EB, 0x02F1, 0x037C, 0x01BE, 0x00DF, 0x026B, 0x0331, 0x039C,
    0x01CE, 0x00E7, 0x0277, 0x033F, 0x039B, 0x03C9, 0x03E0, 0x01F0, 0x00F8, 0x007C, 0x003E, 0x001F, 0x020B, 0x0301, 0x0384, 0x01C2,
    0x00E1, 0x0274, 0x013A, 0x009D, 0x024A, 0x0125, 0x0296, 0x014B, 0x02A1, 0x0354, 0x01AA, 0x00D5, 0x026E, 0x0137, 0x029F, 0x034B,
    0x03A1, 0x03D4, 0x01EA, 0x00F5, 0x027E, 0x013F, 0x029B, 0x0349, 0x03A0, 0x01D0, 0x00E8, 0x0074, 0x003A, 0x001D, 0x020A, 0x0105,
    0x0286, 0x0143, 0x02A5, 0x0356, 0x01AB, 0x02D1, 0x036C, 0x01B6, 0x00DB, 0x0269, 0x0330, 0x0198, 0x00CC, 0x0066, 0x0033, 0x021D,
    0x030A, 0x0185, 0x02C6, 0x0163, 0x02B5, 0x035E, 0x01AF, 0x02D3, 0x036D, 0x03B2, 0x01D9, 0x02E8, 0x0174, 0x00BA, 0x005D, 0x022A,
    0x0115, 0x028E, 0x0147, 0x02A7, 0x0357, 0x03AF, 0x03D3, 0x03ED, 0x03F2, 0x01F9, 0x02F8, 0x017C, 0x00BE, 0x005F, 0x022B, 0x0311,
    0x038C, 0x01C6, 0x00E3, 0x0275, 0x033E, 0x019F, 0x02CB, 0x0361, 0x03B4, 0x01DA, 0x00ED, 0x0272, 0x0139, 0x0298, 0x014C, 0x00A6,
    0x0053, 0x022D, 0x0312, 0x0189, 0x02C0, 0x0160, 0x00B0, 0x0058, 0x002C, 0x0016, 0x000B, 0x0201, 0x0304, 0x0182, 0x00C1, 0x0264,
    0x0132, 0x0099, 0x0248, 0x0124, 0x0092, 0x0049, 0x0220, 0x0110, 0x0088, 0x0044, 0x0022, 0x0011, 0x020C, 0x0106, 0x0083, 0x0245,
    0x0326, 0x0193, 0x02CD, 0x0362, 0x01B1, 0x02DC, 0x016E, 0x00B7, 0x025F, 0x032B, 0x0391, 0x03CC, 0x01E6, 0x00F3, 0x027D, 0x033A,
    0x019D, 0x02CA, 0x0165, 0x02B6, 0x015B, 0x02A9, 0x0350, 0x01A8, 0x00D4, 0x006A, 0x0035, 0x021E, 0x010F, 0x0283, 0x0345, 0x03A6,
    0x01D3, 0x02ED, 0x0372, 0x01B9, 0x02D8, 0x016C, 0x00B6, 0x005B, 0x0229, 0x0310, 0x0188, 0x00C4, 0x0062, 0x0031, 0x021C, 0x010E,
    0x0087, 0x0247, 0x0327, 0x0397, 0x03CF, 0x03E3, 0x03F5, 0x03FE, 0x01FF, 0x02FB, 0x0379, 0x03B8, 0x01DC, 0x00EE, 0x0077, 0x023F,
    0x031B, 0x0389, 0x03C0, 0x01E0, 0x00F0, 0x0078, 0x003C, 0x001E, 0x000F, 0x0203, 0x0305, 0x0386, 0x01C3, 0x02E5, 0x0376, 0x01BB,
    0x02D9, 0x0368, 0x01B4, 0x00DA, 0x006D, 0x0232, 0x0119, 0x0288, 0x0144, 0x00A2, 0x0051, 0x022C, 0x0116, 0x008B, 0x0241, 0x0324,
    0x0192, 0x00C9, 0x0260, 0x0130, 0x0098, 0x004C, 0x0026, 0x0013, 0x020D, 0x0302, 0x0181, 0x02C4, 0x0162, 0x00B1, 0x025C, 0x012E,
    0x0097, 0x024F, 0x0323, 0x0395, 0x03CE, 0x01E7, 0x02F7, 0x037F, 0x03BB, 0x03D9, 0x03E8, 0x01F4, 0x00FA, 0x007D, 0x023A, 0x011D,
    0x028A, 0x0145, 0x02A6, 0x0153, 0x02AD, 0x0352, 0x01A9, 0x02D0, 0x0168, 0x00B4, 0x005A, 0x002D, 0x0212, 0x0109, 0x0280, 0x0140,
    0x00A0, 0x0050, 0x0028, 0x0014, 0x000A, 0x0005, 0x0206, 0x0103, 0x0285, 0x0346, 0x01A3, 0x02D5, 0x036E, 0x01B7, 0x02DF, 0x036B,
    0x03B1, 0x03DC, 0x01EE, 0x00F7, 0x027F, 0x033B, 0x0399, 0x03C8, 0x01E4, 0x00F2, 0x0079, 0x0238, 0x011C, 0x008E, 0x0047, 0x0227,
    0x0317, 0x038F, 0x03C3, 0x03E5, 0x03F6, 0x01FB, 0x02F9, 0x0378, 0x01BC, 0x00DE, 0x006F, 0x0233, 0x031D, 0x038A, 0x01C5, 0x02E6,
    0x0173, 0x02BD, 0x035A, 0x01AD, 0x02D2, 0x0169, 0x02B0, 0x0158, 0x00AC, 0x0056, 0x002B, 0x0211, 0x030C, 0x0186, 0x00C3, 0x0265,
    0x0336, 0x019B, 0x02C9, 0x0360, 0x01B0, 0x00D8, 0x006C, 0x0036, 0x001B, 0x0209, 0x0300, 0x0180, 0x00C0, 0x0060, 0x0030, 0x0018,
    0x000C, 0x0006, 0x0003, 0x0205, 0x0306, 0x0183, 0x02C5, 0x0366, 0x01B3, 0x02DD, 0x036A, 0x01B5, 0x02DE, 0x016F, 0x02B3, 0x035D,
    0x03AA, 0x01D5, 0x02EE, 0x0177, 0x02BF, 0x035B, 0x03A9, 0x03D0, 0x01E8, 0x00F4, 0x007A, 0x003D, 0x021A, 0x010D, 0x0282, 0x0141,
    0x02A4, 0x0152, 0x00A9, 0x0250, 0x0128, 0x0094, 0x004A, 0x0025, 0x0216, 0x010B, 0x0281, 0x0344, 0x01A2, 0x00D1, 0x026C, 0x0136,
    0x009B, 0x0249, 0x0320, 0x0190, 0x00C8, 0x0064, 0x0032, 0x0019, 0x0208, 0x0104, 0x0082, 0x0041, 0x0224, 0x0112, 0x0089, 0x0240,
    0x0120, 0x0090, 0x0048, 0x0024, 0x0012, 0x0009, 0x0200, 0x0100, 0x0080, 0x0040, 0x0020, 0x0010, 0x0008, 0x0004, 0x0002, 0x0001,
  ]
};

pub static SYNDROME_TO_ERROR_POS: phf::Map<u32, &'static [u32]> = phf_map! {
  3u32 => &[
    0x0000, 0x0007, 0x0006, 0x0004, 0x0005, 0x0001, 0x0003, 0x0002
  ],
  4u32 => &[
    0x0000, 0x000F, 0x000E, 0x000B, 0x000D, 0x0007, 0x000A, 0x0005, 0x000C, 0x0001, 0x0006, 0x0008, 0x0009, 0x0002, 0x0004, 0x0003,
  ],
  5u32 => &[
    0x0000, 0x001F, 0x001E, 0x000D, 0x001D, 0x001A, 0x000C, 0x0014, 0x001C, 0x0002, 0x0019, 0x0004, 0x000B, 0x0017, 0x0013, 0x0008,
    0x001B, 0x0015, 0x0001, 0x000E, 0x0018, 0x0009, 0x0003, 0x0005, 0x000A, 0x0006, 0x0016, 0x000F, 0x0012, 0x0011, 0x0007, 0x0010,
  ],
  6u32 => &[
    0x0000, 0x003F, 0x003E, 0x0039, 0x003D, 0x0033, 0x0038, 0x0025, 0x003C, 0x001F, 0x0032, 0x001C, 0x0037, 0x000F, 0x0024, 0x002D,
    0x003B, 0x0027, 0x001E, 0x002F, 0x0031, 0x000B, 0x001B, 0x0009, 0x0036, 0x0012, 0x000E, 0x0019, 0x0023, 0x0016, 0x002C, 0x0007,
    0x003A, 0x0001, 0x0026, 0x0034, 0x001D, 0x0020, 0x002E, 0x0010, 0x0030, 0x0028, 0x000A, 0x000C, 0x001A, 0x0013, 0x0008, 0x0017,
    0x0035, 0x0002, 0x0011, 0x0021, 0x000D, 0x0029, 0x0018, 0x0014, 0x0022, 0x0003, 0x0015, 0x002A, 0x002B, 0x0004, 0x0006, 0x0005,
  ],
  7u32 => &[
    0x0000, 0x007F, 0x007E, 0x0060, 0x007D, 0x0041, 0x005F, 0x0018, 0x007C, 0x0078, 0x0040, 0x0070, 0x005E, 0x002B, 0x0017, 0x0022,
    0x007B, 0x0003, 0x0077, 0x0006, 0x003F, 0x0030, 0x006F, 0x000C, 0x005D, 0x0074, 0x002A, 0x0059, 0x0016, 0x0051, 0x0021, 0x004C,
    0x007A, 0x002D, 0x0002, 0x0043, 0x0076, 0x0053, 0x0005, 0x0032, 0x003E, 0x003C, 0x002F, 0x0055, 0x006E, 0x003A, 0x000B, 0x0068,
    0x005C, 0x0009, 0x0073, 0x0063, 0x0029, 0x0066, 0x0058, 0x0046, 0x0015, 0x006C, 0x0050, 0x0026, 0x0020, 0x0038, 0x004B, 0x0011,
    0x0079, 0x0071, 0x002C, 0x0023, 0x0001, 0x0061, 0x0042, 0x0019, 0x0075, 0x005A, 0x0052, 0x004D, 0x0004, 0x0007, 0x0031, 0x000D,
    0x003D, 0x0056, 0x003B, 0x0069, 0x002E, 0x0044, 0x0054, 0x0033, 0x006D, 0x0027, 0x0039, 0x0012, 0x000A, 0x0064, 0x0067, 0x0047,
    0x005B, 0x004E, 0x0008, 0x000E, 0x0072, 0x0024, 0x0062, 0x001A, 0x0028, 0x0013, 0x0065, 0x0048, 0x0057, 0x006A, 0x0045, 0x0034,
    0x0014, 0x0049, 0x006B, 0x0035, 0x004F, 0x000F, 0x0025, 0x001B, 0x001F, 0x001E, 0x0037, 0x001D, 0x004A, 0x0036, 0x0010, 0x001C,
  ],
  8u32 => &[
    0x0000, 0x00FF, 0x00FE, 0x00E6, 0x00FD, 0x00CD, 0x00E5, 0x0039, 0x00FC, 0x0020, 0x00CC, 0x0011, 0x00E4, 0x0097, 0x0038, 0x00B4,
    0x00FB, 0x009B, 0x001F, 0x00F1, 0x00CB, 0x0072, 0x0010, 0x007E, 0x00E3, 0x003E, 0x0096, 0x0007, 0x0037, 0x00F7, 0x00B3, 0x008E,
    0x00FA, 0x0075, 0x009A, 0x00D0, 0x001E, 0x00DB, 0x00F0, 0x00DE, 0x00CA, 0x006C, 0x0071, 0x0025, 0x000F, 0x00ED, 0x007D, 0x00BA,
    0x00E2, 0x004A, 0x003D, 0x0082, 0x0095, 0x00D8, 0x0006, 0x0046, 0x0036, 0x0065, 0x00F6, 0x0087, 0x00B2, 0x001B, 0x008D, 0x0059,
    0x00F9, 0x0040, 0x0074, 0x009D, 0x0099, 0x0022, 0x00CF, 0x0002, 0x001D, 0x0067, 0x00DA, 0x004C, 0x00EF, 0x006E, 0x00DD, 0x0077,
    0x00C9, 0x002F, 0x006B, 0x0031, 0x0070, 0x0069, 0x0024, 0x0042, 0x000E, 0x002D, 0x00EC, 0x00A3, 0x007C, 0x00C7, 0x00B9, 0x00BF,
    0x00E1, 0x00BD, 0x0049, 0x005C, 0x003C, 0x00B7, 0x0081, 0x0091, 0x0094, 0x00C5, 0x00D7, 0x00AB, 0x0005, 0x007A, 0x0045, 0x00C2,
    0x0035, 0x00A1, 0x0064, 0x0060, 0x00F5, 0x00EA, 0x0086, 0x00D4, 0x00B1, 0x002B, 0x001A, 0x0053, 0x008C, 0x000C, 0x0058, 0x00A8,
    0x00F8, 0x008F, 0x003F, 0x0008, 0x0073, 0x007F, 0x009C, 0x00F2, 0x0098, 0x00B5, 0x0021, 0x0012, 0x00CE, 0x003A, 0x0001, 0x00E7,
    0x001C, 0x005A, 0x0066, 0x0088, 0x00D9, 0x0047, 0x004B, 0x0083, 0x00EE, 0x00BB, 0x006D, 0x0026, 0x00DC, 0x00DF, 0x0076, 0x00D1,
    0x00C8, 0x00C0, 0x002E, 0x00A4, 0x006A, 0x0043, 0x0030, 0x0032, 0x006F, 0x0078, 0x0068, 0x004D, 0x0023, 0x0003, 0x0041, 0x009E,
    0x000D, 0x00A9, 0x002C, 0x0054, 0x00EB, 0x00D5, 0x00A2, 0x0061, 0x007B, 0x00C3, 0x00C6, 0x00AC, 0x00B8, 0x0092, 0x00BE, 0x005D,
    0x00E0, 0x00D2, 0x00BC, 0x0027, 0x0048, 0x0084, 0x005B, 0x0089, 0x003B, 0x00E8, 0x00B6, 0x0013, 0x0080, 0x00F3, 0x0090, 0x0009,
    0x0093, 0x005E, 0x00C4, 0x00AD, 0x00D6, 0x0062, 0x00AA, 0x0055, 0x0004, 0x009F, 0x0079, 0x004E, 0x0044, 0x0033, 0x00C1, 0x00A5,
    0x0034, 0x00A6, 0x00A0, 0x004F, 0x0063, 0x0056, 0x005F, 0x00AE, 0x00F4, 0x000A, 0x00E9, 0x0014, 0x0085, 0x008A, 0x00D3, 0x0028,
    0x00B0, 0x0051, 0x002A, 0x0016, 0x0019, 0x0018, 0x0052, 0x0017, 0x008B, 0x0029, 0x000B, 0x0015, 0x0057, 0x00AF, 0x00A7, 0x0050,
  ],
  9u32 => &[
    0x0000, 0x01FF, 0x01FE, 0x017D, 0x01FD, 0x00FB, 0x017C, 0x00DD, 0x01FC, 0x005B, 0x00FA, 0x0114, 0x017B, 0x012A, 0x00DC, 0x0079,
    0x01FB, 0x01F6, 0x005A, 0x01EC, 0x00F9, 0x01BA, 0x0113, 0x00A8, 0x017A, 0x00B3, 0x0129, 0x01D8, 0x00DB, 0x0092, 0x0078, 0x0086,
    0x01FA, 0x0004, 0x01F5, 0x0008, 0x0059, 0x00BA, 0x01EB, 0x0010, 0x00F8, 0x01C0, 0x01B9, 0x0031, 0x0112, 0x0156, 0x00A7, 0x006A,
    0x0179, 0x01F1, 0x00B2, 0x0174, 0x0128, 0x016A, 0x01D7, 0x0020, 0x00DA, 0x0026, 0x0091, 0x014F, 0x0077, 0x0046, 0x0085, 0x0138,
    0x01F9, 0x00B6, 0x0003, 0x005E, 0x01F4, 0x0029, 0x0007, 0x01C3, 0x0058, 0x01A0, 0x00B9, 0x01A3, 0x01EA, 0x00CD, 0x000F, 0x0190,
    0x00F7, 0x0055, 0x01BF, 0x016F, 0x01B8, 0x00F2, 0x0030, 0x01E2, 0x0111, 0x019D, 0x0155, 0x0144, 0x00A6, 0x0163, 0x0069, 0x00E8,
    0x0178, 0x000C, 0x01F0, 0x0181, 0x00B1, 0x0185, 0x0173, 0x0062, 0x0127, 0x018D, 0x0169, 0x0098, 0x01D6, 0x01CB, 0x001F, 0x0038,
    0x00D9, 0x01E7, 0x0025, 0x00AD, 0x0090, 0x0050, 0x014E, 0x00D4, 0x0076, 0x00CA, 0x0045, 0x013E, 0x0084, 0x01AE, 0x0137, 0x003F,
    0x01F8, 0x01BC, 0x00B5, 0x0094, 0x0002, 0x00FD, 0x005D, 0x012C, 0x01F3, 0x016C, 0x0028, 0x0048, 0x0006, 0x00BC, 0x01C2, 0x0158,
    0x0057, 0x00F4, 0x019F, 0x0165, 0x00B8, 0x002B, 0x01A2, 0x00CF, 0x01E9, 0x0052, 0x00CC, 0x01B0, 0x000E, 0x0187, 0x018F, 0x01CD,
    0x00F6, 0x002D, 0x0054, 0x0189, 0x01BE, 0x00FF, 0x016E, 0x00BE, 0x01B7, 0x01DF, 0x00F1, 0x0018, 0x002F, 0x0101, 0x01E1, 0x0103,
    0x0110, 0x01B5, 0x019C, 0x0123, 0x0154, 0x01DD, 0x0143, 0x0149, 0x00A5, 0x00EF, 0x0162, 0x010B, 0x0068, 0x0016, 0x00E7, 0x00C4,
    0x0177, 0x0152, 0x000B, 0x0034, 0x01EF, 0x01DB, 0x0180, 0x0117, 0x00B0, 0x0141, 0x0184, 0x009B, 0x0172, 0x0147, 0x0061, 0x01A6,
    0x0126, 0x010E, 0x018C, 0x001B, 0x0168, 0x01B3, 0x0097, 0x004B, 0x01D5, 0x019A, 0x01CA, 0x011E, 0x001E, 0x0121, 0x0037, 0x009E,
    0x00D8, 0x0066, 0x01E6, 0x01C7, 0x0024, 0x0014, 0x00AC, 0x00E1, 0x008F, 0x00E5, 0x004F, 0x011B, 0x014D, 0x00C2, 0x00D3, 0x0130,
    0x0075, 0x00A3, 0x00C9, 0x01D2, 0x0044, 0x00ED, 0x013D, 0x008B, 0x0083, 0x0160, 0x01AD, 0x0197, 0x0136, 0x0109, 0x003E, 0x0070,
    0x01F7, 0x01ED, 0x01BB, 0x00A9, 0x00B4, 0x01D9, 0x0093, 0x0087, 0x0001, 0x017E, 0x00FC, 0x00DE, 0x005C, 0x0115, 0x012B, 0x007A,
    0x01F2, 0x0175, 0x016B, 0x0021, 0x0027, 0x0150, 0x0047, 0x0139, 0x0005, 0x0009, 0x00BB, 0x0011, 0x01C1, 0x0032, 0x0157, 0x006B,
    0x0056, 0x0170, 0x00F3, 0x01E3, 0x019E, 0x0145, 0x0164, 0x00E9, 0x00B7, 0x005F, 0x002A, 0x01C4, 0x01A1, 0x01A4, 0x00CE, 0x0191,
    0x01E8, 0x00AE, 0x0051, 0x00D5, 0x00CB, 0x013F, 0x01AF, 0x0040, 0x000D, 0x0182, 0x0186, 0x0063, 0x018E, 0x0099, 0x01CC, 0x0039,
    0x00F5, 0x0166, 0x002C, 0x00D0, 0x0053, 0x01B1, 0x0188, 0x01CE, 0x01BD, 0x0095, 0x00FE, 0x012D, 0x016D, 0x0049, 0x00BD, 0x0159,
    0x01B6, 0x0124, 0x01DE, 0x014A, 0x00F0, 0x010C, 0x0017, 0x00C5, 0x002E, 0x018A, 0x0100, 0x00BF, 0x01E0, 0x0019, 0x0102, 0x0104,
    0x010F, 0x001C, 0x01B4, 0x004C, 0x019B, 0x011F, 0x0122, 0x009F, 0x0153, 0x0035, 0x01DC, 0x0118, 0x0142, 0x009C, 0x0148, 0x01A7,
    0x00A4, 0x01D3, 0x00EE, 0x008C, 0x0161, 0x0198, 0x010A, 0x0071, 0x0067, 0x01C8, 0x0015, 0x00E2, 0x00E6, 0x011C, 0x00C3, 0x0131,
    0x0176, 0x0022, 0x0151, 0x013A, 0x000A, 0x0012, 0x0033, 0x006C, 0x01EE, 0x00AA, 0x01DA, 0x0088, 0x017F, 0x00DF, 0x0116, 0x007B,
    0x00AF, 0x00D6, 0x0140, 0x0041, 0x0183, 0x0064, 0x009A, 0x003A, 0x0171, 0x01E4, 0x0146, 0x00EA, 0x0060, 0x01C5, 0x01A5, 0x0192,
    0x0125, 0x014B, 0x010D, 0x00C6, 0x018B, 0x00C0, 0x001A, 0x0105, 0x0167, 0x00D1, 0x01B2, 0x01CF, 0x0096, 0x012E, 0x004A, 0x015A,
    0x01D4, 0x008D, 0x0199, 0x0072, 0x01C9, 0x00E3, 0x011D, 0x0132, 0x001D, 0x004D, 0x0120, 0x00A0, 0x0036, 0x0119, 0x009D, 0x01A8,
    0x00D7, 0x0042, 0x0065, 0x003B, 0x01E5, 0x00EB, 0x01C6, 0x0193, 0x0023, 0x013B, 0x0013, 0x006D, 0x00AB, 0x0089, 0x00E0, 0x007C,
    0x008E, 0x0073, 0x00E4, 0x0133, 0x004E, 0x00A1, 0x011A, 0x01A9, 0x014C, 0x00C7, 0x00C1, 0x0106, 0x00D2, 0x01D0, 0x012F, 0x015B,
    0x0074, 0x0134, 0x00A2, 0x01AA, 0x00C8, 0x0107, 0x01D1, 0x015C, 0x0043, 0x003C, 0x00EC, 0x0194, 0x013C, 0x006E, 0x008A, 0x007D,
    0x0082, 0x0081, 0x015F, 0x0080, 0x01AC, 0x015E, 0x0196, 0x007F, 0x0135, 0x01AB, 0x0108, 0x015D, 0x003D, 0x0195, 0x006F, 0x007E,
  ],
  10u32 => &[
    0x0000, 0x03FF, 0x03FE, 0x03B2, 0x03FD, 0x0365, 0x03B1, 0x0043, 0x03FC, 0x03F5, 0x0364, 0x02BA, 0x03B0, 0x0195, 0x0042, 0x0318,
    0x03FB, 0x02CB, 0x03F4, 0x0337, 0x0363, 0x0086, 0x02B9, 0x0148, 0x03AF, 0x03E7, 0x0194, 0x03A8, 0x0041, 0x026D, 0x0317, 0x024B,
    0x03FA, 0x01FE, 0x02CA, 0x01D8, 0x03F3, 0x03D7, 0x0336, 0x0220, 0x0362, 0x01F9, 0x0085, 0x039A, 0x02B8, 0x035B, 0x0147, 0x00A3,
    0x03AE, 0x02FD, 0x03E6, 0x027E, 0x0193, 0x02EA, 0x03A7, 0x01BE, 0x0040, 0x00FB, 0x026C, 0x0157, 0x0316, 0x03CB, 0x024A, 0x0039,
    0x03F9, 0x03EB, 0x01FD, 0x00FF, 0x02C9, 0x0175, 0x01D7, 0x037E, 0x03F2, 0x02C5, 0x03D6, 0x00AE, 0x0335, 0x010A, 0x021F, 0x002B,
    0x0361, 0x032A, 0x01F8, 0x02B0, 0x0084, 0x0231, 0x0399, 0x0074, 0x02B7, 0x0171, 0x035A, 0x02F7, 0x0146, 0x028E, 0x00A2, 0x029D,
    0x03AD, 0x015C, 0x02FC, 0x01B1, 0x03E5, 0x018B, 0x027D, 0x0020, 0x0192, 0x01D3, 0x02E9, 0x00C9, 0x03A6, 0x0324, 0x01BD, 0x038A,
    0x003F, 0x0056, 0x00FA, 0x01EA, 0x026B, 0x0214, 0x0156, 0x030E, 0x0315, 0x037A, 0x03CA, 0x01AC, 0x0249, 0x034D, 0x0038, 0x0050,
    0x03F8, 0x0003, 0x03EA, 0x02CE, 0x01FC, 0x0201, 0x00FE, 0x0300, 0x02C8, 0x03EE, 0x0174, 0x032D, 0x01D6, 0x015F, 0x037D, 0x0059,
    0x03F1, 0x0006, 0x02C4, 0x0009, 0x03D5, 0x019D, 0x00AD, 0x0340, 0x0334, 0x02C1, 0x0109, 0x03E0, 0x021E, 0x0253, 0x002A, 0x01C7,
    0x0360, 0x019A, 0x0329, 0x010F, 0x01F7, 0x0164, 0x02AF, 0x00EB, 0x0083, 0x03D2, 0x0230, 0x00DE, 0x0398, 0x01F2, 0x0073, 0x013E,
    0x02B6, 0x033D, 0x0170, 0x000F, 0x0359, 0x017D, 0x02F6, 0x02D7, 0x0145, 0x00AA, 0x028D, 0x0186, 0x00A1, 0x007C, 0x029C, 0x00F4,
    0x03AC, 0x02BE, 0x015B, 0x039E, 0x02FB, 0x00B2, 0x01B0, 0x00CD, 0x03E4, 0x0331, 0x018A, 0x00E2, 0x027C, 0x00E6, 0x001F, 0x0128,
    0x0191, 0x03DD, 0x01D2, 0x016A, 0x02E8, 0x025B, 0x00C8, 0x00BD, 0x03A5, 0x0106, 0x0323, 0x0278, 0x01BC, 0x0061, 0x0389, 0x023C,
    0x003E, 0x0250, 0x0055, 0x02A2, 0x00F9, 0x01CC, 0x01E9, 0x0241, 0x026A, 0x021B, 0x0213, 0x0124, 0x0155, 0x02AA, 0x030D, 0x0135,
    0x0314, 0x01C4, 0x0379, 0x02DD, 0x03C9, 0x0263, 0x01AB, 0x0373, 0x0248, 0x0027, 0x034C, 0x001B, 0x0037, 0x00D5, 0x004F, 0x01E4,
    0x03F7, 0x0197, 0x0002, 0x0367, 0x03E9, 0x026F, 0x02CD, 0x0088, 0x01FB, 0x035D, 0x0200, 0x03D9, 0x00FD, 0x03CD, 0x02FF, 0x02EC,
    0x02C7, 0x010C, 0x03ED, 0x0177, 0x0173, 0x0290, 0x032C, 0x0233, 0x01D5, 0x0326, 0x015E, 0x018D, 0x037C, 0x034F, 0x0058, 0x0216,
    0x03F0, 0x0161, 0x0005, 0x0203, 0x02C3, 0x0255, 0x0008, 0x019F, 0x03D4, 0x01F4, 0x019C, 0x0166, 0x00AC, 0x007E, 0x033F, 0x017F,
    0x0333, 0x00E8, 0x02C0, 0x00B4, 0x0108, 0x0063, 0x03DF, 0x025D, 0x021D, 0x02AC, 0x0252, 0x01CE, 0x0029, 0x00D7, 0x01C6, 0x0265,
    0x035F, 0x03CF, 0x0199, 0x0271, 0x0328, 0x0351, 0x010E, 0x0292, 0x01F6, 0x0080, 0x0163, 0x0257, 0x02AE, 0x00D9, 0x00EA, 0x0065,
    0x0082, 0x00DB, 0x03D1, 0x0353, 0x022F, 0x0097, 0x00DD, 0x0099, 0x0397, 0x022D, 0x01F1, 0x02E4, 0x0072, 0x0095, 0x013D, 0x011F,
    0x02B5, 0x01EF, 0x033C, 0x0283, 0x016F, 0x02E2, 0x000E, 0x0014, 0x0358, 0x0395, 0x017C, 0x00B9, 0x02F5, 0x022B, 0x02D6, 0x03BD,
    0x0144, 0x013B, 0x00A9, 0x0390, 0x028C, 0x011D, 0x0185, 0x03C3, 0x00A0, 0x0070, 0x007B, 0x00C4, 0x029B, 0x0093, 0x00F3, 0x020E,
    0x03AB, 0x033A, 0x02BD, 0x03B5, 0x015A, 0x0281, 0x039D, 0x01DB, 0x02FA, 0x02B3, 0x00B1, 0x0102, 0x01AF, 0x01ED, 0x00CC, 0x01B4,
    0x03E3, 0x000C, 0x0330, 0x02D1, 0x0189, 0x0012, 0x00E1, 0x0112, 0x027B, 0x016D, 0x00E5, 0x03A1, 0x001E, 0x02E0, 0x0127, 0x02A5,
    0x0190, 0x017A, 0x03DC, 0x036A, 0x01D1, 0x00B7, 0x0169, 0x0206, 0x02E7, 0x0356, 0x025A, 0x0274, 0x00C7, 0x0393, 0x00BC, 0x0286,
    0x03A4, 0x02D4, 0x0105, 0x03B8, 0x0322, 0x03BB, 0x0277, 0x036D, 0x01BB, 0x02F3, 0x0060, 0x031F, 0x0388, 0x0229, 0x023B, 0x0150,
    0x003D, 0x00A7, 0x024F, 0x031C, 0x0054, 0x038E, 0x02A1, 0x002F, 0x00F8, 0x0142, 0x01CB, 0x005D, 0x01E8, 0x0139, 0x0240, 0x012C,
    0x0269, 0x0183, 0x021A, 0x02F0, 0x0212, 0x03C1, 0x0123, 0x0069, 0x0154, 0x028A, 0x02A9, 0x01B8, 0x030C, 0x011B, 0x0134, 0x0130,
    0x0313, 0x0079, 0x01C3, 0x014D, 0x0378, 0x00C2, 0x02DC, 0x0345, 0x03C8, 0x009E, 0x0262, 0x0238, 0x01AA, 0x006E, 0x0372, 0x0117,
    0x0247, 0x00F1, 0x0026, 0x0226, 0x034B, 0x020C, 0x001A, 0x01A5, 0x0036, 0x0299, 0x00D4, 0x0385, 0x004E, 0x0091, 0x01E3, 0x0308,
    0x03F6, 0x02BB, 0x0196, 0x0319, 0x0001, 0x03B3, 0x0366, 0x0044, 0x03E8, 0x03A9, 0x026E, 0x024C, 0x02CC, 0x0338, 0x0087, 0x0149,
    0x01FA, 0x039B, 0x035C, 0x00A4, 0x01FF, 0x01D9, 0x03D8, 0x0221, 0x00FC, 0x0158, 0x03CC, 0x003A, 0x02FE, 0x027F, 0x02EB, 0x01BF,
    0x02C6, 0x00AF, 0x010B, 0x002C, 0x03EC, 0x0100, 0x0176, 0x037F, 0x0172, 0x02F8, 0x028F, 0x029E, 0x032B, 0x02B1, 0x0232, 0x0075,
    0x01D4, 0x00CA, 0x0325, 0x038B, 0x015D, 0x01B2, 0x018C, 0x0021, 0x037B, 0x01AD, 0x034E, 0x0051, 0x0057, 0x01EB, 0x0215, 0x030F,
    0x03EF, 0x032E, 0x0160, 0x005A, 0x0004, 0x02CF, 0x0202, 0x0301, 0x02C2, 0x03E1, 0x0254, 0x01C8, 0x0007, 0x000A, 0x019E, 0x0341,
    0x03D3, 0x00DF, 0x01F3, 0x013F, 0x019B, 0x0110, 0x0165, 0x00EC, 0x00AB, 0x0187, 0x007D, 0x00F5, 0x033E, 0x0010, 0x017E, 0x02D8,
    0x0332, 0x00E3, 0x00E7, 0x0129, 0x02BF, 0x039F, 0x00B3, 0x00CE, 0x0107, 0x0279, 0x0062, 0x023D, 0x03DE, 0x016B, 0x025C, 0x00BE,
    0x021C, 0x0125, 0x02AB, 0x0136, 0x0251, 0x02A3, 0x01CD, 0x0242, 0x0028, 0x001C, 0x00D6, 0x01E5, 0x01C5, 0x02DE, 0x0264, 0x0374,
    0x035E, 0x03DA, 0x03CE, 0x02ED, 0x0198, 0x0368, 0x0270, 0x0089, 0x0327, 0x018E, 0x0350, 0x0217, 0x010D, 0x0178, 0x0291, 0x0234,
    0x01F5, 0x0167, 0x007F, 0x0180, 0x0162, 0x0204, 0x0256, 0x01A0, 0x02AD, 0x01CF, 0x00D8, 0x0266, 0x00E9, 0x00B5, 0x0064, 0x025E,
    0x0081, 0x0258, 0x00DA, 0x0066, 0x03D0, 0x0272, 0x0352, 0x0293, 0x022E, 0x02E5, 0x0096, 0x0120, 0x00DC, 0x0354, 0x0098, 0x009A,
    0x0396, 0x00BA, 0x022C, 0x03BE, 0x01F0, 0x0284, 0x02E3, 0x0015, 0x0071, 0x00C5, 0x0094, 0x020F, 0x013C, 0x0391, 0x011E, 0x03C4,
    0x02B4, 0x0103, 0x01EE, 0x01B5, 0x033B, 0x03B6, 0x0282, 0x01DC, 0x016E, 0x03A2, 0x02E1, 0x02A6, 0x000D, 0x02D2, 0x0013, 0x0113,
    0x0357, 0x0275, 0x0394, 0x0287, 0x017B, 0x036B, 0x00B8, 0x0207, 0x02F4, 0x0320, 0x022A, 0x0151, 0x02D5, 0x03B9, 0x03BC, 0x036E,
    0x0143, 0x005E, 0x013A, 0x012D, 0x00A8, 0x031D, 0x038F, 0x0030, 0x028B, 0x01B9, 0x011C, 0x0131, 0x0184, 0x02F1, 0x03C2, 0x006A,
    0x009F, 0x0239, 0x006F, 0x0118, 0x007A, 0x014E, 0x00C3, 0x0346, 0x029A, 0x0386, 0x0092, 0x0309, 0x00F2, 0x0227, 0x020D, 0x01A6,
    0x03AA, 0x024D, 0x0339, 0x014A, 0x02BC, 0x031A, 0x03B4, 0x0045, 0x0159, 0x003B, 0x0280, 0x01C0, 0x039C, 0x00A5, 0x01DA, 0x0222,
    0x02F9, 0x029F, 0x02B2, 0x0076, 0x00B0, 0x002D, 0x0101, 0x0380, 0x01AE, 0x0052, 0x01EC, 0x0310, 0x00CB, 0x038C, 0x01B3, 0x0022,
    0x03E2, 0x01C9, 0x000B, 0x0342, 0x032F, 0x005B, 0x02D0, 0x0302, 0x0188, 0x00F6, 0x0011, 0x02D9, 0x00E0, 0x0140, 0x0111, 0x00ED,
    0x027A, 0x023E, 0x016C, 0x00BF, 0x00E4, 0x012A, 0x03A0, 0x00CF, 0x001D, 0x01E6, 0x02DF, 0x0375, 0x0126, 0x0137, 0x02A4, 0x0243,
    0x018F, 0x0218, 0x0179, 0x0235, 0x03DB, 0x02EE, 0x0369, 0x008A, 0x01D0, 0x0267, 0x00B6, 0x025F, 0x0168, 0x0181, 0x0205, 0x01A1,
    0x02E6, 0x0121, 0x0355, 0x009B, 0x0259, 0x0067, 0x0273, 0x0294, 0x00C6, 0x0210, 0x0392, 0x03C5, 0x00BB, 0x03BF, 0x0285, 0x0016,
    0x03A3, 0x02A7, 0x02D3, 0x0114, 0x0104, 0x01B6, 0x03B7, 0x01DD, 0x0321, 0x0152, 0x03BA, 0x036F, 0x0276, 0x0288, 0x036C, 0x0208,
    0x01BA, 0x0132, 0x02F2, 0x006B, 0x005F, 0x012E, 0x031E, 0x0031, 0x0387, 0x030A, 0x0228, 0x01A7, 0x023A, 0x0119, 0x014F, 0x0347,
    0x003C, 0x01C1, 0x00A6, 0x0223, 0x024E, 0x014B, 0x031B, 0x0046, 0x0053, 0x0311, 0x038D, 0x0023, 0x02A0, 0x0077, 0x002E, 0x0381,
    0x00F7, 0x02DA, 0x0141, 0x00EE, 0x01CA, 0x0343, 0x005C, 0x0303, 0x01E7, 0x0376, 0x0138, 0x0244, 0x023F, 0x00C0, 0x012B, 0x00D0,
    0x0268, 0x0260, 0x0182, 0x01A2, 0x0219, 0x0236, 0x02EF, 0x008B, 0x0211, 0x03C6, 0x03C0, 0x0017, 0x0122, 0x009C, 0x0068, 0x0295,
    0x0153, 0x0370, 0x0289, 0x0209, 0x02A8, 0x0115, 0x01B7, 0x01DE, 0x030B, 0x01A8, 0x011A, 0x0348, 0x0133, 0x006C, 0x012F, 0x0032,
    0x0312, 0x0024, 0x0078, 0x0382, 0x01C2, 0x0224, 0x014C, 0x0047, 0x0377, 0x0245, 0x00C1, 0x00D1, 0x02DB, 0x00EF, 0x0344, 0x0304,
    0x03C7, 0x0018, 0x009D, 0x0296, 0x0261, 0x01A3, 0x0237, 0x008C, 0x01A9, 0x0349, 0x006D, 0x0033, 0x0371, 0x020A, 0x0116, 0x01DF,
    0x0246, 0x00D2, 0x00F0, 0x0305, 0x0025, 0x0383, 0x0225, 0x0048, 0x034A, 0x0034, 0x020B, 0x01E0, 0x0019, 0x0297, 0x01A4, 0x008D,
    0x0035, 0x01E1, 0x0298, 0x008E, 0x00D3, 0x0306, 0x0384, 0x0049, 0x004D, 0x004C, 0x0090, 0x004B, 0x01E2, 0x008F, 0x0307, 0x004A,
  ],
};
