use std::any::TypeId;
use std::ptr::null_mut;
use std::mem::MaybeUninit;

use super::null::Null;
use super::ptr::*;

#[derive(Clone, Copy, PartialEq, Eq)]
pub struct Tag {
    raw: TypeId
}
pub trait Taged: 'static {
    #[inline]
    fn tag() -> Tag {
        Tag {
            raw: TypeId::of::<Self>()
        }
    }
}
impl<T: 'static> Taged for T {
}

pub struct Var<T: Copy + Eq = Tag> {
    pub tag: T
}
pub struct VarPtr<T: Copy + Eq = Tag> {
    raw: *mut u8,
    tag: T
}
impl<T: Copy + Eq> PtxType for Var<T> {
    type Pointer = VarPtr<T>;
}
impl<T: Copy + Eq> Ptx<Var<T>> {
    #[inline]
    pub fn tag(&self) -> T {
        self.raw.tag
    }
}

impl<T: Copy + Eq> Clone for VarPtr<T> {
    #[inline]
    fn clone(&self) -> Self {
        Self { 
            raw: self.raw,
            tag: self.tag
        }
    }
}
impl<T: Copy + Eq> Clone for Ptx<Var<T>> {
    #[inline]
    fn clone(&self) -> Self {
        Self { 
            raw: self.raw.clone()
        }
    }
}
impl<T: Copy + Eq> Copy for VarPtr<T> {
}
impl<T: Copy + Eq> Copy for Ptx<Var<T>> {
}

impl<T: 'static> Ptr<T> {
    #[inline]
    pub fn cast(&self) -> Ptx<Var> {
        Ptx::<Var> {
            raw: VarPtr {
                raw: self.raw as *mut u8,
                tag: T::tag()
            }
        }
    }
}
impl<T: Copy + Eq> Var<T> {
    #[inline]
    pub fn from<D>(ptr: &Ptr<D>, tag: T) -> Ptx<Var<T>> {
        Ptx::<Var<T>> {
            raw: VarPtr::<T> {
                raw: ptr.raw as *mut u8,
                tag
            }
        }
    }
}
impl<T: Copy + Eq> Ptx<Var<T>> {
    #[inline]
    pub fn cast<Target>(&self) -> Ptr<Target> {
        Ptr::<Target> {
            raw: self.raw.raw as *mut Target
        }
    }
}
impl Ptx<Var> {
    #[inline]
    pub fn is<T: 'static>(&self) -> bool {
        self.tag() == T::tag()
    }
}

impl<T: Copy + Eq> Null for Ptx<Var<T>> {
    #[inline]
    fn null() -> Self {
        unsafe {
            Self {
                raw: VarPtr::<T> {
                    raw: null_mut(),
                    tag: MaybeUninit::uninit().assume_init()
                }
            }
        }
    }
    #[inline]
    fn is_null(&self) -> bool {
        self.raw.raw.is_null()
    }
}

impl<T: Copy + Eq> Default for Ptx<Var<T>> {
    #[inline]
    fn default() -> Self {
        <Self as Null>::null()
    }
}
