use std::alloc::{alloc, dealloc, Layout};
use std::ops::{Deref, DerefMut};
use std::ptr::null_mut;

use super::null::Null;

pub struct Ptr<T> {
    pub(super) raw: *mut T
}
impl<T> Clone for Ptr<T> {
    #[inline]
    fn clone(&self) -> Self {
        Self { raw: self.raw.clone() }
    }
}
impl<T> Copy for Ptr<T> {
}

pub trait AllocPtr: Sized {
    #[inline]
    fn alloc_uninit() -> Ptr<Self> {
        unsafe {
            Ptr::<Self> {
                raw: alloc(Layout::new::<Self>()) as *mut Self
            }
        }
    }
    #[inline]
    fn alloc_value(value: Self) -> Ptr<Self> {
        let mut ptr = Self::alloc_uninit();
        *ptr = value;
        ptr
    }
}
impl<T> AllocPtr for T {  
}
impl<T> Ptr<T> {
    #[inline]
    pub fn dealloc(&self) {
        unsafe {
            dealloc(self.raw as *mut u8, Layout::new::<T>())
        }
    }
}

pub trait NullPtr: Sized {
    #[inline]
    fn null() -> Ptr<Self> {
        Ptr::<Self> {
            raw: null_mut()
        }
    }
}
impl<T> NullPtr for T { 
}
impl<T> Null for Ptr<T> {
    #[inline]
    fn null() -> Self {
        T::null()
    }
    #[inline]
    fn is_null(&self) -> bool {
        self.raw.is_null()
    }
}

impl<T> Deref for Ptr<T> {
    type Target = T;
    
    #[inline]
    fn deref(&self) -> &Self::Target {
        unsafe {
            &*self.raw
        }    
    }
}
impl<T> DerefMut for Ptr<T> {
    #[inline]
    fn deref_mut(&mut self) -> &mut Self::Target {
        unsafe {
            &mut *self.raw
        }
    }
}

pub trait AllocDefaultPtr: Sized + Default {
    #[inline]
    fn alloc() -> Ptr<Self> {
        Self::alloc_value(Self::default())
    }
}
impl<T: Default> AllocDefaultPtr for T {
}
impl<T> Default for Ptr<T> {
    #[inline]
    fn default() -> Self {
        T::null()
    }
}

pub trait PtxType {
    type Pointer: Copy;
}
pub struct Ptx<T: PtxType + ?Sized> {
    pub(super) raw: T::Pointer
}