//! Diving deeper into mathematics, huh? In here you'll find mathematical sets - Sometimes pretty handy!
/// Brings mathematical sets into Rust.
#[derive(Debug, Clone)]
pub struct Set<'a, T> {
    elements: Vec<T>,
    superset: Option<&'a Set<'a, T>>,
}

// Main impl
impl<'a, T: Copy + Ord> Set<'a, T> {
    /// Creates a new Set.
    ///
    /// # Arguments
    /// * `values` - The values for the set.
    ///
    /// # Returns
    /// A new set.
    /// 
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// 
    /// let set = Set::new(&vec![0, 1, 1, 2, 3, 4, 5]);
    /// 
    /// assert_eq!(set.elements(), &vec![0, 1, 2, 3, 4, 5]);
    /// ```
    pub fn new(values: &[T]) -> Set<'a, T> {        
        let mut res: Set<T> = Set { elements: values.to_vec(),
                                    superset: None };
        res.elements.sort_unstable();
        res.elements.dedup();
        res
    }
    /// Creates a new Set using a parent-Set to which it applies a closure.
    ///
    /// # Arguments
    /// * `parent` - The Set from which the new set emerges.
    /// * `f` - The closure after which the new set is created.
    ///
    /// # Returns
    /// A child Set.
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// let test1:       Set<u8> = Set::new(&vec![0,1,2,3,4]);
    /// let from_parent: Set<u8> = Set::<u8>::new_subset(&test1, |x| x % 2 == 0);
    /// assert_eq!(from_parent, Set::new(&vec![0,2,4]));
    /// assert_eq!(test1.elements(), &vec![0,1,2,3,4])
    /// ```
    pub fn new_subset<F: Fn(T) -> bool>(parent: &'a Set<T>, f: F) -> Set<'a, T> {
            let mut res: Set<T> = Set { elements: Vec::new(),
                                        superset: Some(parent) };
            for elem in &parent.elements {
                if f(*elem) {
                    res.elements.push(*elem);
                }
            }
            res
    }
    /// Does a mathematical union on two sets.
    /// `self ∪ other`.
    /// # Arguments
    /// * `self` - The first set.
    /// * `other` - The second set.
    ///
    /// # Returns
    /// A new `Set<T>`: `self ∪ other`.
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// use lib_rapid::math::sets::new_set;
    /// use lib_rapid::compsci::general::BinaryInsert;
    /// let s:  Set<i32> = Set::new(&vec![0,1,2,3,4,5,6,7,8,9,10]);
    /// let s1: Set<i32> = Set::new(&vec![11,12,13,13,11,0,0,0]);
    /// 
    /// let c:  Set<i32> = s.union(&s1);
    /// assert_eq!(c, new_set!(0,1,2,3,4,5,6,7,8,9,10,11,12,13));
    /// ```
    pub fn union(&self, other: &Set<T>) -> Set<T> {
        let mut res: Set<T> = Set {elements: Vec::new(),
                                   superset: None };

        res.elements.append(&mut self.elements.clone());
        res.elements.append(&mut other.elements.clone());

        res.elements.sort_unstable(); 
        res.elements.dedup();
        res
    }
    /// Does a mathematical intersection on two sets.
    ///
    /// # Arguments
    /// * `self` - The first set.
    /// * `other` - The second set.
    ///
    /// # Returns
    /// A new `Set<T>`: `self ∩ other`.
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// use lib_rapid::math::sets::new_set;
    /// use lib_rapid::compsci::general::BinaryInsert; // Used for "new_set!"
    /// 
    /// let s:  Set<i32> = Set::new(&vec![0,1,2,3,4,5,6,7,8,9,10]);
    /// let s2: Set<i32> = Set::new(&vec![0,1,2,3,11,0,0,0]);
    /// 
    /// let c:  Set<i32> = s.intersection(&s2);
    /// assert_eq!(c, new_set!(0, 1, 2, 3));
    /// ```
    pub fn intersection(&self, other: &Set<T>) -> Set<T> {
        let mut res: Set<T> = self.clone();

        for _ in &self.elements {
            res.elements.retain(|x| other.elements.contains(x));
        }
        res
    }
    /// Lets you check for an element in a set.
    ///
    /// # Arguments
    /// * `elem` - The element to check for.
    ///
    /// # Returns
    /// A boolean value which determines if `elem ∈ self`. 
    /// 
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// use lib_rapid::math::sets::new_set;
    /// 
    /// let set = new_set!(0, 1, 2, 3, 4, 5, 6);
    /// 
    /// assert_eq!(false, set.has_element(7));
    /// assert_eq!(false, set.has_element(-1));
    /// assert_eq!(true, set.has_element(1));
    /// ```
    pub fn has_element(&self, elem: T) -> bool {
        match self.elements.binary_search(&elem) {
            Ok(_)  => { return true; }
            Err(_) => { return false; }
        }
    }
    /// Lets you insert an element into a set. Does not insert already present values.
    ///
    /// # Arguments
    /// * `elem` - The element to insert.
    ///
    /// # Returns
    /// Nothing.
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// use lib_rapid::math::sets::new_set;
    /// let mut s: Set<i32> = Set::new(&vec![0,1,2,3,4,5,6,7,8,9,10]);
    /// 
    /// s.insert(5);
    /// assert_eq!(s.elements(), &vec![0,1,2,3,4,5,6,7,8,9,10]);
    /// ```
    pub fn insert(&mut self, elem: T) {
        self.elements.binary_insert_no_dup(elem)
    }
    /// Lets you check wether a set has a parent (emerged from another set) or not.
    ///
    /// # Returns
    /// A boolean value which determines if the set is a subset of any other set.
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// use lib_rapid::math::sets::new_set;
    /// 
    /// let set = new_set!(0, 1, 2, 3, 4, 5, 6);
    /// let subset = Set::new_subset(&set, |x| x % 2 == 0);
    /// 
    /// assert_eq!(true, subset.has_superset());
    /// assert_eq!(false, set.has_superset());
    /// ```
    pub fn has_superset(&self) -> bool {
        self.superset.is_some()
    }
    /// Gets you the optional superset.
    ///
    /// # Returns
    /// A `Option<&Set<T>>`.
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// use lib_rapid::math::sets::new_set;
    /// 
    /// let set = new_set!(0, 1, 2, 3, 4, 5, 6);
    /// let subset = Set::new_subset(&set, |x| x % 2 == 0);
    /// 
    /// assert_eq!(&set, subset.get_superset().unwrap());
    /// ```
    pub fn get_superset(&self) -> Option<&Set<T>> {
        self.superset
    }
    /// Gets the cardinality of a set.
    ///
    /// # Returns
    /// A `usize`: `|self|`.
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// use lib_rapid::math::sets::new_set;
    /// 
    /// let set = new_set!(0, 1, 2, 3, 4, 5, 6);
    /// 
    /// assert_eq!(7, set.cardinality());
    /// ```
    pub fn cardinality(&self) -> usize {
        self.elements.len()
    }
    /// Lets you set the elements of a set.
    ///
    /// # Arguments
    /// * `vals` - The Vec to change the values to.
    ///
    /// # Returns
    /// Nothing. 
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// use lib_rapid::math::sets::new_set;
    /// 
    /// let mut set = new_set!(0, 1, 2, 3, 4, 5, 6);
    /// set.set_elements(&vec![0, 2, 4, 6]);
    /// 
    /// assert_eq!(&vec![0, 2, 4, 6], set.elements());
    /// ```
    pub fn set_elements(&mut self, vals: &[T]) {
        self.elements = vals.to_vec();
        self.elements.sort_unstable();
    }
    /// Lets you get the elements of a set.
    ///
    /// # Arguments
    /// * none
    ///
    /// # Returns
    /// A `&[T]` containing all elements. 
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// use lib_rapid::math::sets::new_set;
    /// 
    /// let mut set = new_set!(0, 1, 2, 3, 4, 5, 6);
    /// 
    /// assert_eq!(&vec![0, 1, 2, 3, 4, 5, 6], set.elements());
    /// ```
    pub fn elements(&self) -> &[T] {
        &self.elements
    }
}

/// Creates a new `Set` more elegantly from values.
///
/// # Returns
/// A new `Set`.
/// # Examples
/// ```
/// use lib_rapid::new_set;
/// use lib_rapid::math::sets::Set; 
/// 
/// let set: Set<i32> = new_set!(0,1,2,3,4,5,6,-1);
/// assert_eq!(set.to_string(), "{ -1; 0; 1; 2; 3; 4; 5; 6 }");
/// assert_eq!(set.to_full_string(), "{ -1; 0; 1; 2; 3; 4; 5; 6 }");
#[macro_export]
macro_rules! new_set {
    ( $( $a:expr ),* ) => {
        {
            use lib_rapid::compsci::general::BinaryInsert;
            let mut temp = Vec::new();
            $(
                temp.binary_insert_no_dup($a);
            )*
            Set::new(&temp)
        }
    };
}
pub use new_set;

use crate::compsci::general::BinaryInsert;

impl<T: ToString> Set<'_, T> {
    /// Lets you print a set with all its parents recursively.
    ///
    /// # Returns
    /// Nothing. 
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// let s:  Set<i32> = Set::new(&vec![0,1,2,3,4,5,6,7,8,9,10]);
    /// let s1: Set<i32> = Set::new_subset(&s, |x| x % 2 == 0);
    /// let s2: Set<i32> = Set::new_subset(&s1, |x| x == 4);
    /// 
    /// s2.full_println(); // Prints this set and the superset, see to_full_string.
    /// println!("{}", s2); // Only prints this set
    /// ```
    pub fn full_println(&self) {
        println!("{}", self.rec_to_string(&mut String::new()));
    }
    /// Converts a set with all subsets to a string.
    ///
    /// # Returns
    /// A String containing the result. 
    /// # Examples
    /// ```
    /// use lib_rapid::math::sets::Set;
    /// let s:  Set<i32> = Set::new(&vec![0,1,2,3,4,5,6,7,8,9,10]);
    /// let s1: Set<i32> = Set::new_subset(&s, |x| x % 2 == 0);
    /// let s2: Set<i32> = Set::new_subset(&s1, |x| x == 4);
    ///
    /// assert_eq!(s2.to_full_string(), "{ 4 } ⊆ { 0; 2; 4; 6; 8; 10 } ⊆ { 0; 1; 2; 3; 4; 5; 6; 7; 8; 9; 10 }".to_string());
    /// ```
    pub fn to_full_string(&self) -> String {
        self.rec_to_string(&mut String::new())
    }

    fn rec_to_string(&self, string: &mut String) -> String {
        string.push_str(&self.to_string()); // The child-set at the bottom
        match self.superset {
            Some(x)  => { string.push_str(" ⊆ "); // Add subset-character
                          x.rec_to_string(string); } // Recursively append parent sets
            None     => { }
        }
        string.to_string()
    }
}

// Indexing for Sets
impl<T> std::ops::Index<usize> for Set<'_, T> {
    type Output = T;
    fn index(&self, index: usize) -> &Self::Output {
        &self.elements[index]
    }
}

// Implement Printing
impl<T: ToString> std::fmt::Display for Set<'_, T> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        let mut res: String = String::from('{');
        for elem in &self.elements {
            res.push(' ');
            res += &elem.to_string();
            res.push(';');
        }
        res.pop();
        write!(f, "{} }}", res)
    }
}

// Implement Equality
impl<T: PartialEq> PartialEq for Set<'_, T> {
    fn eq(&self, other: &Self) -> bool {
        self.elements == other.elements
    }

    fn ne(&self, other: &Self) -> bool {
        self.elements != other.elements
    }
}

impl<T: Copy> Iterator for Set<'_, T> {
    type Item = T;

    fn next(&mut self) -> Option<Self::Item> {
        match self.elements.iter().next() {
            Some(x) => { return Some(*x); }
            None    => { return None; }
        }
    }
}