/**
The conversion algorithm to be chosen. Used by `temp_conversion`.
*/
pub enum TempConversion {
    CelsiusToFahrenheit,
    FahrenheitToCelsius,

    FahrenheitToKelvin,
    KelvinToFahrenheit,

    CelsiusToKelvin,
    KelvinToCelsius,
}
/**
 Trait for the cross sum of a given number.
 */
pub trait CrossSum<T> {
    /**
    Calculates the cross sum of a number.

    # Returns
    A `usize` containing the result.
    */
    fn cross_sum(&self) -> usize;
}
/**
 Trait for temperature conversion.
 */
pub trait TempConversionTrait {
    /**
    Performs a temperature conversion.

    # Arguments
    * `&self` - The value to be converted.
    * `mode` - The mode ( e.g. CelsiusToFahrenheit ).

    # Returns
    A `Self` containing the result.
    */
    fn temp_conversion(&self, mode: TempConversion) -> Self;
}
/**
Trait for left-shifting decimal-numbers.
*/
#[deprecated(note = "This feature is deprecated, as it has not been proven to be faster than multiplying by 10. Use at your own risk.")]
pub trait DecimalLeftShift<T> {
    /**
    Multiplies by 10 (shifts the decimal places to the left by 1) while being more efficient.

    # Returns
    The new shifted number.
    */
    #[deprecated(note = "This feature is deprecated, as it has not been proven to be faster than multiplying by 10. Use at your own risk.")]
    fn dec_lshift(&self) -> T;
}

/**
Trait for mapping numbers to another number range.
*/
pub trait MapToNumRange<T> {
    /**
    Maps a given number of a range onto another range.

    # Arguments
    * `self` - The value which is to be mapped.
    * `start1` - The original start value of the number range.
    * `end1` - The original end value of the number range.
    * `start2` - The new start value of the number range.
    * `end2` - The new start value of the number range.

    # Returns
    A number containing the new mapped value.

    # Examples
    ```
    use lib_rapid::math::rapidmath::MapToNumRange;

    let result: f32 = 5f32.map_to(0., 10., 0., 1.); // Original value 5 in the range from 0-10
    std::println!("{}", result.to_string()) // Prints "0.5"
    ```
    */
    fn map_to(&self, start1: T, end1: T, start2: T, end2: T) -> T;
}

impl<T: std::fmt::Display> CrossSum<T> for T {
    fn cross_sum(&self) -> usize {
        let self_str: String = self.to_string();
        self_str.chars().map(|c| c.to_digit(10).unwrap()).sum::<u32>() as usize
    }
}

impl<T: std::ops::Add<Output = T> + 
        std::ops::Sub<Output = T> + 
        std::ops::Mul<Output = T> + 
        std::ops::Div<Output = T> + 
        Copy> MapToNumRange<T> for T {
            fn map_to(&self, start1: T, end1: T, start2: T, end2: T) -> T {
                (start2 + (end2 - start2)) * ((*self - start1) / end1 - start1)
            }
        }

impl TempConversionTrait for f64 {
    fn temp_conversion(&self, mode: TempConversion) -> f64 {
        match mode {
            TempConversion::CelsiusToFahrenheit => { return self * 1.8 + 32.0; }
            TempConversion::CelsiusToKelvin     => { return self + 273.15; }
            TempConversion::FahrenheitToCelsius => { return (self - 32.0) / 1.8; }
            TempConversion::FahrenheitToKelvin  => { return (self - 32.0) / 1.8 + 273.15; }
            TempConversion::KelvinToCelsius     => { return self - 273.15; }
            TempConversion::KelvinToFahrenheit  => { return (self - 273.15) * 1.8 + 32.0; }
        }
    }
}

impl TempConversionTrait for f32 {
    fn temp_conversion(&self, mode: TempConversion) -> f32 {
        match mode {
            TempConversion::CelsiusToFahrenheit => { return self * 1.8 + 32.0; }
            TempConversion::CelsiusToKelvin     => { return self + 273.15; }
            TempConversion::FahrenheitToCelsius => { return (self - 32.0) / 1.8; }
            TempConversion::FahrenheitToKelvin  => { return (self - 32.0) / 1.8 + 273.15; }
            TempConversion::KelvinToCelsius     => { return self - 273.15; }
            TempConversion::KelvinToFahrenheit  => { return (self - 273.15) * 1.8 + 32.0; }
        }
    }
}