use super::InternalTokenRewriterResult;
use crate::blobs::{Blob, HasBlob};
use crate::containers::{ExternalPtr as Ptr, ExternalSharedByteList as SharedByteList};
use crate::Token;

/// Enum of what token rewriter should do with a token.
#[repr(C)]
pub struct RewriteAction {
    pub(crate) blob: Blob<RewriteAction>,
}

extern "C" {
    fn lib_ruby_parser__external__rewrite_action__drop(blob: *mut Blob<RewriteAction>);
    fn lib_ruby_parser__external__rewrite_action__is_drop(blob: *const Blob<RewriteAction>)
        -> bool;
    fn lib_ruby_parser__external__rewrite_action__is_keep(blob: *const Blob<RewriteAction>)
        -> bool;
}

impl Drop for RewriteAction {
    fn drop(&mut self) {
        unsafe { lib_ruby_parser__external__rewrite_action__drop(&mut self.blob) }
    }
}

impl RewriteAction {
    pub(crate) fn is_drop(&self) -> bool {
        unsafe { lib_ruby_parser__external__rewrite_action__is_drop(&self.blob) }
    }

    pub(crate) fn is_keep(&self) -> bool {
        unsafe { lib_ruby_parser__external__rewrite_action__is_keep(&self.blob) }
    }
}

impl PartialEq for RewriteAction {
    fn eq(&self, other: &Self) -> bool {
        if self.is_keep() {
            other.is_keep()
        } else if self.is_drop() {
            other.is_drop()
        } else {
            panic!("Unknown RewriteAction variant")
        }
    }
}

impl Eq for RewriteAction {}

impl std::fmt::Debug for RewriteAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        if self.is_drop() {
            write!(f, "Drop")
        } else if self.is_keep() {
            write!(f, "Keep")
        } else {
            panic!("Unknown RewriteAction variant")
        }
    }
}

/// Enum of what token rewriter should do with the state of the lexer
#[repr(C)]
pub struct LexStateAction {
    pub(crate) blob: Blob<LexStateAction>,
}

extern "C" {
    fn lib_ruby_parser__external__lex_state_action__drop(blob: *mut Blob<LexStateAction>);
    fn lib_ruby_parser__external__lex_state_action__is_set(
        blob: *const Blob<LexStateAction>,
    ) -> bool;
    fn lib_ruby_parser__external__lex_state_action__is_keep(
        blob: *const Blob<LexStateAction>,
    ) -> bool;
    fn lib_ruby_parser__external__lex_state_action__get_next_state(
        blob: *const Blob<LexStateAction>,
    ) -> i32;
}

impl Drop for LexStateAction {
    fn drop(&mut self) {
        unsafe { lib_ruby_parser__external__lex_state_action__drop(&mut self.blob) }
    }
}

impl LexStateAction {
    pub(crate) fn is_set(&self) -> bool {
        unsafe { lib_ruby_parser__external__lex_state_action__is_set(&self.blob) }
    }

    pub(crate) fn is_keep(&self) -> bool {
        unsafe { lib_ruby_parser__external__lex_state_action__is_keep(&self.blob) }
    }

    pub(crate) fn next_state(&self) -> i32 {
        unsafe { lib_ruby_parser__external__lex_state_action__get_next_state(&self.blob) }
    }
}

impl std::fmt::Debug for LexStateAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        if self.is_keep() {
            write!(f, "Keep")
        } else if self.is_set() {
            write!(f, "Set({})", self.next_state())
        } else {
            panic!("Unknown LexStateAction variant")
        }
    }
}

impl PartialEq for LexStateAction {
    fn eq(&self, other: &Self) -> bool {
        if self.is_keep() {
            other.is_keep()
        } else if self.is_set() {
            if other.is_set() {
                self.next_state() == other.next_state()
            } else {
                false
            }
        } else {
            panic!("Unknown LexStateAction variant")
        }
    }
}

impl Eq for LexStateAction {}

/// Output of the token rewriter
#[repr(C)]
pub struct TokenRewriterResult {
    pub(crate) blob: Blob<TokenRewriterResult>,
}

extern "C" {
    fn lib_ruby_parser__external__token_rewriter_result__drop(blob: *mut Blob<TokenRewriterResult>);
    fn lib_ruby_parser__external__token_rewriter_result__into_internal(
        blob: Blob<TokenRewriterResult>,
    ) -> InternalTokenRewriterResult;
}

impl Drop for TokenRewriterResult {
    fn drop(&mut self) {
        unsafe { lib_ruby_parser__external__token_rewriter_result__drop(&mut self.blob) }
    }
}

impl TokenRewriterResult {
    pub(crate) fn into_internal(self) -> InternalTokenRewriterResult {
        unsafe { lib_ruby_parser__external__token_rewriter_result__into_internal(self.into_blob()) }
    }
}

impl std::fmt::Debug for TokenRewriterResult {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.debug_struct("TokenRewriterResult").finish()
    }
}

/// Output of the token rewriter
#[repr(C)]
pub struct TokenRewriter {
    pub(crate) blob: Blob<TokenRewriter>,
}

extern "C" {
    fn lib_ruby_parser__external__token_rewriter__drop(blob: *mut Blob<TokenRewriter>);
    fn lib_ruby_parser__external__token_rewriter__call(
        blob: *const Blob<TokenRewriter>,
        token: Blob<Ptr<Token>>,
        input: Blob<SharedByteList>,
    ) -> Blob<TokenRewriterResult>;
}

impl Drop for TokenRewriter {
    fn drop(&mut self) {
        unsafe { lib_ruby_parser__external__token_rewriter__drop(&mut self.blob) }
    }
}

impl TokenRewriter {
    pub(crate) fn call(&self, token: Ptr<Token>, input: SharedByteList) -> TokenRewriterResult {
        TokenRewriterResult::from_blob(unsafe {
            lib_ruby_parser__external__token_rewriter__call(
                &self.blob,
                token.into_blob(),
                input.into_blob(),
            )
        })
    }
}
